class HorizontalSlicedSprite extends Sprite {
    constructor(bitmap) {
        super();
        this._sleft = 0;
        this._sright = 0;
        this._drawW = 0;
        this._bitmapRect = new Rectangle(0, 0, 0, 0);
        this._slicedBitmap = bitmap;
        this.createParts(bitmap);
        this._loadedBitmap = false;
        this._slicedBitmap.addLoadListener(this.onLoadedBitmap.bind(this));
    }
    createParts(bitmap) {
        // left
        {
            const sprite = new Sprite(bitmap);
            sprite.bitmap = bitmap;
            this.addChild(sprite);
            this._spriteLeft = sprite;
        }
        // center
        {
            const sprite = new TilingSprite();
            sprite.bitmap = bitmap;
            this.addChild(sprite);
            this._spriteCenter = sprite;
        }
        // right
        {
            const sprite = new Sprite(bitmap);
            sprite.bitmap = bitmap;
            this.addChild(sprite);
            this._spriteRight = sprite;
        }
        [this._spriteLeft, this._spriteCenter, this._spriteRight].forEach((s) => {
            s.visible = false;
        });
    }
    setSlice(left, right) {
        this._sleft = left;
        this._sright = right;
        this.refresh();
    }
    setSize(width) {
        this._drawW = width;
        this.refresh();
    }
    setBitmapRect(bitmapRect) {
        this._bitmapRect = bitmapRect;
        this.refresh();
    }
    onLoadedBitmap() {
        this._loadedBitmap = true;
        [this._spriteLeft, this._spriteCenter, this._spriteRight].forEach((s) => {
            s.visible = true;
        });
        this.refresh();
    }
    refresh() {
        if (!this._loadedBitmap)
            return;
        this._drawW = Math.max(this._drawW, this._sleft + this._sright);
        const cw = this._drawW - this._sleft - this._sright;
        const sx = this._bitmapRect.left;
        const sy = this._bitmapRect.top;
        const cfw = this._bitmapRect.width - this._sright - this._sleft;
        const height = this._bitmapRect.height;
        this._spriteLeft.setFrame(sx, sy, this._sleft, height);
        this._spriteCenter.setFrame(sx + this._sleft, sy, cfw, height);
        this._spriteCenter.move(this._sleft, 0, cw, height);
        this._spriteRight.setFrame(sx + this._sleft + cfw, sy, this._sright, height);
        this._spriteRight.move(this._sleft + cw, 0);
        this.setFrame(0, 0, this._drawW, height);
    }
}
class VerticalSlicedSprite extends Sprite {
    constructor(bitmap) {
        super();
        this._stop = 0;
        this._sbottom = 0;
        this._drawH = 0;
        this._slicedBitmap = bitmap;
        this.createParts(bitmap);
        this._loadedBitmap = false;
        this._slicedBitmap.addLoadListener(this.onLoadedBitmap.bind(this));
    }
    createParts(bitmap) {
        // top
        {
            const sprite = new Sprite(bitmap);
            sprite.bitmap = bitmap;
            this.addChild(sprite);
            this._spriteTop = sprite;
        }
        // center
        {
            const sprite = new TilingSprite();
            sprite.bitmap = bitmap;
            this.addChild(sprite);
            this._spriteCenter = sprite;
        }
        // bottom
        {
            const sprite = new Sprite(bitmap);
            sprite.bitmap = bitmap;
            this.addChild(sprite);
            this._spriteBottom = sprite;
        }
        [this._spriteTop, this._spriteCenter, this._spriteBottom].forEach((s) => {
            s.visible = false;
        });
    }
    setSlice(top, bottom) {
        this._stop = top;
        this._sbottom = bottom;
        this.refresh();
    }
    setSize(height) {
        this._drawH = height;
        this.refresh();
    }
    onLoadedBitmap() {
        this._loadedBitmap = true;
        [this._spriteTop, this._spriteCenter, this._spriteBottom].forEach((s) => {
            s.visible = true;
        });
        this.refresh();
    }
    refresh() {
        if (!this._loadedBitmap) {
            return;
        }
        this._drawH = Math.max(this._drawH, this.minSize());
        const ch = this._drawH - this._stop - this._sbottom;
        const cfh = this._slicedBitmap.height - this._stop - this._sbottom;
        const width = this._slicedBitmap.width;
        this._spriteTop.setFrame(0, 0, width, this._stop);
        this._spriteCenter.setFrame(0, this._stop, width, cfh);
        this._spriteCenter.move(0, this._stop, width, ch);
        this._spriteBottom.setFrame(0, this._stop + cfh, width, this._sbottom);
        this._spriteBottom.move(0, this._stop + ch);
        this.setFrame(0, 0, width, this._drawH);
    }
    minSize() {
        return this._stop + this._sbottom;
    }
}
class TouchMover {
    constructor(target) {
        this._target = target;
        this._pressed = false;
        this._prevPos = new Point(0, 0);
        this._offset = new Point(0, 0);
    }
    get isTouching() {
        return this._pressed;
    }
    get offset() {
        return this._offset;
    }
    update() {
        this._offset.set(0, 0);
        if (TouchInput.isReleased()) {
            this._pressed = false;
        }
        else if (this.isBeingTouched()) {
            if (TouchInput.isTriggered()) {
                this._pressed = true;
                this._prevPos.set(TouchInput.x, TouchInput.y);
            }
        }
        if (this._pressed && TouchInput.isMoved()) {
            const p1 = this._target.worldTransform.applyInverse(new Point(TouchInput.x, TouchInput.y));
            const p2 = this._target.worldTransform.applyInverse(new Point(this._prevPos.x, this._prevPos.y));
            this._offset.set(p1.x - p2.x, p1.y - p2.y);
            this._prevPos.x = TouchInput.x;
            this._prevPos.y = TouchInput.y;
        }
    }
    isBeingTouched() {
        const touchPos = new Point(TouchInput.x, TouchInput.y);
        const localPos = this._target.worldTransform.applyInverse(touchPos);
        return this.hitTest(localPos.x, localPos.y);
    }
    hitTest(x, y) {
        const target = this._target;
        const rect = new Rectangle(target.anchor.x * target.width, target.anchor.y * target.height, target.width, target.height);
        return rect.contains(x, y);
    }
}
class VerticalScrollBar extends Sprite {
    constructor(drawH, wholeH) {
        super();
        this._drawH = Math.min(drawH, wholeH);
        this._wholeH = wholeH;
        this._enableTouch = true;
        this._value = 0;
        this.createFrameSprite();
        this.createScrollSprite();
        this._mover = new TouchMover(this._scrollSprite);
        this.setFrame(0, 0, VerticalScrollBar.WIDTH, this._drawH);
    }
    createFrameSprite() {
        const frameBitmap = ImageManager.loadBitmap("img/system/", "scrollbar_2");
        const frameSprite = new VerticalSlicedSprite(frameBitmap);
        frameSprite.setSlice(16, 16);
        frameSprite.setSize(this._drawH);
        this.addChild(frameSprite);
        this._frameSprite = frameSprite;
    }
    createScrollSprite() {
        const bitmap = ImageManager.loadBitmap("img/system/", "scrollbar_1");
        const sprite = new VerticalSlicedSprite(bitmap);
        this._scrollSprite = sprite;
        sprite.setSlice(16, 16);
        sprite.setSize(this.scrollH);
        this.addChild(sprite);
    }
    update() {
        super.update();
        this.updateTouch();
    }
    updateTouch() {
        if (!this._enableTouch)
            return;
        if (!this.worldVisible)
            return;
        this._mover.update();
        const offset = this._mover.offset;
        this._scrollSprite.y = Math.min(Math.max(this._scrollSprite.y + offset.y, 0), this.scrollMoveMax);
        this._value = this._scrollSprite.y / this.scrollMoveMax;
    }
    /**
     *
     * @param value 0.0-1.0
     */
    setValue(value) {
        this._value = Math.min(Math.max(value, 0), 1);
        this._scrollSprite.move(0, this.scrollMoveMax * this._value);
    }
    get value() {
        return this._value;
    }
    get scrollH() {
        const rate = this._drawH / this._wholeH;
        return Math.max(Math.floor(this._drawH * rate), this._scrollSprite.minSize());
    }
    get scrollMoveMax() {
        return this._drawH - this.scrollH;
    }
    get isHolding() {
        return this._mover.isTouching;
    }
    get dir() {
        return this._mover.offset.y;
    }
    get enableTouch() {
        return this._enableTouch;
    }
    set enableTouch(value) {
        this._enableTouch = value;
    }
}
VerticalScrollBar.WIDTH = 22;
//# sourceMappingURL=ScrollBar.js.map