var Iz;
(function (Iz) {
    class GridKeyMapper {
        set(mapper) {
            Object.assign(this, mapper);
        }
    }
    Iz.GridKeyMapper = GridKeyMapper;
    class HandleController {
        constructor() {
            this._handlers = {};
        }
        setHandler(symbol, handler) {
            if (!this._handlers[symbol]) {
                this._handlers[symbol] = [];
            }
            this._handlers[symbol].push(handler);
        }
        isHandled(symbol) {
            return !!this._handlers[symbol];
        }
        callHandler(symbol, e) {
            if (this.isHandled(symbol)) {
                this._handlers[symbol].forEach((handler) => {
                    handler(e);
                });
            }
        }
        removeHandler(symbol, handler) {
            if (this.isHandled(symbol)) {
                this._handlers[symbol].remove(handler);
            }
        }
        clear() {
            this._handlers = {};
        }
    }
    Iz.GridEventType = {
        Ok: "ok",
        Cancel: "cancel",
        CursorMoved: "cursorMoved",
        CursorOutLeft: "cursorOutLeft",
        CursorOutUp: "cursorOutUp",
        CursorOutRight: "cursorOutRight",
        CursorOutDown: "cursorOutDown",
        CursorLoopLeft: "cursorLoopLeft",
        CursorLoopRight: "cursorLoopRight",
    };
    Iz.GridWrapMode = {
        None: 0,
        Wrap: 1, // 左端or右端にいる時に横移動した場合、前or次の段に移動
        Loop: 2, // 右端から左端、左端から右端に移動
    };
    Iz.GridVerticalWrapMode = {
        None: 0,
        Loop: 1,
    };
    class LoopGrid extends Sprite {
        constructor(options) {
            super();
            if (options.initDeselect) {
                this._index = this._prevIndex = LoopGrid.INIT_DESELECT_INDEX;
            }
            else {
                this._index = this._prevIndex = 0;
            }
            this._active = false;
            this._frameSize = {
                width: options.frameWidth,
                height: options.frameHeight,
            };
            this._emptyCellCreator = options.emptyCellCreator;
            this._cellUpdater = options.cellUpdater;
            this._padding = { left: 0, top: 0, right: 0, bottom: 0 };
            Object.assign(this._padding, options.padding);
            this._spacing = { left: 0, top: 0, right: 0, bottom: 0 };
            Object.assign(this._spacing, options.spacing);
            this._maxItems = options.maxItems;
            this._maxColumn = options.col;
            this._maxRow = options.row;
            this._wrap = Iz.GridWrapMode.None;
            this._verticalWrap = Iz.GridVerticalWrapMode.None;
            this._fixedCursorStep = false;
            this._gridKeyMapper = new GridKeyMapper();
            this._gridKeyMapper.set({
                ok: "ok",
                cancel: "cancel",
                left: "left",
                right: "right",
                up: "up",
                down: "down",
            });
            this._handler = new HandleController();
            this._step = 0;
            this._outofRangeCellNum = options.outofRangeCellNum ?? 1;
            this._cellStates = [];
            for (let i = 0; i < this._maxItems; i++) {
                this._cellStates.push({ selectable: true });
            }
            this._addStep = 0;
            this._prevStep = 0;
            this._targetStep = 0;
            this.width =
                this.blockWidth * this._maxColumn + this._padding.left + this._padding.right;
            this.height =
                this.blockHeight * this._maxRow + this._padding.top + this._padding.bottom;
            this.createContainter(this.width, this.height);
            // NOTE:
            // WindowLayerのマスク処理と競合する
            // 1列しかない時はマスクを作らないことで断定対応している
            if (this._outofRangeCellNum > 0) {
                this.createMask(this._padding.left, this._padding.top, this.blockWidth * this._maxColumn, this.blockHeight * this._maxRow);
            }
            this.createFrames();
            this.createScrollBar();
            // this.updateStep(0);
            this.onInitialized();
        }
        get frameSize() {
            return this._frameSize;
        }
        get index() {
            return this._index;
        }
        get prevIndex() {
            return this._prevIndex;
        }
        get loopRow() {
            return this._maxRow + this._outofRangeCellNum * 2;
        }
        get loopBlocks() {
            return this.loopRow * this._maxColumn;
        }
        get blockWidth() {
            return this._frameSize.width + this._spacing.left + this._spacing.right;
        }
        get blockHeight() {
            return this._frameSize.height + this._spacing.top + this._spacing.bottom;
        }
        get maxColumn() {
            return this._maxColumn;
        }
        get maxRow() {
            return this._maxRow;
        }
        get maxItems() {
            return this._maxItems;
        }
        set maxItems(value) {
            this._maxItems = value;
            this._cellStates = [];
            for (let i = 0; i < this._maxItems; i++) {
                this._cellStates.push({ selectable: true });
            }
            // this._index = Math.min(Math.max(this._index, 0), this._maxItems-1);
            // this._step = Math.min(Math.max(this._step, 0), this.limitStep);
            this._index = 0;
            this._prevStep = 0;
            this._step = 0;
            this.locate(this._index, this._step);
        }
        get itemsRow() {
            return Math.ceil(this._maxItems / this._maxColumn);
        }
        get itemsBlocks() {
            return this.itemsRow * this._maxColumn;
        }
        get wrap() {
            return this._wrap;
        }
        set wrap(value) {
            this._wrap = value;
        }
        get verticalWrap() {
            return this._verticalWrap;
        }
        set verticalWrap(value) {
            this._verticalWrap = value;
        }
        get fixedCursorStep() {
            return this._fixedCursorStep;
        }
        set fixedCursorStep(value) {
            this._fixedCursorStep = value;
        }
        get cellStates() {
            return this._cellStates;
        }
        get step() {
            return this._step;
        }
        get limitStep() {
            return Math.max(Math.ceil(this._maxItems / this._maxColumn) - this._maxRow, 0);
        }
        get enableScrollBar() {
            return this._enableScrollBar;
        }
        set enableScrollBar(value) {
            this._enableScrollBar = value;
            this._scrollBar.enableTouch = value;
            this._scrollBar.visible = value;
        }
        get wheelMove() {
            return this._wheelMove;
        }
        set wheelMove(value) {
            this._wheelMove = value;
        }
        /*
        get padding() {
            return this._padding;
        }
        get spacing() {
            return this._spacing;
        }
        set spacing(value: Partial<BoxSpace>) {
            Object.assign(this._spacing, value);
        }
        */
        getWidth(withScrollBar) {
            if (!withScrollBar) {
                return this.width;
            }
            return this.width + LoopGrid.SCROLLBAR_OFFSET_X + this._scrollBar.width;
        }
        frame(index = this._index) {
            const i = this.cellIndex(index);
            return this._container.children[i];
        }
        frames() {
            return this._container.children;
        }
        cell(index = this._index) {
            return this.frame(index).children[0];
        }
        cells() {
            return this.frames().map((frame) => frame.children[0]);
        }
        cellIndex(index) {
            const row = Math.floor(index / this._maxColumn) % this.loopRow;
            return row * this._maxColumn + (index % this._maxColumn);
        }
        indexFromCellIndex(cellIndex) {
            //
            // セルの配列インデクスから、アイテムのインデクスに変換する
            // NOTE:
            //
            // 例: maxColumn:2 maxRow:3 outofRangeCellNum:2 の時
            // 数値はCellインデクス カッコ内の数値がアイテムのインデクス
            //
            // [step=0]          [step=1]...
            //  10(10) 11(11)     12(12) 13(13)
            //  12(12) 13(13)     0(14) 1(15)
            // -----------       -----------
            //  0(0) 1(1)         2(2) 3(3)
            //  2(2) 3(3)         4(4) 5(5)
            //  4(4) 5(5)         6(6) 7(7)
            // -----------       -----------
            //  6(6) 7(7)         8(8) 9(9)
            //  8(8) 9(9)         10(10) 11(11)
            //
            const loopAddIndex = Math.floor((this.loopRow - 1 + this._step - Math.floor(cellIndex / this._maxColumn)) /
                this.loopRow) * this.loopBlocks;
            return cellIndex + loopAddIndex;
        }
        row() {
            return Math.floor(this._index / this._maxColumn) + 1;
        }
        getRowIndex(index) {
            return Math.floor(index / this._maxColumn);
        }
        startX() {
            return this._padding.left + this._spacing.left; // + this._frameSize.width/2;
        }
        blockX(index) {
            const column = index % this._maxColumn;
            return this.startX() + this.blockWidth * column;
        }
        startY() {
            return this._padding.top + this._spacing.top; // + this._frameSize.height/2;
        }
        blockY(index) {
            const offset = this.getStepOffset(index);
            return this.startY() + offset * this.blockHeight;
        }
        nextBlockY(index) {
            const offset = this.getStepOffsetStandard(this.index, this._targetStep);
            return this.startY() + offset * this.blockHeight;
        }
        createContainter(width, height) {
            const bitmap = new Bitmap(width, height);
            // bitmap.fillRect(0,0,width,height,"red"); // 領域確認用
            this._container = new Sprite(bitmap);
            this._container.relativeAnchor.enable = true;
            this.addChild(this._container);
        }
        createFrames() {
            let index = 0;
            for (let y = 0; y < this.loopRow; y++) {
                for (let x = 0; x < this._maxColumn; x++) {
                    const bitmap = new Bitmap(this._frameSize.width, this._frameSize.height);
                    // bitmap.fillRect(0,0,this._frameSize.width,this._frameSize.height,"yellow");  // 領域確認用
                    const frame = new Sprite(bitmap);
                    this._container.addChild(frame);
                    const cell = this._emptyCellCreator();
                    this.frame(index).addChild(cell);
                    ++index;
                }
            }
        }
        createScrollBar() {
            const height = this.height -
                this._padding.top -
                this._padding.bottom -
                this._spacing.top -
                this._spacing.bottom;
            const wholeHeight = this.blockHeight *
                Math.max(Math.ceil(this._maxItems / this._maxColumn), this._maxRow) -
                this._spacing.top -
                this._spacing.bottom;
            const scrollBar = new VerticalScrollBar(height, wholeHeight);
            scrollBar.move(this.width + LoopGrid.SCROLLBAR_OFFSET_X, this._padding.top + this._spacing.top);
            this.addChild(scrollBar);
            this._scrollBar = scrollBar;
            this._enableScrollBar = false;
            this.enableScrollBar = this._enableScrollBar;
        }
        createMask(x, y, width, height) {
            if (this._maskGraphics) {
                this.removeChild(this._maskGraphics);
            }
            const graphics = new PIXI.Graphics();
            graphics.beginFill(0xffffff);
            graphics.drawRect(x, y, width, height);
            graphics.endFill();
            this._container.mask = graphics;
            this._maskGraphics = graphics;
            this.addChild(this._maskGraphics);
        }
        updateMain() {
            this._prevIndex = this._index;
            this.processCursorMove();
            this.processScrollBar();
            this.processTouch();
            this.processHandling();
            this.processUpdatePosition();
            this._prevStep = this._step;
        }
        isMoving() {
            return this._addStep !== 0;
        }
        processTouch() {
            if (!this.isMovable())
                return;
            if (TouchInput.isHovered()) {
                const cells = this.cells();
                for (let i = 0; i < cells.length; i++) {
                    const cell = cells[i];
                    const index = this.indexFromCellIndex(i);
                    if (this.index === index)
                        continue;
                    if (!cell.isTouched())
                        continue;
                    if (!this.isDrawRange(index))
                        continue;
                    this.select(index);
                    break;
                }
            }
        }
        processUpdatePosition() {
            this._step += this._addStep;
            if (this._addStep > 0) {
                if (this._step >= this._targetStep) {
                    this._addStep = 0;
                    this.updateStep(this._targetStep);
                }
                else if (Math.floor(this._prevStep) !== Math.floor(this._step)) {
                    this.refresh(Math.floor(this._step));
                }
            }
            else if (this._addStep < 0) {
                if (this._step <= this._targetStep) {
                    this._addStep = 0;
                    this.updateStep(this._targetStep);
                }
                else if (Math.floor(this._prevStep) !== Math.floor(this._step)) {
                    this.refresh(Math.ceil(this._step));
                }
            }
            this.frames().forEach((frame, i) => {
                frame.x = this.blockX(i);
                frame.y = this.blockY(i);
            });
        }
        processCursorMove() {
            if (this.isMovable()) {
                const prevIndex = this._index;
                if (Input.isRepeated(this._gridKeyMapper.down)) {
                    this.cursorDown();
                }
                if (Input.isRepeated(this._gridKeyMapper.up)) {
                    this.cursorUp();
                }
                let touch = false;
                // right
                {
                    let res = false;
                    // let touch = false;
                    if (Input.isRepeated(this._gridKeyMapper.right)) {
                        res = true;
                        touch = false;
                    }
                    else if (this.wheelMove &&
                        !TouchInput.isHovered() &&
                        TouchInput.wheelY >= 20) {
                        res = true;
                        touch = true;
                    }
                    if (res) {
                        this.cursorRight(touch);
                    }
                }
                // left
                {
                    let res = false;
                    // let touch = false;
                    if (Input.isRepeated(this._gridKeyMapper.left)) {
                        res = true;
                        touch = false;
                    }
                    else if (this.wheelMove &&
                        !TouchInput.isHovered() &&
                        TouchInput.wheelY <= -20) {
                        res = true;
                        touch = true;
                    }
                    if (res) {
                        this.cursorLeft(touch);
                    }
                }
                if (this._index !== prevIndex) {
                    this.cursorMoved(touch);
                }
            }
        }
        processScrollBar() {
            if (this._scrollBar.isHolding) {
                // down
                if (this._scrollBar.dir > 0) {
                    const targetStep = this._step + 1;
                    const value = this._scrollBar.value;
                    const current = value * this.limitStep;
                    if (current >= targetStep) {
                        this.setStep(Math.floor(current));
                        this._scrollBar.setValue(value);
                    }
                }
                // up
                else if (this._scrollBar.dir < 0) {
                    const value = this._scrollBar.value;
                    const current = value * this.limitStep;
                    if (current < this._step) {
                        if (current <= this._step - 1) {
                            this.setStep(Math.ceil(current));
                            this._scrollBar.setValue(value);
                        }
                    }
                }
            }
        }
        cursorMoved(touch) {
            this.playCursorSound();
            const type = Iz.GridEventType.CursorMoved;
            const e = this.createGridEvent(type, touch);
            this._handler.callHandler(type, e);
        }
        createGridEvent(type, touch) {
            const e = {
                type: type,
                touch: touch,
            };
            return e;
        }
        isDrawRange(index) {
            if (index >= this.maxItems)
                return false;
            const offset = this.getStepOffset(index);
            return offset >= 0 && offset < this._maxRow;
        }
        getStepOffset(index) {
            return this.getStepOffsetStandard(index, this._step);
        }
        getStepOffsetStandard(index, baseStep) {
            // NOTE:
            // 初期位置(step=0)を基準とした相対位置を算出
            //
            // 例: maxColumn:2 maxRow:3 outofRangeCellNum:2 の時
            //   [相対位置]
            // ■ ■ -2
            // ■ ■ -1
            // ---
            // ■ ■ 0 <- step=0時の基準位置
            // ■ ■ 1
            // ■ ■ 2
            // ---
            // ■ ■ 3
            // ■ ■ 4
            //
            const step = baseStep % this.loopRow;
            const frameRowIndex = Math.floor(index / this._maxColumn) % this.loopRow;
            const offset = -step +
                frameRowIndex +
                Math.floor((step - frameRowIndex + this._maxRow + this._outofRangeCellNum - 1) /
                    this.loopRow) *
                    this.loopRow;
            return offset;
        }
        setStep(step) {
            const clmapStep = Math.min(Math.max(step, 0), this.limitStep);
            this.deselect();
            this._index = clmapStep * this._maxColumn;
            this.updateStep(clmapStep);
        }
        adjustPosition(maxItems) {
            // グリッドの長さを変えた際の位置を自動調節する.
            // if(index >= maxItems) {
            //     index = Math.max(0,maxItems-1);
            // }
            const nextMaxStep = Math.max(0, this.getRowIndex(maxItems - 1) - (this.maxRow - 1));
            const nextStep = Math.min(this._step, nextMaxStep);
            // const nextIndex = Math.(this._index,maxItems-1);
            const nextIndex = Math.min(Math.max(this._index, 0), Math.max(0, maxItems - 1));
            this._maxItems = maxItems;
            const newStates = new Array(this._maxItems);
            for (let i = 0; i < this._maxItems; i++) {
                if (this._cellStates[i]) {
                    newStates[i] = this._cellStates[i];
                }
                else {
                    newStates[i] = { selectable: true };
                }
            }
            this._cellStates = newStates;
            this.locate(nextIndex, nextStep);
            // console.log(`index:${this._index} step:${this._step}`);
        }
        locate(index, step) {
            this.deselect();
            this._index = index;
            this._prevIndex = index;
            this.updateStep(step);
            this.select(index);
            this.refresh();
        }
        updateStep(step) {
            // console.log(`step: ${this._step} -> ${step}`);
            this._step = step;
            this._targetStep = step;
            this.refresh();
            this.onUpdateStep(this._step);
            this._scrollBar.setValue(this.limitStep > 0 ? this._step / this.limitStep : 0);
            this._prevStep = this._step;
        }
        cursorDown() {
            let looped = false;
            let targetIndex = this._index < 0 ? 0 : this._index;
            const startIndex = targetIndex;
            const offset = this._maxColumn;
            while (true) {
                targetIndex += offset;
                // 同じアイテムを選択(1周)
                if (looped && targetIndex >= startIndex) {
                    break;
                }
                // 範囲外
                if (targetIndex >= this._maxItems) {
                    if (this._verticalWrap === Iz.GridVerticalWrapMode.Loop) {
                        // 押し続けている場合は一度止める
                        if (this._fixedCursorStep ||
                            Input.isLongPressed(this._gridKeyMapper.down)) {
                            break;
                        }
                        // 一番上のアイテムが選択されるように調整
                        targetIndex = targetIndex % this._maxColumn;
                        looped = true;
                    }
                    else {
                        // 最下段で選択できるものがあれば選択
                        const targetRowIndex = this.getRowIndex(targetIndex);
                        let existSelectable = false;
                        for (let i = this._maxItems - 1; i >= targetRowIndex * this._maxColumn; i--) {
                            if (this._cellStates[i].selectable) {
                                existSelectable = true;
                                targetIndex = i;
                                break;
                            }
                        }
                        if (!existSelectable) {
                            this.processEvent(Iz.GridEventType.CursorOutDown);
                            break;
                        }
                    }
                }
                // step固定
                const isNextStep = this.getRowIndex(targetIndex) - this._step >= this._maxRow;
                if (this._fixedCursorStep && isNextStep) {
                    break;
                }
                // 選択チェック
                if (this._cellStates[targetIndex].selectable) {
                    if (this._verticalWrap === Iz.GridVerticalWrapMode.Loop && looped) {
                        this.locate(targetIndex, Math.max(0, this.getRowIndex(targetIndex) + 1 - this._maxRow));
                    }
                    else {
                        this.select(targetIndex);
                        if (this.maxItems > this._maxRow * this._maxColumn) {
                            if (isNextStep) {
                                const stepOffset = this.row() - (this._step + this._maxRow);
                                this._addStep = stepOffset / 5;
                                this._targetStep = this._step + stepOffset;
                            }
                        }
                    }
                    break;
                }
            }
        }
        cursorUp() {
            let looped = false;
            let targetIndex = this._index < 0 ? 0 : this._index;
            const startIndex = targetIndex;
            const offset = -this._maxColumn;
            while (true) {
                targetIndex += offset;
                // 同じアイテムを選択(1周)
                if (looped && targetIndex <= startIndex) {
                    break;
                }
                // 範囲外
                if (targetIndex < 0) {
                    if (this._verticalWrap === Iz.GridVerticalWrapMode.Loop) {
                        if (this._fixedCursorStep || Input.isLongPressed(this._gridKeyMapper.up)) {
                            break;
                        }
                        // 一番下のアイテムが選択されるように調整
                        targetIndex += this.itemsBlocks;
                        if (targetIndex >= this._maxItems) {
                            targetIndex -= this._maxColumn;
                        }
                        looped = true;
                    }
                    else {
                        this.processEvent(Iz.GridEventType.CursorOutUp);
                        break;
                    }
                }
                // step固定
                const isNextStep = this.getRowIndex(targetIndex) < this._step;
                if (this._fixedCursorStep && isNextStep) {
                    break;
                }
                // 選択チェック
                if (this._cellStates[targetIndex].selectable) {
                    if (this._verticalWrap === Iz.GridVerticalWrapMode.Loop && looped) {
                        this.locate(targetIndex, Math.max(0, this.getRowIndex(targetIndex) + 1 - this._maxRow));
                    }
                    else {
                        this.select(targetIndex);
                        if (this.maxItems > this._maxRow * this._maxColumn) {
                            if (isNextStep) {
                                const stepOffset = this.row() - 1 - this._step;
                                this._addStep = stepOffset / 5;
                                this._targetStep = this._step + stepOffset;
                            }
                        }
                    }
                    break;
                }
            }
        }
        cursorRight(touch) {
            if (this._index === -1) {
                this.select(0);
                return;
            }
            const offset = 1;
            let targetIndex = this._index + offset;
            while (true) {
                const isNextRow = this.getRowIndex(this._index) !== this.getRowIndex(targetIndex);
                const isEndItem = targetIndex >= this._maxItems;
                if (this.wrap === Iz.GridWrapMode.Loop) {
                    if (isNextRow || isEndItem) {
                        let nextIndex = this.getRowIndex(this.index) * this.maxColumn;
                        while (nextIndex < this._index) {
                            if (this._cellStates[nextIndex].selectable) {
                                break;
                            }
                            nextIndex++;
                        }
                        if (nextIndex !== this._index) {
                            this.select(nextIndex);
                            this.processEvent(Iz.GridEventType.CursorLoopRight, touch);
                            break;
                        }
                        break;
                    }
                }
                else if (targetIndex >= this._maxItems) {
                    this.processEvent(Iz.GridEventType.CursorOutRight, touch);
                    break;
                }
                if (this.wrap === Iz.GridWrapMode.None && isNextRow) {
                    this.processEvent(Iz.GridEventType.CursorOutRight, touch);
                    break;
                }
                const isNextStep = this.getRowIndex(targetIndex) - this._step >= this._maxRow;
                if (this._fixedCursorStep && isNextStep) {
                    break;
                }
                if (this._cellStates[targetIndex].selectable) {
                    this.select(targetIndex);
                    if (this.maxItems > this._maxRow * this._maxColumn) {
                        if (isNextStep) {
                            const stepOffset = this.row() - (this._step + this._maxRow);
                            this._addStep = stepOffset / 5;
                            this._targetStep = this._step + stepOffset;
                        }
                    }
                    break;
                }
                targetIndex += offset;
            }
        }
        cursorLeft(touch) {
            if (this._index === -1) {
                this.select(0);
                return;
            }
            const offset = -1;
            let targetIndex = this._index + offset;
            while (true) {
                const isNextRow = this.getRowIndex(this._index) !== this.getRowIndex(targetIndex);
                const isEndItem = targetIndex < 0;
                if (this.wrap === Iz.GridWrapMode.Loop) {
                    if (isNextRow || isEndItem) {
                        let nextIndex = Math.min((this.getRowIndex(this.index) + 1) * this.maxColumn - 1, this._maxItems - 1);
                        while (nextIndex > this._index) {
                            if (this._cellStates[nextIndex].selectable) {
                                break;
                            }
                            nextIndex--;
                        }
                        if (nextIndex !== this._index) {
                            this.select(nextIndex);
                            this.processEvent(Iz.GridEventType.CursorLoopLeft, touch);
                            break;
                        }
                        break;
                    }
                }
                else if (targetIndex < 0) {
                    this.processEvent(Iz.GridEventType.CursorOutLeft, touch);
                    break;
                }
                if (this.wrap === Iz.GridWrapMode.None && isNextRow) {
                    this.processEvent(Iz.GridEventType.CursorOutLeft, touch);
                    break;
                }
                const isNextStep = this.getRowIndex(targetIndex) < this._step;
                if (this._fixedCursorStep && isNextStep) {
                    break;
                }
                if (this._cellStates[targetIndex].selectable) {
                    this.select(targetIndex);
                    if (this.maxItems > this._maxRow * this._maxColumn) {
                        if (isNextStep) {
                            const stepOffset = this.row() - 1 - this._step;
                            this._addStep = stepOffset / 5;
                            this._targetStep = this._step + stepOffset;
                        }
                    }
                    break;
                }
                targetIndex += offset;
            }
        }
        playCursorSound() {
            SoundManager.playCursor();
        }
        activate() {
            this._active = true;
            if (this._index >= 0) {
                this.select(this._index);
            }
            this._scrollBar.enableTouch = this._enableScrollBar;
        }
        deactivate() {
            this._active = false;
            this._scrollBar.enableTouch = false;
        }
        isActive() {
            return this._active;
        }
        setHandler(symbol, handler) {
            this._handler.setHandler(symbol, handler);
        }
        removeHandler(symbol, handler) {
            this._handler.removeHandler(symbol, handler);
        }
        clearHandler() {
            this._handler.clear();
        }
        isMovable() {
            return this.isActive() && !this.isMoving() && this._maxItems > 0;
        }
        isInputCancel() {
            return Input.isTriggered(this._gridKeyMapper.cancel) || TouchInput.isCancelled();
        }
        processHandling() {
            if (this.isActive() && !this.isMoving()) {
                // keyboard
                if (this.isOkEnabled() && Input.isTriggered(this._gridKeyMapper.ok)) {
                    return this.processEvent(Iz.GridEventType.Ok);
                }
                if (this.isCancelEnabled() && this.isInputCancel()) {
                    return this.processEvent(Iz.GridEventType.Cancel);
                }
                // touch
                if (this.isOkEnabled() && TouchInput.isClicked() && this.cell().isTouched()) {
                    return this.processEvent(Iz.GridEventType.Ok, true);
                }
            }
        }
        isOkEnabled() {
            return this._handler.isHandled(Iz.GridEventType.Ok) && this._index >= 0;
        }
        isCancelEnabled() {
            return this._handler.isHandled(Iz.GridEventType.Cancel);
        }
        updateInputData() {
            Input.update();
            TouchInput.update();
        }
        processEvent(type, touch) {
            const e = this.createGridEvent(type, !!touch);
            this._handler.callHandler(type, e);
            this.updateInputData();
        }
        select(index) {
            this.deselect();
            this._index = index;
            // console.log(`index: ${this._lastIndex} -> ${this._index}`);
        }
        deselect() {
            // this._lastIndex = this._index;
        }
        gridKeyMapper() {
            const mapper = new GridKeyMapper();
            mapper.set(this._gridKeyMapper);
            return mapper;
        }
        setGridKeyMapper(mapper) {
            this._gridKeyMapper = mapper;
        }
        onInitialized() { }
        onUpdateStep(step) { }
        // 表示を更新する
        refresh(step = this._step) {
            const beginIndex = Math.max(0, step - this._outofRangeCellNum) * this._maxColumn;
            const endIndex = beginIndex + this.loopBlocks;
            for (let i = beginIndex; i < endIndex; i++) {
                if (i < this._maxItems) {
                    this.frame(i).visible = true;
                    this._cellUpdater(this.cell(i), i);
                }
                else {
                    this.frame(i).visible = false;
                }
            }
        }
    }
    // NOTE:
    // [stepとCellの相対関係]
    //
    // 例: maxColumn:2 maxRow:3 outofRangeCellNum:2 の時
    // 数値はCellインデクス
    // 点線で囲われた部分のみマスク処理で見えるが、範囲外にもループ処理用にCellを用意
    //
    // [step=0]     [step=1]...
    //  10 11        12 13
    //  12 13        0 1
    // ------       ------
    //  0 1          2 3
    //  2 3          4 5
    //  4 5          6 7
    // ------       ------
    //  6 7          8 9
    //  8 9          10 11
    //
    LoopGrid.INIT_DESELECT_INDEX = -1;
    LoopGrid.SCROLLBAR_OFFSET_X = 8;
    Iz.LoopGrid = LoopGrid;
})(Iz || (Iz = {}));
(() => {
    "use strict";
    if (Utils.isOptionValid("test_menu")) {
        const _Window_MenuCommand_addOriginalCommands = Window_MenuCommand.prototype.addOriginalCommands;
        Window_MenuCommand.prototype.addOriginalCommands = function () {
            _Window_MenuCommand_addOriginalCommands.call(this);
            this.addCommand("[test] grid", "test_grid", true);
        };
        const _Scene_Menu_createCommandWindow = Scene_Menu.prototype.createCommandWindow;
        Scene_Menu.prototype.createCommandWindow = function () {
            _Scene_Menu_createCommandWindow.call(this);
            this._commandWindow.setHandler("test_grid", () => SceneManager.push(GridScene));
        };
    }
    const FrameSize = { x: 48, y: 48 };
    class GridSprite extends Sprite {
        constructor() {
            const bitmap = new Bitmap(FrameSize.x, FrameSize.y);
            super(bitmap);
        }
        set(index, selectable) {
            this.bitmap?.fillAll(selectable ? "yellow" : "grey");
            this.bitmap?.drawText(`${index}`, 0, 0, FrameSize.x, 26, "left");
        }
    }
    class GridScene extends Scene_Base {
        create() {
            super.create();
            this.createBg();
            this.creatGrid();
        }
        createBg() { }
        creatGrid() {
            this._items = [];
            for (let i = 0; i < 41; i++) {
                this._items[i] = i + 1;
            }
            const option = {
                emptyCellCreator: () => new GridSprite(),
                cellUpdater: this.updateCell.bind(this),
                maxItems: this._items.length,
                frameWidth: FrameSize.x,
                frameHeight: FrameSize.y,
                col: 3,
                row: 4,
            };
            this._grid = new Iz.LoopGrid(option);
            this._grid.wrap = Iz.GridWrapMode.Wrap;
            this._grid.verticalWrap = Iz.GridVerticalWrapMode.Loop;
            for (let i = 20; i < 28; i++) {
                this._grid.cellStates[i].selectable = false;
            }
            this.addChild(this._grid);
            this._grid.refresh();
            this._grid.activate();
        }
        update() {
            super.update();
            this._grid.updateMain();
            if (Input.isTriggered("cancel")) {
                this.popScene();
            }
            if (this._grid.prevIndex !== this._grid.index) {
                console.log(`index: ${this._grid.index}`);
            }
        }
        updateCell(cell, index) {
            cell.set(index, this._grid.cellStates[index].selectable);
        }
    }
})();
//# sourceMappingURL=Sprite_Grid.js.map