(() => {
    "use strict";
    Bitmap.prototype.setShadow = function (shadow) {
        this.context.shadowOffsetX = shadow.offsetX;
        this.context.shadowOffsetY = shadow.offsetY;
        this.context.shadowColor = shadow.color;
        this.context.shadowBlur = shadow.blur ?? 0;
    };
})();
/**
 * 簡易なテキストスプライト
 *
 */
class Sprite_Text extends Sprite {
    constructor(w = 32, h = 32) {
        super();
        this.align = "left"; // left,center,right
        this.createBitmap(w, h);
        this.setFontFace($gameSystem.mainFontFace());
        this.setFontSize($gameSystem.mainFontSize());
        this.setOutlineWidth(3);
        this.textWidth = w;
        this.textHeight = h;
    }
    get text() {
        return this._text;
    }
    createBitmap(w, h) {
        this.bitmap = new Bitmap(w, h);
    }
    setText(text, force = false) {
        this._text = text;
        this.redraw();
    }
    setFontFace(fontFace) {
        if (this.bitmap)
            this.bitmap.fontFace = fontFace;
    }
    setFontSize(size) {
        if (this.bitmap)
            this.bitmap.fontSize = size;
    }
    setNumberFontFace() {
        if (this.bitmap) {
            this.bitmap.fontFace = $gameSystem.numberFontFace();
        }
    }
    setAlign(align) {
        this.align = align;
    }
    setTextColor(color) {
        if (this.bitmap)
            this.bitmap.textColor = color;
    }
    setOutlineColor(color) {
        if (this.bitmap)
            this.bitmap.outlineColor = color;
    }
    setOutlineWidth(width) {
        if (this.bitmap)
            this.bitmap.outlineWidth = width;
    }
    measureTextWidth() {
        if (this.bitmap) {
            return this.bitmap.measureTextWidth(this._text);
        }
        return 0;
    }
    redraw() {
        // this.setupFont();
        if (this.bitmap) {
            this.bitmap.clear();
            this.bitmap.drawText(this._text, 0, 0, this.textWidth, this.textHeight, this.align);
        }
    }
    setFontSetting(key) {
        this.applyFontSetting(key);
    }
    applyFontSetting(key) {
        const bitmap = this.bitmap;
        if (!bitmap)
            return;
        const setting = Iz.Life.getFontSetting(key);
        if (!setting)
            return;
        Iz.Life.applyFont(bitmap, setting.font);
        if (setting.size) {
            this.setFontSize(setting.size);
        }
        this.setTextColor(setting.color);
        bitmap.outlineWidth = setting.outline_width;
        this.setOutlineColor(setting.outline_color);
        if (setting.use_shadow) {
            const shadow = {
                offsetX: setting.shadow_x,
                offsetY: setting.shadow_y,
                color: setting.shadow_color,
                blur: setting.shadow_blur,
            };
            bitmap.setShadow(shadow);
        }
    }
}
class TextState {
}
/**
 * テキストスプライト
 * Wnidow_Baseの文字列処理を移植
 *
 */
class Sprite_TextEx extends Sprite {
    constructor(w = 32, h = 32) {
        super();
        this._text = "";
        this.align = "left"; // left,center,right
        this.fontSettingKey = "";
        this._fontFace = $gameSystem.mainFontFace();
        this._fontSize = $gameSystem.mainFontSize();
        this.createBitmap(w, h);
        this.resetFontSettings();
    }
    get text() {
        return this._text;
    }
    createBitmap(w, h) {
        this.bitmap = new Bitmap(w, h);
    }
    setText(text) {
        this._text = text ? text : "";
        this.redraw();
    }
    setFontSize(size) {
        this._fontSize = size;
        if (this.bitmap)
            this.bitmap.fontSize = this._fontSize;
    }
    setTextAlign(x_, y_) {
        const x = x_ ?? 0;
        const y = y_ ?? 0;
        const textSize = this.textSizeEx(this._text);
        const textRect = new Rectangle(0, 0, textSize.width, textSize.height);
        const textAnchorPosition = {
            x: textRect.x + textRect.width * x,
            y: textRect.y + textRect.height * y,
        };
        this.anchor.set(textAnchorPosition.x / this._frame.width, textAnchorPosition.y / this._frame.height);
    }
    setOutlineWidth(width) {
        if (this.bitmap)
            this.bitmap.outlineWidth = width;
    }
    setShadow(offsetX, offsetY, color, blur) {
        const shadow = {
            offsetX: offsetX,
            offsetY: offsetY,
            color: color,
            blur: blur,
        };
        this.bitmap?.setShadow(shadow);
    }
    update() {
        super.update();
    }
    // redraw() {
    //     this.setupFont();
    //     this.bitmap?.clear();
    //     this.bitmap?.drawText(this.text,0,0,this.textWidth,this.textHeight,this.align);
    // }
    redraw() {
        this.bitmap?.clear();
        this.drawTextEx(this._text, 0, 0, this.textWidth(this._text));
    }
    setupFont() {
        // if(this.bitmap) this.bitmap.fontFace = this.fontFace;
        // if(this.bitmap) this.bitmap.fontSize = this.fontSize;
        // if(this.bitmap) this.bitmap.textColor = this.textColor();
        // if(this.bitmap) this.bitmap.outlineColor = this.outlineColor();
        // if(this.bitmap) this.bitmap.outlineWidth = this.outlineWidth();
    }
    drawTextEx(text, x, y, width) {
        this.resetFontSettings();
        const textState = this.createTextState(text, x, y, width);
        this.processAllText(textState);
        return textState.outputWidth;
    }
    drawIcon(iconIndex, x, y) {
        const iconBitmap = ImageManager.loadSystem("IconSet");
        const pw = ImageManager.iconWidth;
        const ph = ImageManager.iconHeight;
        const sx = (iconIndex % 16) * pw;
        const sy = Math.floor(iconIndex / 16) * ph;
        this.bitmap?.blt(iconBitmap, sx, sy, pw, ph, x, y);
    }
    processAllText(textState) {
        while (textState.index < textState.text.length) {
            this.processCharacter(textState);
        }
        this.flushTextState(textState);
    }
    textWidth(text) {
        if (this.bitmap) {
            return this.bitmap.measureTextWidth(text);
        }
        return 0;
    }
    resetFontSettings() {
        if (!this.bitmap)
            return;
        this.bitmap.fontFace = this._fontFace;
        this.bitmap.fontSize = this._fontSize;
        this.resetTextColor();
    }
    setFontSetting(key) {
        this.fontSettingKey = key;
    }
    applyFontSetting(key) {
        const bitmap = this.bitmap;
        if (!bitmap)
            return;
        const setting = Iz.Life.getFontSetting(key);
        if (!setting)
            return;
        Iz.Life.applyFont(bitmap, setting.font);
        if (setting.size) {
            this.setFontSize(setting.size);
        }
        this.changeTextColor(setting.color);
        bitmap.outlineWidth = setting.outline_width;
        this.changeOutlineColor(setting.outline_color);
        if (setting.use_shadow) {
            const shadow = {
                offsetX: setting.shadow_x,
                offsetY: setting.shadow_y,
                color: setting.shadow_color,
                blur: setting.shadow_blur,
            };
            bitmap.setShadow(shadow);
        }
    }
    resetTextColor() {
        this.applyFontSetting(this.fontSettingKey);
    }
    textSizeEx(text) {
        this.resetFontSettings();
        const textState = this.createTextState(text, 0, 0, 0);
        textState.drawing = false;
        this.processAllText(textState);
        return { width: textState.outputWidth, height: textState.outputHeight };
    }
    flushTextState(textState) {
        if (!this.bitmap)
            return;
        const text = textState.buffer;
        const rtl = textState.rtl;
        const width = this.textWidth(text);
        const height = textState.height;
        const x = rtl ? textState.x - width : textState.x;
        const y = textState.y;
        if (textState.drawing) {
            this.bitmap.drawText(text, x, y, width, height, this.align);
        }
        textState.x += rtl ? -width : width;
        textState.buffer = this.createTextBuffer(rtl);
        const outputWidth = Math.abs(textState.x - textState.startX);
        if (textState.outputWidth < outputWidth) {
            textState.outputWidth = outputWidth;
        }
        textState.outputHeight = y - textState.startY + height;
    }
    processCharacter(textState) {
        const c = textState.text[textState.index++];
        if (c.charCodeAt(0) < 0x20) {
            this.flushTextState(textState);
            this.processControlCharacter(textState, c);
        }
        else {
            textState.buffer += c;
        }
    }
    processControlCharacter(textState, c) {
        if (c === "\n") {
            this.processNewLine(textState);
        }
        if (c === "\x1b") {
            const code = this.obtainEscapeCode(textState);
            this.processEscapeCharacter(code, textState);
        }
    }
    processNewLine(textState) {
        textState.x = textState.startX;
        textState.y += textState.height;
        textState.height = this.calcTextHeight(textState);
    }
    obtainEscapeCode(textState) {
        const nonParamCode = Iz.obtainEscapeCodeNonParam(textState);
        if (nonParamCode) {
            return nonParamCode;
        }
        const regExp = /^[$.|^!><{}\\]|^[A-Z]+/i;
        const arr = regExp.exec(textState.text.slice(textState.index));
        if (arr) {
            textState.index += arr[0].length;
            return arr[0].toUpperCase();
        }
        return "";
    }
    obtainEscapeParam(textState) {
        const regExp = /^\[\d+\]/;
        const arr = regExp.exec(textState.text.slice(textState.index));
        if (arr) {
            textState.index += arr[0].length;
            return Number.parseInt(arr[0].slice(1));
        }
        return 0;
    }
    obtainEscapeCharacter(textState) {
        const regExp = /^\[\w+\]/;
        const s = textState.text.slice(textState.index);
        const arr = regExp.exec(s);
        if (arr) {
            textState.index += arr[0].length;
            return arr[0].substring(1, arr[0].length - 1);
        }
        return "";
    }
    processEscapeCharacter(code, textState) {
        switch (code) {
            case "FSET":
                this.setFontSetting(this.obtainEscapeCharacter(textState));
                break;
            case "FRESET":
                this.obtainEscapeParam(textState);
                this.resetTextColor();
                break;
            case "C":
                this.processColorChange(this.obtainEscapeParam(textState));
                break;
            case "I":
                this.processDrawIcon(this.obtainEscapeParam(textState), textState);
                break;
            case "PX":
                textState.x = this.obtainEscapeParam(textState);
                break;
            case "PY":
                textState.y = this.obtainEscapeParam(textState);
                break;
            case "FS":
                if (this.bitmap)
                    this.bitmap.fontSize = this.obtainEscapeParam(textState);
                break;
            case "{":
                this.makeFontBigger();
                break;
            case "}":
                this.makeFontSmaller();
                break;
            case "N":
                this.processNewLine(textState);
                break;
        }
    }
    processColorChange(colorIndex) {
        this.changeTextColor(ColorManager.textColor(colorIndex));
    }
    processDrawIcon(iconIndex, textState) {
        if (textState.drawing) {
            this.drawIcon(iconIndex, textState.x + 2, textState.y + 2);
        }
        textState.x += ImageManager.iconWidth + 4;
    }
    changeTextColor(color) {
        if (this.bitmap)
            this.bitmap.textColor = color;
    }
    changeOutlineColor(color) {
        if (this.bitmap)
            this.bitmap.outlineColor = color;
    }
    // ------------
    createTextState(text, x, y, width) {
        const rtl = Utils.containsArabic(text);
        const textState = new TextState();
        textState.text = this.convertEscapeCharacters(text);
        textState.index = 0;
        textState.x = rtl ? x + width : x;
        textState.y = y;
        textState.width = width;
        textState.height = this.calcTextHeight(textState);
        textState.startX = textState.x;
        textState.startY = textState.y;
        textState.rtl = rtl;
        textState.buffer = this.createTextBuffer(rtl);
        textState.drawing = true;
        textState.outputWidth = 0;
        textState.outputHeight = 0;
        return textState;
    }
    convertEscapeCharacters(text_) {
        /* eslint no-control-regex: 0 */
        let text = text_.replace(/\\/g, "\x1b");
        text = text.replace(/\x1b\x1b/g, "\\");
        while (text.match(/\x1bV\[(\d+)\]/gi)) {
            text = text.replace(/\x1bV\[(\d+)\]/gi, (_, p1) => `${$gameVariables.value(Number.parseInt(p1))}`);
        }
        text = text.replace(/\x1bN\[(\d+)\]/gi, (_, p1) => this.actorName(Number.parseInt(p1)));
        text = text.replace(/\x1bP\[(\d+)\]/gi, (_, p1) => this.partyMemberName(Number.parseInt(p1)));
        text = text.replace(/\x1bG/gi, TextManager.currencyUnit);
        text = text.replaceAll("\x1b" + "n", "\n");
        return text;
    }
    actorName(n) {
        const actor = n >= 1 ? $gameActors.actor(n) : null;
        return actor ? actor.name() : "";
    }
    partyMemberName(n) {
        const actor = n >= 1 ? $gameParty.members()[n - 1] : null;
        return actor ? actor.name() : "";
    }
    calcTextHeight(textState) {
        if (this.bitmap) {
            const lineSpacing = this.lineSpacing();
            const lastFontSize = this.bitmap.fontSize;
            const lines = textState.text.slice(textState.index).split("\n");
            const textHeight = this.maxFontSizeInLine(lines[0]) + lineSpacing;
            this.bitmap.fontSize = lastFontSize;
            return textHeight;
        }
        return this.lineHeight();
    }
    lineHeight() {
        return this._fontSize + this.lineSpacing();
    }
    lineSpacing() {
        return 16;
    }
    maxFontSizeInLine(line) {
        if (!this.bitmap)
            return this.lineHeight();
        let maxFontSize = this.bitmap.fontSize;
        const regExp = /\x1b({|}|FS)(\[(\d+)])?/gi;
        for (;;) {
            const array = regExp.exec(line);
            if (!array) {
                break;
            }
            const code = String(array[1]).toUpperCase();
            if (code === "{") {
                this.makeFontBigger();
            }
            else if (code === "}") {
                this.makeFontSmaller();
            }
            else if (code === "FS") {
                this.bitmap.fontSize = Number.parseInt(array[3]);
            }
            if (this.bitmap.fontSize > maxFontSize) {
                maxFontSize = this.bitmap.fontSize;
            }
        }
        return maxFontSize;
    }
    makeFontBigger() {
        if (!this.bitmap)
            return;
        if (this.bitmap.fontSize <= 96) {
            this.bitmap.fontSize += 12;
        }
    }
    makeFontSmaller() {
        if (!this.bitmap)
            return;
        if (this.bitmap.fontSize >= 24) {
            this.bitmap.fontSize -= 12;
        }
    }
    createTextBuffer(rtl) {
        // U+202B: RIGHT-TO-LEFT EMBEDDING
        return rtl ? "\u202B" : "";
    }
}
//# sourceMappingURL=Sprite_Text.js.map