/*:
 * @target MZ
 * @plugindesc シーケンス(状態)制御
 *
 * @help
 * State Pattern の提供・支援
 * (※Stateという名前が既にMZで使用されているため、Sequenceという名前で実装)
 *
 *
 */
// sequence
var Iz;
(function (Iz) {
    Iz.SequenceResponse = {
        Complete: 0,
        FinishBehaviour: 1,
    };
    class BehaviourSequence {
        constructor() {
            this._list = new Map();
            this._key = "";
            this._frame = 0;
            this._requestFinish = false;
        }
        get key() {
            return this._key;
        }
        get current() {
            return this._list.get(this._key);
        }
        getInitSequenceKey() {
            let res = "";
            this._list.forEach((info, key) => {
                if (info.init) {
                    res = key;
                }
            });
            return res;
        }
        onStart(arg) {
            this.change(this.getInitSequenceKey(), arg);
        }
        onUpdate() {
            if (this._requestFinish) {
                return true;
            }
            ++this._frame;
            if (!this.current)
                return false;
            const end = this.current.sequence.onUpdate();
            if (end) {
                const result = this.current.sequence.getResult();
                if (result.response === Iz.SequenceResponse.Complete) {
                    const nextInfo = this.current.flowList.find((flow) => flow.dstSequence === result.nextSequence);
                    if (nextInfo) {
                        this.change(result.nextSequence, result.output);
                    }
                    else {
                        // 初期シーケンスに移行
                        const initKey = this.getInitSequenceKey();
                        this.change(initKey, {});
                    }
                }
                else if (result.response === Iz.SequenceResponse.FinishBehaviour) {
                    return true;
                }
                else {
                    // 不正な値 初期シーケンスに移行
                    const initKey = this.getInitSequenceKey();
                    this.change(initKey, {});
                }
            }
            return false;
        }
        onFinish() {
            this._requestFinish = false;
            this.change("", {});
        }
        getResult() {
            return {
                response: Iz.SequenceResponse.Complete,
                nextSequence: "",
                output: {},
            };
        }
        set(key, sequence, flowList) {
            const firstSequence = this._list.size === 0;
            this._list.set(key, {
                sequence: sequence,
                flowList: flowList,
                init: firstSequence,
            });
        }
        finish() {
            this._requestFinish = true;
        }
        suspend(arg) {
            // 初期シーケンスに戻る
            this.change(this.getInitSequenceKey(), arg ?? {});
        }
        change(key, arg) {
            if (this.current) {
                this.current.sequence.onFinish();
            }
            const next = this._list.get(key);
            if (next) {
                next.sequence.onStart(arg);
                this._key = key;
            }
            else {
                this._key = "";
            }
            this._frame = 0;
            // console.log("sequence: " + key);
        }
        isBusy() {
            return this._frame <= 0 || this._requestFinish;
        }
    }
    Iz.BehaviourSequence = BehaviourSequence;
    class RootSequence extends BehaviourSequence {
        start(arg) {
            this.onStart(arg);
        }
        update() {
            this.onUpdate();
        }
    }
    Iz.RootSequence = RootSequence;
})(Iz || (Iz = {}));
//# sourceMappingURL=Sequence.js.map