/*:
 * @target MZ
 * @plugindesc 農場プラグイン
 * @help
 *
 *
 * @command Newday
 * @text Newday
 * @desc 日付経過して植物の成長を行う
 *
 *
 * @command HarvestFromThisEvent
 * @text HarvestFromThisEvent
 * @desc 収穫する　呼び出したイベントを対象
 *
 *
 * @command CreateObject
 * @text CreateObject
 * @desc 耕地にオブジェクト(木や石)を生成 削除する時はRemoveObjectSelfを呼んでください
 *
 * @arg mapId
 * @text mapId
 * @desc マップID 省略すると現在のマップが対象になります
 *
 * @arg x
 * @text x
 * @desc x座標(整数値)
 *
 * @arg y
 * @text y
 * @desc y座標(整数値)
 *
 * @arg templateId
 * @text templateId
 * @desc テンプレートIDもしくは名前　数値を指定するとIDとして解釈されます
 *
 *
 * @command CreateObjectRandom
 * @text CreateObjectRandom
 * @desc 耕地にオブジェクト(木や石)をランダム生成
 *
 * @arg mapId
 * @text mapId
 * @desc マップID 省略すると現在のマップが対象になります
 *
 * @arg templateId
 * @text templateId
 * @desc テンプレートIDもしくは名前　数値を指定するとIDとして解釈されます
 *
 * @arg ranges
 * @text ranges
 * @desc 生成範囲  複数指定可  座標が重複した箇所は１回分としてカウントされます
 * @type struct<RandomPoint>[]
 *
 *
 * @command RemoveObjectSelf
 * @text RemoveObjectSelf
 * @desc 耕地から自身を削除
 *
 */
/*~struct~RandomPoint:
 * @param left
 * @text left
 * @desc 左上x座標
 * @type number
 *
 * @param top
 * @text top
 * @desc 左上y座標
 * @type number
 *
 * @param right
 * @text right
 * @desc 右下x座標
 * @type number
 *
 * @param bottom
 * @text bottom
 * @desc 右下y座標
 * @type number
 *
 */
var Iz;
(function (Iz) {
    var Life;
    (function (Life) {
        Life.PluginName = "farm";
        Life.HoeList = [];
        Life.WaterPotList = [];
        Life.GloveList = [];
        Life.SprinklerList = [];
        Life.HoeTag = "hoe";
        Life.WaterPotTag = "waterpot";
        Life.GloveTag = "glove";
        Life.SeedTag = "seed";
        Life.SprinklerTag = "sprinkler";
        Life.WaterRegionId = 3;
        Life.FarmRegionId = 2;
        Life.PlantTemplateMapId = 94;
        const PlantItemList = new Map();
        Life.FarmEventTag = "farm";
        Life.FarmEventTag_Soil = "soil";
        Life.FarmEventTag_Plant = "plant";
        // チャージフレーム数設定
        const HoeActionFrameList = [...$parameter.charge.hoe_frames];
        const PlantActionFrameList = [...$parameter.charge.plant_frames];
        const WaterPotActionFrameList = [...$parameter.charge.water_pot_frames];
        // チャージ消費HP係数設定(Lv.1~Lv.5)
        const HoeChargeHpRate = [...$parameter.charge.hoe_consume_hp_rate];
        const PlantChargeHpRate = [...$parameter.charge.plant_consume_hp_rate];
        const WaterPotChargeHpRate = [...$parameter.charge.water_pot_consume_hp_rate];
        Life.HoeAnimationId = 1;
        Life.PlantAnimationId = 7;
        Life.WaterPotAnimationId = 6;
        Life.FarmObjectLabelName = "farm";
        const FarmEventMapId = 101;
        const FarmMapId = 1;
        const farmSyncMaps = new Map();
        farmSyncMaps.set(FarmEventMapId, FarmMapId);
        function getFarmSrcMap(mapId) {
            return farmSyncMaps.get(mapId) ?? mapId;
        }
        Life.getFarmSrcMap = getFarmSrcMap;
        // 土地の状態(0:何もなし 1:耕作済 2:濡れた状態)
        Life.Soil = {
            None: 0,
            Hoed: 1,
            Wetted: 2,
        };
        Life.FarmConsumeHpType = {
            Hoe: 1,
            Plant: 2,
            WaterPot: 3,
        };
        function getRegion(x, y) {
            const RegionLayer = 5;
            const region = $gameMap.data()[x + (y + RegionLayer * $gameMap.height()) * $gameMap.width()];
            return region;
        }
        Life.getRegion = getRegion;
        function isRegion(x, y, targetRegion) {
            const region = getRegion(x, y);
            return region === targetRegion;
        }
        Life.isRegion = isRegion;
        function isHoe(itemId) {
            return Life.HoeList.includes(itemId);
        }
        Life.isHoe = isHoe;
        function isWaterPot(itemId) {
            return Life.WaterPotList.includes(itemId);
        }
        Life.isWaterPot = isWaterPot;
        function getPlantItem(itemId) {
            const plantItemId = PlantItemList.get(itemId);
            if (plantItemId) {
                return $dataItems[plantItemId];
            }
            return undefined;
        }
        function isSeed(itemId) {
            const plant = getPlantItem(itemId);
            return !!plant;
        }
        Life.isSeed = isSeed;
        function getPlantId(seedItemId) {
            const plant = getPlantItem(seedItemId);
            return plant?.id ?? 0;
        }
        Life.getPlantId = getPlantId;
        function isGlove(itemId) {
            return Life.GloveList.includes(itemId);
        }
        Life.isGlove = isGlove;
        function isSprinkler(itemId) {
            return Life.SprinklerList.includes(itemId);
        }
        Life.isSprinkler = isSprinkler;
        function getSprinkerId(itemId) {
            const item = $dataItems[itemId];
            if (!item)
                return 0;
            const sprinklerId = Number(item.meta[Life.SprinklerTag]);
            if (!sprinklerId)
                return 0;
            return sprinklerId;
        }
        Life.getSprinkerId = getSprinkerId;
        function getSprinklerItemId(sprinklerId) {
            const s = Life.SprinklerList.find((itemId) => itemId && Number($dataItems[itemId].meta[Life.SprinklerTag]) === sprinklerId);
            return s ?? 0;
        }
        Life.getSprinklerItemId = getSprinklerItemId;
        function getSprinkerTargets(x, y, sprinklerId) {
            // スプリンクラーIDに応じた対象となるマス
            const list = [];
            $dataSprinklerRange
                .filter((s) => s && s.id === sprinklerId)
                .forEach((range) => {
                for (let ty = y + range.y; ty < y + range.y + range.height; ty++) {
                    for (let tx = x + range.x; tx < x + range.x + range.width; tx++) {
                        if (!list.some((t) => t.x === tx && t.y === ty)) {
                            list.push({ x: tx, y: ty });
                        }
                    }
                }
            });
            return list;
        }
        Life.getSprinkerTargets = getSprinkerTargets;
        function isSoilEvent(event) {
            if (!event)
                return false;
            return event.event().meta[Life.FarmEventTag] === Life.FarmEventTag_Soil;
        }
        Life.isSoilEvent = isSoilEvent;
        function isPlantEvent(event) {
            if (!event)
                return false;
            return event.event().meta[Life.FarmEventTag] === Life.FarmEventTag_Plant;
        }
        Life.isPlantEvent = isPlantEvent;
        const plantGrowthMap = new Map();
        function setupGrowthMap() {
            // 植物のレベルごとの必要成長度を算出する
            plantGrowthMap.clear();
            $dataCrops
                .filter((data) => data)
                .forEach((data) => {
                const growthList = [0];
                const growthLevelMax = Math.max(1, data.growth_level_max);
                if (growthLevelMax <= 1) {
                    plantGrowthMap.set(data.id, growthList);
                    return;
                }
                const growthMax = Math.max(1, data.growth_max);
                for (let i = 1; i < growthLevelMax - 1; i++) {
                    const targetGrowth = Math.floor((growthMax / (growthLevelMax - 1)) * i);
                    growthList.push(targetGrowth);
                }
                growthList.push(growthMax);
                plantGrowthMap.set(data.id, growthList);
            });
        }
        function getPlantLevel(itemId, growth) {
            const growthList = plantGrowthMap.get($dataCrops[itemId]?.id ?? 0) ?? [];
            for (let i = growthList.length - 1; i >= 0; i--) {
                if (growth >= growthList[i]) {
                    return i + 1;
                }
            }
            return 1;
        }
        Life.getPlantLevel = getPlantLevel;
        function hoeLevel() {
            return Iz.Life.calcToolLevel(Life.HoeList);
        }
        Life.hoeLevel = hoeLevel;
        function waterPotLevel() {
            return Iz.Life.calcToolLevel(Life.WaterPotList);
        }
        Life.waterPotLevel = waterPotLevel;
        function gloveLevel() {
            return Iz.Life.calcToolLevel(Life.GloveList);
        }
        Life.gloveLevel = gloveLevel;
        function getFarmConsumeHp(type, chargeLevel) {
            switch (type) {
                case Life.FarmConsumeHpType.Hoe: {
                    const base = $parameter.charge.hoe_consume_hp_base;
                    const rate = HoeChargeHpRate[chargeLevel - 1] ?? 1;
                    return base * rate;
                }
                case Life.FarmConsumeHpType.Plant: {
                    const base = $parameter.charge.plant_consume_hp_base;
                    const rate = PlantChargeHpRate[chargeLevel - 1] ?? 1;
                    return base * rate;
                }
                case Life.FarmConsumeHpType.WaterPot: {
                    const base = $parameter.charge.water_pot_consume_hp_base;
                    const rate = WaterPotChargeHpRate[chargeLevel - 1] ?? 1;
                    return base * rate;
                }
                default:
                    return 1;
            }
        }
        Life.getFarmConsumeHp = getFarmConsumeHp;
        function getWaterLimit() {
            const id = Life.WaterPotList[waterPotLevel() - 1];
            if (id) {
                return Number($dataItems[id].meta.water);
            }
            const DefaultValue = 10;
            return DefaultValue;
        }
        Life.getWaterLimit = getWaterLimit;
        function isRainy() {
            return Life.getWeatherType() === Life.WeatherType.Rainy;
        }
        Life.isRainy = isRainy;
        function getGrowthValueWetted() {
            return $parameter.farm.growth_per_day_water ?? 0;
        }
        Life.getGrowthValueWetted = getGrowthValueWetted;
        // 半歩移動の場合にも対応
        function getTakeWaterPosition(character) {
            const x = character.x;
            const y = character.y;
            const d = character.direction();
            if (character.isMapPassable(x, y, d))
                return undefined;
            if (d === 2 || d === 8) {
                const targetY = character.isHalfPosY(y)
                    ? $gameMap.roundHalfYWithDirection(y, d)
                    : $gameMap.roundYWithDirection(y, d);
                const targetTileXList = [];
                if (character.isHalfPosX(x)) {
                    targetTileXList.push(Math.floor(x));
                    targetTileXList.push(Math.ceil(x));
                }
                else {
                    targetTileXList.push(x);
                }
                for (let i = 0; i < targetTileXList.length; i++) {
                    const targetX = targetTileXList[i];
                    if (Iz.Life.isRegion(targetX, targetY, Iz.Life.WaterRegionId)) {
                        return { x: targetX, y: targetY };
                    }
                }
            }
            else if (d === 4 || d === 6) {
                const targetX = character.isHalfPosX(x)
                    ? $gameMap.roundHalfXWithDirection(x, d)
                    : $gameMap.roundXWithDirection(x, d);
                const targetTileYList = [];
                if (character.isHalfPosY(y)) {
                    targetTileYList.push(Math.floor(y));
                    targetTileYList.push(Math.ceil(y));
                }
                else {
                    targetTileYList.push(y);
                }
                for (let i = 0; i < targetTileYList.length; i++) {
                    const targetY = targetTileYList[i];
                    if (Iz.Life.isRegion(targetX, targetY, Iz.Life.WaterRegionId)) {
                        return { x: targetX, y: targetY };
                    }
                }
            }
            return undefined;
        }
        Life.getTakeWaterPosition = getTakeWaterPosition;
        function getHoeTherePosition(excludeEvents = true) {
            const positions = Life.getTherePositions($gamePlayer.x, $gamePlayer.y, $gamePlayer.direction());
            for (let i = 0; i < positions.length; ++i) {
                const position = positions[i];
                if (excludeEvents && $gameMap.eventsXyUnitNt(position.x, position.y).length > 0) {
                    continue;
                }
                if (Iz.Life.isRegion(position.x, position.y, Iz.Life.FarmRegionId)) {
                    return [position];
                }
            }
            return [];
        }
        Life.getHoeTherePosition = getHoeTherePosition;
        function getPlantTherePosition() {
            if (Iz.Life.getThereEvent()) {
                return [];
            }
            const events = Iz.Life.getThereEvents(Iz.Life.EventPriotity.Below);
            const soilEvent = events.find((e) => {
                if (Iz.Life.isSoilEvent(e)) {
                    const crop = $gameFarm.getCrop(e.x, e.y, $gameMap.mapId());
                    if (crop && crop.plantItemId === 0) {
                        return true;
                    }
                }
                return false;
            });
            return soilEvent ? [{ x: soilEvent.x, y: soilEvent.y }] : [];
        }
        Life.getPlantTherePosition = getPlantTherePosition;
        function getWaterPotTherePosition() {
            if (Iz.Life.getThereEvent()) {
                return [];
            }
            const events = Iz.Life.getThereEvents(Iz.Life.EventPriotity.Below);
            const soilEvent = events.find((e) => {
                if (Iz.Life.isSoilEvent(e)) {
                    const crop = $gameFarm.getCrop(e.x, e.y, $gameMap.mapId());
                    if (crop && crop.soil === Iz.Life.Soil.Hoed) {
                        return true;
                    }
                }
                return false;
            });
            return soilEvent ? [{ x: soilEvent.x, y: soilEvent.y }] : [];
        }
        Life.getWaterPotTherePosition = getWaterPotTherePosition;
        function getFarmLevelingEvent() {
            const soilEvent = Iz.Life.getThereEvents(Iz.Life.EventPriotity.Below).find((e) => e.eventType() === Iz.Life.EventType.Soil);
            return soilEvent;
        }
        Life.getFarmLevelingEvent = getFarmLevelingEvent;
        function getFarmActionPositionList(basePos, level) {
            return Iz.Life.getActionPositionList(basePos, level);
        }
        function getHoePositionList(basePos, level) {
            if (level <= 1) {
                return getHoeTherePosition();
            }
            return getFarmActionPositionList(basePos, level);
        }
        function checkHoeAction(chargeLevel, checkOnly) {
            const basePos = Iz.Life.getUsePosition($gamePlayer.x, $gamePlayer.y, $gamePlayer.direction());
            const positionList = getHoePositionList(basePos, chargeLevel);
            let successCount = 0;
            positionList.forEach((position) => {
                if (checkOnly) {
                    if ($gameFarm.executableHoe(position.x, position.y, $gameMap.mapId())) {
                        successCount++;
                    }
                }
                else {
                    if ($gameFarm.execHoe(position.x, position.y, $gameMap.mapId())) {
                        successCount++;
                    }
                }
            });
            return successCount;
        }
        function executableHoeAction(chargeLevel) {
            return checkHoeAction(chargeLevel, true) > 0;
        }
        Life.executableHoeAction = executableHoeAction;
        function execHoeAction(chargeLevel) {
            return checkHoeAction(chargeLevel, false);
        }
        Life.execHoeAction = execHoeAction;
        const HoeAnimation = {
            begin: "hoe_charging",
            exec: "hoe_exec",
        };
        class HoeAction extends Iz.Life.ChargeAction {
            constructor() {
                super(HoeActionFrameList);
            }
            onExec() {
                const hoedCount = execHoeAction(this.getChargeLevel());
                const success = hoedCount > 0;
                if (success) {
                    Life.Status.addWorkLevelExpFarm(Life.Status.WorkFarmType.Hoe);
                    Life.Stats.farm.addHoedCount(hoedCount);
                }
                return success;
            }
            getChargeType() {
                return Life.ActionType.Hoe;
            }
            getToolLevel() {
                return Iz.Life.hoeLevel();
            }
            getPositionList(basePos, level) {
                return getHoePositionList(basePos, level);
            }
            getConsumeToolHp() {
                return getFarmConsumeHp(Life.FarmConsumeHpType.Hoe, this.getChargeLevel());
            }
            animation() {
                return HoeAnimation;
            }
        }
        Life.HoeAction = HoeAction;
        class PlantAction extends Iz.Life.ChargeAction {
            constructor() {
                super(PlantActionFrameList);
            }
            onExec() {
                const itemId = Iz.Life.ItemSlot.itemId;
                if (!Iz.Life.isSeed(itemId)) {
                    return false;
                }
                const basePos = Iz.Life.getUsePosition($gamePlayer.x, $gamePlayer.y, $gamePlayer.direction());
                const positionList = this.getPositionList(basePos, this.getChargeLevel());
                let successCount = 0;
                positionList.forEach((position) => {
                    if ($gameFarm.execPlant(position.x, position.y, itemId)) {
                        successCount++;
                    }
                });
                const success = successCount > 0;
                if (success) {
                    // Status.addWorkLevelExp(Status.WorkType.Farm, this.getConsumeHp());
                    Life.Stats.farm.addPlantedCount(successCount);
                }
                return success;
            }
            getChargeType() {
                return Life.ActionType.Plant;
            }
            getToolLevel() {
                return Iz.Life.gloveLevel();
            }
            getPositionList(basePos, level) {
                if (level <= 1) {
                    return getPlantTherePosition();
                }
                return getFarmActionPositionList(basePos, level);
            }
            getConsumeToolHp() {
                return getFarmConsumeHp(Iz.Life.FarmConsumeHpType.Plant, this.getChargeLevel());
            }
        }
        Life.PlantAction = PlantAction;
        const WaterPotAnimation = {
            begin: "waterpot_charging",
            exec: "waterpot_exec",
        };
        class WaterPotAction extends Iz.Life.ChargeAction {
            constructor() {
                super(WaterPotActionFrameList);
            }
            onExec() {
                if ($gameFarm.water <= 0) {
                    Iz.System.notify(Iz.Text.get("SYSTEM_WATER_SHORTAGE"));
                    return false;
                }
                const basePos = Iz.Life.getUsePosition($gamePlayer.x, $gamePlayer.y, $gamePlayer.direction());
                const positionList = this.getPositionList(basePos, this.getChargeLevel());
                let successCount = 0;
                positionList.forEach((p) => {
                    if ($gameFarm.execWater(p.x, p.y)) {
                        successCount++;
                    }
                });
                const success = successCount > 0;
                if (success) {
                    Life.Status.addWorkLevelExpFarm(Life.Status.WorkFarmType.WaterPot);
                    Life.Stats.farm.addWateredCount(successCount);
                }
                return success;
            }
            getChargeType() {
                return Life.ActionType.WaterPot;
            }
            getToolLevel() {
                return Iz.Life.waterPotLevel();
            }
            getPositionList(basePos, level) {
                if (level <= 1) {
                    return getWaterPotTherePosition();
                }
                return getFarmActionPositionList(basePos, level);
            }
            getConsumeToolHp() {
                return getFarmConsumeHp(Iz.Life.FarmConsumeHpType.WaterPot, this.getChargeLevel());
            }
            animation() {
                return WaterPotAnimation;
            }
        }
        Life.WaterPotAction = WaterPotAction;
        function getDataKey(x, y, mapId) {
            return `${mapId}_${x}_${y}`;
        }
        Life.getDataKey = getDataKey;
        Life.Temp = {
            lastHoeEventId: 0,
        };
        function initializeFarm() {
            Iz.Life.getToolItems(Life.HoeTag).forEach((data) => {
                Life.HoeList[data.level - 1] = data.id;
            });
            Iz.Life.getToolItems(Life.WaterPotTag).forEach((data) => {
                Life.WaterPotList[data.level - 1] = data.id;
            });
            Iz.Life.getToolItems(Life.GloveTag).forEach((data) => {
                Life.GloveList[data.level - 1] = data.id;
            });
            Iz.Life.getToolItems(Life.SprinklerTag).forEach((data) => {
                Life.SprinklerList[data.level - 1] = data.id;
            });
            PlantItemList.clear();
            $dataItems
                .filter((data) => data)
                .forEach((data) => {
                const seedId = Number(data.meta[Life.SeedTag]);
                if (seedId) {
                    PlantItemList.set(seedId, data.id);
                }
            });
            setupGrowthMap();
        }
        Life.initializeFarm = initializeFarm;
    })(Life = Iz.Life || (Iz.Life = {}));
})(Iz || (Iz = {}));
(function (Iz) {
    var Life;
    (function (Life) {
        var FarmUtil;
        (function (FarmUtil) {
            // 耕作済み農地
            function getHoedEvents() {
                return $gameFarm.crops.filter((info) => info.soil !== Iz.Life.Soil.None);
            }
            FarmUtil.getHoedEvents = getHoedEvents;
            // 種植え済み農地
            function getPlantedEvents() {
                return $gameFarm.crops.filter((info) => info.plantItemId > 0);
            }
            FarmUtil.getPlantedEvents = getPlantedEvents;
            // 水やり済み農地
            function getWettedEvents() {
                return $gameFarm.crops.filter((info) => info.soil === Iz.Life.Soil.Wetted);
            }
            FarmUtil.getWettedEvents = getWettedEvents;
            // 耕作地の水やり一括設定
            function setWettedAllFarms(wetted, applyInGameMap = true) {
                $gameFarm.crops.forEach((crop) => {
                    let isChange = false;
                    if (wetted && crop.soil === Iz.Life.Soil.Hoed) {
                        crop.soil = Iz.Life.Soil.Wetted;
                        isChange = true;
                    }
                    else if (!wetted && crop.soil === Iz.Life.Soil.Wetted) {
                        crop.soil = Iz.Life.Soil.Hoed;
                        isChange = true;
                    }
                    if (!isChange)
                        return;
                    if (!applyInGameMap)
                        return;
                    if (crop.mapId !== $gameMap.mapId())
                        return;
                    const event = $gameMap.eventsXy(crop.x, crop.y).find((e) => Life.isSoilEvent(e));
                    if (!event)
                        return;
                    $gameSelfSwitches.setValue([$gameMap.mapId(), event.eventId(), "A"], !wetted);
                    $gameSelfSwitches.setValue([$gameMap.mapId(), event.eventId(), "B"], wetted);
                });
            }
            FarmUtil.setWettedAllFarms = setWettedAllFarms;
            function setSprinkler(x, y, mapId, itemId) {
                const sprinklerId = Life.getSprinkerId(itemId);
                if (sprinklerId <= 0) {
                    return;
                }
                $gameFarm.setSprinkler({ x: x, y: y, mapId: mapId, sprinklerId: sprinklerId });
            }
            FarmUtil.setSprinkler = setSprinkler;
            function getPlantedNum() {
                return $gameFarm.crops.filter((c) => c.plantItemId > 0).length;
            }
            FarmUtil.getPlantedNum = getPlantedNum;
        })(FarmUtil = Life.FarmUtil || (Life.FarmUtil = {}));
    })(Life = Iz.Life || (Iz.Life = {}));
})(Iz || (Iz = {}));
(function (Iz) {
    var Life;
    (function (Life) {
        class FarmSprinkers {
            constructor() {
                this._sprinklers = new Map();
            }
            get list() {
                return [...this._sprinklers.values()];
            }
            get(x, y, mapId) {
                return this._sprinklers.get(Iz.Life.getDataKey(x, y, mapId));
            }
            set(data) {
                this._sprinklers.set(Iz.Life.getDataKey(data.x, data.y, data.mapId), data);
            }
            pick(x, y, mapId) {
                const data = this.get(x, y, mapId);
                if (!data)
                    return;
                this._sprinklers.delete(Iz.Life.getDataKey(x, y, mapId));
                if ($gameMap.mapId() === mapId) {
                    const event = $gameMap
                        .eventsXyNt(x, y)
                        .find((e) => e.eventType() === Iz.Life.EventType.Sprinkler);
                    if (event) {
                        $gameMap.eraseEvent(event.eventId());
                    }
                }
                $gameParty.gainItem($dataItems[Iz.Life.getSprinklerItemId(data.sprinklerId)], 1);
            }
            copy(srcMap, dstMap) {
                const dstSprinklers = this.list
                    .filter((s) => s.mapId === srcMap)
                    .map((s) => {
                    const sprinkler = Object.assign({}, s);
                    sprinkler.mapId = dstMap;
                    return sprinkler;
                });
                const excludeDstSprinklers = this.list.filter((s) => s.mapId !== dstMap);
                const newSprinklers = [...excludeDstSprinklers, ...dstSprinklers];
                newSprinklers.sort((a, b) => {
                    if (a.mapId !== b.sprinklerId)
                        return a.mapId - b.sprinklerId;
                    if (a.y !== b.y)
                        return a.y - b.y;
                    if (a.x !== b.x)
                        return a.x - b.x;
                    return a.sprinklerId - b.sprinklerId;
                });
                this._sprinklers.clear();
                newSprinklers.forEach((s) => {
                    const key = Iz.Life.getDataKey(s.x, s.y, s.mapId);
                    this._sprinklers.set(key, s);
                });
            }
            newday() { }
            clear() {
                this._sprinklers.clear();
            }
            setData(list) {
                this._sprinklers.clear();
                list.forEach((data) => {
                    const newData = {
                        x: 0,
                        y: 0,
                        mapId: 0,
                        sprinklerId: 0,
                    };
                    Iz.Obj.apply(newData, data);
                    this._sprinklers.set(Iz.Life.getDataKey(newData.x, newData.y, newData.mapId), newData);
                });
            }
            getData() {
                return [...this._sprinklers.values()];
            }
        }
        Life.FarmSprinkers = FarmSprinkers;
        function getSpawnObjects(points, mapId) {
            return Iz.Life.Spawn.objects.filter((object) => {
                if (object.mapId !== mapId)
                    return false;
                const col = Iz.Life.Spawn.getCollider(object.x, object.y, object.templateId);
                return points.some((p) => HM.isInColliderExcludeBorder(col, p.x, p.y));
            });
        }
        Life.getSpawnObjects = getSpawnObjects;
    })(Life = Iz.Life || (Iz.Life = {}));
})(Iz || (Iz = {}));
class Game_Farm {
    constructor(data) {
        this._crops = new Map();
        this._sprinkler = new Iz.Life.FarmSprinkers();
        this._water = 1;
        if (data) {
            data.crops.forEach((c) => {
                this._crops.set(Iz.Life.getDataKey(c.x, c.y, c.mapId), c);
            });
            this._water = data.water;
            this._sprinkler.setData(data.sprinklers);
        }
    }
    get crops() {
        return [...this._crops.values()];
    }
    get water() {
        return this._water;
    }
    get sprinklers() {
        return this._sprinkler.list;
    }
    getCrop(x, y, mapId) {
        return this._crops.get(Iz.Life.getDataKey(x, y, mapId));
    }
    executableHoe(x, y, mapId) {
        if (!Iz.Life.isRegion(x, y, Iz.Life.FarmRegionId))
            return false;
        // オブジェクトチェック
        const col = new HM.Collider({ x: x, y: y, expansionArea: HM.getExpansionArea({}) });
        const collidedObjects = Iz.Life.getSpawnObjects([col.base, col.leftTop, col.leftBottom, col.rightTop, col.rightBottom], mapId);
        if (collidedObjects.length > 0)
            return false;
        // イベントチェック
        if ($gameMap.eventsXyUnitNt(x, y).length > 0)
            return false;
        // すでに耕作されている場合は実行しない
        const data = this.getCrop(x, y, mapId);
        if (data)
            return false;
        return true;
    }
    execHoe(x, y, mapId = $gameMap.mapId()) {
        if (!this.executableHoe(x, y, mapId)) {
            return false;
        }
        const data = {
            mapId: mapId,
            x: x,
            y: y,
            soil: Iz.Life.Soil.None,
            plantItemId: 0,
            growingDays: 1,
            growth: 0,
        };
        data.soil = Iz.Life.isRainy() ? Iz.Life.Soil.Wetted : Iz.Life.Soil.Hoed;
        this._crops.set(Iz.Life.getDataKey(data.x, data.y, data.mapId), data);
        // イベントに反映
        const event = $gameMap.createSoilEvent(x, y);
        if (event) {
            $gameSelfSwitches.setValue([mapId, event.eventId(), data.soil === Iz.Life.Soil.Wetted ? "B" : "A"], true);
            Iz.Life.Temp.lastHoeEventId = event.eventId();
            $gameVariables.setValue(this.hoeEventVariableId(), event.eventId());
            const Delay = 1;
            $gameTemp.requestDelayAnimation([event], Iz.Life.HoeAnimationId, Delay, false);
        }
        return true;
    }
    executablePlant(x, y, mapId, seedItemId) {
        if (!Iz.Life.isRegion(x, y, Iz.Life.FarmRegionId))
            return false;
        // 耕作されていない場合は実行しない
        const data = this.getCrop(x, y, mapId);
        if (!data)
            return false;
        if (!(data.soil === Iz.Life.Soil.Hoed || data.soil === Iz.Life.Soil.Wetted))
            return false;
        // 植えられている場合は実行しない
        if (data.plantItemId !== 0)
            return false;
        const plantItemId = Iz.Life.getPlantId(seedItemId);
        if (!plantItemId)
            return false;
        return true;
    }
    execPlant(x, y, seedItemId) {
        const mapId = $gameMap.mapId();
        if (!this.executablePlant(x, y, mapId, seedItemId))
            return false;
        const data = this.getCrop(x, y, mapId);
        const plantItemId = Iz.Life.getPlantId(seedItemId);
        const event = $gameMap.createPlantEvent(x, y, plantItemId);
        if (!event)
            return false;
        data.plantItemId = plantItemId;
        data.growingDays = 1;
        data.growth = 0;
        const seedNum = $gameParty.numItems($dataItems[seedItemId]);
        if (seedNum <= 0)
            return false;
        $gameParty.gainItem($dataItems[seedItemId], -1);
        // イベントに反映
        if (event) {
            const Delay = 1;
            $gameTemp.requestDelayAnimation([event], Iz.Life.PlantAnimationId, Delay, false);
        }
        return true;
    }
    executableWater(x, y, mapId) {
        if (this._water <= 0) {
            return false;
        }
        // 耕作地チェック
        const data = this.getCrop(x, y, mapId);
        if (!data)
            return false;
        if (data.soil !== Iz.Life.Soil.Hoed)
            return false;
        return true;
    }
    execWater(x, y) {
        if (!this.executableWater(x, y, $gameMap.mapId()))
            return false;
        const data = this.getCrop(x, y, $gameMap.mapId());
        if (!data)
            return false;
        data.soil = Iz.Life.Soil.Wetted;
        // イベントに反映
        const events = $gameMap.eventsXy(x, y);
        const event = events.find((e) => Iz.Life.isSoilEvent(e));
        if (event) {
            $gameSelfSwitches.setValue([$gameMap.mapId(), event.eventId(), "B"], true);
            const Delay = 1;
            $gameTemp.requestDelayAnimation([event], Iz.Life.WaterPotAnimationId, Delay, false);
        }
        this._water--;
        this.setWater(this._water);
        return true;
    }
    setWater(waterNum) {
        const limit = Iz.Life.getWaterLimit();
        this._water = Math.min(Math.max(waterNum, 0), limit);
    }
    getSprinkler(x, y, mapId) {
        return this._sprinkler.get(x, y, mapId);
    }
    settableSprinkler(x, y, mapId) {
        if (!Iz.Life.isRegion(x, y, Iz.Life.FarmRegionId))
            return false;
        // オブジェクトチェック
        const col = new HM.Collider({ x: x, y: y, expansionArea: HM.getExpansionArea({}) });
        const collidedObjects = Iz.Life.getSpawnObjects([col.base, col.leftTop, col.leftBottom, col.rightTop, col.rightBottom], mapId);
        if (collidedObjects.length > 0)
            return false;
        // イベントチェック
        if ($gameMap.eventsXyUnitNt(x, y).some((e) => e.eventType() !== Iz.Life.EventType.Soil))
            return false;
        // 更地の場合のみ可
        const data = this.getCrop(x, y, mapId);
        if (data?.plantItemId)
            return false;
        return true;
    }
    setSprinkler(data) {
        if (!this.settableSprinkler(data.x, data.y, data.mapId)) {
            return false;
        }
        let success = true;
        if ($gameMap.mapId() === data.mapId) {
            const event = $gameMap.createSprinklerEvent(data.x, data.y, data.sprinklerId);
            if (!event) {
                success = false;
            }
        }
        if (success) {
            this._sprinkler.set(data);
        }
        return success;
    }
    pickSprinkler(x, y, mapId) {
        this._sprinkler.pick(x, y, mapId);
    }
    applySprinkler(data) {
        const targets = Iz.Life.getSprinkerTargets(data.x, data.y, data.sprinklerId);
        const targetCrops = this.crops
            .filter((c) => c.mapId === data.mapId)
            .filter((c) => targets.find((t) => t.x === c.x && t.y === c.y))
            .filter((c) => c.soil === Iz.Life.Soil.Hoed);
        targetCrops.forEach((c) => {
            c.soil = Iz.Life.Soil.Wetted;
            this.refreshEvent(c.x, c.y, c.mapId);
        });
    }
    harvest(x, y, mapId) {
        const target = this.getCrop(x, y, mapId);
        if (!target)
            return false;
        const itemId = target.plantItemId;
        if (itemId === 0)
            return false;
        const baseAmount = 1;
        const amount = Iz.Life.Status.calcHarvestAmount(baseAmount);
        $gameParty.gainItem($dataItems[itemId], amount);
        Iz.System.notifyGainItem(itemId, amount);
        Iz.Life.Status.addWorkLevelExpFarm(Iz.Life.Status.WorkFarmType.Harvest, itemId);
        Iz.Life.Status.addGoddessPointCrops(itemId);
        Iz.Life.Stats.farm.addHarvest(itemId, amount);
        this.removePlant(x, y, mapId);
        return true;
    }
    leveling(x, y, mapId, force = false) {
        const target = this.getCrop(x, y, mapId);
        if (!target)
            return false;
        if (target.plantItemId !== 0 && !force)
            return false;
        this.removeCrop(x, y, mapId);
    }
    hoeEventVariableId() {
        return $parameter.farm.last_hoe_event_variable_id ?? 0;
    }
    lastHoeEvent() {
        return $gameMap._events[Iz.Life.Temp.lastHoeEventId];
    }
    createObjectRandom(ranges, templateId, mapId) {
        let res = { success: false, index: -1, event: undefined };
        const p = Iz.Life.getRandomPoint(ranges, this.sprinklers);
        if (p) {
            res = this.createObject(p.x, p.y, templateId, mapId);
        }
        return res;
    }
    createObject(x, y, templateId, mapId) {
        let res = { success: false, index: -1, event: undefined };
        const template = Iz.Life.Spawn.getTemplate(templateId);
        if (!template) {
            return res;
        }
        // 作物とは重複しない
        const isHitInfo = this.crops
            .filter((info) => info.mapId === mapId && info.plantItemId > 0)
            .some((info) => {
            const infoCol = Iz.Life.Spawn.getCollider(info.x, info.y);
            const objectCol = Iz.Life.Spawn.getCollider(x, y, templateId);
            return HM.isCollidedExcludeBorder(infoCol, objectCol);
        });
        if (isHitInfo) {
            return res;
        }
        // 他オブジェクトと重ならない
        const isObjectHit = Iz.Life.Spawn.objects
            .filter((object) => object.mapId === mapId)
            .some((object) => {
            if (!Iz.Life.Spawn.getTemplate(object.templateId))
                return res;
            const col1 = Iz.Life.Spawn.getCollider(object.x, object.y, object.templateId);
            const col2 = Iz.Life.Spawn.getCollider(x, y, templateId);
            return HM.isCollidedExcludeBorder(col1, col2);
        });
        if (isObjectHit) {
            return res;
        }
        res = Iz.Life.Spawn.createObject({
            mapId: mapId,
            x: x,
            y: y,
            templateId: templateId,
            labels: {
                [Iz.Life.FarmObjectLabelName]: "object",
            },
        });
        if (res.success) {
            // 耕作地は削除
            this.crops
                .filter((info) => info.mapId === mapId)
                .filter((info) => {
                const infoCol = Iz.Life.Spawn.getCollider(info.x, info.y);
                const objectCol = Iz.Life.Spawn.getCollider(x, y, templateId);
                return HM.isCollidedExcludeBorder(infoCol, objectCol);
            })
                .forEach((info) => {
                $gameFarm.removeCrop(info.x, info.y, info.mapId);
            });
        }
        return res;
    }
    removeObject(index) {
        Iz.Life.Spawn.removeObject(index);
    }
    removePlant(x, y, mapId) {
        const info = this.getCrop(x, y, mapId);
        if (!info)
            return;
        // info.soil = Iz.Life.isRainy() ? Iz.Life.Soil.Wetted : Iz.Life.Soil.Hoed;
        info.plantItemId = 0;
        info.growingDays = 0;
        info.growth = 0;
        const events = $gameMap.eventsXy(x, y);
        const plantEvent = events.find((e) => Iz.Life.isPlantEvent(e));
        if (plantEvent) {
            $gameMap.eraseEvent(plantEvent.eventId());
        }
        this.refreshEvent(x, y, mapId);
    }
    removeCrop(x, y, mapId) {
        const info = this.getCrop(x, y, mapId);
        if (!info)
            return;
        this._crops.delete(Iz.Life.getDataKey(x, y, mapId));
        if ($gameMap.mapId() === mapId) {
            const events = $gameMap.eventsXy(x, y);
            const plantEvent = events.find((e) => Iz.Life.isPlantEvent(e));
            if (plantEvent) {
                $gameMap.eraseEvent(plantEvent.eventId());
            }
            const soilEvent = events.find((e) => Iz.Life.isSoilEvent(e));
            if (soilEvent) {
                $gameMap.eraseEvent(soilEvent.eventId());
            }
        }
    }
    refreshEvent(x, y, mapId) {
        if (mapId !== $gameMap.mapId())
            return;
        const crop = this.getCrop(x, y, mapId);
        if (!crop)
            return;
        const events = $gameMap.eventsXy(x, y);
        const soilEvent = events.find((e) => Iz.Life.isSoilEvent(e));
        if (soilEvent) {
            $gameSelfSwitches.setValue([$gameMap.mapId(), soilEvent.eventId(), "A"], crop.soil !== Iz.Life.Soil.Wetted);
            $gameSelfSwitches.setValue([$gameMap.mapId(), soilEvent.eventId(), "B"], crop.soil === Iz.Life.Soil.Wetted);
        }
    }
    getFarmData() {
        const data = {
            crops: this.crops,
            water: this._water,
            sprinklers: this.sprinklers,
        };
        return data;
    }
    copyCrops(srcMap, dstMap) {
        const dstCrops = this.crops
            .filter((c) => c.mapId === srcMap)
            .map((c) => {
            const crop = Object.assign({}, c);
            crop.mapId = dstMap;
            return crop;
        });
        // コピー先に存在しているものと置き換え
        const excludeDstCrops = this.crops.filter((c) => c.mapId !== dstMap);
        const newCrops = [...excludeDstCrops, ...dstCrops];
        newCrops.sort((a, b) => {
            if (a.mapId !== b.mapId)
                return a.mapId - b.mapId;
            if (a.y !== b.y)
                return a.y - b.y;
            if (a.x !== b.x)
                return a.x - b.x;
            if (a.plantItemId !== b.plantItemId)
                return a.plantItemId - b.plantItemId;
            if (a.soil !== b.soil)
                return a.soil - b.soil;
            if (a.growth !== b.growth)
                return a.growth - b.growth;
            return a.growingDays - b.growingDays;
        });
        this._crops.clear();
        newCrops.forEach((c) => {
            const key = Iz.Life.getDataKey(c.x, c.y, c.mapId);
            this._crops.set(key, c);
        });
    }
    copySprinklers(srcMap, dstMap) {
        this._sprinkler.copy(srcMap, dstMap);
    }
    copyObjects(srcMap, dstMap) {
        // コピー先のマップのfarmオブジェクトは削除
        for (let i = 0; i < Iz.Life.Spawn.objects.length; i++) {
            const o = Iz.Life.Spawn.objects[i];
            if (o.mapId !== dstMap)
                continue;
            if (!o.includeLabel(Iz.Life.FarmObjectLabelName))
                continue;
            Iz.Life.Spawn.removeObject(i);
        }
        Iz.Life.Spawn.objects
            .filter((o) => o.mapId === srcMap)
            .filter((o) => o.includeLabel(Iz.Life.FarmObjectLabelName))
            .forEach((o) => {
            Iz.Life.Spawn.createObject({
                mapId: dstMap,
                x: o.x,
                y: o.y,
                templateId: o.templateId,
                labels: Object.assign({}, o.labels),
            });
        });
    }
    copyMap(srcMap, dstMap) {
        this.copyCrops(srcMap, dstMap);
        this.copySprinklers(srcMap, dstMap);
        this.copyObjects(srcMap, dstMap);
    }
    clearMap(mapId) {
        const newCrops = this.crops.filter((c) => c.mapId !== mapId);
        this._crops.clear();
        newCrops.forEach((c) => {
            this._crops.set(Iz.Life.getDataKey(c.x, c.y, c.mapId), c);
        });
        const newSprinklers = this.sprinklers.filter((s) => s.mapId !== mapId);
        this._sprinkler.clear();
        newSprinklers.forEach((s) => {
            this._sprinkler.set(s);
        });
        for (let i = 0; i < Iz.Life.Spawn.objects.length; i++) {
            const o = Iz.Life.Spawn.objects[i];
            if (o.mapId !== mapId)
                continue;
            if (!o.includeLabel(Iz.Life.FarmObjectLabelName))
                continue;
            Iz.Life.Spawn.removeObject(i);
        }
    }
    applySprinklers() {
        this.sprinklers.forEach((s) => {
            this.applySprinkler(s);
        });
    }
    newday() {
        this.applySprinklers();
        this._crops.forEach((info) => {
            let addGrowth = 0;
            if (info.soil === Iz.Life.Soil.Wetted) {
                info.soil = Iz.Life.Soil.Hoed;
                addGrowth += Iz.Life.getGrowthValueWetted();
            }
            const data = $dataCrops[info.plantItemId];
            if (data) {
                addGrowth += data.growth_add_base;
                info.growingDays++;
                info.growth = Math.min(info.growth + addGrowth, data.growth_max);
            }
        });
        this._sprinkler.newday();
        this.applySprinklers();
    }
}
(function (Iz) {
    var Life;
    (function (Life) {
        function getRandomPoint(ranges, excludes = []) {
            let positions = [];
            ranges.forEach((range) => {
                for (let y = range.top; y <= range.bottom; y++) {
                    for (let x = range.left; x <= range.right; x++) {
                        positions.push({ x: x, y: y });
                    }
                }
            });
            if (positions.length <= 0)
                return;
            positions = positions.filter((p) => !excludes.find((e) => e.x === p.x && e.y === p.y));
            positions = positions.filter(function (a, i, array) {
                return (array.findIndex(function (b) {
                    return b.x === a.x && b.y === a.y;
                }) === i);
            });
            positions.sort((a, b) => 0.5 - Math.random());
            const p = positions[0];
            return p;
        }
        Life.getRandomPoint = getRandomPoint;
    })(Life = Iz.Life || (Iz.Life = {}));
})(Iz || (Iz = {}));
(() => {
    "use strict";
    const script = document.currentScript;
    PluginManagerEx.registerCommand(script, "Newday", (args) => {
        $gameFarm.newday();
    });
    // thisをGame_Interpreterとして取得するためラムダ式は使わない
    PluginManagerEx.registerCommand(script, "HarvestFromThisEvent", harvestFromThisEvent);
    function harvestFromThisEvent(args) {
        // thisを使うための措置
        // @ts-ignore
        const eventId = this._eventId;
        const event = $gameMap.event(eventId);
        $gameFarm.harvest(event.x, event.y, event._mapId);
        Iz.System.playPickSe();
    }
    PluginManagerEx.registerCommand(script, "CreateObject", (args) => {
        const templateId = TR.getTemplateId(args.templateId);
        $gameFarm.createObject(args.x, args.y, templateId, args.mapId ? args.mapId : $gameMap.mapId());
    });
    PluginManagerEx.registerCommand(script, "CreateObjectRandom", (args) => {
        const ranges = args.ranges;
        const templateId = TR.getTemplateId(args.templateId);
        $gameFarm.createObjectRandom(ranges, templateId, args.mapId ? args.mapId : $gameMap.mapId());
    });
    PluginManagerEx.registerCommand(script, "RemoveObjectSelf", commandRemoveObjectSelf);
    function commandRemoveObjectSelf(args) {
        // @ts-ignore
        const event = this.character(0);
        $gameFarm.removeObject(event.getSpawnObjectIndex());
    }
    const _createGameObjects = DataManager.createGameObjects;
    DataManager.createGameObjects = function () {
        _createGameObjects.call(this);
        window.$gameFarm = new Game_Farm();
        Iz.Life.initializeFarm();
    };
    const _DataManager_makeSaveContents = DataManager.makeSaveContents;
    DataManager.makeSaveContents = function () {
        const contents = _DataManager_makeSaveContents.call(this);
        contents.farm = $gameFarm.getFarmData();
        return contents;
    };
    const _DataManager_extractSaveContents = DataManager.extractSaveContents;
    DataManager.extractSaveContents = function (contents) {
        _DataManager_extractSaveContents.call(this, contents);
        $gameFarm = new Game_Farm(contents.farm);
    };
    const _Scene_Map_create = Scene_Map.prototype.create;
    Scene_Map.prototype.create = function () {
        if ($gamePlayer.isTransferring()) {
            const newMapId = $gamePlayer.newMapId();
            if (newMapId !== $gameMap.mapId()) {
                const copySrcMapId = Iz.Life.getFarmSrcMap(newMapId);
                if (copySrcMapId !== newMapId) {
                    $gameFarm.copyMap(copySrcMapId, newMapId);
                }
                const clearSrcMapId = Iz.Life.getFarmSrcMap($gameMap.mapId());
                if (clearSrcMapId !== $gameMap.mapId()) {
                    $gameFarm.clearMap($gameMap.mapId());
                }
            }
        }
        _Scene_Map_create.call(this);
    };
    const _Scene_Map_onMapLoaded = Scene_Map.prototype.onMapLoaded;
    Scene_Map.prototype.onMapLoaded = function () {
        _Scene_Map_onMapLoaded.call(this);
        $gameMap.refreshFarmEvents();
    };
    const _Game_Map_setup = Game_Map.prototype.setup;
    Game_Map.prototype.setup = function (mapId) {
        _Game_Map_setup.call(this, mapId);
        // 農作などでイベント追加の可能性があるのでリフレッシュさせとく
        this._needsRefresh = true;
    };
    const _Game_Map_setupEvents = Game_Map.prototype.setupEvents;
    Game_Map.prototype.setupEvents = function () {
        _Game_Map_setupEvents.call(this);
        this.setupFarmEvents();
    };
    Game_Map.prototype.setupFarmEvents = function () {
        const targets = $gameFarm.crops.filter((info) => info.mapId === $gameMap.mapId());
        // 土壌
        targets.forEach((info) => {
            this.createSoilEvent(info.x, info.y);
        });
        // 植物
        targets
            .filter((info) => info.plantItemId !== 0)
            .forEach((info) => {
            this.createPlantEvent(info.x, info.y, info.plantItemId);
        });
        // スプリンクラー
        const sprinklers = $gameFarm.sprinklers.filter((s) => s.mapId === $gameMap.mapId());
        sprinklers.forEach((s) => {
            this.createSprinklerEvent(s.x, s.y, s.sprinklerId);
        });
    };
    Game_Map.prototype.refreshFarmEvents = function () {
        // 土壌
        const mapId = this.mapId();
        const soilEvents = $gameMap.events().filter((e) => Iz.Life.isSoilEvent(e));
        soilEvents.forEach((event) => {
            const info = $gameFarm.getCrop(event.x, event.y, mapId);
            if (info) {
                $gameSelfSwitches.setValue([mapId, event.eventId(), "A"], false);
                $gameSelfSwitches.setValue([mapId, event.eventId(), "B"], false);
                $gameSelfSwitches.setValue([mapId, event.eventId(), info.soil === Iz.Life.Soil.Hoed ? "A" : "B"], true);
            }
        });
    };
    Game_Map.prototype.createSoilEvent = function (x, y) {
        const SoilEventId = 114;
        const event = this.spawnEventAndGet(SoilEventId, x, y, true);
        event?.setEventType(Iz.Life.EventType.Soil);
        return event;
    };
    Game_Map.prototype.createPlantEvent = function (x, y, plantItemId) {
        const tagName = Iz.Life.FarmEventTag_Plant;
        const target = TR.getTemplateMapEvents(Iz.Life.PlantTemplateMapId).find((e) => !!e && e.meta[tagName] && Number(e.meta[tagName]) === plantItemId);
        if (!target) {
            console.warn(`id:${plantItemId} の植物のスポーンイベントがありません`);
            return undefined;
        }
        const originalEventId = target.id;
        const event = this.spawnEventAndGet(originalEventId, x, y, true, Iz.Life.PlantTemplateMapId);
        event?.setEventType(Iz.Life.EventType.Plant);
        return event;
    };
    Game_Map.prototype.createSprinklerEvent = function (x, y, sprinklerId) {
        const data = $dataSprinkler.get(sprinklerId);
        if (!data) {
            return undefined;
        }
        const SprinklerEventId = data.template_event_id;
        if (SprinklerEventId <= 0) {
            return undefined;
        }
        const event = this.spawnEventAndGet(SprinklerEventId, x, y, true);
        if (event) {
            event?.setEventType(Iz.Life.EventType.Sprinkler);
        }
        return event;
    };
    const _Game_Player_performTransfer = Game_Player.prototype.performTransfer;
    Game_Player.prototype.performTransfer = function () {
        _Game_Player_performTransfer.call(this);
        $gameFarm.applySprinklers();
    };
    const _Game_Player_onRegistAction = Game_Player.prototype.onRegistAction;
    Game_Player.prototype.onRegistAction = function () {
        _Game_Player_onRegistAction.call(this);
        const controller = this.getActionController();
        controller.setAction(new Iz.Life.HoeAction(), Iz.Life.ActionType.Hoe);
        controller.setAction(new Iz.Life.PlantAction(), Iz.Life.ActionType.Plant);
        controller.setAction(new Iz.Life.WaterPotAction(), Iz.Life.ActionType.WaterPot);
    };
    const _Game_PrefabEvent_screenZ = Game_PrefabEvent.prototype.screenZ;
    Game_PrefabEvent.prototype.screenZ = function () {
        // NOTE:
        // 土壌と植物イベントで順番が入れ替わらないようにするため、
        // 植物イベントはプライオリティが「通常キャラの下」よりも上になるようにする
        switch (this._eventType) {
            case Iz.Life.EventType.Plant:
                // return _priorityType * 2 + 1;
                return 2;
            default:
                return _Game_PrefabEvent_screenZ.call(this);
        }
    };
    const _Game_PrefabEvent_findProperPageIndex = Game_PrefabEvent.prototype.findProperPageIndex;
    Game_PrefabEvent.prototype.findProperPageIndex = function () {
        if (this.eventType() === Iz.Life.EventType.Plant) {
            const crop = $gameFarm.getCrop(this.x, this.y, $gameMap.mapId());
            if (crop) {
                const level = Iz.Life.getPlantLevel(crop.plantItemId, crop.growth);
                return level - 1;
            }
            return -1;
        }
        return _Game_PrefabEvent_findProperPageIndex.call(this);
    };
    // NOTE:
    // MessageWindow.tsのupdateBitmapでエラーが起きないようにする
    // 吹き出しを植物の上にも出せるようにしたい
    const _Sprite_Character_isImageChanged = Sprite_Character.prototype.isImageChanged;
    Sprite_Character.prototype.isImageChanged = function () {
        if (this._character.eventType() === Iz.Life.EventType.Plant) {
            return false;
        }
        return _Sprite_Character_isImageChanged.call(this);
    };
    const _Sprite_Character_updateBitmap = Sprite_Character.prototype.updateBitmap;
    Sprite_Character.prototype.updateBitmap = function () {
        if (this._character.eventType() === Iz.Life.EventType.Plant) {
            // updateFrameでsetAtlasするので不要
        }
        else {
            _Sprite_Character_updateBitmap.call(this);
        }
    };
    const _Sprite_Character_updateFrame = Sprite_Character.prototype.updateFrame;
    Sprite_Character.prototype.updateFrame = function () {
        if (this._character.eventType() === Iz.Life.EventType.Plant) {
            const crop = $gameFarm.getCrop(this._character.x, this._character.y, $gameMap.mapId());
            const frameKey = crop
                ? `plant_${crop.plantItemId}_${Iz.Life.getPlantLevel(crop.plantItemId, crop.growth)}`
                : "";
            // NOTE:
            // 植物が大量にあるとアトラスの更新処理が重くなるのでキャッシュする
            if (this._plantFrameKey !== frameKey) {
                this.setAtlas(Iz.Life.ATLAS_KEY_FARM_PLANT, frameKey);
                this._plantFrameKey = frameKey;
            }
        }
        else {
            _Sprite_Character_updateFrame.call(this);
        }
    };
})();
//# sourceMappingURL=farm.js.map