/*:ja
 * @target MZ
 * @plugindesc NPC自律移動プラグイン
 * @author INAZUMA GAMES
 *
 * @help
 * 広域マップクラスを定義し、マップを跨いでのNPCの自律移動を実現します。
 *
 * プラグインコマンド一覧
 * NPC_NEWDAY
 * NPC_JUMPCURRENTTIME
 * NPC_JUMPTIME
 * NPC_PROGRESS_SETUP
 * NPC_APPEAR
 * NPC_LEAVE
 * NPC_MOVE_INTERRUPT
 * NPC_CLEARDESTINATION
 * NPC_SETTARGET
 * NPC_CLEARTARGET
 * NPC_SETPOSITION
 * NPC_PAUSE
 * NPC_RESUME
 * GLOBALMAP_PAUSE
 * GLOBALMAP_RESUME
 * IS_SAME_LOCATION
 * NPC_CHANGE_SCHEDULE
 * NPC_CHANGE_COSTUME
 *
 * [タグ設定]
 * <GM_Collider:>
 * イベントと接触判定が行われるようになります（※通常はNPCとイベントの接触判定はありません）
 *
 * <GM_OnExitExecutePage:n>
 * NPCが場所移動した際に直下のイベントを起動する設定です
 * 出現条件を満たすものが優先で、満たすものが無い場合に指定のページを起動します
 * nはNPCExecuteEventSwitchで指定したページを抽出したリストのインデックスです
 * 例：
 * 4ページ中2・3番目がNPCExecuteEventの時、2ページ目は0、3ページ目は1となります
 *
 *
 * @command NPC_NEWDAY
 * @text NPC_NEWDAY
 * @desc 各 NPC のスケジュールを現在の日付に初期化する
 *
 *
 * @command NPC_JUMPCURRENTTIME
 * @text NPC_JUMPCURRENTTIME [actor] [all]
 * @desc 指定したIDのNPCを現在時刻のスケジュールに基づいた位置に移動させる。allがtrueの場合、全てのNPCが対象になる
 *
 * @arg actor
 * @type actor
 * @text actor
 * @desc
 *
 * @arg all
 * @type boolean
 * @text 全対象
 * @desc
 *
 *
 * @command NPC_JUMPTIME
 * @text NPC_JUMPTIME [actor] [hour] [minute] [all]
 * @desc 各 NPC のスケジュールを指定した時間に更新する。allがtrueの場合、全てのNPCが対象になる
 *
 * @arg actor
 * @type actor
 * @text actor
 * @desc
 *
 * @arg hour
 * @type number
 * @text hour
 * @desc
 *
 * @arg minute
 * @type number
 * @text minute
 * @desc
 *
 * @arg all
 * @type boolean
 * @text 全対象
 * @desc
 *
 *
 * @arg actor
 * @type actor
 * @text actor
 * @desc
 *
 * @arg minute
 * @type number
 * @text minute
 * @desc
 *
 * @arg all
 * @type boolean
 * @text 全対象
 * @desc
 *
 *
 * @command NPC_PROGRESS_SETUP
 * @text NPC_PROGRESS_SETUP [actor] [hour] [minute]
 * @desc NPCの進行度を指定時刻に合うようにする
 *
 * @arg actor
 * @type actor
 * @text actor
 * @desc
 *
 * @arg hour
 * @type number
 * @text hour
 * @desc
 *
 * @arg minute
 * @type number
 * @text minute
 * @desc
 *
 *
 * @command NPC_APPEAR
 * @text NPC_APPEAR [actor] [hour]
 * @desc 指定したactorのNPCをマップに出現させる。引数２を省略した場合、現在時間になる
 *
 * @arg actor
 * @type actor
 * @text actor
 * @desc
 *
 * @arg hour
 * @type number
 * @text hour
 * @desc
 *
 *
 * @command NPC_LEAVE
 * @text NPC_LEAVE [actor]
 * @desc 指定したactorのNPCをマップから退場させる
 *
 * @arg actor
 * @type actor
 * @text actor
 * @desc
 *
 *
 * @command NPC_MOVE_INTERRUPT
 * @text NPC_MOVE_INTERRUPT [actor] [mapId] [x] [y] [direction]
 * @desc 指定したactorのNPCを指定のマップ、位置に移動させる
 *
 * @arg actor
 * @type actor
 * @text actor
 * @desc
 *
 * @arg mapId
 * @type number
 * @text mapId
 * @desc
 *
 * @arg x
 * @type number
 * @text x
 * @desc
 *
 * @arg y
 * @type number
 * @text y
 * @desc
 *
 * @arg direction
 * @type number
 * @text direction
 * @desc
 *
 *
 * @command NPC_CLEARDESTINATION
 * @text NPC_CLEARDESTINATION [actor] [all]
 * @desc 指定したIDのNPCの目的地を消去する。allがtrueの場合、全てのNPCが対象になる
 *
 * @arg actor
 * @type actor
 * @text actor
 * @desc
 *
 * @arg all
 * @type boolean
 * @text 全対象
 * @desc
 *
 *
 * @command NPC_SETTARGET
 * @text NPC_SETTARGET [actor] [actor]
 * @desc 第１引数で指定したactorのNPCが第2引数で指定したactorのキャラに向かうようになる。
 *
 * @arg actor1
 * @type actor
 * @text actor1
 * @desc
 *
 * @arg actor2
 * @type actor
 * @text actor2
 * @desc
 *
 *
 * @command NPC_CLEARTARGET
 * @text NPC_CLEARTARGET [actor] [all]
 * @desc 指定したIDのNPCのターゲットを解除する。allがtrueの場合、全てのNPCが対象になる
 *
 * @arg actor
 * @type actor
 * @text actor
 * @desc
 *
 * @arg all
 * @type boolean
 * @text 全対象
 * @desc
 *
 *
 * @command NPC_SETPOSITION
 * @text NPC_SETPOSITION [actor] [x] [y] [direction]
 * @desc 指定したactorのNPCを指定の位置に移動させる
 *
 * @arg actor
 * @type actor
 * @text actor
 * @desc
 *
 * @arg x
 * @type number
 * @text x
 * @desc
 *
 * @arg y
 * @type number
 * @text y
 * @desc
 *
 * @arg direction
 * @type number
 * @text direction
 * @desc
 *
 *
 * @command NPC_PAUSE
 * @text NPC_PAUSE [actor] [hour] [minute]
 * @desc 指定したactorのNPCを指定した時間停止させる。トイレなど目的地に到着した後に一定時間動作を停止させる時に使用する
 *
 * @arg actor
 * @type actor
 * @text actor
 * @desc
 *
 * @arg hour
 * @type number
 * @text hour
 * @desc
 *
 * @arg minute
 * @type number
 * @text minute
 * @desc
 *
 *
 * @command NPC_RESUME
 * @text NPC_RESUME [actor]
 * @desc 指定したactorのNPCのポーズを解除する
 *
 * @arg actor
 * @type actor
 * @text actor
 * @desc
 *
 * @arg all
 * @type boolean
 * @text 全対象
 * @desc
 *
 *
 * @command GLOBALMAP_PAUSE
 * @text GLOBALMAP_PAUSE
 * @desc グローバルマップのアップデートを停止する
 *
 * @command GLOBALMAP_RESUME
 * @text GLOBALMAP_RESUME
 * @desc グローバルマップのアップデートを再開させる
 *
 *
 * @command IS_SAME_LOCATION
 * @text IS_SAME_LOCATION [Actor1] [Actor2] [Switch]
 * @desc Actor1 と Actor2 のロケーションを比較し、同じロケーションなら Switch を ON に、そうでない場合は OFF にします
 *
 * @arg actor1
 * @type actor
 * @text Actor1
 * @desc
 * @default 1
 * @arg actor2
 * @type actor
 * @text Actor2
 * @desc
 * @default 2
 * @arg switch
 * @type switch
 * @text Switch
 * @desc
 * @default 1
 *
 *
 * @command NPC_CHANGE_SCHEDULE
 * @text NPC_CHANGE_SCHEDULE
 * @desc NPCのスケジュールを変更します
 *
 * @arg actor
 * @type actor
 * @text actor
 * @desc
 *
 * @arg schedule
 * @type number
 * @text schedule
 * @desc スケジュール
 *
 * @command NPC_CHANGE_COSTUME
 * @text NPC_CHANGE_COSTUME [actor] [costume] [actorType]
 * @desc NPCのコスチュームを変更する
 *
 * @arg actor
 * @type actor
 * @text actor
 * @desc
 *
 * @arg costume
 * @type
 *
 * @arg actorType
 * @type select
 * @text アクターの指定方法
 * @desc 未指定:actorを使用 1:全員 2:executeCharacter(※NPCExecuteEventでのみ有効)
 *
 * @option 全員
 * @value 1
 * @option ExecuteCharacter
 * @value 2
 *
 */
// --------------------------------
// RouteExploration
// --------------------------------
var Iz;
(function (Iz) {
    var GlobalMap;
    (function (GlobalMap) {
        class ExplorationNode {
            constructor(mapId, x, y) {
                this._mapId = mapId;
                this._x = x;
                this._y = y;
                this._parent = null;
                this._heuristicCost = 0;
                this._costToStart = 0;
            }
            get mapId() {
                return this._mapId;
            }
            get x() {
                return this._x;
            }
            get y() {
                return this._y;
            }
            get parent() {
                return this._parent;
            }
            set parent(value) {
                this._parent = value;
            }
            get totalCost() {
                return this._heuristicCost + this._costToStart;
            }
            get heuristicCost() {
                return this._heuristicCost;
            }
            get moveCost() {
                return this._moveCost;
            }
            get costToStart() {
                return this._costToStart;
            }
            set costToStart(value) {
                this._costToStart += value;
            }
            equal(node) {
                return this._x === node.x && this._y === node.y;
            }
            setGoal(node) {
                this._heuristicCost = this.distance(this._x, this._y, node.x, node.y);
            }
            closer(node) {
                return this._heuristicCost < node.heuristicCost ? this : node;
            }
            //ループ非対応
            distance(x1, y1, x2, y2) {
                return Math.abs(x1 - x2) + Math.abs(y1 - y2);
            }
            calc(notRoadRegionCost) {
                const mapId = this._mapId;
                const x = this._x;
                const y = this._y;
                if (GlobalMap.isHalfMove) {
                    if (Number.isInteger(x)) {
                        if (Number.isInteger(y)) {
                            const cost = $globalMap.map(mapId).getRoadRegion(x, y);
                            this._moveCost = cost !== undefined ? cost : notRoadRegionCost;
                        }
                        else {
                            // 上下
                            const r1 = $globalMap.map(mapId).getRoadRegion(x, y - 0.5);
                            const r2 = $globalMap.map(mapId).getRoadRegion(x, y + 0.5);
                            const isRoad = r1 !== undefined && r2 !== undefined;
                            this._moveCost = isRoad ? (r1 + r2) / 2 : notRoadRegionCost;
                        }
                    }
                    else if (Number.isInteger(y)) {
                        // 左右
                        const r1 = $globalMap.map(mapId).getRoadRegion(x - 0.5, y);
                        const r2 = $globalMap.map(mapId).getRoadRegion(x + 0.5, y);
                        const isRoad = r1 !== undefined && r2 !== undefined;
                        this._moveCost = isRoad ? (r1 + r2) / 2 : notRoadRegionCost;
                    }
                    else {
                        // 上下左右
                        const r1 = $globalMap.map(mapId).getRoadRegion(x - 0.5, y - 0.5);
                        const r2 = $globalMap.map(mapId).getRoadRegion(x + 0.5, y - 0.5);
                        const r3 = $globalMap.map(mapId).getRoadRegion(x - 0.5, y + 0.5);
                        const r4 = $globalMap.map(mapId).getRoadRegion(x + 0.5, y + 0.5);
                        const isRoad = r1 !== undefined &&
                            r2 !== undefined &&
                            r3 !== undefined &&
                            r4 !== undefined;
                        this._moveCost = isRoad ? (r1 + r2 + r3 + r4) / 4 : notRoadRegionCost;
                    }
                }
                else {
                    const cost = $globalMap.map(mapId).getRoadRegion(x, y);
                    this._moveCost = cost !== undefined ? cost : notRoadRegionCost;
                }
            }
        }
        GlobalMap.ExplorationNode = ExplorationNode;
        const InvalidNode = new ExplorationNode(-1, -1, -1);
        /*
         * A*アルゴリズムの理解を深めるために
         * 既存の下記メソッドを再実装
         * CharacterBase.prototype.findDirectionTo
         *
         * 参考URL
         * https://yttm-work.jp/algorithm/algorithm_0015.html
         */
        class RouteExploration {
            constructor(mapId, notRoadRegionCost) {
                this._mapId = mapId;
                this._openList = [];
                this._closedList = [];
                this._routeNodes = [];
                this._startNode = InvalidNode;
                this._goalNode = InvalidNode;
                this._notRoadRegionCost = notRoadRegionCost;
            }
            onFinish() {
                this._openList = [];
                this._closedList = [];
                this._startNode = InvalidNode;
                this._goalNode = InvalidNode;
            }
            node(x, y) {
                if (x < 0) {
                    return;
                }
                if (y < 0) {
                    return;
                }
                const newNode = new ExplorationNode(this._mapId, x, y);
                newNode.calc(this._notRoadRegionCost);
                newNode.setGoal(this._goalNode);
                return newNode;
            }
            adjacentNodes(node) {
                const unitCost = GlobalMap.getUnitCost();
                const adjNodes = [];
                const adjPos = [
                    { x: node.x, y: node.y + unitCost },
                    { x: node.x - unitCost, y: node.y },
                    { x: node.x + unitCost, y: node.y },
                    { x: node.x, y: node.y - unitCost },
                ];
                for (let i = 3; i >= 0; i--) {
                    const target = adjPos[i];
                    if (target.y + unitCost >= $globalMap.map(node.mapId).height ||
                        target.x + unitCost >= $globalMap.map(node.mapId).width ||
                        target.x < 0 ||
                        target.y < 0) {
                        adjPos.splice(i, 1);
                    }
                }
                for (let i = 0; i < adjPos.length; i++) {
                    const adjNode = this.node(adjPos[i].x, adjPos[i].y);
                    if (adjNode) {
                        adjNodes.push(adjNode);
                    }
                }
                return adjNodes.filter((n) => n);
            }
            nextNode() {
                return this._routeNodes[0];
            }
            shift() {
                return this._routeNodes.shift();
            }
            start(start, goal) {
                this._startNode = start;
                this._goalNode = goal;
                this._startNode.calc(this._notRoadRegionCost);
                this._goalNode.calc(this._notRoadRegionCost);
                this._openList.push(this._startNode);
                let closerNode = this._startNode;
                let find = false;
                while (this._openList.length > 0) {
                    const minimumCostNode = this._openList.shift();
                    this._closedList.push(minimumCostNode);
                    if (this._goalNode.equal(minimumCostNode)) {
                        find = true;
                        break;
                    }
                    closerNode = closerNode.closer(minimumCostNode);
                    this.addAdjacentNodeToOpenList(minimumCostNode);
                }
                // NotFound
                if (!find) {
                    this._routeNodes.unshift(closerNode);
                }
                else {
                    const node = this._closedList.pop();
                    this._routeNodes.unshift(node);
                }
                if (this._routeNodes[0].equal(this._startNode)) {
                    this.onFinish();
                    return;
                }
                while (this._routeNodes[0].parent &&
                    !this._routeNodes[0].parent.equal(this._startNode)) {
                    this._routeNodes.unshift(this._routeNodes[0].parent);
                }
                // ネストが深いとJsonEx.depthの上限を超えることがあるので参照を消す
                this._routeNodes.forEach((node) => {
                    node.parent = null;
                });
                this.onFinish();
            }
            updateCloseList(newNode, oldNode) {
                if (newNode.totalCost < oldNode.totalCost) {
                    const index = this._closedList.indexOf(oldNode);
                    this._closedList.splice(index, 1);
                    this.addNodeToOpenList(newNode);
                }
            }
            updateOpenList(newNode, oldNode) {
                if (newNode.totalCost < oldNode.totalCost) {
                    const index = this._openList.indexOf(oldNode);
                    this._openList.splice(index, 1);
                    this.addNodeToOpenList(newNode);
                }
            }
            //オープンリストの最適な位置へ挿入する
            addNodeToOpenList(node) {
                let i = 0;
                for (i = 0; i < this._openList.length; i++) {
                    if (node.totalCost <= this._openList[i].totalCost) {
                        break;
                    }
                }
                this._openList.splice(i, 0, node);
            }
            //未探査の隣接ノードをオープンリストに追加する
            addAdjacentNodeToOpenList(curNode) {
                const adjNodes = this.adjacentNodes(curNode);
                for (let i = 0; i < adjNodes.length; i++) {
                    adjNodes[i].costToStart = adjNodes[i].moveCost + curNode.costToStart;
                    if (adjNodes[i].totalCost === Number.POSITIVE_INFINITY) {
                        continue;
                    }
                    adjNodes[i].parent = curNode;
                    let oldNode = this._closedList.find((closed) => closed.equal(adjNodes[i]));
                    if (oldNode) {
                        this.updateCloseList(adjNodes[i], oldNode);
                        continue;
                    }
                    oldNode = this._openList.find((open) => open.equal(adjNodes[i]));
                    if (oldNode) {
                        this.updateOpenList(adjNodes[i], oldNode);
                        continue;
                    }
                    this.addNodeToOpenList(adjNodes[i]);
                }
            }
        }
        GlobalMap.RouteExploration = RouteExploration;
    })(GlobalMap = Iz.GlobalMap || (Iz.GlobalMap = {}));
})(Iz || (Iz = {}));
// --------------------------------
// NPC
// --------------------------------
(function (Iz) {
    var GlobalMap;
    (function (GlobalMap) {
        const CollidedWithCount = 60;
        class MB_NonPlayer extends Game_Event {
            static receiveItemIdStoredVariableId() {
                return 1;
            }
            static receiveItemNameStoredVariableId() {
                return 2;
            }
            constructor(data, messenger) {
                super(0, 0);
                this._characterId = data.id;
                this._validity = data.validity;
                this._scheduler = new GlobalMap.MB_CharacterScheduler(this);
                this._messenger = messenger;
                this._hitWithPlayerCount = 0;
                this._playerThroughCount = 0;
                this._requestPlayerThrough = false;
                this.setupEvent();
            }
            initialize(...args) {
                Game_Character.prototype.initialize.call(this);
                // NOTE:
                // HalfMove.jsのGame_Event.prototype.initialize()が呼ばれないので
                // ここで設定  直したい
                this._triggerExpansion = true;
                this._throughDisable = undefined;
            }
            getExpansionArea() {
                return [0, 0.5, 0.5, 0];
            }
            initMembers() {
                super.initMembers();
                this._received = false;
                this._eventId = -1;
                this._moveSpeed = 3;
                this._characterCollidCount = 0;
                this._route = null;
                this._locked = false;
                this._validEventTags = {};
                this._costume = 0;
                this._imgName = "";
                this._collidedPlayer = false;
                this._collidingPlayer = false;
                $globalMapParameter.npc_events.forEach((define) => {
                    this._validEventTags[define.name] = define.enable;
                });
            }
            setSchedule(schedule) {
                this._scheduler.setSchedule(schedule);
            }
            isAppear() {
                return this.validity && this._scheduler.isAppear();
            }
            jumpSpecifiedTime(hour, minute = 0) {
                this.unlock();
                this.setDirectionFix(false);
                this.clearDestination();
                if (this.paused) {
                    this.resume();
                }
                this._scheduler.jumpSpecifiedTime(hour, minute);
            }
            clearDestination() {
                this._scheduler.clearDestination();
                // this._route = [];
                this._route = null;
            }
            newDay(hour, dayOfWeek) {
                this.setEventEnable(GlobalMap.EventTag_Greeting, true);
                this._collidedPlayer = false;
                this._collidingPlayer = false;
                this._scheduler.newday(hour, dayOfWeek);
                this.refresh();
            }
            setEventEnable(tag, enable) {
                if (!Object.hasOwn(this._validEventTags, tag)) {
                    return;
                }
                this._validEventTags[tag] = enable;
                this.setupEvent();
            }
            getEventEnable(tag) {
                if (!Object.hasOwn(this._validEventTags, tag)) {
                    return false;
                }
                return this._validEventTags[tag];
            }
            setupEvent() {
                let event = undefined;
                for (let i = $globalMapParameter.npc_events.length - 1; i >= 0; i--) {
                    const tag = $globalMapParameter.npc_events[i].name;
                    let id = 0;
                    if (this._validEventTags[tag] === false) {
                        continue;
                    }
                    if (tag === GlobalMap.EventTag_Greeting) {
                        if (!this.getEventEnable(GlobalMap.EventTag_Greeting)) {
                            continue;
                        }
                    }
                    else if (tag === GlobalMap.EventTag_MapId) {
                        id = this._mapId;
                    }
                    else if (tag === GlobalMap.EventTag_DayOfWeek) {
                        id = Iz.GlobalMap.getChronus().getDaysOfWeek();
                    }
                    const e = $globalMap.findEvent(this.dataNPC.name, tag, id);
                    if (e) {
                        event = e;
                        break;
                    }
                }
                if (!event) {
                    throw new Error(`NPC:${this.dataNPC.name} 適切なNPCイベントが見つかりませんでした。`);
                }
                this._event = event;
                this.refresh();
            }
            //=============================================================================
            // データ取得
            //=============================================================================
            get dataNPC() {
                return $dataNonPlayers[this._characterId];
            }
            get characterId() {
                return this._characterId;
            }
            get validity() {
                return this._validity;
            }
            set validity(value) {
                this._validity = value;
            }
            get name() {
                return this.dataNPC.name;
            }
            get scheduler() {
                return this._scheduler;
            }
            get defaultCostume() {
                return 2;
            }
            get collidedPlayer() {
                return this._collidedPlayer;
            }
            get paused() {
                return this._scheduler.paused;
            }
            get destination() {
                return this._scheduler.destination();
            }
            get hasDestinationStack() {
                return this._scheduler.hasDestinationStack();
            }
            event() {
                return this._event;
            }
            characterName() {
                const costume = this.getCostume();
                const imgName = this._imgName ? this._imgName : "walk";
                const characterStr = this.characterId.toString().padStart(3, "0");
                const costumeStr = costume.toString().padStart(2, "0");
                const c = `chara${characterStr}_${costumeStr}/$${imgName}`;
                return c;
            }
            getCostume() {
                const cos = this.getOriginalCostume();
                return GlobalMap.convertCostume(cos);
            }
            getOriginalCostume() {
                const cos = this._costume ? this._costume : this.defaultCostume;
                return cos;
            }
            setCostume(costume) {
                this._costume = costume ? costume : this.defaultCostume;
            }
            mapId() {
                return this._mapId;
            }
            //=============================================================================
            // データ設定
            //=============================================================================
            setPosition(x, y, d = 2, mapId = this._mapId) {
                super.setPosition(x, y);
                this._mapId = mapId;
                this.setDirectionFix(false);
                this.setDirection(d);
            }
            performPosition(x, y, d = 2, mapId = this._mapId) {
                const prevX = this._realX;
                const prevY = this._realY;
                const prevMapId = this._mapId;
                this.setPosition(x, y, d, mapId);
                this._hitWithPlayerCount = 0;
                this._playerThroughCount = 0;
                this._requestPlayerThrough = false;
                this.send({
                    type: GlobalMap.MessageType.TriggerExit,
                    character: this.characterId,
                    data: {
                        mapId: prevMapId,
                        col: new HM.Collider({
                            x: prevX,
                            y: prevY,
                            expansionArea: this.getExpansionArea(),
                        }),
                    },
                });
                this.send({
                    type: GlobalMap.MessageType.TriggerEnter,
                    character: this.characterId,
                    data: {
                        mapId: this._mapId,
                        col: new HM.Collider({
                            x: this._realX,
                            y: this._realY,
                            expansionArea: this.getExpansionArea(),
                        }),
                    },
                });
            }
            setTarget(target) {
                this._scheduler.setTarget(target);
            }
            setEvent(event) {
                this._event = event;
                this.refresh();
            }
            setEventId(id) {
                this._eventId = id;
            }
            //=============================================================================
            // Update処理
            //=============================================================================
            update() {
                if (!$gameMap.isEventRunning() && !this._locked) {
                    const wasMoving = this.isMoving();
                    Game_Character.prototype.update.call(this);
                    if (!this.isMoving()) {
                        this.updateNonmoving(wasMoving);
                    }
                    if (this._playerThroughCount > 0) {
                        this._playerThroughCount--;
                    }
                }
                this.updateParallel();
            }
            updateMove() {
                const prevRealX = this._realX;
                const prevRealY = this._realY;
                super.updateMove();
                this._characterCollidCount = 0;
                const moveData = {
                    prev: {
                        mapId: this._mapId,
                        col: new HM.Collider({
                            x: prevRealX,
                            y: prevRealY,
                            expansionArea: this.getExpansionArea(),
                        }),
                    },
                    current: {
                        mapId: this._mapId,
                        col: new HM.Collider({
                            x: this._realX,
                            y: this._realY,
                            expansionArea: this.getExpansionArea(),
                        }),
                    },
                };
                this.send({
                    type: GlobalMap.MessageType.Move,
                    character: this.characterId,
                    data: moveData,
                });
            }
            updateNonmoving(wasMoving) {
                if (wasMoving) {
                    this.checkEventTriggerHere();
                }
            }
            updateStop() {
                if (this._locked) {
                    this.resetStopCount();
                    return;
                }
                const hour = Iz.GlobalMap.getChronus().getHour();
                const minute = Iz.GlobalMap.getChronus().getMinute();
                this._scheduler.update(hour, minute);
                // プレイヤーが傍を離れたら再開
                if (this._collidingPlayer) {
                    if ($gameMap.mapId() !== this.mapId()) {
                        this.resume();
                    }
                    else {
                        const distance = $globalMapParameter.collide_pause_reset_distance;
                        const dx = $gamePlayer.x - this.x;
                        const dy = $gamePlayer.y - this.y;
                        if (dx * dx + dy * dy > distance * distance) {
                            this.resume();
                        }
                    }
                }
                //そのうち自律移動についても考える。
                // if (false)
                //     this.updateSelfMovement();
            }
            updateParallel() {
                if (this._interpreter) {
                    this._interpreter.update();
                    if (!this._interpreter.isRunning()) {
                        this._interpreter = null;
                    }
                }
            }
            checkEventTriggerHere(onPerform = false) {
                const events = this.executeEvent(this.x, this.y);
                events.forEach((event) => {
                    if (event?.isTile()) {
                        this.setupInterpreter(event);
                    }
                });
                if (onPerform) {
                    events
                        .filter((e) => Object.hasOwn(e.meta, GlobalMap.ColliderTag) &&
                        Object.hasOwn(e.meta, GlobalMap.OnPerformTag))
                        .forEach((e) => {
                        if (e) {
                            e.setPageIndex(Number(e.meta[GlobalMap.OnPerformTag]));
                            this.setupInterpreter(e);
                        }
                    });
                }
            }
            checkEventTriggerHereOnDeparture() {
                const events = this.executeEvent(this._realX, this._realY);
                events.forEach((event) => {
                    if (event?.isTile()) {
                        this.setupInterpreter(event);
                    }
                });
            }
            getHereEventOnDeparture() {
                const events = this.executeEvent(this.x, this.y);
                const hereEvents = events.filter((event) => {
                    if (event?.isTile()) {
                        return true;
                    }
                    return false;
                });
                return hereEvents;
            }
            setHereEventsOnDeparture(events) {
                events.forEach((e) => {
                    this.setupInterpreter(e);
                });
            }
            checkEventTriggerTouch(x, y) {
                if (!this._interpreter) {
                    const events = this.executeEvent(x, y);
                    events.forEach((event) => {
                        if (event?.isNormalPriority()) {
                            this.setupInterpreter(event);
                            return;
                        }
                    });
                }
                if (this.isSameMap()) {
                    super.checkEventTriggerTouch(x, y);
                }
            }
            executeEvent(x, y) {
                const map = $globalMap.map(this._mapId);
                const events = map?.nonPlayerExecuteEvents(x, y) ?? [];
                events.forEach((event) => event.refresh(this));
                return events;
            }
            samePosition(mapId, x, y) {
                return this.mapId() === mapId && this._realX === x && this._realY === y;
            }
            performTransfer(mapId, x, y, d, jumpTime) {
                const lastMapId = this._mapId;
                this.sendDeparture();
                if (jumpTime) {
                    this.sendJumpTime();
                }
                this.performPosition(x, y, d, mapId);
                this.checkEventTriggerHere(true);
                if (lastMapId !== mapId) {
                    if (this.isSameMap()) {
                        const scene = SceneManager._scene;
                        scene.addMapObject(this);
                        $gameMap.addNpc(this);
                        this.setupEvent();
                        this.refresh();
                    }
                    if (lastMapId === $gameMap.mapId()) {
                        const scene = SceneManager._scene;
                        scene.removeCharacter(this);
                        this.endBalloon();
                        $gameMap.removeNpc(this._eventId);
                    }
                }
                this.sendArrive();
                this.onPerformTransfer();
            }
            onPerformTransfer() {
                this.unlock();
            }
            setupInterpreter(event) {
                if (!event.page()) {
                    return;
                }
                if (event.isParallel()) {
                    this._interpreter = new Game_Interpreter();
                    this._interpreter.setup(event.list(), event.eventId());
                    this._interpreter.setExecuteCharacter(this.characterId);
                }
                else {
                    if ($gameMap.isEventRunning()) {
                        $gameMap.terminateInterpreter(null).setupChild(event.list(), event.eventId());
                        $gameMap.terminateInterpreter(null).setExecuteCharacter(this.characterId);
                    }
                    else {
                        $gameMap.interpreter().setup(event.list(), event.eventId());
                        $gameMap.interpreter().setExecuteCharacter(this.characterId);
                    }
                }
            }
            //=============================================================================
            // フロー制御
            //=============================================================================
            start() {
                const startingEvent = $globalMap.nonPlayers().find((npc) => npc.isStarting());
                if (!startingEvent) {
                    super.start();
                }
            }
            // TODO:
            // recieve〇〇の使用有無
            /*
            receiveGift(item) {
                if (this.isAlreadyReceived()) {
                    this._pageIndex = 7;
                }
                else {
        
                    const preference = this.favorite(item.id);
                    this._pageIndex = preference + 3;
                    this._data.gainAffection(preference * 100);
                    this._received = true;
        
                    $gameVariables._data[MB_NonPlayer.receiveItemIdStoredVariableId()] = item.id;
                    $gameVariables._data[MB_NonPlayer.receiveItemNameStoredVariableId()] = item.name;
        
                    $gamePlayer.consumptionItem();
        
                }
                this._event = $globalMap.findEventByGift(this._data.name);
                super.start();
                //$gameMap.addFinishEventListener(this.findEvent.bind(this));
            }
            */
            interrupt(mapId, x, y, d) {
                this._scheduler.interrupt(mapId, x, y, d);
            }
            pause(hour, minute) {
                this._scheduler.pause(hour, minute);
            }
            resume() {
                this._scheduler.resume();
            }
            onResume() {
                if (this._collidingPlayer) {
                    this._collidingPlayer = false;
                }
            }
            isAlreadyReceived() {
                return this._received;
            }
            isAlreadyGreeted() {
                return !this.getEventEnable(GlobalMap.EventTag_Greeting);
            }
            isSameMap() {
                return this._mapId === $gameMap.mapId();
            }
            //=============================================================================
            // 既存処理の上書き
            //=============================================================================
            findDirectionTo() {
                const postion = this._route?.nextNode();
                if (postion) {
                    const deltaX = postion.x - this._x;
                    if (deltaX < 0) {
                        return 4;
                    }
                    if (0 < deltaX) {
                        return 6;
                    }
                    const deltaY = postion.y - this._y;
                    if (deltaY < 0) {
                        return 8;
                    }
                    if (0 < deltaY) {
                        return 2;
                    }
                }
                return 0;
            }
            canPass2(x, y, d) {
                const x2 = $gameMap.roundXWithDirection(x, d);
                const y2 = $gameMap.roundYWithDirection(y, d);
                if (!$gameMap.isValid(x2, y2)) {
                    return false;
                }
                if (this.isThrough() || this.isDebugThrough()) {
                    return true;
                }
                if (!this.isMapPassable(x, y, d)) {
                    return false;
                }
                return true;
            }
            //A*アルゴリズムのサーチリミット
            //50マス先までサーチできる、広大なマップでは注意が必要
            searchLimit() {
                return GlobalMap.AstarSearchLimit;
            }
            canPass(x, y, d) {
                const x2 = $gameMap.roundXWithDirection(x, d);
                const y2 = $gameMap.roundYWithDirection(y, d);
                if (this.isSameMap()) {
                    if (!$gameMap.isValid(x2, y2)) {
                        return false;
                    }
                    // AStar時に検証しているのでここでは見ない
                    if (!GlobalMap.isHalfMove) {
                        if (!$globalMap.map(this._mapId).isRoadRegion(x2, y2)) {
                            return false;
                        }
                    }
                    if (this.isThrough() || this.isDebugThrough()) {
                        return true;
                    }
                    if (!this.isMapPassable(x, y, d)) {
                        return false;
                    }
                    if (this.isCollidedWithCharacters(x2, y2)) {
                        if (this._characterCollidCount > CollidedWithCount) {
                            this._characterCollidCount = 0;
                            // this._throughCount = 180;
                            return true;
                        }
                        this._characterCollidCount++;
                        const PlayerHitThreshold = 2;
                        if (this.isHitWithPlayer() &&
                            this._characterCollidCount === PlayerHitThreshold) {
                            this.hitToPlayer();
                        }
                        return false;
                    }
                    return true;
                }
                if (!GlobalMap.isHalfMove) {
                    if (!$globalMap.map(this._mapId).isRoadRegion(x2, y2)) {
                        return false;
                    }
                }
                if (this.isCollidedWithNPCExecuteEvents(x2, y2)) {
                    if (5 < this._characterCollidCount) {
                        this._characterCollidCount = 0;
                        return true;
                    }
                    this._characterCollidCount++;
                    return false;
                }
                return true;
            }
            isHitWithPlayer() {
                return this.isCollidedFromPlayer();
            }
            availableCollidedWithPlayer() {
                // NOTE:
                // 同じマップで指定回数接触すると立ち止まらなくなる
                if (this._hitWithPlayerCount > 3)
                    return false;
                //
                if (this._playerThroughCount > 0)
                    return false;
                return true;
            }
            hitToPlayer() {
                this._hitWithPlayerCount++;
                this._requestPlayerThrough = true;
            }
            updateAnimation() {
                this.updateAnimationCount();
                if (this._animationCount >= this.animationWait()) {
                    this.updatePattern();
                    this._animationCount = 0;
                }
            }
            // イベントどうしの衝突を無しに
            // ドアは判定させる
            isCollidedWithEvents(x, y) {
                const events = $gameMap.eventsXyNtInUnit(x, y);
                if (events.some((e) => e.isNormalPriority() &&
                    !(e instanceof MB_NonPlayer) &&
                    Object.hasOwn(e.event().meta, GlobalMap.ColliderTag))) {
                    return super.isCollidedWithEvents(x, y);
                }
                return false;
            }
            isCollidedWithPlayerCharacters(x, y) {
                if (!this.availableCollidedWithPlayer()) {
                    return false;
                }
                return super.isCollidedWithPlayerCharacters(x, y);
            }
            isCollidedWithNPCExecuteEvents(x, y) {
                const events = $globalMap.map(this._mapId).nonPlayerExecuteEvents(x, y);
                for (let i = 0; i < events.length; i++) {
                    events[i].refresh(this);
                    if (events[i].isNormalPriority()) {
                        return true;
                    }
                }
                return false;
            }
            // 目的地に移動中(ターゲット移動時は除外)
            isMovingToDestination() {
                return this._scheduler.isExistDestination() && !this._scheduler.target;
            }
            refresh() {
                const newPageIndex = this._erased ? -1 : this.findProperPageIndex();
                if (this._pageIndex !== newPageIndex) {
                    this._pageIndex = newPageIndex;
                    this.setupPage();
                }
                this.refreshBushDepth();
            }
            onLoadContents() {
                if (this.isSameMap()) {
                    $gameMap._events[this._eventId] = this;
                }
                this.setupPageSettings();
                this.setupEvent();
                if (this._scheduler.isExistDestination()) {
                    this._scheduler.setRoute();
                }
            }
            setupProgress(hour, minute) {
                this._scheduler.setupProgress(hour, minute);
            }
            setupPageSettings() {
                const page = this.page();
                if (!page) {
                    return;
                }
                const image = page.image;
                if (image.tileId > 0) {
                    this.setTileImage(image.tileId);
                }
                else {
                    this.setImage(image.characterName, image.characterIndex);
                }
                if (this._originalDirection !== image.direction) {
                    this._originalDirection = image.direction;
                    this._prelockDirection = 0;
                    this.setDirectionFix(false);
                    this.setDirection(image.direction);
                }
                if (this._originalPattern !== image.pattern) {
                    this._originalPattern = image.pattern;
                    this.setPattern(image.pattern);
                }
                //this.setMoveSpeed(page.moveSpeed);
                this.setMoveFrequency(page.moveFrequency);
                this.setPriorityType(page.priorityType);
                this.setWalkAnime(page.walkAnime);
                this.setStepAnime(page.stepAnime);
                this.setDirectionFix(page.directionFix);
                this.setThrough(page.through);
                this.setMoveRoute(page.moveRoute);
                this._moveType = page.moveType;
                this._trigger = page.trigger;
                if (this._trigger === 4) {
                    this._interpreter = new Game_Interpreter();
                }
                else {
                    this._interpreter = null;
                }
            }
            lock(toWardPlayer = true) {
                if (!this._locked) {
                    this._prelockDirection = this.direction();
                    if (toWardPlayer) {
                        this.turnTowardPlayer();
                    }
                    this._locked = true;
                }
            }
            location() {
                return $globalMap.getLocation(this._mapId, this.x, this.y);
            }
            startRouteExloration(x, y) {
                const start = new GlobalMap.ExplorationNode(this._mapId, this._x, this._y);
                const goal = new GlobalMap.ExplorationNode(this._mapId, x, y);
                this._route = new GlobalMap.RouteExploration(this._mapId, GlobalMap.NotRoadRegionCost);
                this._route.start(start, goal);
            }
            moveStraight(d) {
                super.moveStraight(d);
                if (this.isMovementSucceeded()) {
                    // NOTE:
                    // 妨害後は数歩分の間非接触にする
                    if (this._requestPlayerThrough) {
                        this._requestPlayerThrough = false;
                        this._playerThroughCount = 60;
                    }
                    this._route?.shift();
                    this.checkEventTriggerHereOnDeparture();
                }
            }
            distancePerFrame() {
                const speed = super.distancePerFrame();
                if (this.mapId() !== $gameMap.mapId()) {
                    return speed * $globalMapParameter.npc_off_map_speed_rate;
                }
                return speed;
            }
            collidePlayer() {
                this._collidedPlayer = true;
                this._collidingPlayer = true;
                const d = $gamePlayer.reverseDir($gamePlayer.direction());
                this.setDirection(d);
                this._prelockDirection = d;
                const MaxMinutes = $globalMapParameter.collide_pause_max_minutes;
                this.pause(0, MaxMinutes);
            }
            onArrival() {
                this._collidedPlayer = false;
                this._collidingPlayer = false;
                this.sendArrive();
            }
            send(msg) {
                this._messenger.send(msg);
            }
            sendDeparture() {
                this.send({
                    type: GlobalMap.MessageType.Departure,
                    character: this._characterId,
                    data: {
                        mapId: this.mapId(),
                        col: new HM.Collider({
                            x: this._realX,
                            y: this._realY,
                            expansionArea: this.getExpansionArea(),
                        }),
                    },
                });
            }
            sendArrive() {
                this.send({
                    type: GlobalMap.MessageType.Arrive,
                    character: this.characterId,
                    data: {
                        mapId: this._mapId,
                        col: new HM.Collider({
                            x: this._realX,
                            y: this._realY,
                            expansionArea: this.getExpansionArea(),
                        }),
                    },
                });
            }
            sendJumpTime() {
                this.send({
                    type: GlobalMap.MessageType.JumpTime,
                    character: this._characterId,
                    data: {
                        mapId: this.mapId(),
                        col: new HM.Collider({
                            x: this._realX,
                            y: this._realY,
                            expansionArea: this.getExpansionArea(),
                        }),
                    },
                });
            }
            onSyncActor(sync) {
                this._scheduler.onSyncActor(sync);
            }
            makeSaveContents() {
                return {
                    characterId: this._characterId,
                    validity: this._validity,
                    x: this._x,
                    realX: this._realX,
                    y: this._y,
                    realY: this._realY,
                    direction: this._direction,
                    mapId: this._mapId,
                    eventId: this._eventId,
                    moveSpeed: this._moveSpeed,
                    pageIndex: this._pageIndex,
                    scheduler: this._scheduler.makeSaveContents(),
                    validEventTags: this._validEventTags,
                    costume: this._costume,
                    imgName: this._imgName,
                    collidedPlayer: this._collidedPlayer,
                    collidingPlayer: this._collidingPlayer,
                };
            }
            extractSaveContents(contents) {
                this._characterId = contents.characterId;
                this._validity = contents.validity;
                this._x = contents.x;
                this._y = contents.y;
                this._realX = contents.realX;
                this._realY = contents.realY;
                this._direction = contents.direction;
                this._mapId = contents.mapId;
                this._eventId = contents.eventId;
                this._moveSpeed = contents.moveSpeed;
                this._pageIndex = contents.pageIndex;
                this._scheduler.extractSaveContents(contents.scheduler);
                Object.assign(this._validEventTags, contents.validEventTags);
                this._costume = contents.costume;
                this._imgName = contents.imgName;
                this._collidedPlayer = contents.collidedPlayer;
                this._collidingPlayer = contents.collidingPlayer;
            }
        }
        GlobalMap.MB_NonPlayer = MB_NonPlayer;
    })(GlobalMap = Iz.GlobalMap || (Iz.GlobalMap = {}));
})(Iz || (Iz = {}));
// --------------------------------
// TimeTable
// --------------------------------
(function (Iz) {
    var GlobalMap;
    (function (GlobalMap) {
        class MB_TimeTableRange {
            constructor(first, size) {
                this._first = first;
                this._size = size;
            }
            get firstIndex() {
                return this._first;
            }
            get size() {
                return this._size;
            }
            get lastIndex() {
                return this._first + this._size - 1;
            }
        }
        GlobalMap.MB_TimeTableRange = MB_TimeTableRange;
        class MB_TimeTable {
            constructor(charaId) {
                this._charaId = charaId;
                this._ranges = new Map();
                const charaTable = $dataTimeTable[charaId];
                const keys = [...new Set(charaTable.map((data) => data.group))];
                keys.forEach((key) => {
                    const firstIndex = charaTable.findIndex((table) => table.group === key);
                    const size = charaTable.filter((table) => table.group === key).length;
                    const range = new MB_TimeTableRange(firstIndex, size);
                    this._ranges.set(key, range);
                });
            }
            sheduleToGroup(schedule) {
                const data = $dataSchedule
                    .filter((s) => !!s && s.actor_id === this._charaId)
                    .find((s) => s.schedule === schedule);
                return data ? data.table_group : -1;
            }
            range(schedule) {
                // TODO: 無い場合の扱い
                const group = this.sheduleToGroup(schedule);
                const res = this._ranges.get(group);
                if (!res) {
                    throw new Error(`スケジュールがありません charaId:${this._charaId} schedule:${schedule} group:${group}`);
                }
                return res;
            }
            hasField(schedule, row) {
                const range = this.range(schedule);
                const index = range.firstIndex + row;
                const f = $dataTimeTable[this._charaId][index];
                return !!f;
            }
            field(schedule, row) {
                const range = this.range(schedule);
                const index = range.firstIndex + row;
                return $dataTimeTable[this._charaId][index];
            }
            time(schedule, row) {
                // NOTE:
                // 00:00~任意の時間まで表現できるようにする
                return GlobalMap.toTime(this.hour(schedule, row), this.minute(schedule, row));
            }
            hour(schedule, row) {
                return this.field(schedule, row).hour;
            }
            minute(schedule, row) {
                return this.field(schedule, row).minute;
            }
            mapId(schedule, row) {
                return this.field(schedule, row).mapId;
            }
            destination(schedule, row) {
                const field = this.field(schedule, row);
                return new GlobalMap.Destination({
                    mapId: field.mapId,
                    x: field.x,
                    y: field.y,
                    direction: field.direction,
                });
            }
            moveSpeed(schedule, row) {
                return this.field(schedule, row).moveSpeed;
            }
            syncActor(schedule, row) {
                return this.field(schedule, row).syncActor;
            }
        }
        GlobalMap.MB_TimeTable = MB_TimeTable;
        const EstimateTimeOffsetMax = 10;
        class MB_CharacterScheduler {
            constructor(character) {
                this._characterId = character.characterId;
                this._currentTime = 0;
                this._pause = false;
                this._resumeTime = 0;
                this._schedule = 0;
                this._destination = new GlobalMap.Destination();
                this._destinationStack = [];
                this._targetCharacterId = 0;
                this._progress = 0;
                this._estimateTimeOffset = 0;
                this._route = [];
                const dest = this.timeTable.destination(this._schedule, 0);
                const speed = this.timeTable.moveSpeed(this._schedule, 0);
                // NOTE:
                // validityがfalseでもスケジュールに従ってマップIDや座標が更新されるので
                // なんとかしたい
                if (character.validity) {
                    character.setPosition(dest.x, dest.y, dest.direction, dest.mapId);
                }
                else {
                    character.setPosition(-1, -1, 0, -1);
                }
                character.setMoveSpeed(speed);
            }
            get timeTable() {
                return $globalMap.timeTable(this._characterId);
            }
            get character() {
                return $globalMap.character(this._characterId);
            }
            get target() {
                return this._targetCharacterId <= 0
                    ? null
                    : $globalMap.character(this._targetCharacterId);
            }
            get paused() {
                return this._pause;
            }
            get schedule() {
                return this._schedule;
            }
            get progress() {
                return this._progress;
            }
            setSchedule(schedule) {
                this._schedule = Math.min(Math.max(schedule, 0), Iz.GlobalMap.getChronus().getDaysOfWeek() - 1);
            }
            jumpSpecifiedTime(hour, minute) {
                if (this.isAppear()) {
                    // 出発時のイベントを先に処理するため、インタープリターのセットアップを後に行う
                    const hereEvents = this.character.getHereEventOnDeparture();
                    this.setupProgress(hour, minute);
                    const dest = this.timeTable.destination(this._schedule, this._progress);
                    const speed = this.timeTable.moveSpeed(this._schedule, this._progress);
                    this.character.performTransfer(dest.mapId, dest.x, dest.y, dest.direction, true);
                    this.character.setMoveSpeed(speed);
                    this.setTime(hour, minute);
                    this.updateEstimatedTimeOffset();
                    this.character.setHereEventsOnDeparture(hereEvents);
                }
                else {
                    this.character.setPosition(-1, -1, 0, -1);
                }
            }
            setupProgress(hour, minute) {
                // 進行度を指定時刻に合うようにする
                const targetTime = GlobalMap.toGMTime(hour, minute);
                const size = this.timeTable.range(this._schedule).size;
                let progress = 0;
                for (let i = 0; i < size; i++) {
                    const estimatedTime = this.timeTable.time(this._schedule, i);
                    if (estimatedTime <= targetTime) {
                        progress = i;
                    }
                    else {
                        break;
                    }
                }
                // console.log(`[NPC:${this._characterId}] ${this._progress} -> ${progress}`);
                this._progress = progress;
            }
            clearDestination() {
                this._destination.clear();
                this._destinationStack = [];
                this._route = [];
            }
            setTime(hour, minute) {
                this._currentTime = GlobalMap.toGMTime(hour, minute);
            }
            setTarget(target) {
                this._targetCharacterId = $globalMap.characterId(target);
                if (this.target) {
                    if (this.isExistDestination()) {
                        this._destinationStack.push(new GlobalMap.Destination(this._destination));
                    }
                }
                else {
                    if (!this.checkDestinationStack()) {
                        const hour = Iz.GlobalMap.getChronus().getHour();
                        const minute = Iz.GlobalMap.getChronus().getMinute();
                        this.setTime(hour, minute);
                        if (!this.checkAppointments()) {
                            const dest = this.timeTable.destination(this._schedule, this._progress);
                            dest.disableArrivalProgress = true;
                            this._destination.set(dest);
                            this.setRoute();
                        }
                    }
                }
            }
            destination() {
                return this._destination;
            }
            hasDestinationStack() {
                return this._destinationStack.length > 0;
            }
            position() {
                return {
                    x: this.character.x,
                    y: this.character.y,
                    mapId: this.character.mapId(),
                    direction: this.character.direction(),
                };
            }
            newday(hour, dayOfWeek) {
                // this._currentTime = hour;
                this.setTime(hour, 0);
                this._schedule = dayOfWeek >= Iz.GlobalMap.getChronus().getDaysOfWeek() ? 0 : dayOfWeek;
                this._destination.clear();
                this._progress = 0;
                this._estimateTimeOffset = 0;
                this._route = [];
            }
            update(hour, minute) {
                // 登場しない日は npc の配列から除去したほうがいいかもね。
                if (!this.isAppear()) {
                    return;
                }
                // if ($gameMap.isEventRunning()) {
                //     return;
                // }
                if (this._pause) {
                    if (GlobalMap.toGMTime(hour, minute) < this._resumeTime) {
                        return;
                    }
                    this.resume();
                }
                if (this.target) {
                    let mapId = $gameMap.mapId();
                    mapId = this.target instanceof Game_Event ? this.target.mapId() : mapId;
                    this._destination.set({
                        mapId: mapId,
                        x: this.target.x,
                        y: this.target.y,
                        direction: this.target.direction(),
                        disableArrivalProgress: true,
                    });
                    this.setRoute();
                    // console.log(`[NPC:${this._characterId}] search targt ${this._targetCharacterId}`);
                }
                if (this.isExistDestination()) {
                    this.goTowardsDestination();
                }
                else if (this.checkDestinationStack()) {
                    this.goTowardsDestination();
                }
                else {
                    if (this._currentTime !== GlobalMap.toGMTime(hour, minute)) {
                        this.setTime(hour, minute);
                        const size = this.timeTable.range(this._schedule).size;
                        if (!this.isTerminated()) {
                            const res = this.checkAppointments();
                            if (res) {
                                this.checkSyncActor(this._schedule, this._progress + 1);
                            }
                        }
                    }
                }
            }
            checkSyncActor(schedule, progress) {
                const syncActor = this.timeTable.syncActor(schedule, progress);
                if (syncActor <= 0)
                    return;
                $globalMap.onSyncActor({
                    srcActor: this._characterId,
                    targetActor: syncActor,
                    hour: this.timeTable.hour(schedule, progress),
                    minute: this.timeTable.minute(schedule, progress),
                });
            }
            checkAppointments() {
                const nextProgress = this._progress + 1;
                const tableEstimatedTime = this.timeTable.time(this._schedule, nextProgress);
                const estimatedTime = GlobalMap.clampTime(tableEstimatedTime + this._estimateTimeOffset);
                if (estimatedTime <= this._currentTime) {
                    this._destination = this.timeTable.destination(this._schedule, nextProgress);
                    this.character.setMoveSpeed(this.timeTable.moveSpeed(this._schedule, nextProgress));
                    this.setRoute();
                    // console.log(`[NPC:${this._characterId}] start: ${this._progress} -> ${nextProgress}`);
                    this.updateEstimatedTimeOffset();
                    this.character.sendDeparture();
                    return true;
                }
                return false;
            }
            updateEstimatedTimeOffset() {
                // -10~0
                this._estimateTimeOffset = -Math.randomInt(EstimateTimeOffsetMax + 1);
            }
            arrival() {
                this.character.setDirection(this._destination.direction);
                this.character.checkEventTriggerTouchFront(this._destination.direction);
                if (!this._destination.disableArrivalProgress) {
                    this._progress++;
                }
                this._destination.clear();
                this.character.onArrival();
                // console.log(`[NPC:${this._characterId}] arrival: ${this._progress}`);
            }
            setRoute() {
                if (this._destination.mapId !== this.character.mapId()) {
                    this._route = $globalMap.routeFromMapToMap(this._destination.mapId, this.character.mapId());
                    this._route.shift();
                    const entrance = $globalMap.entranceFromMaptoMap(this._route[0], this.character.mapId());
                    this.character.startRouteExloration(entrance.x, entrance.y);
                }
                else {
                    this.character.startRouteExloration(this._destination.x, this._destination.y);
                    this._route = [];
                }
            }
            goTowardsDestination() {
                if (this._route.length > 0) {
                    this.goTowardsRelayPoint();
                }
                else {
                    // let direction = this._character.findDirectionTo(this._destination.x, this._destination.y);
                    const direction = this.character.findDirectionTo();
                    if (direction > 0) {
                        //this._character.checkEventTriggerTouchFront(this._character.direction());
                        this.character.moveStraight(direction);
                    }
                    else {
                        this.arrival();
                    }
                }
            }
            goTowardsRelayPoint() {
                // let entrance = $globalMap.entranceFromMaptoMap(this._route[0], this.character.mapId());
                // const direction = this._character.findDirectionTo(entrance.x, entrance.y);
                const direction = this.character.findDirectionTo();
                if (direction > 0) {
                    //this._character.checkEventTriggerTouchFront(this._character.direction());
                    this.character.moveStraight(direction);
                }
                else {
                    const mapId = this._route.shift();
                    const exit = $globalMap.exitFromMaptoMap(mapId, this.character.mapId());
                    this.character.performTransfer(mapId, exit.x, exit.y, this.character.direction(), false);
                    if (this._route.length > 0) {
                        const entrance = $globalMap.entranceFromMaptoMap(this._route[0], this.character.mapId());
                        this.character.startRouteExloration(entrance.x, entrance.y);
                    }
                    else {
                        this.character.startRouteExloration(this._destination.x, this._destination.y);
                    }
                }
            }
            checkDestinationStack() {
                if (0 < this._destinationStack.length) {
                    this._destination = this._destinationStack.pop();
                    this.setRoute();
                    this.character.sendDeparture();
                    return true;
                }
                return false;
            }
            pause(hour, minute) {
                this._pause = true;
                this._resumeTime = GlobalMap.toGMTime(GlobalMap.getChronus().getHour() + hour, GlobalMap.getChronus().getMinute() + minute);
            }
            resume() {
                this._pause = false;
                this._resumeTime = 0;
                this.character.onResume();
            }
            interrupt(mapId, x, y, direction) {
                if (this.target) {
                    this._targetCharacterId = 0;
                    this._destination.clear();
                }
                if (this.isExistDestination()) {
                    this._destinationStack.push(new GlobalMap.Destination(this._destination));
                }
                this._destination.set({
                    x: x,
                    y: y,
                    mapId: mapId,
                    direction: direction,
                    disableArrivalProgress: true,
                });
                this.setRoute();
            }
            isExistDestination() {
                return this._destination.mapId > 0;
            }
            isAppear() {
                return this.timeTable.range(this._schedule).size > 0;
            }
            isTerminated() {
                return this._progress >= this.timeTable.range(this._schedule).size - 1;
            }
            onSyncActor(sync) {
                if (this._pause)
                    return;
                if (this.target)
                    return;
                if (this.isExistDestination())
                    return;
                if (this._destinationStack.length > 0)
                    return;
                const progress = this._progress + 1;
                if (!this.timeTable.hasField(this._schedule, progress))
                    return;
                if (this.timeTable.hour(this._schedule, progress) !== sync.hour)
                    return;
                if (this.timeTable.minute(this._schedule, progress) !== sync.minute)
                    return;
                this._estimateTimeOffset = -EstimateTimeOffsetMax;
                this.checkAppointments();
            }
            makeSaveContents() {
                return {
                    currentTime: this._currentTime,
                    pause: this._pause,
                    resumeTime: this._resumeTime,
                    schedule: this._schedule,
                    destination: GlobalMap.createPlainDestination(this._destination),
                    destinationStack: this._destinationStack.map((dest) => GlobalMap.createPlainDestination(dest)),
                    targetCharacterId: this._targetCharacterId,
                    progress: this._progress,
                    estimateTimeOffset: this._estimateTimeOffset,
                    route: this._route,
                };
            }
            extractSaveContents(contents) {
                this._currentTime = contents.currentTime;
                this._pause = contents.pause;
                this._resumeTime = contents.resumeTime;
                this._schedule = contents.schedule;
                this._destination = new GlobalMap.Destination(contents.destination);
                this._destinationStack = contents.destinationStack.map((dest) => new GlobalMap.Destination(dest));
                this._targetCharacterId = contents.targetCharacterId;
                this._progress = contents.progress;
                this._estimateTimeOffset = contents.estimateTimeOffset;
                this._route = contents.route;
            }
            changeSchedule(schedule, hour, minute) {
                this._schedule = schedule;
                this.clearDestination();
                this.setupProgress(hour, minute);
            }
        }
        GlobalMap.MB_CharacterScheduler = MB_CharacterScheduler;
    })(GlobalMap = Iz.GlobalMap || (Iz.GlobalMap = {}));
})(Iz || (Iz = {}));
// --------------------------------
// GlobalMap
// --------------------------------
(function (Iz) {
    var GlobalMap;
    (function (GlobalMap) {
        GlobalMap.ColliderEventType = {
            OnTriggerEnter: 1,
            // OnTriggerStay: 2,
            OnTriggerExit: 3,
            // OnColliderEnter: 4,
            // OnColliderStay: 5,
            // OnColliderExit: 6,
            OnDeparture: 7,
            //
            OnArrive: 9,
            OnJumpTime: 10,
        };
        const OnColliderEvent = 1101;
        const OnTriggerEnterSwitchId = 1111;
        const OnTriggerExitSwitchId = 1113;
        const OnDepartureSwitchId = 1117;
        const OnArriveSwitchId = 1119;
        const OnJumpTimeSwitchId = 1120;
        function isColliderEvent(e) {
            for (let i = 0; i < e.pages.length; i++) {
                const page = e.pages[i];
                const cond = page.conditions;
                if (cond.switch1Id === OnColliderEvent)
                    return true;
            }
            return false;
        }
        class ColliderEvent {
            constructor() {
                this._eventId = 0;
                this._eventPages = new Map();
            }
            get id() {
                return this._eventId;
            }
            setup(e) {
                this._eventId = e.id;
                this._eventPages.clear();
                for (let i = 0; i < e.pages.length; i++) {
                    const page = e.pages[i];
                    const cond = page.conditions;
                    if (cond.switch2Valid && cond.switch2Id === OnTriggerEnterSwitchId) {
                        this._eventPages.set(GlobalMap.ColliderEventType.OnTriggerEnter, i);
                    }
                    else if (cond.switch2Valid && cond.switch2Id === OnTriggerExitSwitchId) {
                        this._eventPages.set(GlobalMap.ColliderEventType.OnTriggerExit, i);
                    }
                    else if (cond.switch2Valid && cond.switch2Id === OnDepartureSwitchId) {
                        this._eventPages.set(GlobalMap.ColliderEventType.OnDeparture, i);
                    }
                    else if (cond.switch2Valid && cond.switch2Id === OnArriveSwitchId) {
                        this._eventPages.set(GlobalMap.ColliderEventType.OnArrive, i);
                    }
                    else if (cond.switch2Valid && cond.switch2Id === OnJumpTimeSwitchId) {
                        this._eventPages.set(GlobalMap.ColliderEventType.OnJumpTime, i);
                    }
                }
            }
            getPageIndex(event) {
                const page = this._eventPages.get(event);
                if (page === undefined) {
                    return -1;
                }
                return page;
            }
        }
        class MB_EventCollider {
            constructor(e) {
                this._event = new ColliderEvent();
                this._event.setup(e);
                this._data = e;
                this._collider = new HM.Collider({
                    x: this._data.x,
                    y: this._data.y,
                    expansionArea: HM.getExpansionArea(this._data.meta),
                });
            }
            get event() {
                return this._event;
            }
            get data() {
                return this._data;
            }
            get collider() {
                return this._collider;
            }
        }
        GlobalMap.MB_EventCollider = MB_EventCollider;
        class MB_MapCollider {
            constructor(mapId, list) {
                this._mapId = mapId;
                this._events = new Map();
                list.forEach((e) => {
                    if (!e)
                        return;
                    if (!isColliderEvent(e))
                        return;
                    this._events.set(e.id, new MB_EventCollider(e));
                });
            }
            onMove(character, data) {
                for (const pair of this._events) {
                    const col = pair[1].collider;
                    // 現在接触
                    if (HM.isCollidedExcludeBorder(col, data.current.col)) {
                        if (!HM.isCollidedExcludeBorder(col, data.prev.col)) {
                            this.onTriggerEnterEvent(character, pair[0]);
                        }
                    }
                    // 現在非接触
                    else {
                        if (HM.isCollidedExcludeBorder(col, data.prev.col)) {
                            this.onTriggerExitEvent(character, pair[0]);
                        }
                    }
                }
            }
            onDeparture(character, data) {
                for (const pair of this._events) {
                    const col = pair[1].collider;
                    if (HM.isCollidedExcludeBorder(col, data.col)) {
                        this.onDepartureEvent(character, pair[0]);
                    }
                }
            }
            onTriggerExit(character, data) {
                for (const pair of this._events) {
                    const col = pair[1].collider;
                    if (HM.isCollidedExcludeBorder(col, data.col)) {
                        this.onTriggerExitEvent(character, pair[0]);
                    }
                }
            }
            onTriggerEnter(character, data) {
                for (const pair of this._events) {
                    const col = pair[1].collider;
                    if (HM.isCollidedExcludeBorder(col, data.col)) {
                        this.onTriggerEnterEvent(character, pair[0]);
                    }
                }
            }
            onArrive(character, data) {
                for (const pair of this._events) {
                    const col = pair[1].collider;
                    if (HM.isCollidedExcludeBorder(col, data.col)) {
                        this.onArriveEvent(character, pair[0]);
                    }
                }
            }
            onJumpTime(character, data) {
                for (const pair of this._events) {
                    const col = pair[1].collider;
                    if (HM.isCollidedExcludeBorder(col, data.col)) {
                        this.onJumpTimeEvent(character, pair[0]);
                    }
                }
            }
            getEventPage(eventId, type) {
                const e = this._events.get(eventId);
                if (!e)
                    return undefined;
                const pageIndex = e.event.getPageIndex(type);
                if (pageIndex < 0)
                    return undefined;
                return e.data.pages[pageIndex];
            }
            getCollidedEvents(col) {
                const eventIds = [];
                for (const pair of this._events) {
                    const eventCol = pair[1].collider;
                    if (HM.isCollidedExcludeBorder(col, eventCol)) {
                        eventIds.push(pair[0]);
                    }
                }
                return eventIds;
            }
            onExecEvent(character, eventId, eventType) {
                const eventCollider = this._events.get(eventId);
                if (!eventCollider)
                    return;
                const page = eventCollider.event.getPageIndex(eventType);
                if (page < 0)
                    return;
                $globalMap.requestColliderEvent({
                    mapId: this._mapId,
                    eventId: eventId,
                    executeCharacter: character,
                    type: eventType,
                });
            }
            onTriggerEnterEvent(character, eventId) {
                this.onExecEvent(character, eventId, GlobalMap.ColliderEventType.OnTriggerEnter);
            }
            onTriggerExitEvent(character, eventId) {
                this.onExecEvent(character, eventId, GlobalMap.ColliderEventType.OnTriggerExit);
            }
            onArriveEvent(character, eventId) {
                this.onExecEvent(character, eventId, GlobalMap.ColliderEventType.OnArrive);
            }
            onDepartureEvent(character, eventId) {
                this.onExecEvent(character, eventId, GlobalMap.ColliderEventType.OnDeparture);
            }
            onJumpTimeEvent(character, eventId) {
                this.onExecEvent(character, eventId, GlobalMap.ColliderEventType.OnJumpTime);
            }
        }
        GlobalMap.MB_MapCollider = MB_MapCollider;
        let MessageType;
        (function (MessageType) {
            MessageType[MessageType["None"] = 0] = "None";
            MessageType[MessageType["Move"] = 1] = "Move";
            MessageType[MessageType["Departure"] = 2] = "Departure";
            MessageType[MessageType["Arrive"] = 3] = "Arrive";
            MessageType[MessageType["TriggerExit"] = 4] = "TriggerExit";
            MessageType[MessageType["TriggerEnter"] = 5] = "TriggerEnter";
            MessageType[MessageType["CollidePlayer"] = 11] = "CollidePlayer";
            MessageType[MessageType["Happening"] = 12] = "Happening";
            MessageType[MessageType["JumpTime"] = 20] = "JumpTime";
        })(MessageType = GlobalMap.MessageType || (GlobalMap.MessageType = {}));
        class Messenger {
            send(msg) {
                $globalMap.onSendMessage(msg);
            }
        }
        GlobalMap.Messenger = Messenger;
        // 並列用
        class Game_ColliderEvent {
            constructor() {
                this.initialize();
            }
            initialize() {
                this._interpreter = new Game_Interpreter();
                this._list = [];
            }
            request(data) {
                this._list.push(data);
            }
            update() {
                this.updateInterpreter();
            }
            updateInterpreter() {
                for (;;) {
                    this._interpreter.update();
                    if (this._interpreter.isRunning()) {
                        return;
                    }
                    this._interpreter.clear();
                    const data = this._list.shift();
                    if (!data)
                        return;
                    const page = $globalMap
                        .mapCollider(data.mapId)
                        .getEventPage(data.eventId, data.type);
                    if (!page)
                        return;
                    this._interpreter.setup(page.list);
                    this._interpreter.setExecuteCharacter(data.executeCharacter);
                }
            }
        }
        GlobalMap.Game_ColliderEvent = Game_ColliderEvent;
    })(GlobalMap = Iz.GlobalMap || (Iz.GlobalMap = {}));
})(Iz || (Iz = {}));
(function (Iz) {
    var GlobalMap;
    (function (GlobalMap) {
        //==================================================
        // MB_NPCExecuteEvent
        // NPCが実行するイベントデータを保持、アクセサを提供するクラス
        //==================================================
        class MB_NPCExecuteEvent {
            constructor(mapId, eventId, event) {
                this._mapId = mapId;
                this._eventId = eventId;
                this._character = null;
                this._event = event;
                this._pageIndex = -1;
            }
            get x() {
                return this._event.x;
            }
            get y() {
                return this._event.y;
            }
            get meta() {
                return this._event.meta;
            }
            eventId() {
                return this._eventId;
            }
            mapId() {
                return this._mapId;
            }
            list() {
                return this.page().list;
            }
            page() {
                return this._event.pages[this._pageIndex];
            }
            setCharacter(chara) {
                this._character = chara;
            }
            setPosition(x, y) {
                this._event.x = x;
                this._event.y = y;
            }
            refresh(chara) {
                this.setCharacter(chara);
                this._pageIndex = this.findProperPageIndex();
            }
            findProperPageIndex() {
                const pages = this._event.pages;
                for (let i = pages.length - 1; i >= 0; i--) {
                    const page = pages[i];
                    if (this.meetsConditions(page)) {
                        return i;
                    }
                }
                return -1;
            }
            meetsConditions(page) {
                const c = page.conditions;
                // if (c.switch1Valid) {
                //     if (!$gameSwitches.value(c.switch1Id)) {
                //         return false;
                //     }
                // }
                if (c.switch2Valid) {
                    if (!$gameSwitches.value(c.switch2Id)) {
                        return false;
                    }
                }
                if (c.variableValid) {
                    if ($gameVariables.value(c.variableId) < c.variableValue) {
                        return false;
                    }
                }
                if (c.selfSwitchValid) {
                    const key = [this._mapId, this._eventId, c.selfSwitchCh];
                    if ($gameSelfSwitches.value(key) !== true) {
                        return false;
                    }
                }
                if (c.itemValid) {
                    const item = $dataItems[c.itemId];
                    if (!$gameParty.hasItem(item)) {
                        return false;
                    }
                }
                if (c.actorValid) {
                    if (!(c.actorId === this._character?.characterId)) {
                        return false;
                    }
                }
                return true;
            }
            isParallel() {
                const page = this.page();
                if (!page) {
                    return false;
                }
                return page.trigger === 4;
            }
            isTile() {
                const page = this.page();
                if (!page) {
                    return false;
                }
                return page.priorityType === 0;
            }
            isNormalPriority() {
                const page = this.page();
                if (!page) {
                    return false;
                }
                return page.priorityType === 1;
            }
            setPageIndex(index) {
                this._pageIndex = index;
            }
            collider() {
                return new HM.Collider({
                    x: this.x,
                    y: this.y,
                    expansionArea: HM.getExpansionArea(this._event.meta),
                });
            }
        }
        GlobalMap.MB_NPCExecuteEvent = MB_NPCExecuteEvent;
        //==================================================
        // MB_Map
        // 各マップの接続やリージョン情報
        // NPC実行イベントの配列を保持し、アクセサを提供するクラス
        //==================================================
        const NullMapCollider = new GlobalMap.MB_MapCollider(0, []);
        class MB_Map {
            constructor(mapId) {
                this._mapId = mapId;
                // this._data = {};
                this._nonPlayerExecuteEvents = [];
                this._mapIdToEntranceMap = {};
                this.setupDataStr();
                this.defineNpcExecuteEvents();
                this.defineMapCollider(mapId);
                this.defineEntrances();
                this.readMapData();
                this.clearDataStr();
            }
            get width() {
                return this._data.width;
            }
            get height() {
                return this._data.height;
            }
            get mapCollider() {
                return this._mapCollider;
            }
            static getPagesMeetsCondition(event, condition) {
                return event.pages
                    .filter((page) => page.conditions.switch1Id === condition && page.conditions.switch1Valid)
                    .map((page) => page);
            }
            static getLocationTransferMapId(pages) {
                const mapIds = [];
                for (let i = 0; i < pages.length; i++) {
                    const command = pages[i].list.find((cmd) => cmd.code === GlobalMap.LocationTransferCode);
                    if (command) {
                        mapIds.push(command.parameters[1]);
                    }
                }
                return mapIds;
            }
            setupDataStr() {
                // クローン用のデータ文字列を取得
                const fs = require("fs");
                const path = require("path");
                const dir = Iz.GlobalMap.relativePath("data");
                const file = "Map" + ("000" + this._mapId).slice(-3) + ".json";
                this._dataStr = fs.readFileSync(path.join(dir, file), "utf8");
            }
            clearDataStr() {
                this._dataStr = "";
            }
            readMapData() {
                this._data = JSON.parse(this._dataStr);
                this._data.events.filter((e) => e).forEach((e) => DataManager.extractMetadata(e));
            }
            defineNpcExecuteEvents() {
                this.readMapData();
                const events = this._data.events.filter((e) => !!e);
                events.forEach((e) => {
                    const pages = MB_Map.getPagesMeetsCondition(e, $globalMapParameter.execute_event_switch);
                    if (pages.length) {
                        e.pages = pages;
                        GlobalMap.applyInitEventPos(e);
                        this._nonPlayerExecuteEvents.push(new MB_NPCExecuteEvent(this._mapId, e.id, e));
                    }
                });
            }
            defineMapCollider(mapId) {
                this.readMapData();
                const list = this._data.events.filter((e) => !!e);
                this._mapCollider = new GlobalMap.MB_MapCollider(mapId, list);
            }
            defineEntrances() {
                this.readMapData();
                const events = this._data.events.filter((e) => !!e);
                events.forEach((e) => {
                    const pages = MB_Map.getPagesMeetsCondition(e, $globalMapParameter.location_transfer_switch);
                    if (pages.length) {
                        const mapIds = MB_Map.getLocationTransferMapId(pages);
                        mapIds.forEach((mapId) => {
                            if (Object.hasOwn(this._mapIdToEntranceMap, mapId)) {
                                throw new Error(`定義済みのマップ接続を上書きしようとしました。マップID:${this._mapId} イベントID:${e.id} イベント名:${e.name}`);
                            }
                            GlobalMap.applyInitEventPos(e);
                            this._mapIdToEntranceMap[mapId] = { x: e.x, y: e.y };
                        });
                    }
                });
            }
            addNpcExecuteEvent(event, x, y) {
                // NOTE:
                // 使用する場合はlodashを使わない実装にすること
                throw new Error("target is null or undefined.");
                /*
                var clonedeep = require('lodash/cloneDeep');
                event = clonedeep(event);
                const pages = MB_Map.getPagesMeetsCondition(event, MB_Map.npcExecuteEventSwitch());
                if (pages.length) {
                    event.pages = pages;
                    const lastIndex = this._nonPlayerExecuteEvents.length - 1;
                    const eventId = this._nonPlayerExecuteEvents[lastIndex].eventId() + 1;
                    const npcEvent = new MB_NPCExecuteEvent(this._mapId, eventId, event);
                    npcEvent.setPosition(x, y);
                    this._nonPlayerExecuteEvents.push(npcEvent);
                }
                */
            }
            connectMapIds() {
                return Object.keys(this._mapIdToEntranceMap).map((id) => Number.parseInt(id));
            }
            entrance(mapId) {
                return this._mapIdToEntranceMap[mapId];
            }
            nonPlayerExecuteEvents(x, y) {
                if (GlobalMap.isHalfMove) {
                    return this._nonPlayerExecuteEvents.filter((event) => {
                        return HM.isInCollider(event.collider(), x, y);
                    });
                }
                return this._nonPlayerExecuteEvents.filter((event) => event.x === x && event.y === y);
            }
            nonPlayerExecuteEventFromEventId(eventId) {
                return this._nonPlayerExecuteEvents.find((event) => event.eventId() === eventId);
            }
            regionId(x, y) {
                const RegionLayer = 5;
                return this.isValid(x, y) ? this.tileId(x, y, RegionLayer) : 0;
            }
            tileId(x, y, z) {
                const width = this.width;
                const height = this.height;
                return this._data.data[(z * height + y) * width + x] || 0;
            }
            hasConnection(mapId) {
                return this._mapIdToEntranceMap[mapId] !== undefined;
            }
            isRoadRegion(x, y) {
                return this.getRoadRegion(x, y) !== undefined;
            }
            getRoadRegion(x, y) {
                const regionId = this.regionId(x, y);
                return Iz.GlobalMap.calcCosts.get(regionId);
            }
            isValid(x, y) {
                return x >= 0 && x < this.width && y >= 0 && y < this.height;
            }
            makeSaveContents(contents) { }
            extractSaveContents(contents) { }
        }
        GlobalMap.MB_Map = MB_Map;
        //==================================================
        // 各マップの接続情報などを持つ広域マップクラス
        //==================================================
        class MB_GlobalMap {
            constructor() {
                this._data = {};
                this._rootCache = {};
                this._nonPlayers = [];
                this._nonPlayerEvents = {};
                this._timeTables = [];
                this._pause = false;
                this._messenger = new GlobalMap.Messenger();
                this._gameColliderEvent = new GlobalMap.Game_ColliderEvent();
                this._messageHandler = undefined;
            }
            static npcEventMapId() {
                return $globalMapParameter.event_map_id;
            }
            appearNonPlayer(actorId, hour) {
                const npc = this.nonPlayer(actorId);
                npc.validity = true;
                npc.setSchedule(Iz.GlobalMap.getChronus().getWeekIndex());
                npc.jumpSpecifiedTime(hour);
            }
            leaveNonPlayer(actorId) {
                const npc = this.nonPlayer(actorId);
                npc.validity = false;
                if (npc.isSameMap()) {
                    const scene = SceneManager._scene;
                    scene.removeCharacter(npc);
                    $gameMap.removeNpc(npc.eventId());
                }
            }
            setup() {
                this._messageHandler = undefined;
                this.setupEachMaps();
                this.setupTimeTables();
                this.setupNonPlayers();
            }
            setupEachMaps() {
                const fs = require("fs");
                const path = Iz.GlobalMap.relativePath("data");
                // Map の接続定義
                fs.readdirSync(path).forEach((file) => {
                    let mapId = file.match(/[0-9]{3}/);
                    if (mapId) {
                        mapId = Number.parseInt(mapId[0]);
                        this._data[mapId] = new MB_Map(mapId);
                    }
                });
            }
            setupTimeTables() {
                $dataTimeTable.forEach((list, i) => {
                    if (list) {
                        this._timeTables[i] = new GlobalMap.MB_TimeTable(i);
                    }
                });
            }
            setupNonPlayers() {
                const fs = require("fs");
                const path = require("path");
                const dir = Iz.GlobalMap.relativePath("data");
                const file = "Map" + ("000" + MB_GlobalMap.npcEventMapId()).slice(-3) + ".json";
                /**
                 * @type {RPG.Map}
                 */
                const mapdata = JSON.parse(fs.readFileSync(path.join(dir, file), "utf8"));
                mapdata.events.filter((e) => e).forEach((e) => DataManager.extractMetadata(e));
                mapdata.events
                    .filter((e) => e && $dataNonPlayers.find((data) => data && data.name === e.name))
                    .forEach((e) => {
                    if (this._nonPlayerEvents[e.name] === undefined) {
                        this._nonPlayerEvents[e.name] = new GlobalMap.NonPlayerEvent();
                        this._nonPlayerEvents[e.name].events = {};
                        $globalMapParameter.npc_events.forEach((define) => {
                            this._nonPlayerEvents[e.name].events[define.name] = [];
                        });
                    }
                    $globalMapParameter.npc_events.forEach((define) => {
                        if (Object.hasOwn(e.meta, define.name)) {
                            const id = Number.parseInt(e.meta[define.name].toString());
                            if (Number.isInteger(id)) {
                                this._nonPlayerEvents[e.name].events[define.name][id] = e;
                            }
                            else {
                                this._nonPlayerEvents[e.name].events[define.name][0] = e;
                            }
                        }
                    });
                });
                if (!Object.keys(this._nonPlayerEvents).length) {
                    throw new Error(`MapId:data/${MB_GlobalMap.npcEventMapId()}.json ファイルが見つかりません。NPCイベントの取得に失敗しました。`);
                }
                const NpcStartId = 2;
                $dataNonPlayers.forEach((npc) => {
                    if (npc && npc.id >= NpcStartId) {
                        this._nonPlayers[npc.id] = new GlobalMap.MB_NonPlayer(npc, this._messenger);
                    }
                });
            }
            update() {
                if (this._pause) {
                    return;
                }
                this.nonPlayers()
                    .filter((npc) => !npc.isSameMap())
                    .forEach((npc) => npc.update());
            }
            postUpdate() {
                this._gameColliderEvent.update();
            }
            requestColliderEvent(r) {
                this._gameColliderEvent.request(r);
            }
            map(mapId) {
                return this._data[mapId];
            }
            mapCollider(mapId) {
                const map = this.map(mapId);
                if (!map) {
                    return NullMapCollider;
                }
                return map.mapCollider;
            }
            character(actorId) {
                if (actorId === 1) {
                    return $gamePlayer;
                }
                return this._nonPlayers[actorId];
            }
            characterId(character) {
                if (character) {
                    return character instanceof Game_Player ? 1 : character.characterId;
                }
                return 0;
            }
            nonPlayer(actorId) {
                return this._nonPlayers[actorId];
            }
            nonPlayers() {
                return Object.values(this._nonPlayers).filter((npc) => npc.validity);
            }
            nonPlayerIds(all) {
                return Object.keys(this._nonPlayers)
                    .map((s) => Number.parseInt(s))
                    .filter((id) => {
                    if (!all) {
                        return this._nonPlayers[id].validity;
                    }
                    return true;
                });
            }
            timeTable(actorId) {
                return this._timeTables[actorId];
            }
            findEvent(characterName, eventTag, id = 0) {
                const dict = this._nonPlayerEvents[characterName];
                if (!dict) {
                    return undefined;
                }
                const events = dict.events[eventTag];
                if (!events) {
                    return undefined;
                }
                return events[id];
            }
            pause() {
                this._pause = true;
            }
            resume() {
                this._pause = false;
            }
            route(dstMapId, srcMapId = $gameMap.mapId(), relayMapId = srcMapId, routes = [], currentRoute = []) {
                if (currentRoute.indexOf(relayMapId) >= 0) {
                    return false;
                }
                currentRoute.push(relayMapId);
                if (!Object.hasOwn(this._data, relayMapId)) {
                    throw new Error(`MapId:${relayMapId}に有効な場所移動イベントが存在しません。`);
                }
                if (this._data[relayMapId].hasConnection(dstMapId)) {
                    currentRoute.push(dstMapId);
                    routes.push(currentRoute.slice());
                    return currentRoute;
                }
                const connectMapIds = this._data[relayMapId].connectMapIds();
                const results = connectMapIds.filter((mapId) => this.route(dstMapId, srcMapId, mapId, routes, currentRoute.slice()) !== false);
                if (results.length > 0) {
                    return currentRoute;
                }
                return false;
            }
            routeFromMapToMap(dstMapId, srcMapId = $gameMap.mapId()) {
                if (dstMapId === srcMapId) {
                    return [dstMapId];
                }
                if (this._rootCache[srcMapId] === undefined) {
                    this._rootCache[srcMapId] = {};
                }
                else if (this._rootCache[srcMapId][dstMapId] !== undefined) {
                    return this._rootCache[srcMapId][dstMapId].slice();
                }
                this._rootCache[srcMapId][dstMapId] = [];
                const routes = [];
                this.route(dstMapId, srcMapId, srcMapId, routes);
                if (routes.length > 0) {
                    const route = routes
                        .sort((a, b) => {
                        return a.length - b.length;
                    })[0]
                        .slice();
                    this._rootCache[srcMapId][dstMapId] = route;
                    return route.slice();
                }
                return [];
            }
            // 移動元 srcMapId と移動先 dstMapId を引数として
            // 移動先の入り口となる座標を取得する
            // NPCが移動する際に向かう先として使用する
            entranceFromMaptoMap(dstMapId, srcMapId) {
                return this._data[srcMapId].entrance(dstMapId);
            }
            // 移動元 srcMapId と移動先 dstMapId を引数として
            // 移動先の出口となる座標を取得する
            // NPCがマップ移動を行った際の初期座標として使用する
            exitFromMaptoMap(dstMapId, srcMapId) {
                return this._data[dstMapId].entrance(srcMapId);
            }
            makeSaveContents(contents) {
                contents.globalMap = {
                    nonPlayers: this.npcMakeSaveContents(),
                };
                Object.values(this._data).forEach((localMap) => localMap.makeSaveContents(contents));
            }
            npcMakeSaveContents() {
                const nonPlayers = [];
                Object.keys(this._nonPlayers)
                    .map((s) => Number.parseInt(s))
                    .forEach((key) => {
                    nonPlayers[key] = this._nonPlayers[key].makeSaveContents();
                });
                return nonPlayers;
            }
            extractSaveContents(contents) {
                if (!contents.globalMap)
                    return;
                this.setup();
                this.npcExtractSaveContents(contents);
                Object.values(this._data).forEach((localMap) => localMap.extractSaveContents(contents));
            }
            npcExtractSaveContents(contents) {
                Object.keys(this._nonPlayers)
                    .map((s) => Number.parseInt(s))
                    .filter((key) => contents.globalMap.nonPlayers[key])
                    .forEach((key) => this._nonPlayers[key].extractSaveContents(contents.globalMap.nonPlayers[key]));
                this.nonPlayers().forEach((npc) => npc.onLoadContents());
            }
            getLocation(mapId, x, y) {
                const list = $dataLocationMap.get(mapId);
                if (!list)
                    return undefined;
                return list.find((location) => {
                    if (!(x >= location.left))
                        return false;
                    if (!(x <= location.right))
                        return false;
                    if (!(y >= location.top))
                        return false;
                    if (!(y <= location.bottom))
                        return false;
                    return true;
                });
            }
            onSendMessage(msg) {
                if (msg.type === GlobalMap.MessageType.None) {
                    return;
                }
                if (msg.type === GlobalMap.MessageType.Move) {
                    const data = msg.data;
                    if (data.prev.mapId !== data.current.mapId) {
                        const col = data.prev.col;
                        const collidedEvents = this.mapCollider(data.prev.mapId).getCollidedEvents(col);
                        collidedEvents.forEach((eventId) => {
                            this.mapCollider(data.prev.mapId).onTriggerExitEvent(msg.character, eventId);
                        });
                    }
                    this.mapCollider(data.current.mapId).onMove(msg.character, data);
                    return;
                }
                if (msg.type === GlobalMap.MessageType.Departure) {
                    this.mapCollider(msg.data.mapId).onDeparture(msg.character, msg.data);
                    return;
                }
                if (msg.type === GlobalMap.MessageType.Arrive) {
                    this.mapCollider(msg.data.mapId).onArrive(msg.character, msg.data);
                    return;
                }
                if (msg.type === GlobalMap.MessageType.TriggerExit) {
                    this.mapCollider(msg.data.mapId).onTriggerExit(msg.character, msg.data);
                    return;
                }
                if (msg.type === GlobalMap.MessageType.TriggerEnter) {
                    this.mapCollider(msg.data.mapId).onTriggerEnter(msg.character, msg.data);
                    return;
                }
                if (msg.type === GlobalMap.MessageType.CollidePlayer) {
                    this._messageHandler?.(msg);
                    return;
                }
                if (msg.type === GlobalMap.MessageType.Happening) {
                    this._messageHandler?.(msg);
                    return;
                }
                if (msg.type === GlobalMap.MessageType.JumpTime) {
                    this.mapCollider(msg.data.mapId).onJumpTime(msg.character, msg.data);
                    return;
                }
                const _exhaustiveCheck = msg;
            }
            setMessageHandler(handler) {
                this._messageHandler = handler;
            }
            onSyncActor(sync) {
                this.nonPlayer(sync.targetActor)?.onSyncActor(sync);
            }
        }
        GlobalMap.MB_GlobalMap = MB_GlobalMap;
    })(GlobalMap = Iz.GlobalMap || (Iz.GlobalMap = {}));
})(Iz || (Iz = {}));
(function (Iz) {
    var GlobalMap;
    (function (GlobalMap) {
        function relativePath(subPath) {
            const path = require("path");
            const base = path.dirname(process.mainModule.filename);
            return path.join(base, subPath);
        }
        GlobalMap.relativePath = relativePath;
        function convertEscapeCharacters(text) {
            if (!text)
                return "";
            return Window_Base.prototype.convertEscapeCharacters(text);
        }
        GlobalMap.convertEscapeCharacters = convertEscapeCharacters;
    })(GlobalMap = Iz.GlobalMap || (Iz.GlobalMap = {}));
})(Iz || (Iz = {}));
(function (Iz) {
    var GlobalMap;
    (function (GlobalMap) {
        GlobalMap.WEATHER_SIZE = 2;
        GlobalMap.LocationTransferCode = 201;
        GlobalMap.AstarSearchLimit = 50;
        GlobalMap.ColliderTag = "GM_Collider";
        GlobalMap.OnPerformTag = "GM_OnExitExecutePage";
        GlobalMap.NotRoadRegionCost = Number.POSITIVE_INFINITY;
        GlobalMap.EventTag_Default = "default";
        GlobalMap.EventTag_MapId = "mapId";
        GlobalMap.EventTag_DayOfWeek = "dayOfWeek";
        GlobalMap.EventTag_Greeting = "greeting";
        GlobalMap.EventTag_Happening = "happening";
        class NonPlayerEvent {
        }
        GlobalMap.NonPlayerEvent = NonPlayerEvent;
        function IsNPC(character) {
            if (!character)
                return false;
            return character instanceof GlobalMap.MB_NonPlayer;
        }
        GlobalMap.IsNPC = IsNPC;
        GlobalMap.HoursPerDay = 24;
        GlobalMap.MinutesPerHour = 60;
        GlobalMap.TimesPerDay = GlobalMap.HoursPerDay * GlobalMap.MinutesPerHour;
        GlobalMap.BeginingTime = get24Time($globalMapParameter.begining_time.hour, $globalMapParameter.begining_time.minute);
        GlobalMap.EndTime = GlobalMap.BeginingTime + GlobalMap.TimesPerDay;
        function get24Time(hour, minute) {
            return Math.min(Math.max(hour * GlobalMap.MinutesPerHour + minute, 0), GlobalMap.HoursPerDay * GlobalMap.MinutesPerHour);
        }
        function toTime(hour, minute) {
            const time = hour * GlobalMap.MinutesPerHour + minute;
            return time;
        }
        GlobalMap.toTime = toTime;
        function toHourAndMinute(time) {
            const hour = Math.floor(time / GlobalMap.MinutesPerHour);
            const minute = time - hour * GlobalMap.MinutesPerHour;
            const obj = {
                hour: hour,
                minute: minute,
            };
            return obj;
        }
        GlobalMap.toHourAndMinute = toHourAndMinute;
        function toGMTime(hour, minute) {
            let time = Math.min(Math.max(toTime(hour, minute), 0), GlobalMap.EndTime);
            if (time < GlobalMap.BeginingTime) {
                time += GlobalMap.TimesPerDay;
            }
            return time;
        }
        GlobalMap.toGMTime = toGMTime;
        function toGMHourAndMinute(time_) {
            let time = time_;
            // time = Math.min(Math.max(time, 0), EndTime);
            if (time < GlobalMap.BeginingTime) {
                time += GlobalMap.TimesPerDay;
            }
            while (time >= GlobalMap.EndTime) {
                time -= GlobalMap.TimesPerDay;
            }
            const hour = Math.floor(time / GlobalMap.MinutesPerHour);
            const minute = time - hour * GlobalMap.MinutesPerHour;
            const obj = {
                hour: hour,
                minute: minute,
            };
            return obj;
        }
        GlobalMap.toGMHourAndMinute = toGMHourAndMinute;
        function clampTime(time) {
            return Math.min(Math.max(time, GlobalMap.BeginingTime), GlobalMap.EndTime);
        }
        GlobalMap.clampTime = clampTime;
        function convertEscapeCharacterGM(text_) {
            let text = text_;
            text = text.replace(/\x1bGMname\[(\d+)\]/gi, (_, p1) => {
                return Iz.System.getMapName(Number.parseInt(p1));
            });
            text = text.replace(/\\GMname\[(\d+)\]/gi, (_, p1) => {
                return Iz.System.getMapName(Number.parseInt(p1));
            });
            return text;
        }
        GlobalMap.convertEscapeCharacterGM = convertEscapeCharacterGM;
        function createPlainDestination(dest) {
            return {
                mapId: dest.mapId,
                x: dest.x,
                y: dest.y,
                direction: dest.direction,
                disableArrivalProgress: dest.disableArrivalProgress,
            };
        }
        GlobalMap.createPlainDestination = createPlainDestination;
        class Destination {
            constructor(option) {
                this.clear();
                Object.assign(this, option);
            }
            set(value) {
                Object.assign(this, value);
            }
            clear() {
                this.mapId = 0;
                this.x = 0;
                this.y = 0;
                this.direction = 0;
                this.disableArrivalProgress = false;
            }
        }
        GlobalMap.Destination = Destination;
        function isOutdoor(mapId) {
            const data = $dataGlobalMapInfo[mapId];
            return data?.outdoors ?? false;
        }
        GlobalMap.isOutdoor = isOutdoor;
    })(GlobalMap = Iz.GlobalMap || (Iz.GlobalMap = {}));
})(Iz || (Iz = {}));
// Addon関連
(function (Iz) {
    var GlobalMap;
    (function (GlobalMap) {
        const NullChronus = {
            getHour: () => 0,
            getMinute: () => 0,
            getWeekIndex: () => 0,
            getDaysOfWeek: () => 7,
        };
        // NOTE:
        // 外部プラグイン(EventRespawn等)で動的にイベントを生成する仕組みがあると
        // 競合するので適切に置き換えること
        // biome-ignore lint:
        GlobalMap.getLastEventId = () => {
            return $gameMap._events.length - 1;
        };
        // biome-ignore lint:
        GlobalMap.setLastEventId = (eventId) => { };
        // biome-ignore lint:
        GlobalMap.isHalfMove = false;
        function getUnitCost() {
            return GlobalMap.isHalfMove ? 0.5 : 1;
        }
        GlobalMap.getUnitCost = getUnitCost;
        // biome-ignore lint:
        GlobalMap.getChronus = () => {
            return NullChronus;
        };
        // biome-ignore lint:
        GlobalMap.applyInitEventPos = (e) => { };
        // biome-ignore lint:
        GlobalMap.convertCostume = (costume) => {
            return costume;
        };
    })(GlobalMap = Iz.GlobalMap || (Iz.GlobalMap = {}));
})(Iz || (Iz = {}));
(function (Iz) {
    var GlobalMapUtil;
    (function (GlobalMapUtil) {
        // 目的地に到着済みかつ同一マップにいる
        function isArrivedOnSameMap(characterId) {
            if (DataManager.isEventTest()) {
                return true;
            }
            // NOTE:
            // 到着済みの判定は目的地に移動しているかどうかで行う
            // （※ 厳密な判定ではないが、必要十分なため）
            return ($globalMap.nonPlayer(characterId).isSameMap() &&
                !$globalMap.nonPlayer(characterId).isMovingToDestination());
        }
        GlobalMapUtil.isArrivedOnSameMap = isArrivedOnSameMap;
        function isExistNpcOnSameMap(characterId) {
            if (characterId === 0) {
                return !!$globalMap.nonPlayers().find((npc) => npc.mapId() === $gameMap.mapId());
            }
            return !!$globalMap
                .nonPlayers()
                .find((npc) => npc.characterId === characterId && npc.mapId() === $gameMap.mapId());
        }
        GlobalMapUtil.isExistNpcOnSameMap = isExistNpcOnSameMap;
        function getCostume(characterId) {
            const character = $globalMap.character(characterId);
            if (Iz.GlobalMap.IsNPC(character)) {
                return character.getCostume();
            }
            const dir = character.characterName().split("/")[0];
            if (dir) {
                const costume = Number(dir.split("_")[1]);
                if (costume) {
                    return costume;
                }
            }
            return 0;
        }
        GlobalMapUtil.getCostume = getCostume;
        function getOriginalCostume(characterId) {
            const character = $globalMap.character(characterId);
            if (Iz.GlobalMap.IsNPC(character)) {
                return character.getOriginalCostume();
            }
            const dir = character.characterName().split("/")[0];
            if (dir) {
                const costume = Number(dir.split("_")[1]);
                if (costume) {
                    return costume;
                }
            }
            return 0;
        }
        GlobalMapUtil.getOriginalCostume = getOriginalCostume;
        function onLaunchHappening(characterId, eventExecuted = true) {
            const npc = $globalMap.nonPlayer(characterId);
            if (npc) {
                npc.collidePlayer();
                if (eventExecuted) {
                    npc.send({ type: Iz.GlobalMap.MessageType.Happening, character: characterId });
                }
            }
        }
        GlobalMapUtil.onLaunchHappening = onLaunchHappening;
    })(GlobalMapUtil = Iz.GlobalMapUtil || (Iz.GlobalMapUtil = {}));
})(Iz || (Iz = {}));
(function () {
    "use strict";
    // 枠を多少多めに 300まで上限を上げる
    JsonEx.maxDepth = 300;
    const pluginName = "GlobalMap";
    PluginManager.registerCommand(pluginName, "NPC_NEWDAY", (args) => {
        const hour = Iz.GlobalMap.getChronus().getHour();
        const dayOfWeek = Iz.GlobalMap.getChronus().getWeekIndex();
        $globalMap.nonPlayers().forEach((npc) => npc.newDay(hour, dayOfWeek));
    });
    PluginManager.registerCommand(pluginName, "NPC_JUMPCURRENTTIME", (args) => {
        const hour = Iz.GlobalMap.getChronus().getHour();
        const minute = Iz.GlobalMap.getChronus().getMinute();
        if (args.all === "true") {
            $globalMap.nonPlayers().forEach((npc) => npc.jumpSpecifiedTime(hour, minute));
        }
        else {
            const actorId = convertInt(args.actor);
            $globalMap.nonPlayer(actorId).jumpSpecifiedTime(hour, minute);
        }
    });
    PluginManager.registerCommand(pluginName, "NPC_JUMPTIME", (args) => {
        const hour = convertInt(args.hour);
        const minute = convertInt(args.minute);
        if (args.all === "true") {
            $globalMap.nonPlayers().forEach((npc) => npc.jumpSpecifiedTime(hour, minute));
        }
        else {
            const actorId = convertInt(args.actor);
            $globalMap.nonPlayer(actorId).jumpSpecifiedTime(hour, minute);
        }
    });
    PluginManager.registerCommand(pluginName, "NPC_PROGRESS_SETUP", (args) => {
        let hour = Iz.GlobalMap.getChronus().getHour();
        let minute = Iz.GlobalMap.getChronus().getMinute();
        const targetHour = convertInt(args.hour);
        const targetMinute = convertInt(args.minute);
        if (targetHour && targetMinute) {
            hour = targetHour;
            minute = targetMinute;
        }
        const actorId = convertInt(args.actor);
        $globalMap.nonPlayer(actorId).setupProgress(hour, minute);
    });
    PluginManager.registerCommand(pluginName, "NPC_APPEAR", (args) => {
        let hour = convertInt(args.hour);
        if (!hour) {
            hour = Iz.GlobalMap.getChronus().getHour();
        }
        const actorId = convertInt(args.actor);
        $globalMap.appearNonPlayer(actorId, hour);
    });
    PluginManager.registerCommand(pluginName, "NPC_LEAVE", (args) => {
        const actorId = convertInt(args.actor);
        $globalMap.leaveNonPlayer(actorId);
    });
    PluginManager.registerCommand(pluginName, "NPC_MOVE_INTERRUPT", (args) => {
        const actorId = convertInt(args.actor);
        const mapId = convertInt(args.mapId);
        const x = convertInt(args.x);
        const y = convertInt(args.y);
        const d = convertInt(args.direction);
        $globalMap.nonPlayer(actorId).interrupt(mapId, x, y, d);
    });
    PluginManager.registerCommand(pluginName, "NPC_CLEARDESTINATION", (args) => {
        if (args.all === "true") {
            $globalMap.nonPlayers().forEach((npc) => npc.clearDestination());
        }
        else {
            const actorId = convertInt(args.actor);
            $globalMap.nonPlayer(actorId).clearDestination();
        }
    });
    PluginManager.registerCommand(pluginName, "NPC_SETTARGET", (args) => {
        const character = $globalMap.nonPlayer(convertInt(args.actor1));
        const target = $globalMap.character(convertInt(args.actor2));
        character.setTarget(target);
    });
    PluginManager.registerCommand(pluginName, "NPC_CLEARTARGET", (args) => {
        if (args.all === "true") {
            $globalMap.nonPlayers().forEach((npc) => npc.setTarget(null));
        }
        else {
            const actorId = convertInt(args.actor);
            $globalMap.nonPlayer(actorId).setTarget(null);
        }
    });
    PluginManager.registerCommand(pluginName, "NPC_SETPOSITION", (args) => {
        const npc = $globalMap.nonPlayer(convertInt(args.actor));
        const x = convertInt(args.x);
        const y = convertInt(args.y);
        let d = convertInt(args.direction);
        d = d ? d : npc.direction();
        if (npc.isAppear()) {
            npc.sendDeparture();
            npc.performPosition(x, y, d, npc.mapId());
            npc.sendArrive();
        }
        else {
            npc.setPosition(x, y, d, npc.mapId());
        }
    });
    PluginManager.registerCommand(pluginName, "NPC_PAUSE", (args) => {
        const npc = $globalMap.nonPlayer(convertInt(args.actor));
        const hour = convertInt(args.hour);
        const minute = convertInt(args.minute);
        npc.pause(hour, minute);
    });
    PluginManager.registerCommand(pluginName, "NPC_RESUME", (args) => {
        if (args.all === "true") {
            $globalMap.nonPlayers().forEach((npc) => {
                npc.resume();
            });
        }
        else {
            const npc = $globalMap.nonPlayer(convertInt(args.actor));
            npc.resume();
        }
    });
    PluginManager.registerCommand(pluginName, "GLOBALMAP_PAUSE", (args) => {
        $globalMap.pause();
    });
    PluginManager.registerCommand(pluginName, "GLOBALMAP_RESUME", (args) => {
        $globalMap.resume();
    });
    PluginManager.registerCommand(pluginName, "IS_SAME_LOCATION", (args) => {
        const actor1Id = convertInt(args.actor1);
        const actor2Id = convertInt(args.actor2);
        const switchId = convertInt(args.switch);
        $gameSwitches.setValue(switchId, $globalMap.character(actor1Id).isSameLocation($globalMap.character(actor2Id)));
    });
    PluginManager.registerCommand(pluginName, "NPC_CHANGE_SCHEDULE", (args) => {
        const character = $globalMap.nonPlayer(convertInt(args.actor));
        const schedule = args.schedule
            ? convertInt(args.schedule)
            : Iz.GlobalMap.getChronus().getWeekIndex();
        const hour = Iz.GlobalMap.getChronus().getHour();
        const minute = Iz.GlobalMap.getChronus().getMinute();
        character.scheduler.changeSchedule(schedule, hour, minute);
    });
    PluginManager.registerCommand(pluginName, "NPC_CHANGE_COSTUME", function (args) {
        const costume = convertInt(args.costume);
        const actors = extractActors(args, this);
        actors.forEach((actor) => {
            actor.setCostume(costume);
        });
    });
    function convertInt(text) {
        return Number.parseInt(Iz.GlobalMap.convertEscapeCharacters(text));
    }
    function convertFloat(text) {
        return Number.parseFloat(Iz.GlobalMap.convertEscapeCharacters(text));
    }
    const ActorType = {
        ActorId: 0,
        All: 1,
        ExecuteCharacter: 2,
    };
    function extractActors(args, interpreter) {
        // コマンドの引数からNPCを抽出
        if (args.all === "true") {
            return $globalMap.nonPlayers();
        }
        const actorType = convertInt(args.actorType);
        if (actorType === ActorType.All) {
            return $globalMap.nonPlayers();
        }
        if (actorType === ActorType.ExecuteCharacter) {
            const npc = interpreter.executeCharacter();
            if (!npc) {
                return [];
            }
            return [npc];
        }
        const actor = convertInt(args.actor);
        if (!actor)
            return [];
        const npc = $globalMap.nonPlayer(actor);
        if (!npc)
            return [];
        return [npc];
    }
    //========================================
    // DataManager
    //========================================
    const _MB_DataManagerCreateGameObjects = DataManager.createGameObjects;
    DataManager.createGameObjects = function () {
        _MB_DataManagerCreateGameObjects.call(this);
        // biome-ignore lint:
        window["$globalMap"] = new Iz.GlobalMap.MB_GlobalMap();
    };
    const _DataManager_SetupNewGame = DataManager.setupNewGame;
    DataManager.setupNewGame = function () {
        _DataManager_SetupNewGame.call(this);
        $globalMap.setup();
    };
    const _DataManagerMakeSaveContents = DataManager.makeSaveContents;
    DataManager.makeSaveContents = function () {
        // A save data does not contain $gameTemp, $gameMessage, and $gameTroop.
        const contents = _DataManagerMakeSaveContents.call(this);
        if ($globalMapParameter.save_npc_data) {
            $globalMap.makeSaveContents(contents);
        }
        return contents;
    };
    const _DataManagerEXtractSaveContents = DataManager.extractSaveContents;
    DataManager.extractSaveContents = function (contents) {
        _DataManagerEXtractSaveContents.call(this, contents);
        if ($globalMapParameter.save_npc_data) {
            $globalMap.extractSaveContents(contents);
        }
    };
    //========================================
    // Spriteset_Map
    //========================================
    Spriteset_Map.prototype.addCharacter = function (character) {
        const sprite = new Sprite_Character(character);
        const index = this._characterSprites.push(sprite);
        this._tilemap.addChild(this._characterSprites[index - 1]);
    };
    Spriteset_Map.prototype.removeCharacter = function (character) {
        const finded = this._characterSprites.find((sprite) => sprite._character === character);
        if (!finded)
            return;
        const index = this._characterSprites.indexOf(finded);
        if (index < 0) {
            return;
        }
        this._characterSprites.splice(index, 1);
        this._tilemap.removeChild(finded);
    };
    const _SceneMapUpdateMain = Scene_Map.prototype.updateMain;
    Scene_Map.prototype.updateMain = function () {
        $globalMap.update();
        _SceneMapUpdateMain.call(this);
        $globalMap.postUpdate();
    };
    Scene_Map.prototype.addMapObject = function (object) {
        this._spriteset?.addCharacter(object);
    };
    Scene_Map.prototype.removeCharacter = function (character) {
        this._spriteset?.removeCharacter(character);
    };
    //========================================
    // Game_Map
    //========================================
    const _Game_Map_Initialize = Game_Map.prototype.initialize;
    Game_Map.prototype.initialize = function () {
        _Game_Map_Initialize.call(this);
        this.nonPlayerObjects = [];
    };
    Game_Map.prototype.addEvent = function (event) {
        this._events[event.eventId()] = event;
    };
    Game_Map.prototype.removeEvent = function (eventId) {
        this._events[eventId] = undefined;
    };
    const _Game_Map_SetupEvents = Game_Map.prototype.setupEvents;
    Game_Map.prototype.setupEvents = function () {
        _Game_Map_SetupEvents.call(this);
        $globalMap.nonPlayers().forEach((npc) => npc.endBalloon());
        this.nonPlayerObjects = $globalMap.nonPlayers().filter((npc) => npc.isSameMap());
        this.nonPlayerObjects.forEach((npc) => {
            this.addNpc(npc);
        });
    };
    Game_Map.prototype.npcXy = function (x, y) {
        return this.events()
            .filter((event) => event.pos(x, y))
            .find((event) => event instanceof Iz.GlobalMap.MB_NonPlayer);
    };
    Game_Map.prototype.addNpc = function (npc) {
        let id = Iz.GlobalMap.getLastEventId() + 1;
        if (id === 0) {
            id = 1;
        }
        npc.setEventId(id);
        this.addEvent(npc);
        npc.refresh();
        Iz.GlobalMap.setLastEventId(id);
    };
    Game_Map.prototype.removeNpc = function (eventId) {
        const target = this._events[eventId];
        if (Iz.GlobalMap.IsNPC(target)) {
            target.setEventId(-1);
            this.removeEvent(eventId);
        }
    };
    Game_Map.prototype.interpreter = function () {
        return this._interpreter;
    };
    Game_Map.prototype.terminateInterpreter = function (interpreter_) {
        let interpreter = interpreter_;
        interpreter = interpreter ?? this._interpreter;
        const childInterpreter = interpreter.childInterpreter();
        if (childInterpreter) {
            return this.terminateInterpreter(childInterpreter);
        }
        return interpreter;
    };
    // 当たり判定も含めたeventXyNt HalfMove用に対応
    Game_Map.prototype.eventsXyNtInUnit = function (x, y) {
        return this.eventsXyNt(x, y);
    };
    //==============================
    // Game_Interpreter
    //==============================
    Game_Interpreter.prototype.clear = function () {
        this._mapId = 0;
        this._eventId = 0;
        this._list = null;
        this._index = 0;
        this._waitCount = 0;
        this._waitMode = "";
        this._comments = "";
        this._childInterpreter = null;
        this._executeCharacterId = 0;
    };
    Game_Interpreter.prototype.executeCharacter = function () {
        const character = $globalMap.nonPlayer(this._executeCharacterId);
        return character ?? null;
    };
    Game_Interpreter.prototype.setExecuteCharacter = function (character) {
        this._executeCharacterId = character;
        this._mapId = $globalMap.character(character)?.mapId() ?? 0;
    };
    Game_Interpreter.prototype.childInterpreter = function () {
        return this._childInterpreter;
    };
    const _Game_Map_updateInterpreter = Game_Map.prototype.updateInterpreter;
    Game_Map.prototype.updateInterpreter = function () {
        const prevRunning = this._interpreter.isRunning();
        _Game_Map_updateInterpreter.call(this);
        if (prevRunning && !this._interpreter.isRunning()) {
            $globalMap.nonPlayers().forEach((npc) => {
                npc.setupEvent();
            });
        }
    };
    //========================================
    // Game_CharacterBase
    //========================================
    Game_CharacterBase.prototype.location = function () {
        return undefined;
    };
    Game_CharacterBase.prototype.equalLocation = function (character) {
        const location1 = this.location();
        const location2 = character.location();
        if (!location1 || !location2)
            return false;
        return location1.id === location2.id;
    };
    Game_CharacterBase.prototype.closeByCharacter = function (character, distance) {
        const mapA = character.mapId();
        const mapB = this.mapId();
        if (mapA <= 0 || mapB <= 0)
            return false;
        if (mapA !== mapB)
            return false;
        return Math.abs(this.x - character.x) + Math.abs(this.y - character.y) <= distance;
    };
    Game_CharacterBase.prototype.isSameLocation = function (character) {
        return this.equalLocation(character) && this.closeByCharacter(character, 6);
    };
    Game_CharacterBase.prototype.isOppositeDirection = function (character) {
        return this.direction() + character.direction() === 10;
    };
    Game_CharacterBase.prototype.isFaceToFace = function (character) {
        const d = this.direction();
        const x = $gameMap.roundXWithDirection(this.x, d);
        const y = $gameMap.roundYWithDirection(this.y, d);
        return this.isOppositeDirection(character) && character.x === x && character.y === y;
    };
    Game_CharacterBase.prototype.equalDirection = function (character) {
        return this.direction() === character.direction();
    };
    Game_CharacterBase.prototype.isBehindTheCharacter = function (character) {
        const d = this.direction();
        const x = $gameMap.roundXWithDirection(this.x, d);
        const y = $gameMap.roundYWithDirection(this.y, d);
        return this.equalDirection(character) && character.x === x && character.y === y;
    };
    Game_CharacterBase.prototype.isSideTheCharacter = function (character) {
        const d = this.direction();
        const x = $gameMap.roundXWithDirection(this.x, d);
        const y = $gameMap.roundYWithDirection(this.y, d);
        return (!this.isOppositeDirection(character) &&
            !this.equalDirection(character) &&
            character.x === x &&
            character.y === y);
    };
    //==============================
    // Game_Player
    //==============================
    Game_Player.prototype.frontNpc = function () {
        const d = this.direction();
        const x = $gameMap.roundXWithDirection(this.x, d);
        const y = $gameMap.roundYWithDirection(this.y, d);
        return $gameMap.npcXy(x, y);
    };
    Game_Player.prototype.location = function () {
        return $globalMap.getLocation($gameMap.mapId(), this.x, this.y);
    };
    //========================================
    // Game_Event
    //========================================
    Game_Event.prototype.isOverlappingWithPlayer = function () {
        return $gamePlayer.x === this.x && $gamePlayer.y === this.y;
    };
    Game_Event.prototype.isOverlappingWithCharacter = function () {
        const events = $gameMap.eventsXy(this.x, this.y).filter((event) => this !== event);
        return events.length > 0;
    };
    const _Window_Base_convertEscapeCharacters = Window_Base.prototype.convertEscapeCharacters;
    Window_Base.prototype.convertEscapeCharacters = function (text_) {
        let text = text_;
        text = Iz.GlobalMap.convertEscapeCharacterGM(text);
        text = _Window_Base_convertEscapeCharacters.call(this, text);
        return text;
    };
})();
// NPC接触イベント
(function (Iz) {
    var GlobalMapUtil;
    (function (GlobalMapUtil) {
        const HappeningRateVariableId = 1025;
        function passLotteryHappening() {
            const happeningRate = $gameVariables.value(HappeningRateVariableId);
            return Iz.Life.passLotteryPercent(happeningRate);
        }
        GlobalMapUtil.passLotteryHappening = passLotteryHappening;
    })(GlobalMapUtil = Iz.GlobalMapUtil || (Iz.GlobalMapUtil = {}));
})(Iz || (Iz = {}));
(() => {
    "use strict";
    const _Game_Player_checkEventTriggerTouchFront = Game_Player.prototype.checkEventTriggerTouchFront;
    Game_Player.prototype.checkEventTriggerTouchFront = function (d) {
        _Game_Player_checkEventTriggerTouchFront.call(this, d);
        if ($gameMap.isEventRunning())
            return;
        if (this.isMovementSucceeded())
            return;
        const e = Iz.Life.getThereEvent();
        if (!Iz.GlobalMap.IsNPC(e))
            return;
        if (!e.isMovingToDestination())
            return;
        if (e.collidedPlayer)
            return;
        // TODO:
        // 対象が止まってからイベントを起動させたい
        // 確率でハプニング
        if (Iz.GlobalMapUtil.passLotteryHappening()) {
            if (e.getEventEnable(Iz.GlobalMap.EventTag_Happening)) {
                if (this.isDashing()) {
                    const dataEvent = $globalMap.findEvent(e.name, Iz.GlobalMap.EventTag_Happening);
                    if (dataEvent) {
                        e.setEvent(dataEvent);
                        e.start();
                        return;
                    }
                }
            }
        }
        e.collidePlayer();
        e.send({ type: Iz.GlobalMap.MessageType.CollidePlayer, character: e.characterId });
        const QuestionBalloonId = 2;
        $gameTemp.requestBalloon(e, QuestionBalloonId);
    };
})();
//# sourceMappingURL=GlobalMap.js.map