/*:
 * @target MZ
 * @plugindesc ラベル
 * @author INAZUMA GAMES
 *
 * @help
 * <機能>
 * ・イベントにラベルを表示する機能（最大10個）
 *
 *
 *
 * @command Show
 * @text ラベル表示
 * @desc ラベル表示
 *
 * @arg idOrName
 * @text IDか名前
 * @desc 0: このイベント  1: イベントID
 * 数値以外はイベント名として検索します
 *
 * @arg text
 * @text テキストorCharacterLabelマスタのid
 * @desc CharacterLabelマスタにある場合idとして使います ない場合は表示文字列としてそのまま使用します
 *
 * @arg tail
 * @text 吹き出しのしっぽの表示
 * @desc 吹き出しのしっぽの表示
 * @type boolean
 *
 *
 * @command Hide
 * @text ラベル非表示
 * @desc ラベル非表示
 *
 * @arg idOrName
 * @text IDか名前
 * @desc Showコマンドと同様の設定をしてください
 *
 */
var Iz;
(function (Iz) {
    var Life;
    (function (Life) {
        Life.LABEL_MAX = 15;
        const LabelWidth = 160;
        Life.LabelHalfHeight = 21;
        const LabelHeight = Life.LabelHalfHeight * 2;
        class Sprite_Label extends Sprite {
            constructor() {
                super();
                this.create();
                const bitmap = new Bitmap(this._bg.width, this._bg.height);
                // bitmap.fillAll("blue");
                this.bitmap = bitmap;
                this.z = 7; // バルーンと同じ
            }
            get text() {
                return this._text.text;
            }
            get enableTail() {
                return this._tail.visible;
            }
            set enableTail(value) {
                this._tail.visible = value;
            }
            create() {
                this.anchor.x = 0.5;
                this.anchor.y = 0.5;
                this.createBg();
                this.createTail();
                this.createText();
            }
            createBg() {
                const atlas = Iz.Atlas.getAtlas(Life.ATLAS_KEY_COMMON);
                const frame = atlas.getFrame(Life.FRAME_KEY_COMMON_FUKIDASI_BASE);
                this._bg = new NineSliceSprite(atlas.getBitmap());
                this._bg.anchor.x = 0.5;
                this._bg.anchor.y = 0.5;
                this._bg.setBitmapRect(frame.x, frame.y, frame.width, frame.height);
                this._bg.setSlice(20, 20, 20, 20);
                this._bg.setSize(LabelWidth, LabelHeight);
                this.addChild(this._bg);
            }
            createTail() {
                this._tail = new Sprite();
                this._tail.setAtlas(Life.ATLAS_KEY_COMMON, Life.FRAME_KEY_COMMON_FUKIDASI_SIPPO);
                this._tail.anchor.set(0.5, 0);
                this._tail.y = Math.floor(this._bg.height / 2) - 8;
                this.addChild(this._tail);
            }
            createText() {
                this._text = new Sprite_TextEx(LabelWidth, LabelHeight);
                this._text.anchor.set(0.5, 0.5);
                this._text.setFontSetting(Life.FONT_KEY_CHARACTER_LABEL);
                this.addChild(this._text);
            }
            set(text) {
                if (this._text.text !== text) {
                    const textSize = this._text.textSizeEx(text);
                    textSize.width += 20 * 2;
                    textSize.height = Math.min(LabelHeight, textSize.height);
                    this._text.createBitmap(textSize.width, textSize.height);
                    this._text.resetFontSettings();
                    this._text.setText(text);
                    this._text.setTextAlign(0.5, 0.5);
                    this._bg.setSize(textSize.width, textSize.height);
                }
            }
        }
        Life.Sprite_Label = Sprite_Label;
        const NotFoundCharacterId = -2;
        class LabelProxy {
            constructor() {
                this._labels = [];
                this.clear();
            }
            show(idOrName, text, tail) {
                // すでに使っている場合は再利用（上書き）
                const useLabel = this.getUseLabel(idOrName);
                if (useLabel) {
                    useLabel.text = text;
                    useLabel.enableTail = tail ?? false;
                    return true;
                }
                const label = this.getUsableLabel();
                if (!label)
                    return false;
                const id = this.getCharacterId(idOrName);
                if (id < 0)
                    return false;
                label.use = true;
                label.text = text;
                label.id = id;
                label.enableTail = tail ?? false;
                return true;
            }
            hide(idOrName) {
                const label = this.getUseLabel(idOrName);
                if (label) {
                    label.use = false;
                    label.text = "";
                    label.id = 0;
                    label.enableTail = false;
                }
            }
            getLabel(index) {
                return this._labels[index];
            }
            isUsable(idOrName) {
                return !!this.getUseLabel(idOrName);
            }
            getCharacterId(idOrName) {
                const id = Number.parseInt(idOrName);
                if (Number.isInteger(id)) {
                    if (id <= 0) {
                        return NotFoundCharacterId;
                    }
                    return id;
                }
                const event = $gameMap.events().find((e) => e.event().name === idOrName);
                if (event) {
                    return event.eventId();
                }
                return NotFoundCharacterId;
            }
            getUseLabel(idOrName) {
                const id = this.getCharacterId(idOrName);
                return this._labels.find((l) => l.id === id);
            }
            getUsableLabel() {
                return this._labels.find((l) => !l.use);
            }
            clear() {
                for (let i = 0; i < Life.LABEL_MAX; i++) {
                    this._labels[i] = {
                        index: i,
                        use: false,
                        id: 0,
                        text: "",
                        enableTail: false,
                    };
                }
            }
        }
        Life.LabelProxy = LabelProxy;
        Life.Label = new LabelProxy();
    })(Life = Iz.Life || (Iz.Life = {}));
})(Iz || (Iz = {}));
(() => {
    "use strict";
    function showLabel(args) {
        let eventId = args.idOrName;
        if (eventId === "0") {
            // @ts-ignore
            eventId = String(this._eventId);
        }
        const text = $dataCharacterLabel.get(args.text)?.value ?? args.text;
        Iz.Life.Label.show(eventId, text, !!args.tail);
    }
    PluginManagerEx.registerCommand(document.currentScript, "Show", showLabel);
    function hideLabel(args) {
        let eventId = args.idOrName;
        if (eventId === "0") {
            // @ts-ignore
            eventId = String(this._eventId);
        }
        Iz.Life.Label.hide(eventId);
    }
    PluginManagerEx.registerCommand(document.currentScript, "Hide", hideLabel);
    const _Spriteset_Map_initialize = Spriteset_Map.prototype.initialize;
    Spriteset_Map.prototype.initialize = function () {
        _Spriteset_Map_initialize.call(this);
        Iz.Life.Label.clear();
        this._characterLabels = [];
        for (let i = 0; i < Iz.Life.LABEL_MAX; i++) {
            this._characterLabels[i] = new Iz.Life.Sprite_Label();
            this._characterLabels[i].visible = false;
            this._tilemap.addChild(this._characterLabels[i]);
        }
    };
    const _Spriteset_Map_update = Spriteset_Map.prototype.update;
    Spriteset_Map.prototype.update = function () {
        _Spriteset_Map_update.call(this);
        this._characterLabels.forEach((sprite, i) => {
            const labelInfo = Iz.Life.Label.getLabel(i);
            if (sprite.visible !== labelInfo.use) {
                sprite.visible = labelInfo.use;
            }
            if (sprite.text !== labelInfo.text) {
                sprite.set(labelInfo.text);
            }
            if (sprite.enableTail !== labelInfo.enableTail) {
                sprite.enableTail = labelInfo.enableTail;
            }
            if (sprite.visible) {
                const event = $gameMap.event(labelInfo.id);
                if (event) {
                    sprite.x = event.screenX();
                    sprite.y = event.screenY() - $gameMap.tileHeight() - Iz.Life.LabelHalfHeight;
                    sprite.z = 6;
                }
            }
        });
    };
})();
//# sourceMappingURL=CharacterLabel.js.map