/*:
 * @target MZ
 * @plugindesc CG既読フラグの永続保存・差分マージ・全開放 v1.2
 * @author mushizushi
 *
 * @param switchStart
 * @text スイッチ開始ID
 * @default 100
 *
 * @param switchEnd
 * @text スイッチ終了ID
 * @default 199
 *
 * @help
 * 【概要】
 * 指定範囲のスイッチをConfigManagerに保存し、
 * 既読/未読を全プレイ間で共有します。
 * クリア時に新規既読のみ追加され、既読は保持されます。
 *
 * ─────────────────────────────
 * 【主な機能】
 *  - applyToGame()：既読差分をマージ（自動）
 *  - saveNow()：現在の既読状態を保存
 *  - reset()：全消去
 *  - unlockAll()：全スイッチをtrueにして保存
 *  - isSeen(id)：既読判定
 *
 * ─────────────────────────────
 * 【自動動作】
 *  - タイトル起動時：applyToGame() 実行
 *  - マップ開始時：saveNow() 実行
 *
 * ─────────────────────────────
 * 【使い方】
 *  CG閲覧時に：
 *    $gameSwitches.setValue(120, true);
 *    CGPersist.saveNow();
 *
 *  全開放：
 *    CGPersist.unlockAll();
 *
 *  リセット：
 *    CGPersist.reset();
 */

(() => {
  const PN = "CGModePersistMergeMZ";
  const P = PluginManager.parameters(PN);
  const sStart = Number(P.switchStart || 100);
  const sEnd   = Number(P.switchEnd || 199);

  const _makeData = ConfigManager.makeData;
  ConfigManager.cgPersist = ConfigManager.cgPersist || {};
  ConfigManager.makeData = function() {
    const c = _makeData.call(this);
    c.cgPersist = this.cgPersist;
    return c;
  };

  const _applyData = ConfigManager.applyData;
  ConfigManager.applyData = function(cfg) {
    _applyData.call(this, cfg);
    this.cgPersist = cfg?.cgPersist || {};
  };

  const CGPersist = {
    range() { return { s1: Math.min(sStart, sEnd), s2: Math.max(sStart, sEnd) }; },
    saveNow() {
      const { s1, s2 } = this.range();
      for (let i = s1; i <= s2; i++) {
        if ($gameSwitches.value(i)) ConfigManager.cgPersist[i] = true;
      }
      ConfigManager.save();
    },
    applyToGame() {
      const { s1, s2 } = this.range();
      const saved = ConfigManager.cgPersist || {};
      for (let i = s1; i <= s2; i++) {
        const g = $gameSwitches.value(i);
        const s = !!saved[i];
        if (s || g) {
          $gameSwitches.setValue(i, true);
          if (g && !s) ConfigManager.cgPersist[i] = true;
        }
      }
      ConfigManager.save();
    },
    reset() {
      ConfigManager.cgPersist = {};
      ConfigManager.save();
      console.log("[CGPersist] 全データリセット");
    },
    isSeen(id) { return !!ConfigManager.cgPersist[id]; },
    unlockAll() {
      const { s1, s2 } = this.range();
      for (let i = s1; i <= s2; i++) {
        $gameSwitches.setValue(i, true);
        ConfigManager.cgPersist[i] = true;
      }
      ConfigManager.save();
      console.log(`[CGPersist] CG全開放 (${s1}〜${s2})`);
    }
  };
  window.CGPersist = CGPersist;

  const _Scene_Title_start = Scene_Title.prototype.start;
  Scene_Title.prototype.start = function() {
    _Scene_Title_start.call(this);
    try { CGPersist.applyToGame(); } catch(e){ console.warn(e); }
  };
  const _Scene_Map_start = Scene_Map.prototype.start;
  Scene_Map.prototype.start = function() {
    _Scene_Map_start.call(this);
    try { CGPersist.saveNow(); } catch(e){ console.warn(e); }
  };
})();
