/*:
 * @target MZ
 * @plugindesc HR_PictureMask のマスクが画像差し替えや再生成で外れるのを自動再付与。PictureLive2D.js 専用対応（通常ピクチャも対応）。マスク位置追従修正版 v2。
 * @author you
 *
 * @help
 * ◆概要
 *  - HR_PictureMask.js の後ろ(下)に入れるだけで、
 *    ・通常ピクチャ(Sprite_Picture)
 *    ・PictureLive2D.js の Live2D スプライト
 *    に対して、外れた mask を毎フレーム自動で再付与します。
 *  - マスクスプライトの位置を Live2D に追従させるよう修正しました（親座標系の違いに対応）。
 *
 * ◆使い方
 *  - HR_PictureMask.js の setMask で『ピクチャID→ピクチャID』の関連付けをした後、
 *    本パッチは「外れていたら自動再付与」するだけの補強です（通常ピクチャ側）。
 *  - Live2D にもマスクを適用したい場合：
 *      1) マスク形状用のピクチャを表示（例：ID=5）
 *      2) 本プラグインのコマンド setPL2DMask を実行（index と maskPicId を指定）
 *     以後、再生成や差し替えで外れても自動復元します。
 *
 * ◆プラグインコマンド
 *
 * @command setPL2DMask
 * @text PictureLive2Dにマスク設定
 * @desc PictureLive2D のインデックスと、マスクに使うピクチャIDを関連付けます（自動再付与対象）。
 *
 * @arg index
 * @text Live2Dインデックス
 * @type number
 * @min 0
 * @default 0
 * @desc PictureLive2Dのスプライト番号。SceneManager._scene._spriteset._pictureLive2DContainer.children[] の番号。
 *
 * @arg maskPicId
 * @text マスク用ピクチャID
 * @type number
 * @min 1
 * @default 1
 * @desc マスクに使用するピクチャのID。
 *
 * @command clearPL2DMask
 * @text PictureLive2Dマスク解除
 * @desc 指定インデックスの PictureLive2D のマスク関連付けを解除します。
 *
 * @arg index
 * @text Live2Dインデックス
 * @type number
 * @min 0
 * @default 0
 * @desc 解除したい PictureLive2D のスプライト番号。
 */
(function(){
  'use strict';
  const PN = 'HR_PictureMask_Persist_PictureLive2D_fixed_v2';

  // ---- utils ----
  function pictureSprites(){
    const scene = SceneManager._scene;
    return scene && scene._spriteset && scene._spriteset._pictureContainer
      ? scene._spriteset._pictureContainer.children
      : null;
  }
  function getPictureSpriteById(id){
    const list = pictureSprites();
    if(!list) return null;
    return list[id - 1] || null; // ピクチャID→配列index
  }

  // PictureLive2D.js の Live2D スプライト配列を推測取得
  function live2dSprites(){
    const sc = SceneManager._scene;
    if(!sc) return null;
    const ss = sc._spriteset;
    if(ss && ss._pictureLive2DContainer && ss._pictureLive2DContainer.children?.length){
      return ss._pictureLive2DContainer.children;
    }
    if(sc.pictureLive2DSprites?.length) return sc.pictureLive2DSprites;
    if(sc._live2dPictures?.length)     return sc._live2dPictures;
    if(sc._live2dSprites?.length)      return sc._live2dSprites;
    // フォールバック: Spriteset内から Live2D っぽいものを探索
    if(ss){
      const found = [];
      const stack = [ss];
      while(stack.length){
        const node = stack.pop();
        const ch = node.children || node._children || [];
        for(const c of ch){
          if(c && (/(Live2D|L2D)/i).test(c.constructor?.name||'')) found.push(c);
          if(c && (c.children?.length || c._children?.length)) stack.push(c);
        }
      }
      return found.length ? found : null;
    }
    return null;
  }

  // ---- data store (永続化：セーブ互換) ----
  const mapStore = () => {
    $gameSystem._hrPL2DMaskMap = $gameSystem._hrPL2DMaskMap || {};
    return $gameSystem._hrPL2DMaskMap;
  };

  // ---- core: 再付与（通常ピクチャ） ----
  function reapplyPictureMaskIfNeeded(){
    const sprites = pictureSprites();
    if(!sprites) return;
    for(let i=0;i<sprites.length;i++){
      const sp = sprites[i];
      const picId = i+1;
      const gamePic = $gameScreen.picture(picId);
      const maskId = gamePic && (gamePic._maskId || gamePic._maskPicId);
      if(maskId){
        const maskSp = getPictureSpriteById(Number(maskId));
        if(maskSp && sp.mask !== maskSp){ sp.mask = maskSp; }
      }
    }
  }

  // ---- core: 再付与（PictureLive2D） ----
  function reapplyPL2DMaskIfNeeded(){
    const l2ds = live2dSprites();
    if(!l2ds) return;
    const map = mapStore();
    for(const key of Object.keys(map)){
      const idx = Number(key);
      const l2d = l2ds[idx];
      if(!l2d) continue;
      const maskId = Number(map[key]);
      const maskSp = getPictureSpriteById(maskId);
      if(!maskSp) continue;

      // mask を関連付け
      if(l2d.mask !== maskSp){ l2d.mask = maskSp; }

      // --- 親座標系の違いに対応して位置追従 ---
      const parent = maskSp.parent;
      if(parent && parent.toLocal && l2d.getGlobalPosition){
        const g = l2d.getGlobalPosition();       // Live2Dのグローバル座標
        const local = parent.toLocal(g);         // mask親のローカルへ変換
        maskSp.position.set(local.x, local.y);
      }else{
        // フォールバック（同親の場合など）
        if(maskSp.position.copyFrom) maskSp.position.copyFrom(l2d.position);
        else maskSp.position.set(l2d.x||0, l2d.y||0);
      }

      // スケール・回転も可能な範囲で追従
      if(maskSp.scale && l2d.scale) maskSp.scale.copyFrom(l2d.scale);
      maskSp.rotation = l2d.rotation || 0;
    }
  }

  // ---- hooks ----
  const _Scene_Map_update = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function(){
    _Scene_Map_update.call(this);
    reapplyPictureMaskIfNeeded();
    reapplyPL2DMaskIfNeeded();
  };
  const _Scene_Battle_update = Scene_Battle.prototype.update;
  Scene_Battle.prototype.update = function(){
    _Scene_Battle_update.call(this);
    reapplyPictureMaskIfNeeded();
    reapplyPL2DMaskIfNeeded();
  };

  // ピクチャ再表示でも即復元（通常ピクチャ）
  const _Game_Screen_showPicture = Game_Screen.prototype.showPicture;
  Game_Screen.prototype.showPicture = function(pictureId, name, origin, x, y, scaleX, scaleY, opacity, blendMode){
    _Game_Screen_showPicture.apply(this, arguments);
    const gamePic = this.picture(pictureId);
    const maskId = gamePic && (gamePic._maskId || gamePic._maskPicId);
    if(maskId){
      const texSp = getPictureSpriteById(pictureId);
      const maskSp = getPictureSpriteById(Number(maskId));
      if(texSp && maskSp) texSp.mask = maskSp;
    }
  };

  // ---- plugin commands ----
  PluginManager.registerCommand(PN, 'setPL2DMask', args=>{
    const index = Number(args.index||0);
    const maskPicId = Math.max(1, Number(args.maskPicId||1));
    mapStore()[index] = maskPicId;
  });
  PluginManager.registerCommand(PN, 'clearPL2DMask', args=>{
    const index = Number(args.index||0);
    const map = mapStore();
    delete map[index];
    const l2ds = live2dSprites();
    if(l2ds && l2ds[index]) l2ds[index].mask = null;
  });
})();