/*:
 * @target MZ
 * @plugindesc [v1.0.0] ピクチャ移動(ウェイト)中は全入力を無効化。手動ブロックAPI/コマンド付き
 * @author GPT
 *
 * @help InputBlockOnPictureWaitMZ.js
 * 
 * ■概要
 *  - イベントコマンド「ピクチャの移動」で「ウェイト」にチェックが入っている間、
 *    そのフレーム中はすべてのキー入力/パッド/タッチを無効化します。
 *  - 「ウェイト無しの移動」でも入力を止めたい場合のオプション、
 *    さらに Live2D 等の独自描画向けに「手動で一定フレームだけ無効化」する
 *    プラグインコマンドを用意しています。
 * 
 * ■想定シーン
 *  - ピクチャでUIや演出をスライドさせる間に、プレイヤーの操作を完全に抑止したい。
 *  - Live2D等（BMP座標を持たない）のトランジション中も確実に入力を切りたい。
 *
 * ■使い方
 *  1) 本プラグインをON。
 *  2) デフォルトでは「ピクチャ移動（ウェイト有）」の間だけ入力が止まります。
 *  3) Live2D等で任意のフレーム数止めたいときは
 *     プラグインコマンド「BlockForFrames」でブロックしてください。
 *
 * ■注意
 *  - 「全入力無効」は、メニュー呼び出し／決定／キャンセル／移動／タッチを含む
 *    あらゆる入力クエリをfalseにします（Input/TouchInput を横取り）。
 *  - 競合回避のため、できるだけ副作用の少ない形でラップしていますが、
 *    入力状態を直接読む特殊プラグインとは併用結果を確認してください。
 *
 * @param EnableSwitchId
 * @text 有効スイッチID
 * @type switch
 * @desc ここで指定したスイッチがONのときだけ機能を有効化（0は常に有効）
 * @default 0
 *
 * @param AffectOnMap
 * @text マップで有効
 * @type boolean
 * @default true
 *
 * @param AffectOnBattle
 * @text 戦闘で有効
 * @type boolean
 * @default true
 *
 * @param AffectOnMenu
 * @text メニュー系で有効
 * @type boolean
 * @default false
 *
 * @param BlockDuringPictureWait
 * @text ピクチャ「ウェイト有」で無効化
 * @type boolean
 * @desc イベントの「ピクチャの移動(ウェイト)」待ちの間にブロック
 * @default true
 *
 * @param AlsoBlockDuringPictureMoveNoWait
 * @text ウェイト無し移動中も無効化
 * @type boolean
 * @desc ピクチャが移動中(_duration>0)ならウェイト無しでもブロック
 * @default false
 *
 * @param BlockTouchInput
 * @text タッチ入力も無効化
 * @type boolean
 * @default true
 *
 * @command SetEnabled
 * @text 有効/無効を切替
 * @desc ランタイムで機能の有効/無効を切り替え
 * @arg enabled
 * @text 有効にする
 * @type boolean
 * @default true
 *
 * @command BlockForFrames
 * @text nフレーム入力無効
 * @desc 指定フレームのあいだ入力無効（Live2D等の遷移に）
 * @arg frames
 * @text フレーム数
 * @type number
 * @min 1
 * @default 60
 *
 * @command SetScenes
 * @text 対象シーンを設定
 * @desc どのシーンで無効化するか
 * @arg map
 * @text マップ
 * @type boolean
 * @default true
 * @arg battle
 * @text 戦闘
 * @type boolean
 * @default true
 * @arg menu
 * @text メニュー系
 * @type boolean
 * @default false
 *
 * @command ManualBlockToggle
 * @text 手動ブロックON/OFF
 * @desc 明示的に開始/解除（フレーム指定なしの持続ブロック）
 * @arg on
 * @text ONにする
 * @type boolean
 * @default true
 *
 * @------------------------------------------------------------------
 * ■スクリプトAPI（コモンイベントの「スクリプト」等で呼べます）
 *   $gameTemp.requestInputBlock(frames)     // 指定フレーム入力無効
 *   $gameTemp.setInputBlockEnabled(true/false) // 全体の有効・無効
 *   $gameTemp.manualInputBlock(onOffBool)   // 手動ブロックのON/OFF
 *   $gameTemp.isInputBlocked()              // 現在ブロック中か確認(true/false)
 * 
 * 例）Live2Dアニメ切替から 30フレーム止める：
 *   $gameTemp.requestInputBlock(30);
 * 
 * 例）開始〜任意のタイミングまで止め、後で解除：
 *   $gameTemp.manualInputBlock(true);  // 開始
 *   // …後で
 *   $gameTemp.manualInputBlock(false); // 解除
 * -------------------------------------------------------------------
 */

(() => {
  "use strict";

  const PLUGIN_NAME = "InputBlockOnPictureWaitMZ";
  const params = PluginManager.parameters(PLUGIN_NAME);

  const paramEnableSwitchId = Number(params.EnableSwitchId || 0);
  let affectMap    = params.AffectOnMap === "true";
  let affectBattle = params.AffectOnBattle === "true";
  let affectMenu   = params.AffectOnMenu === "true";
  const blockDuringPictureWait = params.BlockDuringPictureWait === "true";
  const alsoBlockDuringPictureMoveNoWait = params.AlsoBlockDuringPictureMoveNoWait === "true";
  const blockTouchInput = params.BlockTouchInput === "true";

  // Runtime on/off (plugin command / API)
  let runtimeEnabled = true;

  // Manual block (toggle) and timer block (frames)
  class InputBlocker {
    constructor() {
      this._timer = 0;          // count-down frames
      this._manual = false;     // manual toggle
    }
    request(frames) {
      this._timer = Math.max(this._timer, Math.floor(frames || 0));
    }
    manual(on) {
      this._manual = !!on;
    }
    update() {
      if (this._timer > 0) this._timer--;
    }
    get activeTimer() {
      return this._timer > 0;
    }
    get activeManual() {
      return this._manual;
    }
  }
  const InputBlock = new InputBlocker();

  // Expose API on $gameTemp
  const _Game_Temp_initialize = Game_Temp.prototype.initialize;
  Game_Temp.prototype.initialize = function() {
    _Game_Temp_initialize.call(this);
    this._inputBlocker = InputBlock;
    this._inputBlockRuntimeEnabled = true;
  };
  Game_Temp.prototype.requestInputBlock = function(frames) {
    this._inputBlocker.request(frames);
  };
  Game_Temp.prototype.manualInputBlock = function(on) {
    this._inputBlocker.manual(on);
  };
  Game_Temp.prototype.setInputBlockEnabled = function(on) {
    runtimeEnabled = !!on;
    this._inputBlockRuntimeEnabled = runtimeEnabled;
  };
  Game_Temp.prototype.isInputBlocked = function() {
    return isInputBlockedNow();
  };

  // Plugin Commands
  PluginManager.registerCommand(PLUGIN_NAME, "SetEnabled", args => {
    const en = args.enabled === "true";
    runtimeEnabled = en;
    if ($gameTemp) $gameTemp._inputBlockRuntimeEnabled = en;
  });

  PluginManager.registerCommand(PLUGIN_NAME, "BlockForFrames", args => {
    const frames = Number(args.frames || 0);
    if ($gameTemp) $gameTemp.requestInputBlock(frames);
  });

  PluginManager.registerCommand(PLUGIN_NAME, "SetScenes", args => {
    affectMap    = args.map === "true";
    affectBattle = args.battle === "true";
    affectMenu   = args.menu === "true";
  });

  PluginManager.registerCommand(PLUGIN_NAME, "ManualBlockToggle", args => {
    const on = args.on === "true";
    if ($gameTemp) $gameTemp.manualInputBlock(on);
  });

  // ---- Core: 判定（今このフレーム入力を潰すか？） ----
  function currentSceneAffects() {
    const s = SceneManager._scene;
    if (!s) return false;
    if (s instanceof Scene_Map)    return affectMap;
    if (s instanceof Scene_Battle) return affectBattle;
    // メニューや他シーン
    return affectMenu;
  }

  function pictureWaitActiveAnywhere() {
    // map interpreters
    const list = [];

    if ($gameMap && $gameMap._interpreter) list.push($gameMap._interpreter);
    // 各イベントの並列実行Interpreter（存在する場合）
    if ($gameMap && $gameMap.events) {
      for (const ev of $gameMap.events()) {
        if (ev && ev._interpreter) list.push(ev._interpreter);
      }
    }
    // battle interpreter
    if ($gameTroop && $gameTroop._interpreter) list.push($gameTroop._interpreter);

    const stack = [];
    for (const it of list) if (it) stack.push(it);

    // 再帰的にchildInterpreterを掘る
    while (stack.length > 0) {
      const it = stack.pop();
      if (it._waitMode === "picture" || it._waitMode === "image" || it._waitMode === "picture2") {
        return true;
      }
      if (it._childInterpreter) stack.push(it._childInterpreter);
    }
    return false;
  }

  function anyPictureIsMoving() {
    if (!$gameScreen || !$gameScreen._pictures) return false;
    for (const p of $gameScreen._pictures) {
      if (p && p._duration && p._duration > 0) return true;
    }
    return false;
  }

  function enableSwitchOk() {
    if (!paramEnableSwitchId) return true;
    if (!$gameSwitches) return true;
    return $gameSwitches.value(paramEnableSwitchId);
    }

  function isInputBlockedNow() {
    if (!runtimeEnabled) return false;
    if (!currentSceneAffects()) return false;
    if (!enableSwitchOk()) return false;

    // 手動 or タイマー
    if (InputBlock.activeManual || InputBlock.activeTimer) return true;

    // ピクチャ移動（ウェイト有）待ち
    if (blockDuringPictureWait && pictureWaitActiveAnywhere()) return true;

    // ウェイト無しでも移動中なら無効化
    if (alsoBlockDuringPictureMoveNoWait && anyPictureIsMoving()) return true;

    return false;
  }

  // 毎フレーム減算
  const _Scene_Base_update = Scene_Base.prototype.update;
  Scene_Base.prototype.update = function() {
    _Scene_Base_update.call(this);
    InputBlock.update();
  };

  // ---- Input/TouchInput をラップして “false を返す” ----
  const wrapInputFn = (obj, fnName) => {
    const _orig = obj[fnName].bind(obj);
    obj[fnName] = function(...args) {
      if (isInputBlockedNow()) return false;
      return _orig(...args);
    };
  };

  wrapInputFn(Input, "isPressed");
  wrapInputFn(Input, "isTriggered");
  wrapInputFn(Input, "isRepeated");
  wrapInputFn(Input, "isLongPressed");
  if (Input.anyPressed) wrapInputFn(Input, "anyPressed");

  if (blockTouchInput) {
    wrapInputFn(TouchInput, "isPressed");
    wrapInputFn(TouchInput, "isTriggered");
    wrapInputFn(TouchInput, "isCancelled");
    if (TouchInput.isReleased) wrapInputFn(TouchInput, "isReleased");
    if (TouchInput.isMoved)    wrapInputFn(TouchInput, "isMoved");
  }

  // メニュー呼び出し等の保険（念押しの抑止）
  const _Scene_Map_isMenuCalled = Scene_Map.prototype.isMenuCalled;
  Scene_Map.prototype.isMenuCalled = function() {
    if (isInputBlockedNow()) return false;
    return _Scene_Map_isMenuCalled.call(this);
  };

  const _Game_Player_canMove = Game_Player.prototype.canMove;
  Game_Player.prototype.canMove = function() {
    if (isInputBlockedNow()) return false;
    return _Game_Player_canMove.call(this);
  };

})();
