/*:
 * @target MZ
 * @plugindesc [v1.1.0] PictureLive2DのsetAnimation発火時に、透明・下段の「空メッセージ」を自動投入して見た目を保ちつつ、変数/スイッチで強制ウェイト＆入力（スキップ/長押し）とHUDを制御（Scene_Map限定）
 * @author
 *  mushizushi + GPT
 *
 * @help
 * ■ 目的
 * - v1.0.0 で「ちゃんと止まっていた」挙動（= 強制ウェイト中は絶対に進まない）を踏襲。
 * - ただし Window_Message を close()/visible=false で消さず、
 *   Live2D再生開始と同時に「透明・下段・空文字」のメッセージを自動投入して
 *   画面上の見た目は従来どおりに保つ（WindowBackImage等と競合しない）。
 *
 * ■ 仕組み
 * - PictureLive2D.js の setAnimation をフック。
 * - 発火直前に「空メッセージ（透明/下段）」を必要に応じて差し込み、
 *   その後、強制ウェイトタイマー＆入力ブロックでテキスト進行を停止。
 * - 強制ウェイトは「変数」で制御、押しっぱ停止・スキップ許可は「スイッチ」で制御。
 * - HUD（「演出中…」など）を任意表示。
 *
 * ■ 強制ウェイト（変数の値）
 *   0: NoWait（※スキップ許可OFFなら安全のため自動的に 20f に変更）
 *   1: 20f
 *   2: 40f
 *   3: 60f
 *   4: 120f
 *
 * ■ 注意
 * - 本プラグインは Window_Message を閉じません。入力ブロックで止めます。
 * - 右クリックで非表示にした場合は（他プラグインがそうするなら）不可視の間は入力ブロックされます。
 * - 互換パッチ（WindowBackImage対策）は不要です。
 *
 * ▼ 並び順
 *   PictureLive2D.js より「下」。
 *
 * @param General
 * @text ▼ 基本（作者設定）
 *
 * @param WaitModeVariableId
 * @text 強制ウェイト設定に使う変数ID
 * @type variable
 * @default 10
 * @parent General
 *
 * @param DefaultWaitMode
 * @text 起動時デフォルト値（変数未設定時）
 * @type select
 * @option 0:NoWait @value 0
 * @option 1:20f    @value 1
 * @option 2:40f    @value 2
 * @option 3:60f    @value 3
 * @option 4:120f   @value 4
 * @default 0
 * @parent General
 *
 * @param StopOnHoldSwitchId
 * @text 押しっぱ中も停止 スイッチID
 * @type switch
 * @default 20
 * @parent General
 * @desc ON: 長押しでの早送り/連打を抑止
 *
 * @param AllowSkipWhileWaitSwitchId
 * @text 強制ウェイト中のスキップ許可 スイッチID
 * @type switch
 * @default 21
 * @parent General
 * @desc ON: 強制中でもOK/タップ/右クリックで即解除（NoWait時向け）
 *
 * @param Inject
 * @text ▼ 空メッセージ自動投入
 *
 * @param InjectEmptyMessage
 * @text Live2D発火時に空メッセージを自動投入する
 * @type boolean
 * @on する
 * @off しない
 * @default true
 * @parent Inject
 *
 * @param InjectOnlyIfNoMessage
 * @text 既にメッセージ表示中なら投入しない
 * @type boolean
 * @on 既存を尊重
 * @off 常に投入
 * @default true
 * @parent Inject
 *
 * @param InjectBackground
 * @text 空メッセージの背景
 * @type select
 * @option ウィンドウ @value 0
 * @option 暗くする @value 1
 * @option 透明 @value 2
 * @default 2
 * @parent Inject
 *
 * @param InjectPosition
 * @text 空メッセージの位置
 * @type select
 * @option 上 @value 0
 * @option 中 @value 1
 * @option 下 @value 2
 * @default 2
 * @parent Inject
 *
 * @param HUD
 * @text ▼ HUD
 *
 * @param HudEnabled
 * @text HUDを表示する
 * @type boolean
 * @on 表示
 * @off 非表示
 * @default true
 * @parent HUD
 *
 * @param HudText
 * @text HUDテキスト
 * @type string
 * @default 演出中…
 * @parent HUD
 *
 * @param HudStyle
 * @text HUDスタイル
 * @type select
 * @option 黒背景 @value black
 * @option ウィンドウ枠あり @value window
 * @option 透明 @value transparent
 * @default black
 * @parent HUD
 *
 * @param HudX
 * @text HUD X
 * @type number
 * @default 40
 * @parent HUD
 *
 * @param HudY
 * @text HUD Y
 * @type number
 * @default 40
 * @parent HUD
 *
 * @param HudWidth
 * @text HUD 幅
 * @type number
 * @default 240
 * @parent HUD
 *
 * @param HudHeight
 * @text HUD 高さ
 * @type number
 * @default 64
 * @parent HUD
 *
 * @param HudFontSize
 * @text HUD フォントサイズ
 * @type number
 * @default 26
 * @parent HUD
 *
 * @param Debug
 * @text ▼ デバッグ
 *
 * @param DebugLog
 * @text デバッグログ出力
 * @type boolean
 * @on 出す
 * @off 出さない
 * @default false
 * @parent Debug
 */

(() => {
  "use strict";

  const PLUGIN_NAME = "L2D_MessageHideByVars_EmptyWinMZ";
  const P = PluginManager.parameters(PLUGIN_NAME);

  const VAR_WAIT_MODE_ID  = Number(P["WaitModeVariableId"] || 10);
  const DEFAULT_WAIT_MODE = Number(P["DefaultWaitMode"] || 0);
  const SW_STOP_ON_HOLD   = Number(P["StopOnHoldSwitchId"] || 20);
  const SW_ALLOW_SKIP     = Number(P["AllowSkipWhileWaitSwitchId"] || 21);

  const INJECT_EMPTY      = P["InjectEmptyMessage"] === "true";
  const INJECT_ONLY_IDLE  = P["InjectOnlyIfNoMessage"] === "true";
  const INJ_BG            = Number(P["InjectBackground"] || 2);
  const INJ_POS           = Number(P["InjectPosition"] || 2);

  const HUD_ENABLED = P["HudEnabled"] === "true";
  const HUD_TEXT    = String(P["HudText"] || "演出中…");
  const HUD_STYLE   = String(P["HudStyle"] || "black");
  const HUD_X       = Number(P["HudX"] || 40);
  const HUD_Y       = Number(P["HudY"] || 40);
  const HUD_W       = Number(P["HudWidth"] || 240);
  const HUD_H       = Number(P["HudHeight"] || 64);
  const HUD_FONT    = Number(P["HudFontSize"] || 26);

  const DEBUG = P["DebugLog"] === "true";
  const dlog = (...xs)=>{ if(DEBUG) console.log(`[${PLUGIN_NAME}]`, ...xs); };

  const waitModeToFrames = (m)=>({1:20,2:40,3:60,4:120}[m]|0);

  // ========= Manager =========
  const L2DEmptyWinMgr = {
    _active:false,
    _until:0, // -1: input until
    _hud:null,
    _lastSceneId:null,

    currentWaitMode(){
      const v = $gameVariables.value(VAR_WAIT_MODE_ID);
      return (v===undefined || v===null) ? (DEFAULT_WAIT_MODE|0) : (v|0);
    },

    inForceWait(){
      if(!this._active) return false;
      if(this._until === -1) return true;
      return Graphics.frameCount < (this._until||0);
    },

    // 入力ブロック判定：強制待機中 or Window_Message不可視/閉
    shouldBlockMessageInput(){
      if(this.inForceWait()) return true;
      const s=SceneManager._scene, w=s && s._messageWindow;
      if(!w) return false;
      if(w.visible===false) return true;
      if(w.openness===0 && !w.isOpening()) return true;
      return false;
    },

    // Live2D発火トリガ（setAnimationフックから呼ばれる）
    onSetAnimation(){
      // 1) 空メッセージ自動投入（必要時）
      if (INJECT_EMPTY){
        if (!INJECT_ONLY_IDLE || !$gameMessage.isBusy()){
          try {
            $gameMessage.setBackground(INJ_BG);   // 2:透明
            $gameMessage.setPositionType(INJ_POS);// 2:下
            $gameMessage.add("");                 // 空行投入
            dlog("inject empty message");
          } catch(e){
            console.error(`[${PLUGIN_NAME}] inject error`, e);
          }
        }
      }

      // 2) 強制ウェイトのセット
      let frames = waitModeToFrames(this.currentWaitMode());
      const allowSkip = $gameSwitches.value(SW_ALLOW_SKIP);
      if (frames===0 && !allowSkip){ // フェイルセーフ
        frames = 20; dlog("failsafe: NoWait & skip OFF -> 20f");
      }

      const now = Graphics.frameCount;
      this._until = (frames>0) ? now + frames : -1;
      this._active = true;

      // HUD
      this.ensureHud(SceneManager._scene);
      this.setHudVisible(true);

      dlog(`start wait: frames=${frames}, until=${this._until}`);
    },

    breakByInput(){
      if(!this._active) return;
      this._active=false;
      this._until=0;
      this.setHudVisible(false);
      dlog("break by input");
    },

    update(){
      const s=SceneManager._scene;
      const id=s ? s.constructor.name : "";
      if(this._lastSceneId !== id){
        this._lastSceneId = id;
        this._hud = null;
      }

      if(this._active){
        if(this._until === -1){
          if($gameSwitches.value(SW_ALLOW_SKIP)){
            if (Input.isTriggered("ok") || TouchInput.isTriggered() || Input.isTriggered("cancel")){
              this.breakByInput(); return;
            }
          }
        }else{
          if(Graphics.frameCount >= (this._until||0)){
            this._active=false; this._until=0; this.setHudVisible(false);
            dlog("end by time"); return;
          }
        }
        this.ensureHud(s); this.setHudVisible(true);
      }else{
        this.setHudVisible(false);
      }
    },

    ensureHud(scene){
      if(!HUD_ENABLED) return;
      if(this._hud && this._hud.parent) return;

      if(HUD_STYLE==="window"){
        const rect = new Rectangle(HUD_X, HUD_Y, HUD_W, HUD_H);
        const win  = new Window_Base(rect);
        win.opacity = 255;
        win.contents.fontSize = HUD_FONT;
        win.drawText(HUD_TEXT, 0, (HUD_H-HUD_FONT)/2, HUD_W, "center");
        this._hud = win;
        scene && scene.addWindow && scene.addWindow(win);
      }else{
        const sp = new Sprite(new Bitmap(HUD_W, HUD_H));
        sp.x = HUD_X; sp.y = HUD_Y;
        if(HUD_STYLE==="black"){
          sp.bitmap.fillRect(0,0,HUD_W,HUD_H,"rgba(0,0,0,0.6)");
        }
        sp.bitmap.fontSize = HUD_FONT;
        sp.bitmap.drawText(HUD_TEXT, 0, (HUD_H-HUD_FONT)/2, HUD_W, "center");
        this._hud = sp;
        scene && scene.addChild && scene.addChild(sp);
      }
    },

    setHudVisible(v){ if(this._hud) this._hud.visible = !!v && HUD_ENABLED; },
  };

  // 公開（デバッグ用）
  window.L2DEmptyWinMgr = L2DEmptyWinMgr;

  // ===== Scene_Map tick =====
  const _Scene_Map_update = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function(){
    _Scene_Map_update.apply(this, arguments);
    L2DEmptyWinMgr.update();
  };

  // ===== Window_Message 入力フック（テキスト進行の停止を担保） =====
  const _WM_isTriggered = Window_Message.prototype.isTriggered;
  Window_Message.prototype.isTriggered = function(){
    if (L2DEmptyWinMgr.shouldBlockMessageInput()) return false;
    if ($gameSwitches.value(SW_STOP_ON_HOLD)){
      if (Input.isPressed("ok") || TouchInput.isPressed()) return false;
    }
    return _WM_isTriggered.apply(this, arguments);
  };

  const _WM_updateShowFast = Window_Message.prototype.updateShowFast;
  Window_Message.prototype.updateShowFast = function(){
    if (L2DEmptyWinMgr.shouldBlockMessageInput()){
      this._showFast=false; this._lineShowFast=false; return;
    }
    if ($gameSwitches.value(SW_STOP_ON_HOLD)){
      this._showFast=false; this._lineShowFast=false; return;
    }
    _WM_updateShowFast.apply(this, arguments);
  };

  // ===== PictureLive2D.setAnimation をフック =====
  function wrapLive2DObject(obj){
    if(!obj || obj.__l2dEmptyWinWrapped) return;
    const proto = Object.getPrototypeOf(obj) || obj;
    const target = obj.setAnimation ? obj : proto;
    if(!target || typeof target.setAnimation !== "function") return;

    const _orig = target.setAnimation;
    target.setAnimation = function(){
      try { L2DEmptyWinMgr.onSetAnimation(); }
      catch(e){ console.error(`[${PLUGIN_NAME}] hook error`, e); }
      return _orig.apply(this, arguments);
    };
    obj.__l2dEmptyWinWrapped = true;
    dlog("wrapped setAnimation on live2d object");
  }

  if (Game_Screen.prototype.live2d){
    const _gs_live2d = Game_Screen.prototype.live2d;
    Game_Screen.prototype.live2d = function(index){
      const ret = _gs_live2d.apply(this, arguments);
      try { wrapLive2DObject(ret); } catch(e){ console.error(`[${PLUGIN_NAME}] live2d wrap error`, e); }
      return ret;
    };
  }else{
    console.warn(`[${PLUGIN_NAME}] Game_Screen.live2d が見つかりません。PictureLive2D.js が必要です。`);
  }
})();
