/*:
 * @target MZ
 * @plugindesc [v1.1.0] Simple: PictureLive2DのsetAnimation時にだけ、変数/スイッチで強制ウェイト＋入力ブロック＋HUDを実行（ウィンドウ状態は一切触らない）
 * @author mushizushi + GPT
 *
 * @help
 * ■ 方針
 * - Window_Message の open/close/visible を一切変更しません。
 * - PictureLive2D.js の setAnimation 呼び出しをフックし、
 *   「強制ウェイト（変数）」「押しっぱ停止（スイッチ）」「強制中入力スキップ許可（スイッチ）」
 *   「HUD表示」だけを行います。
 * - 透明で空のテキストを下段で出す運用は**イベント側で実施**してください。
 *
 * ■ 推奨運用（例）
 *   1) 「文章の表示」：背景=透明 / 位置=下 / 内容=（空行） を置く
 *   2) 直後に Live2D 再生（$gameScreen.live2d(1).setAnimation(...);）
 *   3) 本プラグインが入力をブロック＆HUD表示＆時間管理
 *   4) 強制終了 or 許可スイッチON中のクリックで解除 → 以後は通常入力へ
 *
 * ■ 強制ウェイト（変数）
 *   0:NoWait（※スキップ許可OFFだと自動で20fに変換／ハマり防止）
 *   1:20f / 2:40f / 3:60f / 4:120f
 *
 * ■ 注意
 * - メッセージ進行の可否は「入力ブロック」で制御します。
 * - 右クリックで非表示にしていても、非表示中は進行しません（可視化してから進行）。
 * - WindowBackImage互換パッチは不要です。本プラグインはウィンドウを触りません。
 *
 * @param WaitModeVariableId
 * @text 強制ウェイト設定に使う変数ID
 * @type variable
 * @default 10
 *
 * @param DefaultWaitMode
 * @text 起動時デフォルト値（変数未設定時）
 * @type select
 * @option 0:NoWait @value 0
 * @option 1:20f    @value 1
 * @option 2:40f    @value 2
 * @option 3:60f    @value 3
 * @option 4:120f   @value 4
 * @default 0
 *
 * @param StopOnHoldSwitchId
 * @text 押しっぱ中も停止 スイッチID
 * @type switch
 * @default 20
 *
 * @param AllowSkipWhileWaitSwitchId
 * @text 強制中のスキップ許可 スイッチID
 * @type switch
 * @default 21
 *
 * @param HudEnabled
 * @text HUDを表示する
 * @type boolean
 * @on 表示
 * @off 非表示
 * @default true
 *
 * @param HudText
 * @text HUDテキスト
 * @type string
 * @default 演出中…
 *
 * @param HudStyle
 * @text HUDスタイル
 * @type select
 * @option 黒背景 @value black
 * @option ウィンドウ枠あり @value window
 * @option 透明 @value transparent
 * @default black
 *
 * @param HudX
 * @text HUD X
 * @type number
 * @default 40
 *
 * @param HudY
 * @text HUD Y
 * @type number
 * @default 40
 *
 * @param HudWidth
 * @text HUD 幅
 * @type number
 * @default 240
 *
 * @param HudHeight
 * @text HUD 高さ
 * @type number
 * @default 64
 *
 * @param HudFontSize
 * @text HUD フォントサイズ
 * @type number
 * @default 26
 *
 * @param DebugLog
 * @text デバッグログ出力
 * @type boolean
 * @on 出す
 * @off 出さない
 * @default false
 */

(() => {
  "use strict";

  const PLUGIN_NAME = "L2D_MessageHide_SimpleWaitMZ";
  const P = PluginManager.parameters(PLUGIN_NAME);

  const VAR_WAIT_MODE_ID = Number(P["WaitModeVariableId"] || 10);
  const DEFAULT_WAIT_MODE = Number(P["DefaultWaitMode"] || 0);
  const SW_STOP_ON_HOLD = Number(P["StopOnHoldSwitchId"] || 20);
  const SW_ALLOW_SKIP_WHILE_WAIT = Number(P["AllowSkipWhileWaitSwitchId"] || 21);

  const HUD_ENABLED = P["HudEnabled"] === "true";
  const HUD_TEXT = String(P["HudText"] || "演出中…");
  const HUD_STYLE = String(P["HudStyle"] || "black");
  const HUD_X = Number(P["HudX"] || 40);
  const HUD_Y = Number(P["HudY"] || 40);
  const HUD_W = Number(P["HudWidth"] || 240);
  const HUD_H = Number(P["HudHeight"] || 64);
  const HUD_FONT = Number(P["HudFontSize"] || 26);

  const DEBUG = P["DebugLog"] === "true";
  const dlog = (...a) => { if (DEBUG) console.log(`[${PLUGIN_NAME}]`, ...a); };

  const waitModeToFrames = (m)=>({1:20,2:40,3:60,4:120}[m]|0);

  //=== Manager（ウィンドウ非操作版） =========================
  const L2DSimpleWait = {
    _active:false,
    _until:0, // -1: input until
    _hud:null,
    _lastSceneId:null,

    currentWaitMode(){
      const v = $gameVariables.value(VAR_WAIT_MODE_ID);
      return (v===undefined || v===null) ? (DEFAULT_WAIT_MODE|0) : (v|0);
    },

    inForceWait(){
      if(!this._active) return false;
      if(this._until === -1) return true;
      return Graphics.frameCount < (this._until||0);
    },

    // 入力ブロック：強制待機中、またはメッセージが不可視の間
    shouldBlock(){
      if (this.inForceWait()) return true;
      const s=SceneManager._scene, w=s && s._messageWindow;
      if(!w) return false;
      if (w.visible === false) return true;
      if (w.openness === 0 && !w.isOpening()) return true;
      return false;
    },

    onSetAnimation(){
      let frames = waitModeToFrames(this.currentWaitMode());
      const allowSkip = $gameSwitches.value(SW_ALLOW_SKIP_WHILE_WAIT);
      // フェイルセーフ：NoWait×許可OFF→20f
      if (frames===0 && !allowSkip) { frames=20; dlog("failsafe: NoWait & skip OFF -> 20f"); }

      const now = Graphics.frameCount;
      this._until = (frames>0) ? now + frames : -1;
      this._active = true;

      const s=SceneManager._scene;
      this.ensureHud(s);
      this.setHudVisible(true);
      dlog(`started wait: frames=${frames}, until=${this._until}`);
    },

    breakByInput(){
      if (!this._active) return;
      this._active = false;
      this._until = 0;
      this.setHudVisible(false);
      dlog("break by input");
    },

    update(){
      // HUDの生存管理
      const s=SceneManager._scene;
      const id=s ? s.constructor.name : "";
      if (this._lastSceneId !== id) {
        this._lastSceneId = id;
        this._hud = null;
      }

      if (this._active){
        if (this._until === -1){
          if ($gameSwitches.value(SW_ALLOW_SKIP_WHILE_WAIT)){
            if (Input.isTriggered("ok") || TouchInput.isTriggered() || Input.isTriggered("cancel")){
              this.breakByInput(); return;
            }
          }
        } else {
          if (Graphics.frameCount >= (this._until||0)){
            this._active=false; this._until=0; this.setHudVisible(false);
            dlog("ended by time"); return;
          }
        }
        this.ensureHud(s); this.setHudVisible(true);
      } else {
        this.setHudVisible(false);
      }
    },

    ensureHud(scene){
      if (!HUD_ENABLED) return;
      if (this._hud && this._hud.parent) return;

      if (HUD_STYLE === "window"){
        const rect = new Rectangle(HUD_X, HUD_Y, HUD_W, HUD_H);
        const win = new Window_Base(rect);
        win.opacity = 255;
        win.contents.fontSize = HUD_FONT;
        win.drawText(HUD_TEXT, 0, (HUD_H - HUD_FONT)/2, HUD_W, "center");
        this._hud = win;
        scene && scene.addWindow && scene.addWindow(win);
      } else {
        const sp = new Sprite(new Bitmap(HUD_W, HUD_H));
        sp.x = HUD_X; sp.y = HUD_Y;
        if (HUD_STYLE === "black"){
          sp.bitmap.fillRect(0,0,HUD_W,HUD_H,"rgba(0,0,0,0.6)");
        }
        sp.bitmap.fontSize = HUD_FONT;
        sp.bitmap.drawText(HUD_TEXT, 0, (HUD_H - HUD_FONT)/2, HUD_W, "center");
        this._hud = sp;
        scene && scene.addChild && scene.addChild(sp);
      }
    },

    setHudVisible(v){ if (this._hud) this._hud.visible = !!v && HUD_ENABLED; }
  };

  // 公開（デバッグ用）
  window.L2DSimpleWait = L2DSimpleWait;

  //=== Scene_Map tick ======================================
  const _Scene_Map_update = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function(){
    _Scene_Map_update.apply(this, arguments);
    L2DSimpleWait.update();
  };

  //=== Window_Message 入力フック（進行阻止のみ） ============
  const _WM_isTriggered = Window_Message.prototype.isTriggered;
  Window_Message.prototype.isTriggered = function(){
    // 強制待機中 or 非表示中は進行不可
    if (L2DSimpleWait.shouldBlock()) return false;

    // 押しっぱ停止ONなら長押し早送りを抑止
    if ($gameSwitches.value(SW_STOP_ON_HOLD)){
      if (Input.isPressed("ok") || TouchInput.isPressed()) return false;
    }
    return _WM_isTriggered.apply(this, arguments);
  };

  const _WM_updateShowFast = Window_Message.prototype.updateShowFast;
  Window_Message.prototype.updateShowFast = function(){
    if (L2DSimpleWait.shouldBlock()){
      this._showFast=false; this._lineShowFast=false; return;
    }
    if ($gameSwitches.value(SW_STOP_ON_HOLD)){
      this._showFast=false; this._lineShowFast=false; return;
    }
    _WM_updateShowFast.apply(this, arguments);
  };

  //=== PictureLive2D.setAnimation フック =====================
  function wrapLive2DObject(obj){
    if(!obj || obj.__l2dSimpleWaitWrapped) return;
    const proto = Object.getPrototypeOf(obj) || obj;
    const target = obj.setAnimation ? obj : proto;
    if(!target || typeof target.setAnimation !== "function") return;

    const _orig = target.setAnimation;
    target.setAnimation = function(){
      try { L2DSimpleWait.onSetAnimation(); } catch(e){ console.error(`[${PLUGIN_NAME}] hook error`, e); }
      return _orig.apply(this, arguments);
    };
    obj.__l2dSimpleWaitWrapped = true;
    dlog("wrapped setAnimation on live2d object");
  }

  if (Game_Screen.prototype.live2d){
    const _gs_live2d = Game_Screen.prototype.live2d;
    Game_Screen.prototype.live2d = function(index){
      const ret = _gs_live2d.apply(this, arguments);
      try { wrapLive2DObject(ret); } catch(e){ console.error(`[${PLUGIN_NAME}] live2d wrap error`, e); }
      return ret;
    };
  } else {
    console.warn(`[${PLUGIN_NAME}] Game_Screen.live2d が見つかりません。PictureLive2D.js が必要です。`);
  }
})();
