/*:
 * @target MZ
 * @plugindesc Live2Dモデルをキー操作で移動（範囲制限つき、スイッチ制御・オフセット方式） Live2DInterfaceMZ専用
 * @author you
 *
 * @param moveStep
 * @text 移動ステップ
 * @desc 1回のキー入力で動く距離(px)
 * @type number
 * @default 16
 *
 * @param defaultBox
 * @text デフォルト範囲
 * @desc 移動可能範囲のサイズ(px)。中心から±の距離
 * @type number
 * @default 200
 *
 * @param controlSwitch
 * @text 有効化スイッチID
 * @desc このスイッチがONのときだけ移動できる
 * @type switch
 * @default 1
 *
 * @command setTargetIndex
 * @text 対象モデル番号
 * @desc live2dSprite のキー番号を指定（例: 0）
 * @arg index
 * @type number
 * @default 0
 *
 * @command setBox
 * @text 移動範囲を設定
 * @desc 移動可能範囲のサイズ(px)。中心から±の距離
 * @arg size
 * @type number
 * @default 200
 *
 * @command resetBox
 * @text 移動範囲をリセット
 * @desc 移動範囲をデフォルトに戻します。
 */

(() => {
  "use strict";
  const PN = "Live2DBoxMove";

  const params = PluginManager.parameters(PN);
  const state = {
    step: Number(params.moveStep || 16),
    defaultBox: Number(params.defaultBox || 200),
    boxSize: Number(params.defaultBox || 200),
    targetIndex: 0, // ← デフォルトを 0 に変更
    switchId: Number(params.controlSwitch || 1)
  };

  function currentKeyVector() {
    let l = Input.isPressed("left") || Input.isPressed("a");
    let r = Input.isPressed("right") || Input.isPressed("d");
    let u = Input.isPressed("up") || Input.isPressed("w");
    let d = Input.isPressed("down") || Input.isPressed("s");
    let x = (l ? -1 : 0) + (r ? 1 : 0);
    let y = (u ? -1 : 0) + (d ? 1 : 0);
    return { x, y };
  }

  // Live2D のオフセットを反映する拡張
  const _Sprite_Live2d_update = Sprite_Live2d.prototype.update;
  Sprite_Live2d.prototype.update = function() {
    _Sprite_Live2d_update.call(this);
    if (this._moveOffset) {
      this.x = this._baseX + this._moveOffset.x;
      this.y = this._baseY + this._moveOffset.y;
    } else {
      // 初期化（基準は画面中央）
      this._baseX = Graphics.width / 2;
      this._baseY = Graphics.height / 2;
      this._moveOffset = {x: 0, y: 0};
      this.x = this._baseX;
      this.y = this._baseY;
    }
  };

  function moveLive2D(dx, dy) {
    const s = SceneManager._scene;
    if (!s || !s.live2dSprite) return;
    const sp = s.live2dSprite[state.targetIndex];
    if (!sp) return;

    if (!sp._moveOffset) {
      sp._baseX = Graphics.width / 2;
      sp._baseY = Graphics.height / 2;
      sp._moveOffset = {x: 0, y: 0};
    }

    const half = state.boxSize / 2;
    sp._moveOffset.x = Math.min(half, Math.max(-half, sp._moveOffset.x + dx));
    sp._moveOffset.y = Math.min(half, Math.max(-half, sp._moveOffset.y + dy));
  }

  const _Scene_Map_update = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function () {
    _Scene_Map_update.call(this);

    // スイッチがOFFなら停止
    if (!$gameSwitches.value(state.switchId)) return;

    // 会話中制限を撤廃（$gameMessage.isBusy() のチェック削除）

    const v = currentKeyVector();
    if (v.x === 0 && v.y === 0) return;
    const dx = v.x * state.step;
    const dy = v.y * state.step;
    moveLive2D(dx, dy);
  };

  // Plugin Commands
  PluginManager.registerCommand(PN, "setTargetIndex", args => {
    state.targetIndex = Number(args.index || 0);
  });

  PluginManager.registerCommand(PN, "setBox", args => {
    state.boxSize = Math.max(1, Number(args.size || 200));
  });

  PluginManager.registerCommand(PN, "resetBox", () => {
    state.boxSize = state.defaultBox;
  });

})();
