/*:
 * @target MZ
 * @plugindesc シーンごとにスイッチ・変数を適用するプラグイン（デバッグ対応版）
 * @author あなたの名前
 * 
 * @param Scenes
 * @text シーン設定
 * @type struct<Scene>[]
 * @desc スイッチ・変数のセットを管理するシーンリスト
 * @default []
 * 
 * @command setScene
 * @text シーンを適用（名前）
 * @desc シーン名を指定して適用
 * 
 * @arg scene
 * @type combo
 * @text シーン名
 * @desc 適用するシーンを選択
 * @default 
 * 
 * @command setSceneById
 * @text シーンを適用（ID）
 * @desc シーンIDを指定して適用
 * 
 * @arg sceneId
 * @type number
 * @min 1
 * @text シーンID
 * @desc シーンの登録順で指定（1から）
 * @default 1
 */

 /*~struct~Scene:
 * @param name
 * @text シーン名
 * @desc シーンの名前
 * @type string
 * 
 * @param SwitchList
 * @text スイッチ設定
 * @type struct<SceneSwitch>[]
 * @desc 管理するスイッチリスト
 * @default []
 * 
 * @param VariableList
 * @text 変数設定
 * @type struct<SceneVariable>[]
 * @desc 管理する変数リスト
 * @default []
 */

 /*~struct~SceneSwitch:
 * @param id
 * @text スイッチID
 * @desc 設定するスイッチID
 * @type switch
 * 
 * @param value
 * @text スイッチ状態
 * @desc ON = true, OFF = false
 * @type boolean
 * @default true
 */

 /*~struct~SceneVariable:
 * @param id
 * @text 変数ID
 * @desc 設定する変数ID
 * @type variable
 * 
 * @param value
 * @text 変数の値
 * @desc 設定する数値
 * @type number
 * @default 0
 */

 (() => {
    const pluginName = "MZVariableSwitchPlugin";

    let sceneDataCache = {};
    let sceneList = [];

    function loadSceneData() {
        sceneDataCache = {};
        sceneList = [];

        const params = PluginManager.parameters(pluginName);
        console.log("[DEBUG] 取得したプラグインパラメータ:", params);

        if (!params || Object.keys(params).length === 0) {
            console.error("[ERROR] プラグインパラメータが取得できませんでした！");
            return;
        }

        if (!params.hasOwnProperty("Scenes")) {
            console.error("[ERROR] Scenes のキーが見つかりません！");
            return;
        }

        const rawScenes = params["Scenes"] || "[]";
        console.log("[DEBUG] rawScenes (before JSON.parse):", rawScenes);

        let parsedScenes;
        try {
            parsedScenes = JSON.parse(rawScenes);
            console.log("[DEBUG] parsedScenes (after JSON.parse):", parsedScenes);
        } catch (e) {
            console.error("[ERROR] シーンデータのJSON解析エラー:", e);
            return;
        }

        parsedScenes.forEach((scene, index) => {
            try {
                const sceneObj = JSON.parse(scene);
                const sceneId = index + 1;  // **シーンIDを確実に振る**
                const sceneName = sceneObj.name + `#${sceneId}`; // **同じ名前でも区別する**
                
                const switches = JSON.parse(sceneObj.SwitchList || "[]").map(entry => JSON.parse(entry));
                const variables = JSON.parse(sceneObj.VariableList || "[]").map(entry => JSON.parse(entry));

                sceneDataCache[sceneId] = { // **IDをキーに変更**
                    id: sceneId,
                    name: sceneObj.name, // **元の名前も保持**
                    switches: switches,
                    variables: variables
                };

                sceneList.push({ id: sceneId, name: sceneObj.name });

                console.log(`[DEBUG] シーン登録: ${sceneObj.name} (ID: ${sceneId})`);
            } catch (e) {
                console.error(`[ERROR] シーン '${sceneObj.name}' のデータ解析エラー:`, e);
            }
        });

        console.log("[DEBUG] シーンデータがロードされました:", sceneDataCache);
    }

    loadSceneData();

    PluginManager.registerCommand(pluginName, "setSceneById", args => {
        console.log(`[DEBUG] setSceneById 実行:`, args);

        const sceneId = Number(args.sceneId);
        console.log(`[DEBUG] 取得したシーンID: ${sceneId}`);

        if (isNaN(sceneId) || sceneId < 1 || !sceneDataCache[sceneId]) {
            console.warn(`[ERROR] シーンID '${sceneId}' に該当するシーンが見つかりません！`);
            console.log("[DEBUG] 現在のシーンデータ:", sceneDataCache);
            return;
        }

        console.log(`[DEBUG] シーンID '${sceneId}' は '${sceneDataCache[sceneId].name}' に対応`);
        applySceneValues(sceneDataCache[sceneId]);
    });

    function applySceneValues(sceneData) {
        console.log(`[DEBUG] '${sceneData.name}' の値を適用中...`, sceneData);

        sceneData.switches.forEach(data => {
            const id = Number(data.id);
            const value = (data.value === "true" || data.value === true);
            console.log(`[DEBUG] スイッチ ${id} を ${value ? "ON" : "OFF"} に変更`);
            $gameSwitches.setValue(id, value);
        });

        sceneData.variables.forEach(data => {
            const id = Number(data.id);
            const value = Number(data.value);
            console.log(`[DEBUG] 変数 ${id} を ${value} に変更`);
            $gameVariables.setValue(id, value);
        });

        if (SceneManager._scene && SceneManager._scene.refresh) {
            console.log("[DEBUG] シーンの更新を実行");
            SceneManager._scene.refresh();
        }

        console.log(`[DEBUG] シーン '${sceneData.name}' の適用が完了しました`);
    }
})();
