/*:
 * @target MZ
 * @plugindesc クリック/スペース/エンターでムービーを60Fフェードスキップ（Video/Graphics両対応・引数修正）
 * @author むしずし
 *
 * @command playMovieSkippable
 * @text スキップ対応ムービー再生
 * @desc クリック/スペース/エンターで60Fフェードしながらスキップできるムービーを再生します。
 *
 * @arg filename
 * @text ムービーファイル名
 * @desc moviesフォルダ内のファイル名（拡張子不要でも可）
 * @type file
 * @dir movies
 */

(() => {
  const PLUGIN_NAME = "MovieSkipFade";
  const FADE_FRAMES = 60;

  PluginManager.registerCommand(PLUGIN_NAME, "playMovieSkippable", args => {
    const raw = String(args.filename || "").trim();
    if (!raw) return;

    // 1) ベース名（フォルダ/拡張子を除去）
    const justName = raw
      .replace(/^.*[\\/]/, "")      // ディレクトリ除去
      .replace(/\.(webm|mp4)$/i, ""); // 拡張子除去

    // 2) Graphics系で使うフルパス
    const webmPath = `movies/${justName}.webm`;

    // --- 再生 ---
    // Video.play には「フォルダなし・拡張子なし」で渡すのが正解
    // Graphics.playVideo には「フルパス＋拡張子付き」
    let videoEl = null;
    function tryPlay() {
      if (window.Video && typeof Video.play === "function") {
        // MV/MZ 共通の Video API
        Video.play(justName);
        videoEl = Video._element || null;
      } else if (window.Graphics && typeof Graphics.playVideo === "function") {
        Graphics.playVideo(webmPath);
        videoEl = Graphics._video || null;
      } else {
        console.error("[MovieSkipFade] Video API が見つかりません");
      }
    }
    tryPlay();

    // --- 入力 ---
    let skip = false, fading = false, fadeCount = 0, ended = false;
    const onKeyDown = e => {
      if (e.code === "Space" || e.code === "Enter") skip = true;
    };
    const onPointer = () => { skip = true; };
    window.addEventListener("keydown", onKeyDown);
    window.addEventListener("mousedown", onPointer);
    window.addEventListener("touchstart", onPointer);

    function unbind() {
      window.removeEventListener("keydown", onKeyDown);
      window.removeEventListener("mousedown", onPointer);
      window.removeEventListener("touchstart", onPointer);
    }

    function ensureEl() {
      // 再生直後に要素がまだ作られてない場合があるので拾い直し
      videoEl = (window.Video && Video._element) || (window.Graphics && Graphics._video) || videoEl;
      if (videoEl) {
        videoEl.style.opacity = videoEl.style.opacity || "1";
      }
    }

    function startFade() {
      if (!videoEl || fading) return;
      fading = true;
      fadeCount = FADE_FRAMES;
      videoEl.style.pointerEvents = "none";
    }

    function endVideo() {
      if (ended) return;
      ended = true;
      try {
        const el = videoEl || (window.Video && Video._element) || (window.Graphics && Graphics._video) || null;
        if (el) {
          try { el.pause(); } catch {}
          try { el.currentTime = el.duration || 0; } catch {}
          // MV/MZの内部終了ハンドラがあれば念のため呼ぶ
          if (window.Video && typeof Video._onEnd === "function") {
            try { Video._onEnd(); } catch {}
          }
        }
      } finally {
        unbind();
        PIXI.Ticker.shared.remove(tick);
        if (videoEl) {
          videoEl.style.opacity = "";
          videoEl.style.pointerEvents = "";
          videoEl = null;
        }
      }
    }

    function stepFade() {
      if (!fading || !videoEl) return;
      fadeCount--;
      const t = Math.max(0, fadeCount) / FADE_FRAMES;
      videoEl.style.opacity = String(t);
      if (fadeCount <= 0) endVideo();
    }

    const tick = () => {
      ensureEl();

      // 自然終了
      if (videoEl && videoEl.ended && !ended) {
        endVideo();
        return;
      }

      // スキップ入力でフェード開始
      if (skip && videoEl && !fading) startFade();

      // フェード進行
      if (fading) stepFade();
    };

    PIXI.Ticker.shared.add(tick);

    // 環境チェック（ブラウザ直実行は失敗しやすい）
    const proto = location.protocol;
    if (proto.startsWith("http") || proto.startsWith("chrome-extension")) {
      console.warn(
        "[MovieSkipFade] ブラウザ実行で動画が読めない場合があります。ツクールMZのテストプレイ（NW.js）での実行を推奨します。"
      );
    }
  });
})();
