/*:
 * @target MZ
 * @plugindesc 十字キーでピクチャを移動（ステップ/フレーム/クランプ/対角/連打間隔 対応）
 * @author ChatGPT
 *
 * @help PictureKeyMoveMZ.js
 * 十字キー（ゲームパッド含む）で指定ピクチャをスムーズに移動します。
 * ・1回の入力ごとに「ステップピクセル」分だけ相対移動
 * ・フレーム数でスムーズ補間（picture.movePicture を使用）
 * ・画像の端が画面外に出ないクランプ（原点＆拡大率対応）
 * ・対角移動ON/OFF
 * ・キーリピート：初回遅延＆リピート間隔（フレーム）
 * ・有効/無効の切替、実行中に各種設定変更
 *
 * 【注意】
 * 既定では「マップ画面」でのみ動作します。バトルでも使いたい場合はパラメータをONに。
 * メッセージ表示中も動かしたい場合は「メッセージ中も許可」をONに。
 *
 * @param DefaultPictureId
 * @text 既定ピクチャ番号
 * @type number
 * @min 1
 * @default 1
 *
 * @param StepPixels
 * @text ステップピクセル
 * @type number
 * @min 1
 * @default 16
 *
 * @param Duration
 * @text フレーム数（補間）
 * @type number
 * @min 0
 * @default 6
 * @desc 0で瞬間移動。1以上でスムーズ移動。
 *
 * @param ClampToScreen
 * @text 画面端で止める
 * @type boolean
 * @on クランプする
 * @off しない
 * @default true
 *
 * @param AllowDiagonal
 * @text 対角移動を許可
 * @type boolean
 * @on 許可
 * @off 直交のみ
 * @default true
 *
 * @param InitialDelay
 * @text キー初回遅延(f)
 * @type number
 * @min 0
 * @default 18
 *
 * @param RepeatInterval
 * @text 連打間隔(f)
 * @type number
 * @min 1
 * @default 4
 *
 * @param EnableOnMap
 * @text マップで有効
 * @type boolean
 * @default true
 *
 * @param EnableOnBattle
 * @text バトルで有効
 * @type boolean
 * @default false
 *
 * @param AllowDuringMessage
 * @text メッセージ中も許可
 * @type boolean
 * @default false
 *
 * @command enable
 * @text 有効化
 * @desc キー操作での移動を有効にします
 *
 * @command disable
 * @text 無効化
 * @desc キー操作での移動を無効にします
 *
 * @command setPicture
 * @text ピクチャ番号の設定
 * @arg pictureId
 * @type number
 * @min 1
 * @default 1
 *
 * @command setStep
 * @text ステップ/フレーム/クランプ設定
 * @arg step
 * @text ステップピクセル
 * @type number
 * @min 1
 * @default 16
 * @arg duration
 * @text フレーム数
 * @type number
 * @min 0
 * @default 6
 * @arg clamp
 * @text 画面端で止める
 * @type boolean
 * @default true
 *
 * @command nudge
 * @text 1ステップ移動（イベントから）
 * @desc イベント側から1回ぶん動かします
 * @arg dx
 * @text Xステップ（右＋/左−）
 * @type number
 * @default 1
 * @arg dy
 * @text Yステップ（下＋/上−）
 * @type number
 * @default 0
 *
 * @command diag
 * @text 対角移動ON/OFF
 * @arg allow
 * @type boolean
 * @default true
 */

(() => {
  const PN = "PictureKeyMoveMZ";
  const P = PluginManager.parameters(PN);
  const cfg = {
    pictureId: Number(P.DefaultPictureId || 1),
    step: Number(P.StepPixels || 16),
    duration: Number(P.Duration || 6),
    clamp: String(P.ClampToScreen) === "true",
    allowDiagonal: String(P.AllowDiagonal) === "true",
    initialDelay: Number(P.InitialDelay || 18),
    repeatInterval: Number(P.RepeatInterval || 4),
    enableOnMap: String(P.EnableOnMap) === "true",
    enableOnBattle: String(P.EnableOnBattle) === "true",
    allowDuringMsg: String(P.AllowDuringMessage) === "true",
  };

  // 内部ステート
  const state = {
    enabled: false,
    hold: { left:0, right:0, up:0, down:0 },
  };

  // 画像クランプ
  function clampXY(picture, bitmap, tx, ty) {
    if (!bitmap) return { x: tx, y: ty };
    const scrW = Graphics.width, scrH = Graphics.height;
    const origin = picture._origin || 0; // 0:左上 1:中央
    const sx = ((picture.scaleX ? picture.scaleX() : picture._scaleX || 100) / 100) || 1;
    const sy = ((picture.scaleY ? picture.scaleY() : picture._scaleY || 100) / 100) || 1;
    const w = (bitmap.width || 0) * sx;
    const h = (bitmap.height || 0) * sy;
    let minX, maxX, minY, maxY;
    if (origin === 0) {
      minX = 0; minY = 0;
      maxX = Math.max(0, scrW - w);
      maxY = Math.max(0, scrH - h);
    } else {
      const hw = w / 2, hh = h / 2;
      minX = hw; maxX = Math.max(hw, scrW - hw);
      minY = hh; maxY = Math.max(hh, scrH - hh);
    }
    const clamp = (v, a, b) => Math.max(a, Math.min(b, v));
    return { x: clamp(tx, minX, maxX), y: clamp(ty, minY, maxY) };
  }

  // 相対移動実行
  function moveBy(dx, dy, duration = cfg.duration, clampToScreen = cfg.clamp) {
    const pic = $gameScreen.picture(cfg.pictureId);
    if (!pic) return;

    // 画像ロード待ち→到達点クランプ→movePicture
    const name = pic._name;
    const exec = () => {
      let targetX = (pic.x ? pic.x() : pic._x) + dx;
      let targetY = (pic.y ? pic.y() : pic._y) + dy;
      if (clampToScreen) {
        const bmp = name ? ImageManager.loadPicture(name) : null;
        if (bmp && !bmp.isReady()) return false;
        const c = clampXY(pic, bmp, targetX, targetY);
        targetX = c.x; targetY = c.y;
      }
      const origin  = (pic.origin ? pic.origin() : pic._origin) || 0;
      const scaleX  = (pic.scaleX ? pic.scaleX() : pic._scaleX) || 100;
      const scaleY  = (pic.scaleY ? pic.scaleY() : pic._scaleY) || 100;
      const opacity = (pic.opacity ? pic.opacity() : pic._opacity) || 255;
      let blendMode = (pic.blendMode ? pic.blendMode() : pic._blendMode) || 0;
      blendMode = (blendMode|0); if (blendMode<0||blendMode>3) blendMode = 0;

      $gameScreen.movePicture(cfg.pictureId, origin, targetX, targetY, scaleX, scaleY, opacity, blendMode, Math.max(0, duration|0));
      return true;
    };
    let retry = 60;
    const wait = () => {
      if (exec()) return;
      if (--retry <= 0) { exec(); return; }
      requestAnimationFrame(wait);
    };
    wait();
  }

  // 入力→移動（キーリピート制御）
  function updateKeyMove() {
    if (!state.enabled) return;
    if (!$gamePlayer) return;

    // 画面条件
    const isMap = SceneManager._scene instanceof Scene_Map;
    const isBattle = SceneManager._scene instanceof Scene_Battle;
    if (!((cfg.enableOnMap && isMap) || (cfg.enableOnBattle && isBattle))) return;

    // メッセージ中禁止
    if (!cfg.allowDuringMsg && $gameMessage.isBusy && $gameMessage.isBusy()) return;

    const pressed = {
      left:  Input.isPressed("left"),
      right: Input.isPressed("right"),
      up:    Input.isPressed("up"),
      down:  Input.isPressed("down"),
    };

    // 対角禁止なら直交優先（同時押しを打ち消す）
    if (!cfg.allowDiagonal) {
      if (pressed.left && pressed.right) { pressed.left = pressed.right = false; }
      if (pressed.up && pressed.down) { pressed.up = pressed.down = false; }
    }

    // 方向ごとに初回遅延＆リピート間隔で発火
    const tryAxis = (key, dx, dy) => {
      if (pressed[key]) {
        state.hold[key]++;
        const t = state.hold[key];
        if (t === 1) {
          // 押下直後：即発火
          moveBy(dx, dy);
        } else if (t > cfg.initialDelay && ((t - cfg.initialDelay) % cfg.repeatInterval === 0)) {
          moveBy(dx, dy);
        }
      } else {
        state.hold[key] = 0;
      }
    };

    // 対角を許可する場合：独立に積み上げる（0,1,2,3…で個別リピート）
    // 許可しない場合も同じロジック（直交しか発火しない）
    tryAxis("left",  -cfg.step, 0);
    tryAxis("right",  cfg.step, 0);
    tryAxis("up",     0, -cfg.step);
    tryAxis("down",   0,  cfg.step);
  }

  // Scene更新フック（Map/Battle）
  const _SceneMap_update = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function() {
    updateKeyMove();
    _SceneMap_update.call(this);
  };
  const _SceneBattle_update = Scene_Battle.prototype.update;
  Scene_Battle.prototype.update = function() {
    updateKeyMove();
    _SceneBattle_update.call(this);
  };

  // プラグインコマンド
  PluginManager.registerCommand(PN, "enable",  () => { state.enabled = true; });
  PluginManager.registerCommand(PN, "disable", () => { state.enabled = false; });

  PluginManager.registerCommand(PN, "setPicture", args => {
    cfg.pictureId = Math.max(1, Number(args.pictureId||cfg.pictureId));
  });

  PluginManager.registerCommand(PN, "setStep", args => {
    cfg.step = Math.max(1, Number(args.step||cfg.step));
    cfg.duration = Math.max(0, Number(args.duration||cfg.duration));
    cfg.clamp = String(args.clamp) === "true";
  });

  PluginManager.registerCommand(PN, "nudge", args => {
    const dx = Number(args.dx||0) * cfg.step;
    const dy = Number(args.dy||0) * cfg.step;
    moveBy(dx, dy);
  });

  PluginManager.registerCommand(PN, "diag", args => {
    cfg.allowDiagonal = String(args.allow) === "true";
  });

  // スクリプトAPI（任意）
  window.PictureKeyMove = {
    enable(){ state.enabled = true; },
    disable(){ state.enabled = false; },
    config(){ return {...cfg}; },
    setPicture(id){ cfg.pictureId = Math.max(1, Number(id||cfg.pictureId)); },
    setStep(step, duration=cfg.duration, clamp=cfg.clamp){
      cfg.step = Math.max(1, Number(step||cfg.step));
      cfg.duration = Math.max(0, Number(duration||cfg.duration));
      cfg.clamp = !!clamp;
    },
    nudge(dxSteps, dySteps){ moveBy((dxSteps|0)*cfg.step, (dySteps|0)*cfg.step); },
  };
})();
