/*:
 * @target MZ
 * @plugindesc ピクチャ移動＋クランプ／実寸指定対応。クリップ無効時は左上／中央基準を選択可。
 * @author you
 *
 * @command setKeyMode
 * @arg mode
 * @type select
 * @option arrow
 * @option wasd
 * @option both
 * @default both
 *
 * @command setTargetIds
 * @text 対象ピクチャID設定
 * @desc 操作対象とするピクチャIDをカンマ区切りで指定します。
 * @arg ids
 * @type string
 * @default 1
 *
 * @command setLocalMoveArea
 * @arg size
 * @type number
 * @default 200
 *
 * @command setContentSize
 * @arg id
 * @type number
 * @default 1
 * @arg width
 * @type number
 * @default 1920
 * @arg height
 * @type number
 * @default 1080
 *
 * @command setNoClampOrigin
 * @text クランプ無効時の基準
 * @desc 画像位置基準（左上 or 中央）
 * @arg origin
 * @type select
 * @option leftTop
 * @option center
 * @default center
 *
 * @command setDuringMode
 * @text 反応タイミング
 * @desc 入力を受け付けるタイミング（常時 / 会話中のみ / 会話中以外）
 * @arg mode
 * @type select
 * @option always
 * @option onlyMessage
 * @option noMessage
 * @default onlyMessage
 *
 * @command disableLocalMoveArea
 * @text ローカル移動範囲を解除
 * @desc setLocalMoveArea で設定した±範囲移動を解除します（アンカーも破棄）。
 *
 * @command clearContentSize
 * @text 実寸登録の解除
 * @desc setContentSize で登録した実寸を解除します。
 * @arg id
 * @type number
 * @min 1
 * @default 1
 *
 * @command setClamp
 * @text 画面クランプON/OFF
 * @desc 画面内に収めるクランプを切り替えます（ON推奨）。
 * @arg value
 * @type select
 * @option on
 * @option off
 * @default on
 */
(()=>{
  "use strict";
  const PN = "PictureKeyMoveTest";

  const state = {
    enabled: true,
    allowDuringMessage: true,
    step: 16,
    duration: 6,
    diag: false,
    clamp: true,
    targetIds: [1],
    localArea: null,
    keyMode: "both",
    noClampOrigin: "center",
    duringMode: "onlyMessage"
  };

  const _pkmtContentSizeById = {};
  const _localAnchorById = {};

  function _pkmtGetOverrideSizeFor(pic){
    try{
      const id = pic?._pictureId ?? pic?._id;
      if(id && _pkmtContentSizeById[id]) return _pkmtContentSizeById[id];
    }catch(_){ }
    return null;
  }

  function isMsgBusy(){ return $gameMessage && $gameMessage.isBusy(); }
  function pictureOf(id){ return $gameScreen.picture(id); }

  function currentKeyVector(){
    let l=false,r=false,u=false,d=false;
    if(state.keyMode==="arrow" || state.keyMode==="both"){
      l = l || Input.isPressed("left");
      r = r || Input.isPressed("right");
      u = u || Input.isPressed("up");
      d = d || Input.isPressed("down");
    }
    if(state.keyMode==="wasd" || state.keyMode==="both"){
      l = l || Input.isPressed("a");
      r = r || Input.isPressed("d");
      u = u || Input.isPressed("w");
      d = d || Input.isPressed("s");
    }
    let x = (l? -1:0)+(r? 1:0);
    let y = (u? -1:0)+(d? 1:0);
    if(!state.diag && x!==0) y=0;
    return {x,y};
  }

  function computeBoundsForPicture(pic){
    const name = pic._name||"";
    const id = pic?._pictureId ?? pic?._id;
    let baseW=0, baseH=0;
    if(name){
      const bmp = ImageManager.loadPicture(name);
      if(bmp&&bmp.isReady()){ baseW=bmp.width; baseH=bmp.height; }
    }
    if((!baseW||!baseH)&&id){
      const ov = _pkmtGetOverrideSizeFor(pic);
      if(ov){ baseW=ov.w; baseH=ov.h; }
    }
    if(!baseW||!baseH){ pic._pkmtBounds=null; return; }
    const scrW = Graphics.width, scrH = Graphics.height;
    const origin = pic._origin||0;
    const sx=((pic._scaleX??100)/100), sy=((pic._scaleY??100)/100);
    const w=baseW*sx, h=baseH*sy;
    let minX,maxX,minY,maxY;
    if(origin===0){
      minX = (w<=scrW)?0:Math.floor(scrW-w);
      maxX = 0;
      minY = (h<=scrH)?0:Math.floor(scrH-h);
      maxY = 0;
    }else{
      const hw=w/2, hh=h/2;
      minX = (w<=scrW)?Math.round(scrW/2):Math.ceil(hw);
      maxX = (w<=scrW)?Math.round(scrW/2):Math.floor(scrW-hw);
      minY = (h<=scrH)?Math.round(scrH/2):Math.ceil(hh);
      maxY = (h<=scrH)?Math.round(scrH/2):Math.floor(scrH-hh);
    }
    pic._pkmtBounds = {minX,maxX,minY,maxY};
  }

  function ensureBounds(pic){ if(!pic) return; if(!pic._pkmtBounds) computeBoundsForPicture(pic); }

  function clampXY(pic,x,y){
    const id = pic._pictureId;
    if(state.localArea){
      if(!_localAnchorById[id]) _localAnchorById[id] = {x:pic._x||0, y:pic._y||0};
      const a = _localAnchorById[id];
      const half = state.localArea/2;
      x = Math.min(a.x+half, Math.max(a.x-half, x));
      y = Math.min(a.y+half, Math.max(a.y-half, y));
      return {x,y};
    }
    if(!state.clamp){
      if(state.noClampOrigin==="leftTop"){
        return {x,y};
      }else{
        const cx = Graphics.width/2, cy = Graphics.height/2;
        return {x:x+cx, y:y+cy};
      }
    }
    ensureBounds(pic);
    const b = pic._pkmtBounds; if(!b) return {x,y};
    x = Math.min(b.maxX, Math.max(b.minX, x));
    y = Math.min(b.maxY, Math.max(b.minY, y));
    return {x,y};
  }

  function movePicBy(id,dx,dy,dur){
    const pic = pictureOf(id);
    if(!pic) return;
    const x = (pic._x||0)+dx;
    const y = (pic._y||0)+dy;
    const clamped = clampXY(pic,x,y);
    $gameScreen.movePicture(id, pic._origin||0, clamped.x, clamped.y, pic._scaleX||100, pic._scaleY||100, pic._opacity||255, pic._blendMode||0, dur);
    if(state.localArea) _localAnchorById[id] = {x: clamped.x, y: clamped.y};
  }

  const _Game_Screen_erasePicture = Game_Screen.prototype.erasePicture;
  Game_Screen.prototype.erasePicture = function(id){
    _Game_Screen_erasePicture.call(this,id);
    delete _localAnchorById[id];
    delete _pkmtContentSizeById[id];
  };

  const _Game_Screen_update = Game_Screen.prototype.update;
  Game_Screen.prototype.update = function(){
    _Game_Screen_update.call(this);
    if(!state.enabled) return;
    const busy = isMsgBusy();
    if(state.duringMode === "onlyMessage" && !busy) return;
    if(state.duringMode === "noMessage" && busy) return;
    const v = currentKeyVector();
    if(v.x===0&&v.y===0) return;
    const dx=v.x*state.step, dy=v.y*state.step;
    for(const id of state.targetIds){ movePicBy(id,dx,dy,state.duration); }
  };

  const _Game_Picture_update = Game_Picture.prototype.update;
  Game_Picture.prototype.update = function(){
    _Game_Picture_update.call(this);
    const id = this._pictureId;
    if(state.localArea && id && this._x!=null && this._y!=null){
      _localAnchorById[id] = {x: this._x, y: this._y};
    }
  };

  PluginManager.registerCommand(PN, "setKeyMode", args=>{
    const mode = String(args.mode||"both").toLowerCase();
    const ok = (mode==="arrow" || mode==="wasd" || mode==="both");
    state.keyMode = ok ? mode : "both";
  });
  PluginManager.registerCommand(PN, "setTargetIds", args=>{
    const ids = String(args.ids||"1").split(",").map(s=>Number(s.trim())).filter(n=>n>0);
    if(ids.length>0) state.targetIds = ids;
  });
  PluginManager.registerCommand(PN, "setNoClampOrigin", args=>{
    const o = String(args.origin||"center").toLowerCase();
    state.noClampOrigin = (o==="lefttop")?"leftTop":"center";
  });
  PluginManager.registerCommand(PN, "setDuringMode", args=>{
    const m = String(args.mode||"onlyMessage").toLowerCase();
    if(m==="always"||m==="onlymessage"||m==="nomessage"){
      state.duringMode = (m==="onlymessage")?"onlyMessage":(m==="nomessage")?"noMessage":"always";
    }
  });
  PluginManager.registerCommand(PN, "setLocalMoveArea", args=>{
    const sz = Math.max(1, Number(args.size||200)|0);
    state.localArea = sz;
    state.clamp = false;
    for(const id of state.targetIds){
      const pic = $gameScreen.picture(id);
      if (pic) _localAnchorById[id] = {x: pic._x||0, y: pic._y||0};
    }
  });
  PluginManager.registerCommand(PN, "setContentSize", args=>{
    const id = Math.max(1, Number(args.id||0)|0);
    const w  = Math.max(1, Number(args.width||0)|0);
    const h  = Math.max(1, Number(args.height||0)|0);
    _pkmtContentSizeById[id] = {w,h};
    const pic = $gameScreen.picture(id);
    if(pic){ computeBoundsForPicture(pic); }
  });
  PluginManager.registerCommand(PN, "disableLocalMoveArea", ()=>{
    state.localArea = null;
    for(const id of Object.keys(_localAnchorById)) delete _localAnchorById[id|0];
  });
  PluginManager.registerCommand(PN, "clearContentSize", args=>{
    const id = Math.max(1, Number(args.id||0)|0);
    delete _pkmtContentSizeById[id];
    const pic = $gameScreen.picture(id);
    if(pic){ pic._pkmtBounds = null; }
  });
  PluginManager.registerCommand(PN, "setClamp", args=>{
    state.clamp = String(args.value||"on").toLowerCase()==="on";
  });
  PluginManager.registerCommand(PN, "enable", ()=> state.enabled=true);
  PluginManager.registerCommand(PN, "disable", ()=> state.enabled=false);
})();