/*:
 * @target MZ
 * @plugindesc 【軽量＆厳密】ピクチャ移動（十字キー／px指定）＋「画像外は見せない」クランプ：原点/倍率対応・一発設定(画面サイズ含む)
 * @author ChatGPT
 *
 * @help
 * ■ポイント
 * - 表示/移動の度に、原点(左上/中央)・画像実寸・表示倍率・画面サイズから
 *   可動範囲[minX,maxX,minY,maxY]を前計算してピクチャへ保持。
 * - キー移動/コマンド移動は範囲外へは**そもそもセットしない**。
 * - Game_Picture.updateMove をフックし、補間中も _x/_y を**直接クランプ**（高速）。
 *   → 毎フレームの再movePictureは行いません（重さ改善）。
 * - ズーム機能はありません。
 *
 * ▼最短
 *  1) ピクチャの表示（例：ID=1）
 *  2) プラグインコマンド quickSetupAll
 *     id=1, step=16, dur=6, clamp=ON, allowMsg=ON, diagonal=OFF,
 *     screenW=1180, screenH=720
 *  3) 矢印で移動／moveHorizPx・movePixelsでpx移動（外は絶対見えない）
 *
 * @param pictureId @text 既定ピクチャ番号 @type number @min 1 @default 1
 * @param stepPixels @text ステップ距離(px) @type number @min 1 @default 16
 * @param duration @text フレーム数（補間） @type number @min 0 @default 6
 * @param clampToScreen @text 画面端で止める @type boolean @on する @off しない @default true
 * @param allowDuringMessage @text メッセージ中も移動可 @type boolean @on 許可 @off 不可 @default true
 * @param allowDiagonal @text 斜め移動を許可 @type boolean @on 許可 @off 禁止 @default false
 * @param initialDelay @text キー初回遅延(f) @type number @min 0 @default 18
 * @param repeatInterval @text 連打間隔(f) @type number @min 1 @default 4
 *
 * @param clampScreenWidth
 * @text クランプ計算用 画面幅
 * @type number
 * @min 0
 * @default 0
 * @desc 0=自動(Graphics.width)。独自解像度なら幅を指定（例：1180）。
 *
 * @param clampScreenHeight
 * @text クランプ計算用 画面高
 * @type number
 * @min 0
 * @default 0
 * @desc 0=自動(Graphics.height)。独自解像度なら高さを指定（例：720）。
 *
 * @command quickSetupAll
 * @text まとめて有効化（画面サイズも一度で）
 * @arg id @text ピクチャID @type number @min 1 @default 1
 * @arg step @text ステップ距離(px) @type number @min 1 @default 16
 * @arg dur @text フレーム数 @type number @min 0 @default 6
 * @arg clamp @text 画面端で止める @type boolean @default true
 * @arg allowMsg @text メッセ中も移動可 @type boolean @default true
 * @arg diagonal @text 斜め移動を許可 @type boolean @default false
 * @arg screenW @text 画面幅(0=自動) @type number @min 0 @default 0
 * @arg screenH @text 画面高(0=自動) @type number @min 0 @default 0
 *
 * @command enable  @text 有効化
 * @command disable @text 無効化
 *
 * @command setPicture @text ピクチャ番号の設定
 * @arg id @type number @min 1 @default 1
 *
 * @command setStep @text ステップ/フレーム/クランプ設定
 * @arg step @type number @min 1 @default 16
 * @arg dur  @type number @min 0 @default 6
 * @arg clamp @type boolean @default true
 *
 * @command setAllowDuringMessage @text メッセージ中の移動ON/OFF
 * @arg allow @type boolean @default true
 *
 * @command setDiagonal @text 斜め移動ON/OFF
 * @arg allow @type boolean @default false
 *
 * @command setScreenSize
 * @text クランプ計算用 画面サイズの設定
 * @arg screenW @text 幅(0=自動) @type number @min 0 @default 0
 * @arg screenH @text 高(0=自動) @type number @min 0 @default 0
 *
 * @command nudge @text 1ステップ移動（イベントから）
 * @arg dx @text Xステップ(右＋/左−) @type number @default 1
 * @arg dy @text Yステップ(下＋/上−) @type number @default 0
 *
 * @command moveHorizPx
 * @text ←→ 指定px移動（端で止まる）
 * @arg px @text 水平px（±） @type number @min -999999 @default 200
 * @arg dur @text フレーム数 @type number @min 0 @default 12
 * @arg clamp @text 画面端で止める @type boolean @default true
 *
 * @command movePixels
 * @text 任意px移動（端で止まる）
 * @arg px @text Xpx（右＋/左−） @type number @min -999999 @default 0
 * @arg py @text Ypx（下＋/上−） @type number @min -999999 @default 0
 * @arg dur @text フレーム数 @type number @min 0 @default 12
 * @arg clamp @text 画面端で止める @type boolean @default true
 
 * @command setContentSize
 * @text コンテンツ実寸(幅/高)の手動設定（Live2Dなど用）
 * @desc ビットマップが無いピクチャのクランプ計算に用いる元サイズ。
 * @arg id @text ピクチャID @type number @min 1 @default 1
 * @arg width  @text 幅(px)  @type number @min 1 @default 1920
 * @arg height @text 高(px)  @type number @min 1 @default 1080
*/

(() => {
  const PN = "PictureKeyMoveTest";
  const P  = PluginManager.parameters(PN);

  const cfg = {
    pictureId: Number(P.pictureId || 1),
    step: Number(P.stepPixels || 16),
    duration: Number(P.duration || 6),
    clamp: String(P.clampToScreen) === "true",
    allowDuringMsg: String(P.allowDuringMessage) === "true",
    allowDiagonal: String(P.allowDiagonal) === "true",
    initialDelay: Number(P.initialDelay || 18),
    repeatInterval: Number(P.repeatInterval || 4),
    clampScreenW: Number(P.clampScreenWidth || 0),
    clampScreenH: Number(P.clampScreenHeight || 0),
  };

  const state = {
    enabled: false,
    hold: { left:0, right:0, up:0, down:0 },
  };


// --- Added: manual content-size registry for non-bitmap pictures (e.g., Live2D) ---
const _pkmtContentSizeById = {}; // { [pictureId:number]: {w:number,h:number} }
function _pkmtGetOverrideSizeFor(pic){
  try{
    const id = pic?._pictureId ?? pic?._id;
    if(id && _pkmtContentSizeById[id]) return _pkmtContentSizeById[id];
  }catch(_){}
  return null;
}


  const clampNum = (v,a,b)=>Math.max(a,Math.min(b,v));
  function SCR_W(){ return cfg.clampScreenW>0 ? cfg.clampScreenW : (Graphics.width  || Graphics.boxWidth  || 0); }
  function SCR_H(){ return cfg.clampScreenH>0 ? cfg.clampScreenH : (Graphics.height || Graphics.boxHeight || 0); }

  // --------- Picture へ範囲情報を持たせる ----------
  
function computeBoundsForPicture(pic){
    const name = pic._name||"";
    const id   = pic?._pictureId ?? pic?._id;
    // Base size candidates:
    let baseW = 0, baseH = 0;

    // 1) Try normal bitmap (pictures/ folder)
    if(name){
      const bmp = ImageManager.loadPicture(name);
      if(bmp && bmp.isReady()){
        baseW = bmp.width||0;
        baseH = bmp.height||0;
      }
    }

    // 2) Fallback: user-provided override (for Live2D etc.)
    if((!baseW || !baseH) && id){
      const ov = _pkmtGetOverrideSizeFor(pic);
      if(ov){ baseW = ov.w|0; baseH = ov.h|0; }
    }

    if(!baseW || !baseH){
      // No size yet (e.g., bitmap not loaded & no override) → retry later
      pic._pkmtBounds = null;
      return;
    }

    const scrW = SCR_W(), scrH = SCR_H();
    const origin = pic._origin||0;
    const sx = ((pic._scaleX!=null?pic._scaleX:100)/100)||1;
    const sy = ((pic._scaleY!=null?pic._scaleY:100)/100)||1;
    const w = baseW * sx, h = baseH * sy;

    let minX, maxX, minY, maxY;
    if(origin===0){
      // 左上原点：小さい軸は固定(0)。大きい軸は [scr - img, 0]
      minX = (w<=scrW) ? 0 : Math.floor(scrW - w);
      maxX = 0;
      minY = (h<=scrH) ? 0 : Math.floor(scrH - h);
      maxY = 0;
    }else{
      // 中央原点：小さい軸は中央固定。大きい軸は [半分, 画面-半分]
      const hw=w/2, hh=h/2;
      minX = (w<=scrW) ? Math.round(scrW/2) : Math.ceil(hw);
      maxX = (w<=scrW) ? Math.round(scrW/2) : Math.floor(scrW - hw);
      minY = (h<=scrH) ? Math.round(scrH/2) : Math.ceil(hh);
      maxY = (h<=scrH) ? Math.round(scrH/2) : Math.floor(scrH - hh);
    }
    pic._pkmtBounds = {minX,maxX,minY,maxY};
}

    const bmp = ImageManager.loadPicture(name);
    if(!bmp || !bmp.isReady()){ // 読み込み待ちして後で再試行
      pic._pkmtBounds = null;
      return;
    }
    const scrW = SCR_W(), scrH = SCR_H();
    const origin = pic._origin||0;
    const sx = ((pic._scaleX!=null?pic._scaleX:100)/100)||1;
    const sy = ((pic._scaleY!=null?pic._scaleY:100)/100)||1;
    const w = (bmp.width||0)*sx, h=(bmp.height||0)*sy;

    let minX, maxX, minY, maxY;
    if(origin===0){
      // 左上原点：小さい軸は固定(0)。大きい軸は [scr - img, 0]
      minX = (w<=scrW) ? 0 : Math.floor(scrW - w);
      maxX = 0;
      minY = (h<=scrH) ? 0 : Math.floor(scrH - h);
      maxY = 0;
    }else{
      // 中央原点：小さい軸は中央固定。大きい軸は [半分, 画面-半分]
      const hw=w/2, hh=h/2;
      minX = (w<=scrW) ? Math.round(scrW/2) : Math.ceil(hw);
      maxX = (w<=scrW) ? Math.round(scrW/2) : Math.floor(scrW - hw); // ←★修正：min=hw, max=scrW-hw
      minY = (h<=scrH) ? Math.round(scrH/2) : Math.ceil(hh);
      maxY = (h<=scrH) ? Math.round(scrH/2) : Math.floor(scrH - hh);
    }
    pic._pkmtBounds = {minX,maxX,minY,maxY};
  }

  function ensureBounds(pic){
    if(!pic) return;
    if(!pic._pkmtBounds) computeBoundsForPicture(pic);
  }

  function clampByPic(pic, tx, ty){
    ensureBounds(pic);
    const b = pic._pkmtBounds;
    if(!b) return {x:tx,y:ty};
    return {
      x: Math.round(clampNum(tx, b.minX, b.maxX)),
      y: Math.round(clampNum(ty, b.minY, b.maxY)),
    };
  }

  // --------- 安全 movePicture（ターゲットを事前クランプ） ----------
  function moveClamped(id, origin, x, y, sx, sy, op, bm, dur){
    const pic = $gameScreen.picture(id);
    if(pic && cfg.clamp){
      const c = clampByPic(pic, x, y);
      x=c.x; y=c.y;
    }
    $gameScreen.movePicture(id, origin, x, y, sx, sy, op, bm, dur);
  }

  // --------- 相対移動(px) ----------
  function moveByPixels(dx, dy, duration){
    const id = cfg.pictureId;
    const pic = $gameScreen.picture(id);
    if(!pic) return;
    const origin = pic._origin||0;
    const x = (pic.x?pic.x():pic._x)||0;
    const y = (pic.y?pic.y():pic._y)||0;
    const sx=(pic.scaleX?pic.scaleX():(pic._scaleX!=null?pic._scaleX:100))||100;
    const sy=(pic.scaleY?pic.scaleY():(pic._scaleY!=null?pic._scaleY:100))||100;
    const op=(pic.opacity?pic.opacity():(pic._opacity!=null?pic._opacity:255))||255;
    let bm=(pic.blendMode?pic.blendMode():(pic._blendMode!=null?pic._blendMode:0))|0; if(bm<0||bm>3) bm=0;

    const c = cfg.clamp ? clampByPic(pic, x+dx, y+dy) : {x:x+dx,y:y+dy};
    moveClamped(id, origin, c.x, c.y, sx, sy, op, bm, Math.max(0,(duration|0)));
  }

  // --------- 入力更新（斜め禁止＋リピート） ----------
  function updateKeyMove(){
    if(!state.enabled) return;
    if(!cfg.allowDuringMsg && $gameMessage?.isBusy?.()) return;

    const pressed = {
      left:Input.isPressed("left"), right:Input.isPressed("right"),
      up:Input.isPressed("up"),     down:Input.isPressed("down")
    };

    if(!cfg.allowDiagonal){
      if(pressed.left && pressed.right){ pressed.left=false; pressed.right=false; }
      if(pressed.up && pressed.down){ pressed.up=false; pressed.down=false; }
      const horiz = pressed.left || pressed.right;
      const vert  = pressed.up   || pressed.down;
      if(horiz && vert){
        // 先に押した側を優先
        const holdH = Math.max(state.hold.left,state.hold.right);
        const holdV = Math.max(state.hold.up,state.hold.down);
        if(holdH>=holdV){ pressed.up=false; pressed.down=false; }
        else { pressed.left=false; pressed.right=false; }
      }
    }

    const stepOnce=(k,dx,dy)=>{
      if(pressed[k]){
        state.hold[k]++;
        const t=state.hold[k];
        if(t===1) moveByPixels(dx,dy,cfg.duration);
        else if(t>cfg.initialDelay && ((t-cfg.initialDelay)%cfg.repeatInterval===0))
          moveByPixels(dx,dy,cfg.duration);
      } else state.hold[k]=0;
    };

    stepOnce("left", -cfg.step, 0);
    stepOnce("right", cfg.step, 0);
    stepOnce("up", 0, -cfg.step);
    stepOnce("down", 0, cfg.step);
  }

  // --------- Scene_Map 更新フック（軽量：入力だけ） ----------
  const _SceneMap_update=Scene_Map.prototype.update;
  Scene_Map.prototype.update=function(){
    updateKeyMove();
    _SceneMap_update.call(this);
  };

  // --------- Game_Screen フック：表示/移動/消去 ----------
  const _show = Game_Screen.prototype.showPicture;
  Game_Screen.prototype.showPicture = function(id, name, origin, x, y, sx, sy, op, bm){
    _show.call(this,id,name,origin,x, y, sx, sy, op, bm);
    const pic = this.picture(id);
    if(pic){
      computeBoundsForPicture(pic);
      // 初期表示が外れていたら、その場で補正
      const c = cfg.clamp ? clampByPic(pic, pic._x, pic._y) : {x:pic._x,y:pic._y};
      pic._x = pic._targetX = c.x;
      pic._y = pic._targetY = c.y;
    }
  };

  const _move = Game_Screen.prototype.movePicture;
  Game_Screen.prototype.movePicture = function(id, origin, x, y, sx, sy, op, bm, dur){
    const pic = this.picture(id);
    if(pic){
      // 先にターゲットをクランプしてから親を呼ぶ
      if(cfg.clamp){
        computeBoundsForPicture(pic); // 倍率/原点が変わる可能性に備える
        const c = clampByPic(pic, x, y);
        x=c.x; y=c.y;
      }
    }
    _move.call(this,id,origin,x,y,sx,sy,op,bm,dur);
    // 移動後も最新の範囲を保持（倍率・原点変更に追従）
    const p2 = this.picture(id);
    if(p2) computeBoundsForPicture(p2);
  };

  const _erase = Game_Screen.prototype.erasePicture;
  Game_Screen.prototype.erasePicture = function(id){
    _erase.call(this,id);
    const pic = this.picture(id);
    if(pic) pic._pkmtBounds = null;
  };

  // --------- Game_Picture.updateMove のクランプ（補間中も外を見せない） ----------
  const _picUpdateMove = Game_Picture.prototype.updateMove;
  Game_Picture.prototype.updateMove = function(){
    _picUpdateMove.call(this);
    if(!cfg.clamp) return;
    if(!this._name) return;
    computeBoundsForPicture(this); // 遅延読込にも対応
    const b = this._pkmtBounds; if(!b) return;
    // 現在座標＆ターゲットの両方をクランプ
    const cx = clampNum(this._x, b.minX, b.maxX);
    const cy = clampNum(this._y, b.minY, b.maxY);
    if(this._x!==cx) this._x = cx;
    if(this._y!==cy) this._y = cy;
    const tx = clampNum(this._targetX ?? this._x, b.minX, b.maxX);
    const ty = clampNum(this._targetY ?? this._y, b.minY, b.maxY);
    if(this._targetX!==tx) this._targetX = tx;
    if(this._targetY!==ty) this._targetY = ty;
  };

  // --------- プラグインコマンド ----------
  PluginManager.registerCommand(PN,"quickSetupAll",args=>{
    state.enabled = true;
    cfg.pictureId      = Math.max(1, Number(args.id      ?? cfg.pictureId) | 0);
    cfg.step           = Math.max(1, Number(args.step    ?? cfg.step) | 0);
    cfg.duration       = Math.max(0, Number(args.dur     ?? cfg.duration) | 0);
    cfg.clamp          = String(args.clamp)==="true";
    cfg.allowDuringMsg = String(args.allowMsg)==="true";
    cfg.allowDiagonal  = String(args.diagonal)==="true";
    cfg.clampScreenW   = Math.max(0, Number(args.screenW || cfg.clampScreenW) | 0);
    cfg.clampScreenH   = Math.max(0, Number(args.screenH || cfg.clampScreenH) | 0);

    const pic = $gameScreen.picture(cfg.pictureId);
    if(pic){
      computeBoundsForPicture(pic);
      // その場でフィット
      const c = cfg.clamp ? clampByPic(pic, pic._x, pic._y) : {x:pic._x,y:pic._y};
      pic._x = pic._targetX = c.x;
      pic._y = pic._targetY = c.y;
    }
  });

  PluginManager.registerCommand(PN,"enable", ()=>{ state.enabled=true;  });
  PluginManager.registerCommand(PN,"disable",()=>{ state.enabled=false; });

  PluginManager.registerCommand(PN,"setPicture",args=>{
    cfg.pictureId = Math.max(1, Number(args.id||cfg.pictureId)|0);
    const pic = $gameScreen.picture(cfg.pictureId);
    if(pic){ computeBoundsForPicture(pic); }
  });

  PluginManager.registerCommand(PN,"setStep",args=>{
    cfg.step     = Math.max(1, Number(args.step||cfg.step)|0);
    cfg.duration = Math.max(0, Number(args.dur||cfg.duration)|0);
    cfg.clamp    = String(args.clamp)==="true";
  });

  PluginManager.registerCommand(PN,"setAllowDuringMessage",args=>{
    cfg.allowDuringMsg = String(args.allow)==="true";
  });

  PluginManager.registerCommand(PN,"setDiagonal",args=>{
    cfg.allowDiagonal = String(args.allow)==="true";
  });

  PluginManager.registerCommand(PN,"setScreenSize",args=>{
    cfg.clampScreenW = Math.max(0, Number(args.screenW || cfg.clampScreenW) | 0);
    cfg.clampScreenH = Math.max(0, Number(args.screenH || cfg.clampScreenH) | 0);
    const pic = $gameScreen.picture(cfg.pictureId);
    if(pic){ computeBoundsForPicture(pic); }
  });

  PluginManager.registerCommand(PN,"nudge",args=>{
    const dx=(Number(args.dx||0)|0)*cfg.step;
    const dy=(Number(args.dy||0)|0)*cfg.step;
    moveByPixels(dx,dy,cfg.duration);
  });

  PluginManager.registerCommand(PN,"moveHorizPx",args=>{
    const px  = Number(args.px||0)|0;
    const dur = Math.max(0, Number(args.dur||cfg.duration)|0);
    moveByPixels(px, 0, dur);
  });

  PluginManager.registerCommand(PN,"movePixels",args=>{
    const px  = Number(args.px||0)|0;
    const py  = Number(args.py||0)|0;
    const dur = Math.max(0, Number(args.dur||cfg.duration)|0);
    moveByPixels(px, py, dur);
  
  PluginManager.registerCommand(PN,"setContentSize",args=>{
    const id = Math.max(1, Number(args.id||0)|0);
    const w  = Math.max(1, Number(args.width||0)|0);
    const h  = Math.max(1, Number(args.height||0)|0);
    _pkmtContentSizeById[id] = {w,h};
    const pic = $gameScreen.picture(id);
    if(pic){ computeBoundsForPicture(pic); }
  });
});
})();
