/*:
 * @target MZ
 * @plugindesc ピクチャ相対/絶対スムーズ移動（フレーム指定・画面端クランプ・安全ガード付）
 * @author ChatGPT
 * @help PictureShiftPlus.js (safe r2)
 *
 * ■使い方は以前と同じ（相対/絶対移動・フレーム指定・画面端クランプ）
 * ■今回の修正点
 *  - blendMode/scale/opacity を数値ガードし、0-3 にクランプ
 *  - ログ出力で原因追跡をしやすく
 *  - 画像未読込時の待機ロジックを堅牢化
 *
 * @command slideBy
 * @text 相対スライド移動
 * @arg PictureId @type number @min 1 @default 1 @text ピクチャ番号
 * @arg ShiftX    @type number @default 0 @text 移動X（右＋/左−）
 * @arg ShiftY    @type number @default 0 @text 移動Y（下＋/上−）
 * @arg Duration  @type number @min 0 @default 30 @text フレーム数
 * @arg ClampToScreen @type boolean @on クランプ @off しない @default true @text 画面端で止める
 *
 * @command moveTo
 * @text 絶対座標へ移動
 * @arg PictureId2 @type number @min 1 @default 1 @text ピクチャ番号
 * @arg TargetX    @type number @default 0 @text 目標X座標
 * @arg TargetY    @type number @default 0 @text 目標Y座標
 * @arg Duration2  @type number @min 0 @default 30 @text フレーム数
 * @arg ClampToScreen2 @type boolean @on クランプ @off しない @default true @text 画面端で止める
 */

(() => {
  const PLUGIN_NAME = "PictureShiftPlus";

  // ========== ユーティリティ ==========
  const toNum = (v, def = 0) => {
    const n = Number(v);
    return Number.isFinite(n) ? n : def;
  };
  const clamp = (v, min, max) => Math.max(min, Math.min(max, v));

  function safeBlendMode(picture) {
    // Game_Picture は _blendMode を持つ（0:normal,1:add,2:multiply,3:screen）
    let bm = picture && typeof picture.blendMode === "function"
      ? picture.blendMode()
      : picture && typeof picture._blendMode !== "undefined"
      ? picture._blendMode
      : 0;
    bm = toNum(bm, 0);
    return clamp(bm, 0, 3);
  }
  function safeScaleX(picture) {
    const sx = picture && typeof picture.scaleX === "function" ? picture.scaleX() : picture?._scaleX;
    return clamp(toNum(sx, 100), -10000, 10000);
  }
  function safeScaleY(picture) {
    const sy = picture && typeof picture.scaleY === "function" ? picture.scaleY() : picture?._scaleY;
    return clamp(toNum(sy, 100), -10000, 10000);
  }
  function safeOpacity(picture) {
    const op = picture && typeof picture.opacity === "function" ? picture.opacity() : picture?._opacity;
    return clamp(toNum(op, 255), 0, 255);
  }

  // ========== ペンディング移動要求 ==========
  const _pendingMoves = []; // {type:"by"|"to", picId, dx, dy, tx, ty, duration, clamp}

  function requestMoveBy(picId, dx, dy, duration, clampToScreen) {
    _pendingMoves.push({ type: "by", picId, dx, dy, duration, clamp: clampToScreen });
  }
  function requestMoveTo(picId, tx, ty, duration, clampToScreen) {
    _pendingMoves.push({ type: "to", picId, tx, ty, duration, clamp: clampToScreen });
  }

  function processPendingMoves() {
    if (_pendingMoves.length === 0) return;

    for (let i = _pendingMoves.length - 1; i >= 0; i--) {
      const req = _pendingMoves[i];
      const picture = $gameScreen.picture(toNum(req.picId));
      if (!picture) {
        console.warn(`[${PLUGIN_NAME}] picture #${req.picId} が存在しません。要求を破棄します。`, req);
        _pendingMoves.splice(i, 1);
        continue;
      }

      // 画像読み込み待機：picture._name が空なら待機（非表示/消去直後など）
      const name = picture._name;
      if (!name) {
        // クランプ不要なら即実行してもよいが、ここでは安全のため1フレーム待機
        continue;
      }

      const bitmap = ImageManager.loadPicture(name);
      if (!bitmap || !bitmap.isReady()) {
        // 読み込み完了を待つ
        continue;
      }

      executeMove(req, picture, bitmap);
      _pendingMoves.splice(i, 1);
    }
  }

  // Map/Battle で毎フレーム試行
  const _SceneMap_update = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function () {
    processPendingMoves();
    _SceneMap_update.call(this);
  };
  const _SceneBattle_update = Scene_Battle.prototype.update;
  Scene_Battle.prototype.update = function () {
    processPendingMoves();
    _SceneBattle_update.call(this);
  };

  // ========== クランプ ==========
  function clampTargetXY(picture, bitmap, targetX, targetY) {
    if (!bitmap) return { x: targetX, y: targetY };

    const scrW = Graphics.width;
    const scrH = Graphics.height;

    const origin = picture._origin; // 0:左上 1:中央
    const scaleX = safeScaleX(picture) / 100;
    const scaleY = safeScaleY(picture) / 100;

    const imgW = (bitmap.width || 0) * scaleX;
    const imgH = (bitmap.height || 0) * scaleY;

    let minX, maxX, minY, maxY;
    if (origin === 0) {
      minX = 0;
      minY = 0;
      maxX = Math.max(0, scrW - imgW);
      maxY = Math.max(0, scrH - imgH);
    } else {
      const halfW = imgW / 2;
      const halfH = imgH / 2;
      minX = halfW;
      maxX = Math.max(halfW, scrW - halfW);
      minY = halfH;
      maxY = Math.max(halfH, scrH - halfH);
    }

    return {
      x: clamp(toNum(targetX), minX, maxX),
      y: clamp(toNum(targetY), minY, maxY),
    };
  }

  // ========== 実行 ==========
  function executeMove(req, picture, bitmap) {
    const duration = Math.max(0, toNum(req.duration, 0));

    let targetX, targetY;
    if (req.type === "by") {
      targetX = toNum(picture.x()) + toNum(req.dx);
      targetY = toNum(picture.y()) + toNum(req.dy);
    } else {
      targetX = toNum(req.tx);
      targetY = toNum(req.ty);
    }

    if (req.clamp) {
      const c = clampTargetXY(picture, bitmap, targetX, targetY);
      targetX = c.x;
      targetY = c.y;
    }

    const sx = safeScaleX(picture);
    const sy = safeScaleY(picture);
    const op = safeOpacity(picture);
    const bm = safeBlendMode(picture);

    // デバッグログ
    // console.log(`[${PLUGIN_NAME}] move picture #${req.picId} -> (${targetX},${targetY}) d=${duration} clamp=${!!req.clamp} [scale=${sx}/${sy}, op=${op}, bm=${bm}]`, req);

    picture.move(targetX, targetY, sx, sy, op, bm, duration);
  }

  // ========== プラグインコマンド ==========
  PluginManager.registerCommand(PLUGIN_NAME, "slideBy", args => {
    const picId = toNum(args.PictureId);
    const dx = toNum(args.ShiftX);
    const dy = toNum(args.ShiftY);
    const duration = toNum(args.Duration, 30);
    const clampToScreen = String(args.ClampToScreen) === "true";
    requestMoveBy(picId, dx, dy, duration, clampToScreen);
  });

  PluginManager.registerCommand(PLUGIN_NAME, "moveTo", args => {
    const picId = toNum(args.PictureId2);
    const tx = toNum(args.TargetX);
    const ty = toNum(args.TargetY);
    const duration = toNum(args.Duration2, 30);
    const clampToScreen = String(args.ClampToScreen2) === "true";
    requestMoveTo(picId, tx, ty, duration, clampToScreen);
  });
})();
