/*:
 * @target MZ
 * @plugindesc SNS風チャット表示プラグイン - コメントを表示し、背景や音、表示層や間隔を制御できます。
 *
 * @help
 * このプラグインは、YouTube風のチャットコメントを画面に表示する機能を提供します。
 * コメントにはユーザー名、アバター、寄付金、背景色、サウンドなどを設定でき、
 * コメントの流れる挙動をオンオフできるほか、一定数以上のコメントがある場合には
 * 古いものを押し出して上に詰める動作が可能です。
 *
 * ◆主な機能：
 * - コメントの表示速度調整
 * - 最大表示数の設定
 * - メッセージ同士の間隔（ピクセル指定）
 * - 背景画像、上層画像の表示と位置指定
 * - フォントサイズ、色、寄付額表示などのカスタマイズ
 * - 表示スイッチとコメントフロー制御スイッチ（オンオフ切替）
 * - 寄付額に応じたサウンド再生
 *
 * ◆新規パラメータ：
 * @param MessageFlowSwitch
 * @type switch
 * @desc ONでメッセージが上に流れます。OFFで固定されます。
 *
 * @param MessageAutoEraseSwitch
 * @type switch
 * @desc ONで指定時間経過後にメッセージが消えます。OFFで残り続けます。
 *
 * @param ChatSpeed
 * @type number
 * @min 1
 * @max 10
 * @default 5
 * @desc コメントが上に流れる速度（フレームごと）。
 *
 * @param MaxMessages
 * @type number
 * @default 10
 * @desc 同時に表示できる最大メッセージ数。
 *
 * @param MessageSpacing
 * @type number
 * @default 4
 * @desc 各メッセージの間隔（ピクセル単位、狭めたい場合は小さい値に設定）。
 *
 * @param BackgroundImages
 * @type struct<BackgroundImage>[]
 * @desc 背景画像の設定（ファイル名・位置・透明度など）。
 *
 * @param ShowBackgroundImages
 * @type boolean
 * @default true
 * @desc 背景画像を表示するかどうか。
 *
 * @param WindowWidth
 * @type number
 * @default 300
 * @desc チャットウィンドウの幅（ピクセル単位）。
 *
 * @param WindowHeight
 * @type number
 * @default 720
 * @desc チャットウィンドウの高さ（ピクセル単位）。
 *
 * @param WindowPositionX
 * @type number
 * @default 0
 * @desc チャットウィンドウのX座標。
 *
 * @param WindowPositionY
 * @type number
 * @default 0
 * @desc チャットウィンドウのY座標。
 *
 * @param DefaultFontSize
 * @type number
 * @default 20
 * @desc メッセージのフォントサイズ（デフォルト値）。
 *
 * @param IconHeight
 * @type number
 * @default 32
 * @desc アバター画像の高さ（ピクセル単位）。
 *
 * @param IconOffsetX
 * @type number
 * @default 10
 * @desc アバター画像のXオフセット。
 *
 * @param IconOffsetY
 * @type number
 * @default 10
 * @desc アバター画像のYオフセット。
 *
 * @param CommentOffsetX
 * @type number
 * @default 50
 * @desc コメント本文のXオフセット。
 *
 * @param CommentOffsetY
 * @type number
 * @default 10
 * @desc コメント本文のYオフセット。
 *
 * @param CommentBandHeight
 * @type number
 * @default 42
 * @desc コメント表示エリアの高さ（背景色帯）。
 *
 * @param DonationCurrency
 * @type string
 * @default $
 * @desc 寄付金額に使う通貨記号。
 *
 * @param DisplaySwitch
 * @type switch
 * @desc チャット表示を制御するスイッチ番号。
 *
 * @param LayerPriority
 * @type select
 * @option ゲームウィンドウ
 * @value 1
 * @option ピクチャ
 * @value 2
 * @option チャットツール
 * @value 3
 * @option ゲームマップ
 * @value 4
 * @default 3
 * @desc チャットウィンドウの描画レイヤー優先度。
 *
 * @param TopImagePriority
 * @type number
 * @default 10
 * @desc 上層画像の表示優先度（値が大きいほど前面）。
 *
 * @param SoundEffectCategory1
 * @type file
 * @dir audio/se/
 * @default AudioFileName1
 * @desc カテゴリ1用の効果音ファイル名。
 *
 * @param SoundEffectCategory2
 * @type file
 * @dir audio/se/
 * @default AudioFileName2
 * @desc カテゴリ2用の効果音ファイル名。
 *
 * @param SoundEffectCategory3
 * @type file
 * @dir audio/se/
 * @default AudioFileName3
 * @desc カテゴリ3用の効果音ファイル名。
 *
 * @param SoundEffectCategory4
 * @type file
 * @dir audio/se/
 * @default AudioFileName4
 * @desc カテゴリ4用の効果音ファイル名。
 *
 * @param TopImages
 * @type struct<TopImage>[]
 * @desc チャットレイヤー最前面に表示する上層画像の配列。
 *
 * @command AddMessage
 * @text チャットメッセージ追加
 * @desc 新しいチャットメッセージを追加します。
 *
 * @arg username
 * @type string
 * @default User
 * @desc コメントしたユーザー名。
 *
 * @arg text
 * @type string
 * @default Hello!
 * @desc メッセージ本文。
 *
 * @arg avatar
 * @type file
 * @dir img/pictures/
 * @desc ユーザーアバター画像（任意）。
 *
 * @arg color
 * @type color
 * @default #ffffff
 * @desc メッセージ文字色。
 *
 * @arg fontSize
 * @type number
 * @default 20
 * @desc メッセージ文字サイズ。
 *
 * @arg donation
 * @type number
 * @default 0
 * @desc 寄付金額（0なら非表示）。
 *
 * @arg bgColor
 * @type color
 * @default transparent
 * @desc コメント背景色。透明にするには 'transparent'。
 *
 * @arg soundCategory
 * @type select
 * @option カテゴリ1
 * @value 1
 * @option カテゴリ2
 * @value 2
 * @option カテゴリ3
 * @value 3
 * @option カテゴリ4
 * @value 4
 * @default 1
 * @desc 寄付サウンドカテゴリ。
 *
 * @arg displayTime
 * @type number
 * @default 300
 * @desc 表示持続時間（フレーム数）。
 * @param FadeOutDuration
 * @type number
 * @default 60
 * @desc フェードアウトする際の持続フレーム数（1秒 = 60）

 * @param DonationTotalVariable
 * @type variable
 * @desc 寄付金の合計を加算していく変数番号
  */

/* SNS風チャット表示プラグイン - コメント表示と寄付金加算機能付き */
(() => {
  const parameters = PluginManager.parameters('SNSChatPlugin');
  const chatSpeed = Number(parameters['ChatSpeed'] || 5);
  const maxMessages = Number(parameters['MaxMessages'] || 10);
  const messageGap = Number(parameters['MessageSpacing'] || 4);
  const backgroundImages = JSON.parse(parameters['BackgroundImages'] || '[]');
  const showBackgroundImages = parameters['ShowBackgroundImages'] === 'true';
  const windowWidth = Number(parameters['WindowWidth'] || 420);
  const windowHeight = Number(parameters['WindowHeight'] || 720);
  const windowPositionX = Number(parameters['WindowPositionX'] || 0);
  const windowPositionY = Number(parameters['WindowPositionY'] || 0);
  const defaultFontSize = Number(parameters['DefaultFontSize'] || 20);
  const iconHeight = Number(parameters['IconHeight'] || 32);
  const iconOffsetX = Number(parameters['IconOffsetX'] || 10);
  const iconOffsetY = Number(parameters['IconOffsetY'] || 10);
  const commentOffsetX = Number(parameters['CommentOffsetX'] || 50);
  const commentOffsetY = Number(parameters['CommentOffsetY'] || 10);
  const commentBandHeight = Number(parameters['CommentBandHeight'] || 42);
  const donationCurrency = String(parameters['DonationCurrency'] || '$');
  const displaySwitch = Number(parameters['DisplaySwitch'] || 0);
  const messageFlowSwitch = Number(parameters['MessageFlowSwitch'] || 0);
  const messageAutoEraseSwitch = Number(parameters['MessageAutoEraseSwitch'] || 0);
  const layerPriority = Number(parameters['LayerPriority'] || 3);
  const topImagePriority = Number(parameters['TopImagePriority'] || 10);
  const donationTotalVariable = Number(parameters['DonationTotalVariable'] || 0);
  const soundEffects = [
    String(parameters['SoundEffectCategory1'] || 'AudioFileName1'),
    String(parameters['SoundEffectCategory2'] || 'AudioFileName2'),
    String(parameters['SoundEffectCategory3'] || 'AudioFileName3'),
    String(parameters['SoundEffectCategory4'] || 'AudioFileName4')
  ];
  const topImages = JSON.parse(parameters['TopImages'] || '[]');

  let chatMessages = [];
  let chatBitmap = null;
  let chatSprite = null;
  let backgroundSprites = [];
  let topImageSprites = [];

  function updateChatMessagePositions() {
    for (let i = 0; i < chatMessages.length; i++) {
      const yOffset = i * (commentBandHeight + messageGap);
      chatMessages[i].y = windowPositionY + yOffset;
    }
  }

  function addChatMessage({ username, text, avatar, color, fontSize, donation, bgColor, soundCategory, displayTime }) {
    if (!$gameSwitches.value(displaySwitch)) return;

    const donationAmount = donation || 0;
    const newMessage = {
      username,
      text,
      avatar,
      color: color || '#ffffff',
      fontSize: fontSize || defaultFontSize,
      donation: donationAmount,
      bgColor: bgColor || 'transparent',
      y: windowPositionY + windowHeight,
      duration: displayTime || 300,
    };

    if (chatMessages.length >= maxMessages) {
      chatMessages.shift();
    }

    chatMessages.push(newMessage);
    updateChatMessagePositions();

    const soundEffect = soundEffects[soundCategory - 1];
    if (soundEffect) {
      AudioManager.playSe({ name: soundEffect, pan: 0, pitch: 100, volume: 90 });
    }

    if (donationTotalVariable > 0 && donationAmount > 0) {
      const current = $gameVariables.value(donationTotalVariable);
      $gameVariables.setValue(donationTotalVariable, current + donationAmount);
    }

    drawChatMessages();
  }

  function drawChatMessages() {
    chatBitmap.clear();

    if (!$gameSwitches.value(displaySwitch)) {
      chatSprite.visible = false;
      backgroundSprites.forEach(sprite => (sprite.visible = false));
      topImageSprites.forEach(sprite => (sprite.visible = false));
      return;
    }

    chatSprite.visible = true;
    backgroundSprites.forEach(sprite => (sprite.visible = showBackgroundImages));
    topImageSprites.forEach(sprite => (sprite.visible = true));

    for (let i = 0; i < chatMessages.length; i++) {
      const message = chatMessages[i];
      const yOffset = i * (commentBandHeight + messageGap);

      if (message.bgColor !== 'transparent') {
        chatBitmap.fillRect(windowPositionX, message.y + yOffset, windowWidth, commentBandHeight, message.bgColor);
      }

      if (message.avatar) {
  const avatarImage = ImageManager.loadPicture(message.avatar);
  if (avatarImage.isReady()) {
    chatBitmap.blt(avatarImage, 0, 0, iconHeight, iconHeight, windowPositionX + iconOffsetX, message.y + iconOffsetY + yOffset, iconHeight, iconHeight);
  } else {
    // ロード未完了ならブランク画像（透明）で代用
    const blankBitmap = new Bitmap(iconHeight, iconHeight);
    chatBitmap.blt(blankBitmap, 0, 0, iconHeight, iconHeight, windowPositionX + iconOffsetX, message.y + iconOffsetY + yOffset, iconHeight, iconHeight);
  }
}

      chatBitmap.textColor = message.color || '#ffffff';
      chatBitmap.fontSize = message.fontSize || defaultFontSize;

      const displayDonation = message.donation > 0 ? ` ${donationCurrency}${message.donation}` : '';
      chatBitmap.drawText(
        message.username + displayDonation,
        windowPositionX + commentOffsetX,
        message.y + yOffset + commentOffsetY,
        windowWidth - iconHeight - 20,
        commentBandHeight / 2,
        'left'
      );

      chatBitmap.drawText(
        message.text,
        windowPositionX + commentOffsetX,
        message.y + yOffset + commentBandHeight / 2 + commentOffsetY,
        windowWidth - iconHeight - 20,
        commentBandHeight / 2,
        'left'
      );
    }
  }

  const _Scene_Map_start = Scene_Map.prototype.start;
  Scene_Map.prototype.start = function () {
    _Scene_Map_start.call(this);

    if (showBackgroundImages) {
      backgroundImages.forEach(bgImage => {
        const parsedImage = JSON.parse(bgImage);
        const filename = parsedImage.Filename;
        const opacity = Number(parsedImage.Opacity);
        const xPos = Number(parsedImage.XPosition) || 0;
        const yPos = Number(parsedImage.YPosition) || 0;
        const bgSprite = new Sprite(ImageManager.loadPicture(filename));
        bgSprite.opacity = opacity;
        bgSprite.x = windowPositionX + xPos;
        bgSprite.y = windowPositionY + yPos;
        this.addChildAt(bgSprite, layerPriority - 1);
        backgroundSprites.push(bgSprite);
      });
    }

    chatSprite = new Sprite();
    chatBitmap = new Bitmap(windowWidth, windowHeight);
    chatSprite.bitmap = chatBitmap;
    this.addChildAt(chatSprite, layerPriority);

    chatSprite.x = windowPositionX;
    chatSprite.y = windowPositionY;

    topImages.forEach(topImg => {
      const parsedTopImage = JSON.parse(topImg);
      const filename = parsedTopImage.Filename;
      const opacity = Number(parsedTopImage.Opacity);
      const xPos = Number(parsedTopImage.XPosition) || 0;
      const yPos = Number(parsedTopImage.YPosition) || 0;
      const topSprite = new Sprite(ImageManager.loadPicture(filename));
      topSprite.opacity = opacity;
      topSprite.x = windowPositionX + xPos;
      topSprite.y = windowPositionY + yPos;
      this.addChildAt(topSprite, layerPriority + 1);
      topImageSprites.push(topSprite);
    });

    if (!$gameSwitches.value(displaySwitch)) {
      chatSprite.visible = false;
      backgroundSprites.forEach(sprite => (sprite.visible = false));
      topImageSprites.forEach(sprite => (sprite.visible = false));
    }
  };

  const _Scene_Map_update = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function () {
    _Scene_Map_update.call(this);

    const visible = $gameSwitches.value(displaySwitch);
    chatSprite.visible = visible;
    backgroundSprites.forEach(sprite => (sprite.visible = visible && showBackgroundImages));
    topImageSprites.forEach(sprite => (sprite.visible = visible));

    if (!visible) {
      if (chatMessages.length > 0) {
        chatMessages = [];
        chatBitmap.clear();
      }
      return;
    }

    if (chatMessages.length > 0) {
      if ($gameSwitches.value(messageFlowSwitch)) {
        for (let i = 0; i < chatMessages.length; i++) {
          const message = chatMessages[i];
          if ($gameSwitches.value(messageAutoEraseSwitch)) {
            message.duration--;
            if (message.duration <= 0) {
              message.y -= chatSpeed;
            }
          }
        }
        if ($gameSwitches.value(messageAutoEraseSwitch)) {
          chatMessages = chatMessages.filter(message => message.y + commentBandHeight >= windowPositionY);
        }
      }
      drawChatMessages();
    }
  };

  PluginManager.registerCommand('SNSChatPlugin', 'AddMessage', args => {
    const username = args.username || '';
    const text = args.text || '';
    const avatar = args.avatar || '';
    const color = args.color || '#ffffff';
    const fontSize = Number(args.fontSize) || defaultFontSize;
    const donation = Number(args.donation) || 0;
    const bgColor = args.bgColor || 'transparent';
    const soundCategory = Number(args.soundCategory) || 1;
    const displayTime = Number(args.displayTime) || 300;

    if (!$gameSwitches.value(displaySwitch)) return;

    addChatMessage({ username, text, avatar, color, fontSize, donation, bgColor, soundCategory, displayTime });
  });
})();
