/*:
 * @target MZ
 * @plugindesc 【SNS風チャット表示プラグイン】コメント表示・背景画像・音・色・レイヤー優先度などを自由にカスタマイズ可能。
 * @author ChatGPT
 *
 * このプラグインは、YouTubeや配信風のチャットコメントをゲーム画面上に表示することができます。
 * 各コメントには、ユーザー名、メッセージ本文、アバター、寄付金額、背景色、文字色、効果音などを設定可能です。
 *
 * 主な機能：
 * - チャットメッセージを指定時間表示し、上にスクロールさせる
 * - コメントごとに文字色、フォントサイズ、背景色、アバター画像、効果音を設定
 * - コメントの表示最大数、表示スピード、間隔などの細かな設定
 * - バックグラウンド画像、最前面画像の表示機能
 * - 指定スイッチで表示のON/OFFを切り替え可能
 * - 新規：指定スイッチでコメントのフェードアウト有無を制御可能
 * - RPGツクールMZのカラーパレットで背景色・文字色を指定可能
 *
 * @param ChatSpeed
 * @type number
 * @min 1
 * @max 10
 * @default 5
 * @desc チャットメッセージが上に移動するスピード（大きいほど速い）
 *
 * @param MaxMessages
 * @type number
 * @default 10
 * @desc 同時に表示される最大メッセージ数
 *
 * @param MessageSpacing
 * @type number
 * @default 10
 * @desc メッセージ同士の縦間隔（ピクセル単位）
 *
 * @param BackgroundImages
 * @type struct<BackgroundImage>[]
 * @desc 背景画像の配列設定（画像ファイル、位置、透明度）
 *
 * @param ShowBackgroundImages
 * @type boolean
 * @default true
 * @desc 背景画像を表示するかどうか
 *
 * @param WindowWidth
 * @type number
 * @default 300
 * @desc チャットウィンドウの幅
 *
 * @param WindowHeight
 * @type number
 * @default 720
 * @desc チャットウィンドウの高さ
 *
 * @param WindowPositionX
 * @type number
 * @default 0
 * @desc チャットウィンドウのX座標
 *
 * @param WindowPositionY
 * @type number
 * @default 0
 * @desc チャットウィンドウのY座標
 *
 * @param DefaultFontSize
 * @type number
 * @default 20
 * @desc メッセージのデフォルトフォントサイズ
 *
 * @param IconHeight
 * @type number
 * @default 32
 * @desc アバターアイコンの高さ
 *
 * @param IconOffsetX
 * @type number
 * @default 10
 * @desc アバターのXオフセット
 *
 * @param IconOffsetY
 * @type number
 * @default 10
 * @desc アバターのYオフセット
 *
 * @param CommentOffsetX
 * @type number
 * @default 50
 * @desc コメントテキストのXオフセット
 *
 * @param CommentOffsetY
 * @type number
 * @default 10
 * @desc コメントテキストのYオフセット（1行ごと）
 *
 * @param CommentBandHeight
 * @type number
 * @default 42
 * @desc コメントの背景色バンドの高さ
 *
 * @param DonationCurrency
 * @type string
 * @default $
 * @desc 寄付金額表示の通貨記号
 *
 * @param DisplaySwitch
 * @type switch
 * @desc チャットウィンドウの表示/非表示を切り替えるスイッチ
 *
 * @param MessageFadeSwitch
 * @type switch
 * @desc OFFならチャットメッセージはフェードアウトしない（時間経過で消えない）
 *
 * @param LayerPriority
 * @type select
 * @option Game Window
 * @value 1
 * @option Picture
 * @value 2
 * @option Chat Tool
 * @value 3
 * @option Game Map
 * @value 4
 * @default 3
 * @desc チャットの表示レイヤーの優先順位
 *
 * @param TopImagePriority
 * @type number
 * @default 10
 * @desc 最前面画像の描画優先度（高いほど前面に表示）
 *
 * @param SoundEffectCategory1
 * @type file
 * @dir audio/se/
 * @default AudioFileName1
 * @desc 効果音カテゴリ1の音声ファイル
 *
 * @param SoundEffectCategory2
 * @type file
 * @dir audio/se/
 * @default AudioFileName2
 * @desc 効果音カテゴリ2の音声ファイル
 *
 * @param SoundEffectCategory3
 * @type file
 * @dir audio/se/
 * @default AudioFileName3
 * @desc 効果音カテゴリ3の音声ファイル
 *
 * @param SoundEffectCategory4
 * @type file
 * @dir audio/se/
 * @default AudioFileName4
 * @desc 効果音カテゴリ4の音声ファイル
 *
 * @param TopImages
 * @type struct<TopImage>[]
 * @desc チャットツールの上層に表示する画像の配列
 *
 * @command AddMessage
 * @text チャットメッセージを追加
 * @desc アバター、色、寄付、背景色、効果音などを指定してチャットメッセージを追加します。
 *
 * @arg username
 * @type string
 * @default User
 * @desc メッセージを送信するユーザー名
 *
 * @arg text
 * @type string
 * @default Hello!
 * @desc 表示されるメッセージの内容
 *
 * @arg avatar
 * @type file
 * @dir img/pictures/
 * @desc アバター画像ファイル（省略可）
 *
 * @arg color
 * @type color
 * @default #ffffff
 * @desc メッセージ文字の色
 *
 * @arg fontSize
 * @type number
 * @default 20
 * @desc メッセージのフォントサイズ
 *
 * @arg donation
 * @type number
 * @default 0
 * @desc ユーザーの寄付額（0の場合は非表示）
 *
 * @arg bgColor
 * @type color
 * @default transparent
 * @desc コメント背景色（例：#ffcc00、透明は transparent）
 *
 * @arg soundCategory
 * @type select
 * @option カテゴリ1
 * @value 1
 * @option カテゴリ2
 * @value 2
 * @option カテゴリ3
 * @value 3
 * @option カテゴリ4
 * @value 4
 * @default 1
 * @desc このコメントに対して再生する効果音カテゴリ
 *
 * @arg displayTime
 * @type number
 * @default 300
 * @desc メッセージが表示される時間（フレーム数）
 */

/*~struct~BackgroundImage:
 * @param Filename
 * @type file
 * @dir img/pictures/
 * @desc 背景画像のファイル名
 *
 * @param Opacity
 * @type number
 * @default 255
 * @min 0
 * @max 255
 * @desc 背景画像の不透明度（0: 完全透明、255: 完全不透明）
 *
 * @param XPosition
 * @type number
 * @default 0
 * @desc チャットウィンドウに対するX位置（ピクセル単位）
 *
 * @param YPosition
 * @type number
 * @default 0
 * @desc チャットウィンドウに対するY位置（ピクセル単位）
 */

/*~struct~TopImage:
 * @param Filename
 * @type file
 * @dir img/pictures/
 * @desc 最前面画像のファイル名
 *
 * @param Opacity
 * @type number
 * @default 255
 * @min 0
 * @max 255
 * @desc 最前面画像の不透明度（0: 完全透明、255: 完全不透明）
 *
 * @param XPosition
 * @type number
 * @default 0
 * @desc チャットウィンドウに対するX位置（ピクセル単位）
 *
 * @param YPosition
 * @type number
 * @default 0
 * @desc チャットウィンドウに対するY位置（ピクセル単位）
 */




// （以下、本体処理）
// （以下、本体処理）
(() => {
    const parameters = PluginManager.parameters('SNSChatPlugin');
    const chatSpeed = Number(parameters['ChatSpeed'] || 5);
    const maxMessages = Number(parameters['MaxMessages'] || 10);
    const messageSpacing = Number(parameters['MessageSpacing'] || 10);
    const backgroundImages = JSON.parse(parameters['BackgroundImages'] || '[]');
    const showBackgroundImages = parameters['ShowBackgroundImages'] === 'true';
    const windowWidth = Number(parameters['WindowWidth'] || 420);
    const windowHeight = Number(parameters['WindowHeight'] || 720);
    const windowPositionX = Number(parameters['WindowPositionX'] || 0);
    const windowPositionY = Number(parameters['WindowPositionY'] || 0);
    const defaultFontSize = Number(parameters['DefaultFontSize'] || 20);
    const iconHeight = Number(parameters['IconHeight'] || 32);
    const iconOffsetX = Number(parameters['IconOffsetX'] || 10);
    const iconOffsetY = Number(parameters['IconOffsetY'] || 10);
    const commentOffsetX = Number(parameters['CommentOffsetX'] || 50);
    const commentOffsetY = Number(parameters['CommentOffsetY'] || 10);
    const commentBandHeight = Number(parameters['CommentBandHeight'] || 42);
    const donationCurrency = String(parameters['DonationCurrency'] || '$');
    const displaySwitch = Number(parameters['DisplaySwitch'] || 0);
    const messageFadeSwitch = Number(parameters['MessageFadeSwitch'] || 0);
    const layerPriority = Number(parameters['LayerPriority'] || 3);
    const topImagePriority = Number(parameters['TopImagePriority'] || 10);
    const soundEffects = [
      String(parameters['SoundEffectCategory1'] || 'AudioFileName1'),
      String(parameters['SoundEffectCategory2'] || 'AudioFileName2'),
      String(parameters['SoundEffectCategory3'] || 'AudioFileName3'),
      String(parameters['SoundEffectCategory4'] || 'AudioFileName4')
    ];
    const topImages = JSON.parse(parameters['TopImages'] || '[]');
  
    let chatMessages = [];
    let chatBitmap = null;
    let chatSprite = null;
    let backgroundSprites = [];
    let topImageSprites = [];
  
    function updateChatMessagePositions() {
      for (let i = 0; i < chatMessages.length; i++) {
        const message = chatMessages[i];
        const yOffset = i * (commentBandHeight + messageSpacing);
        message.y = windowPositionY + yOffset;
      }
    }
  
    function addChatMessage({ username, text, avatar, color, fontSize, donation, bgColor, soundCategory, displayTime }) {
      if (!$gameSwitches.value(displaySwitch)) return;
      if (chatMessages.length >= maxMessages) chatMessages.shift();
  
      chatMessages.push({
        username,
        text,
        avatar,
        color: color || '#ffffff',
        fontSize: fontSize || defaultFontSize,
        donation: donation || 0,
        bgColor: bgColor || 'transparent',
        y: windowPositionY + windowHeight,
        duration: displayTime || 300,
      });
  
      const soundEffect = soundEffects[soundCategory - 1];
      if (soundEffect) AudioManager.playSe({ name: soundEffect, pan: 0, pitch: 100, volume: 90 });
  
      updateChatMessagePositions();
      drawChatMessages();
    }
  
    function drawChatMessages() {
      chatBitmap.clear();
  
      if (!$gameSwitches.value(displaySwitch)) {
        chatSprite.visible = false;
        backgroundSprites.forEach(sprite => sprite.visible = false);
        topImageSprites.forEach(sprite => sprite.visible = false);
        return;
      }
  
      chatSprite.visible = true;
      backgroundSprites.forEach(sprite => sprite.visible = showBackgroundImages);
      topImageSprites.forEach(sprite => sprite.visible = true);
  
      for (let i = 0; i < chatMessages.length; i++) {
        const message = chatMessages[i];
        const yOffset = i * (commentBandHeight + messageSpacing);
  
        if (message.bgColor !== 'transparent') {
          chatBitmap.fillRect(windowPositionX, message.y + yOffset, windowWidth, commentBandHeight, message.bgColor);
        }
  
        if (message.avatar) {
          const avatarImage = ImageManager.loadPicture(message.avatar);
          chatBitmap.blt(avatarImage, 0, 0, iconHeight, iconHeight,
            windowPositionX + iconOffsetX,
            message.y + iconOffsetY + yOffset,
            iconHeight, iconHeight);
        }
  
        chatBitmap.textColor = message.color || '#ffffff';
        chatBitmap.fontSize = message.fontSize || defaultFontSize;
  
        const donationText = message.donation > 0 ? ` ${donationCurrency}${message.donation}` : '';
        chatBitmap.drawText(
          message.username + donationText,
          windowPositionX + commentOffsetX,
          message.y + yOffset + commentOffsetY,
          windowWidth - iconHeight - 20,
          commentBandHeight / 2,
          'left'
        );
  
        chatBitmap.drawText(
          message.text,
          windowPositionX + commentOffsetX,
          message.y + yOffset + commentBandHeight / 2 + commentOffsetY,
          windowWidth - iconHeight - 20,
          commentBandHeight / 2,
          'left'
        );
      }
    }
  
    const _Scene_Map_start = Scene_Map.prototype.start;
    Scene_Map.prototype.start = function () {
      _Scene_Map_start.call(this);
  
      if (showBackgroundImages) {
        backgroundImages.forEach(bg => {
          const parsed = JSON.parse(bg);
          const sprite = new Sprite(ImageManager.loadPicture(parsed.Filename));
          sprite.opacity = Number(parsed.Opacity);
          sprite.x = windowPositionX + Number(parsed.XPosition);
          sprite.y = windowPositionY + Number(parsed.YPosition);
          this.addChildAt(sprite, layerPriority - 1);
          backgroundSprites.push(sprite);
        });
      }
  
      chatSprite = new Sprite();
      chatBitmap = new Bitmap(windowWidth, windowHeight);
      chatSprite.bitmap = chatBitmap;
      chatSprite.x = windowPositionX;
      chatSprite.y = windowPositionY;
      this.addChildAt(chatSprite, layerPriority);
  
      topImages.forEach(img => {
        const parsed = JSON.parse(img);
        const sprite = new Sprite(ImageManager.loadPicture(parsed.Filename));
        sprite.opacity = Number(parsed.Opacity);
        sprite.x = windowPositionX + Number(parsed.XPosition);
        sprite.y = windowPositionY + Number(parsed.YPosition);
        this.addChildAt(sprite, layerPriority + 1);
        topImageSprites.push(sprite);
      });
  
      if (!$gameSwitches.value(displaySwitch)) {
        chatSprite.visible = false;
        backgroundSprites.forEach(sprite => sprite.visible = false);
        topImageSprites.forEach(sprite => sprite.visible = false);
      }
    };
  
    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function () {
      _Scene_Map_update.call(this);
      if (chatMessages.length > 0) {
        for (let i = 0; i < chatMessages.length; i++) {
          const message = chatMessages[i];
          if ($gameSwitches.value(messageFadeSwitch)) {
            message.duration--;
            if (message.duration <= 0) message.y -= chatSpeed;
          }
        }
        chatMessages = chatMessages.filter(msg => msg.y + commentBandHeight >= windowPositionY);
        drawChatMessages();
      }
    };
  
    $1
  
    PluginManager.registerCommand('SNSChatPlugin', 'ChangeSoundEffect', args => {
      const category = Number(args.category);
      const filename = args.filename;
      if (category >= 1 && category <= 4) {
        soundEffects[category - 1] = filename;
      }
    });
  
    PluginManager.registerCommand('SNSChatPlugin', 'ChangeBackgroundImage', args => {
      const index = Number(args.index);
      const filename = args.filename;
      const opacity = Number(args.opacity || 255);
      const x = Number(args.x || 0);
      const y = Number(args.y || 0);
  
      if (index >= 0 && index < backgroundSprites.length) {
        const sprite = backgroundSprites[index];
        sprite.bitmap = ImageManager.loadPicture(filename);
        sprite.opacity = opacity;
        sprite.x = windowPositionX + x;
        sprite.y = windowPositionY + y;
      }
    });
    });
  