/*:
 * @target MZ
 * @plugindesc カスタム画像タイトルメニュー（横並び＋CGMODEボタン対応）v1.1
 * @author mushizushi
 * 
 * @help
 * 画像ボタンで構成されたタイトル画面を作成します。
 * 各ボタンは img/titles1/ 内のPNGを使用します。
 *
 * ─────────────────────────────
 * 【構成】
 *  - 1段目（横並び4ボタン）
 *      NewGame / Load / Option / Exit
 *  - 2段目（中央）
 *      CGMODE（ConfigManager.cgUnlocked で解禁判定）
 *
 * ─────────────────────────────
 * 【操作】
 *  - ←→：同段ボタン移動
 *  - ↑↓：段切替（1⇄2）
 *  - 決定：ボタン実行
 *  - ロック時に決定：NG音
 *
 * ─────────────────────────────
 * 【ファイル命名例】
 *   btn_new.png / btn_new_hover.png / btn_new_locked.png
 *   btn_cg.png / btn_cg_hover.png / btn_cg_locked.png
 *   （img/titles1/ に配置）
 *
 * ─────────────────────────────
 * 【解禁判定】
 *   ConfigManager.cgUnlocked === true なら CGMODE 有効
 *
 * ─────────────────────────────
 * 【SE設定】
 * @param seHover
 * @text ホバーSE
 * @default Cursor1
 *
 * @param seOk
 * @text 決定SE
 * @default Decision1
 *
 * @param seNg
 * @text NGSE（ロック押下）
 * @default Buzzer1
 */

(() => {
  const PLUGIN_NAME = "TitleMenuCustomMZ";
  const P = PluginManager.parameters(PLUGIN_NAME);
  const SE_HOVER = P["seHover"] || "Cursor1";
  const SE_OK    = P["seOk"] || "Decision1";
  const SE_NG    = P["seNg"] || "Buzzer1";

  const MENU = {
    width: 1180,
    height: 720,
    y1: 520,        // 1段目Y
    y2: 612,        // 2段目Y
    w: 240, h: 56,  // ボタンサイズ
    cols: 4,
    offsetX: [0, 0, 0, 0],
  };

  const BUTTONS = [
    { key: "new", label: "NewGame", handler: "commandNewGame" },
    { key: "load", label: "Load", handler: "commandContinue" },
    { key: "opt", label: "Options", handler: "commandOptions" },
    { key: "exit", label: "Exit", handler: "commandExit" },
  ];

  // Scene_Title拡張
  const _Scene_Title_create = Scene_Title.prototype.create;
  Scene_Title.prototype.create = function() {
    _Scene_Title_create.call(this);
    this.createMenuButtons();
  };

  Scene_Title.prototype.createMenuButtons = function() {
    this._menuButtons = [];
    const baseY1 = MENU.y1;
    const baseY2 = MENU.y2;
    const colW = MENU.width / MENU.cols;

    for (let i = 0; i < BUTTONS.length; i++) {
      const data = BUTTONS[i];
      const sp = this.createButtonSprite(data.key);
      sp.x = colW * (i + 0.5) + MENU.offsetX[i];
      sp.y = baseY1;
      sp._row = 0;
      sp._col = i;
      sp._data = data;
      this.addChild(sp);
      this._menuButtons.push(sp);
    }

    const cg = this.createButtonSprite("cg");
    cg.x = MENU.width / 2;
    cg.y = baseY2;
    cg._row = 1;
    cg._col = 0;
    cg._data = { key: "cg", handler: "commandCGMode" };
    this.addChild(cg);
    this._cgButton = cg;

    this._row = 0;
    this._col = 0;
    this.updateButtonVisuals();
  };

  Scene_Title.prototype.createButtonSprite = function(key) {
    const s = new Sprite();
    s._key = key;
    s.anchor.set(0.5, 0.5);
    s.setState = (state) => {
      const fn = (state === "hover") ? `btn_${key}_hover` :
                 (state === "locked") ? `btn_${key}_locked` :
                 `btn_${key}`;
      s.bitmap = ImageManager.loadTitle1(fn);
    };
    return s;
  };

  Scene_Title.prototype.updateButtonVisuals = function() {
    const all = [...this._menuButtons, this._cgButton];
    for (const s of all) {
      const selected = (s._row === this._row && s._col === this._col);
      const unlocked = (s._key !== "cg" || ConfigManager.cgUnlocked === true);
      if (!unlocked) s.setState("locked");
      else if (selected) s.setState("hover");
      else s.setState("normal");
    }
  };

  Scene_Title.prototype.update = function() {
    Scene_Base.prototype.update.call(this);
    this.handleInput();
  };

  Scene_Title.prototype.handleInput = function() {
    if (Input.isTriggered("right")) this.changeSelection(1, 0);
    else if (Input.isTriggered("left")) this.changeSelection(-1, 0);
    else if (Input.isTriggered("down")) this.changeSelection(0, 1);
    else if (Input.isTriggered("up")) this.changeSelection(0, -1);
    else if (Input.isTriggered("ok")) this.selectCurrent();
  };

  Scene_Title.prototype.changeSelection = function(dx, dy) {
    const maxCols = MENU.cols;
    const maxRows = 2;
    AudioManager.playSe({ name: SE_HOVER, volume: 90, pitch: 100, pan: 0 });
    this._col = (this._col + dx + maxCols) % maxCols;
    this._row = (this._row + dy + maxRows) % maxRows;
    if (this._row === 1) this._col = 0;
    this.updateButtonVisuals();
  };

  Scene_Title.prototype.selectCurrent = function() {
    const target = (this._row === 0)
      ? this._menuButtons[this._col]
      : this._cgButton;
    const locked = (target._key === "cg" && ConfigManager.cgUnlocked !== true);
    if (locked) {
      AudioManager.playSe({ name: SE_NG, volume: 90, pitch: 100, pan: 0 });
      return;
    }
    AudioManager.playSe({ name: SE_OK, volume: 90, pitch: 100, pan: 0 });
    this.executeCommand(target._data.handler);
  };

  Scene_Title.prototype.executeCommand = function(handler) {
    switch (handler) {
      case "commandNewGame": this.commandNewGame(); break;
      case "commandContinue": this.commandContinue(); break;
      case "commandOptions": this.commandOptions(); break;
      case "commandExit": window.close(); break;
      case "commandCGMode":
        SceneManager.push(Scene_Options); // 仮：ここを Scene_CGGallery に
        break;
    }
  };
})();
