/*~struct~VariableBinding:
 * @param variableId
 * @text 変数ID
 * @type variable
 * @desc 監視対象の変数ID
 *
 * @param commonEventId
 * @text コモンイベントID
 * @type common_event
 * @desc 実行するコモンイベントのID
 *
 * @param cooldown
 * @text クールタイム（フレーム）
 * @type number
 * @min 0
 * @default 60
 * @desc この変数でコモンイベントを実行してから次に反応するまでの待機フレーム数
 */

/*:
 * @target MZ
 * @plugindesc 特定の変数が変化したときにコモンイベントを実行するプラグイン（setValueフック式＋個別クールタイム＋確実な発火制御＋比較変数）
 * @author ChatGPT
 *
 * @param VariableCommonBindings
 * @text 変数とコモンイベントの紐付け
 * @type struct<VariableBinding>[]
 * @desc 監視する変数と対応するコモンイベントIDの組み合わせとクールタイム
 *
 * @param EnableSwitchId
 * @text 実行を有効にするスイッチID
 * @type switch
 * @desc このスイッチがONのときだけ監視が有効になります（0で常時有効）
 *
 * @help
 * ▼概要
 * 指定した変数が setValue で変更されたときに、
 * 指定のコモンイベントを一定間隔のクールタイム付きで実行します。
 *
 * ▼主な仕様
 * - 変数IDごとに個別クールタイム（cooldown）設定可能
 * - 値を監視する変数と比較用変数（shadow）を用いて正確に変更検出
 * - イベントが実行中の場合は予約しない（$gameMap.isEventRunning）
 * - クールタイム中に入力が連続しても、発火は1回のみ
 * - 初回の変更には必ず反応します（即反応）
 */

(() => {
  const pluginName = "VariableWatcher";

  const parameters = PluginManager.parameters(pluginName);
  const variableBindingsRaw = JSON.parse(parameters["VariableCommonBindings"] || "[]");
  const enableSwitchId = Number(parameters["EnableSwitchId"] || 0);

  const variableBindings = variableBindingsRaw.map(binding => {
    const parsed = JSON.parse(binding);
    return {
      variableId: Number(parsed.variableId),
      commonEventId: Number(parsed.commonEventId),
      cooldown: Number(parsed.cooldown || 60),
      lastTriggered: -Infinity,
      shadowValue: undefined
    };
  });

  const watchMap = {};
  variableBindings.forEach(b => {
    watchMap[b.variableId] = b;
  });

  const _Game_Variables_setValue = Game_Variables.prototype.setValue;
  Game_Variables.prototype.setValue = function(variableId, value) {
    const oldValue = this.value(variableId);
    _Game_Variables_setValue.call(this, variableId, value);

    const binding = watchMap[variableId];
    if (!binding) return;

    const now = Graphics.frameCount;
    const shadowOld = binding.shadowValue;
    const changed = (shadowOld === undefined || shadowOld !== value);

    if (!changed) return;
    if (enableSwitchId > 0 && !$gameSwitches.value(enableSwitchId)) return;
    if ($gameMap.isEventRunning()) return;
    if (now - binding.lastTriggered < binding.cooldown) return;

    binding.shadowValue = value; // 更新: 比較用の影変数に記録
    binding.lastTriggered = now;
    $gameTemp.reserveCommonEvent(binding.commonEventId);
  };
})();
