// AutoNamePopup.js Ver.4.0.2
// MIT License (C) 2021 あわやまたな
// http://opensource.org/licenses/mit-license.php

/*:
* @target MV MZ
* @plugindesc  顔画像と連動して名前を表示できます。
* @author あわやまたな (Awaya_Matana)
* @url https://awaya3ji.seesaa.net/
* @help [プラグインコマンド（MV)]
* setNameKeys 名前 ファイル名 インデックス 表情差分 オプション文字列
* 表情差分が未入力の場合はパラメータ「キャラクターの表情差分」を参照します。
* "-1"だとパラメータ「キャラクターの表情差分」、
* "-2"だとパラメータ「アクターの表情差分」を参照します。
*
* [パラメータ設定方法]
* 【インデックス】
* [0][1][2][3]
* [4][5][6][7]
* 顔画像の左上を0とし、下段右寄りになるほど大きくなります。
*
* 【表情差分】
* 例えば同じキャラの喜怒驚泣の表情が並んでいるとします。
* [喜][怒][驚][泣]
* この場合、一番数字が若いインデックス、つまり「[喜]のインデックス」が
* 上記の【インデックス】の設定値になります。
* そして表情差分の数は喜を除いた怒驚泣の個数、つまり「3」が表情差分の
* 設定値になります。

* 要は
* (表情差分の設定値) = (表情差分の最後尾のインデックス) - (一番若いインデックス)
* 表情数が4つなら差分数は3つだし、表情数が8つなら差分数は7つです。
*
* "-1"だとパラメータ「キャラクターの表情差分」、
* "-2"だとパラメータ「アクターの表情差分」を参照します。
*
* 【動作モード】
* 「MVで外部プラグインと連携」
* YEP_MessageCoreなど、MVで名前ウィンドウを表示させるプラグインを
* 使いたい時に選びます。
* YEP_MessageCoreの場合、共通文字列1に"\n<"、共通文字列2に">"を入力する
* ことで自動で名前ウィンドウを表示してくれるようになります。
*
* 【共通文字列】
* 名前の前後に必ず付与される文字列です。
* 例えば、共通文字列1を"\>\c[24]"、共通文字列2を"\c[0]"とした場合、
* 全ての名前が
* \>\c[24]名前\c[0]
* と入力した時と同じ状態になります。
* 文字色を変更したり、制御文字を使用した外部プラグインとの連携に使ったりすることができます。
*
* 【オプション文字列】
* セリフの直前に必ず付与される文字列です。
* 例えば、オプション文字列を"\}"とした場合、
* 常に小声で話すキャラクターを作れます。
* 主に制御文字を使用した外部プラグインとの連携に使う為の機能です。
* MPP_MessageSEの場合、オプション文字列に"\se[n]"と入力すると、
* そのキャラクターが喋るたびにSEが再生されるようになります。
*
* [メモ]
* 基本、アクターの顔画像と名前はデータベースから自動で取得されます。
* よって手動で設定したい箇所のみメモで制御します。
*
* アクターのメモ欄に
* <popupName:n>
* （nは文字列)
* を入力することで文章の表示で表示する名前を設定できます。
* 未入力だと"\n[アクターID]"になります。
*
* <faceIndex:n>
* （nは自然数)
* を入力することで顔画像のインデックスを設定できます。
* 未入力だとデータベースで選択した顔のインデックスになります。
*
* <facialVariations:n>
* （nは自然数)
* を入力することでアクター個別で差分数を設定できます。
* パラメータの「アクターの表情差分」より優先されます。
*
* <optionString:n>
* （nは文字列)
* を入力することでオプション文字列を設定できます。
*
* [機能]
* 【制御文字】
* \EI
* インデント有効時、2行目以降にインデントを付与します。
*
* 【名前を非表示にする】
* 名前欄またはメッセージ冒頭に_（アンダーバー）を入力すると設定を無視して名前を非表示にできます。
*
* 【違う名前を表示する（MZのみ）】
* 文章の表示の名前欄に直接入力します。
* 共通文字列は自動で付与されます。
*
* [仕様]
* パラメータとアクターの顔画像が被った場合、パラメータの設定が優先されます。
* さらにプラグインコマンドで追加した場合はその追加分が最優先になります。
* プラグインコマンド > パラメータ > アクター
* プラグインコマンドで名前を追加するとセーブデータの容量が増すので注意が必要です。
* PluginCommonBaseに対応していますが無くても使えます。
* Ver.4.0.0より、インデントは全角空白ではなく、開始位置をずらす仕様に変更しました。
* よりスピーディな表示を実現します。
*
* [更新履歴]
* 2021/09/03：Ver.1.0.0　公開。
* 2021/09/05：Ver.1.1.0　インデント有効時の行頭の瞬間表示に対応。プラグインコマンドを追加。
* 2021/09/07：Ver.1.1.1　メモでアクターの顔画像のインデックスを直接指定できる機能を追加。
* 2021/09/14：Ver.1.1.2　プラグインコマンドやパラメータでアクターの表情差分を参照する機能を追加。
* 2021/10/24：Ver.2.0.0　設定項目を追加。文字サイズ変更によるインデントのズレを修正。
* 2021/11/03：Ver.2.0.1　設定項目を追加。空欄にインデントを付与してしまう不具合を修正。
* 2021/11/28：Ver.2.0.2　名前未設定の顔グラを使用した際フリーズする不具合を修正。
* 2021/12/06：Ver.3.0.0　名前入力欄に入力したとき、自動で共通文字列を適用する機能を追加。
* 2021/12/30：Ver.4.0.0　コードの見直し。顔差分のインデックスを追加。インデント改善。
* 2021/12/31：Ver.4.0.1　インデントのバグ修正。
* 2021/01/06：Ver.4.0.2　軽微な修正。
*
*
* @command setNameKeys
* @desc 顔画像と名前の関連付けを追加します。
*
* @arg name
* @text 名前
* @desc 表示させる名前。
* 制御文字が使用できます。
* @default
* @type string
*
* @arg faceName
* @text ファイル名
* @desc [0][1][2][3]
* [4][5][6][7]
* @default
* @type file
* @dir image/faces
*
* @arg faceIndex
* @text インデックス
* @desc [0][1][2][3]
* [4][5][6][7]
* @default 0
* @type number
*
* @arg facialVariations
* @text 表情差分
* @desc パラメータ「キャラクターの表情差分」を参照：-1
* パラメータ「アクターの表情差分」を参照：-2
* @default -1
* @type number
* @min -2
*
* @arg optionString
* @text オプション文字列
* @desc セリフ直前に付与する文字列。
* 主に制御文字など。
* @default
* @type string
*
* @param mode
* @text 動作モード
* @desc MV使用時、ネームウィンドウを追加するプラグインと組み合わせるなら「MVで外部プラグインと連携」を選択。
* @default 0
* @type select
* @option 標準
* @value 0
* @option MZでウィンドウ内に表示
* @value 1
* @option MVで外部プラグインと連携
* @value 2
*
* @param autoIndent
* @text 自動インデント
* @desc ウィンドウ内に名前を表示させる際、自動で名前以外の文字列を右に全角1文字ぶんずらします。
* @default true
* @type boolean
*
* @param actorFacialVariations
* @text アクターの表情差分
* @desc アクター共通の表情差分の数を入力します。
* 表情数から1を引くと表情差分の数になります。
* @default 0
* @type number
*
* @param characterFacialVariations
* @text キャラクターの表情差分
* @desc アクター以外共通の表情差分の数を入力します。
* 表情数から1を引くと表情差分の数になります。
* @default 0
* @type number
*
* @param commonString1
* @text 共通文字列1
* @desc 名前より前に付与する共通の文字列。
* 主に制御文字など。
* @default \>
* @type string
*
* @param commonString2
* @text 共通文字列2
* @desc 名前より後に付与する共通の文字列。
* 主に制御文字など。
* @default \<
* @type string
*
* @param nameKeys
* @text 名前キー
* @desc キャラクターの顔と名前の関連付け。
* @default 
* @type struct<key>[]
* 
*/

/*~struct~key:
*
* @param name
* @text 名前
* @desc 表示させる名前。
* 制御文字が使用できます。
* @default
* @type string
*
* @param faceName
* @text ファイル名
* @desc [0][1][2][3]
* [4][5][6][7]
* @default
* @type file
* @dir image/faces
*
* @param faceIndex
* @text インデックス
* @desc [0][1][2][3]
* [4][5][6][7]
* @default 0
* @type number
*
* @param facialVariations
* @text 表情差分
* @desc パラメータ「キャラクターの表情差分」を参照：-1
* パラメータ「アクターの表情差分」を参照：-2
* @default -1
* @type number
* @min -2
*
* @param optionString
* @text オプション文字列
* @desc セリフ直前に付与する文字列。
* 主に制御文字など。
* @default
* @type string
*
*/

'use strict';
{
	const useMZ = Utils.RPGMAKER_NAME === "MZ";
	//プラグイン名取得
	const pluginName = document.currentScript.src.replace(/^.*\/(.*).js$/, (_, p) => p);

	const hasPluginCommonBase = (()=>{
		 for(const i in $plugins) {
			if($plugins[i].name === "PluginCommonBase" && $plugins[i].status) return true;
		}
		return false;
	})();
	
	if (useMZ && hasPluginCommonBase) {
		PluginManagerEx.registerCommand(document.currentScript, "setNameKeys", function (args) {
			$gameSystem.setNameKeys(args);
		});
	} else if (useMZ) {
		PluginManager.registerCommand(pluginName, "setNameKeys", function (args) {
			args.faceIndex = Number(args.faceIndex);
			args.facialVariations = Number(args.facialVariations);
			$gameSystem.setNameKeys(args);
		});
	}

	const _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
	Game_Interpreter.prototype.pluginCommand = function(command, args) {
		_Game_Interpreter_pluginCommand.apply(this, arguments);
		
		command === "setNameKeys" && this[command](args);
		
	};

	Game_Interpreter.prototype.setNameKeys = function(args) {
		if (!args[3]) {
			args[3] = "0";
		}
		const argsObj = {
			name : args[0],
			faceName : args[1],
			faceIndex : Number(args[2] || "0"),
			facialVariations : Number(args[3] || "-1"),
			optionString : args[4]
		}
		$gameSystem.setNameKeys(argsObj);
	};
	
	//プラグインパラメータ取得
	const parameter = PluginManager.parameters(pluginName);
	const paramKeys = parameter['nameKeys'] && JSON.parse(parameter['nameKeys']).map(JSON.parse);
	for (const param of paramKeys) {
		for (const prop in param) {
			if (!isNaN(param[prop])) param[prop] = Number(param[prop])
		}
	}
	const commonString1 = parameter['commonString1'] || "";
	let commonString2 = parameter['commonString2'] || "";
	const actorFacialVariations = Number(parameter['actorFacialVariations']) || 0;
	const characterFacialVariations = Number(parameter['characterFacialVariations']) || 0;
	const mode = Number(parameter['mode']) || 0;
	const autoIndent = parameter['autoIndent'] === "true" && mode !== 2;

	if (mode <  2 && !useMZ || useMZ && mode === 1) {
		commonString2 += "\n";
	}
		
	const nameKeys = {};

	const setNameKeys = function($data) {
		for (const character of $data) {
			if (character === null) continue;
			let charName = "";
			let variations = 0;
			const faceName = character.faceName;
			let faceIndex = Number(character.faceIndex);
			let optionString = "";
			
			if ($data === $dataActors) {
				charName = character.meta["popupName"] || "\\N[" + character.id + "]";
				variations = character.meta["facialVariations"] ? Number(character.meta["facialVariations"]) : -2;
				if (character.meta["faceIndex"]) faceIndex = Number(character.meta["faceIndex"]);
				if (character.meta["optionString"]) optionString = character.meta["optionString"];
			} else {
				charName = character.name;
				variations = Number(character.facialVariations);
				optionString = character.optionString || "";
			}
			
			if (variations < 0) {
				variations = variations === -1 ? characterFacialVariations : actorFacialVariations;
			}

			for (let i = 0; i <= variations; i++) {
				const key = [faceName, faceIndex + i];
				nameKeys[key] = {
					name : charName,
					optionString : optionString,
					variationIndex : i,
					variations : variations
				}
			}
		}
	};

	Game_System.prototype.setNameKeys = function(args) {
		const charName = args.name;
		let variations = args.facialVariations;
		const optionString = args.optionString || "";
		if (variations < 0) {
			variations = variations === -1 ? characterFacialVariations : actorFacialVariations;
		}
		const faceName = args.faceName;
		const faceIndex = args.faceIndex;
		if (!this._nameKeys) this._nameKeys = {};

		for (let i = 0; i <= variations; i++) {
			const key = [faceName, faceIndex + i];
			this._nameKeys[key] = {
				name : charName,
				optionString : optionString,
				variationIndex : i,
				variations : variations
			}
		}
	};

	const propSet = new Set(["variationIndex", "variations"]);
	Game_System.prototype.getNameKeyParam = function(key, prop) {
		return this._nameKeys && this._nameKeys[key] && this._nameKeys[key][prop] || nameKeys[key] && nameKeys[key][prop] || (propSet.has(prop) ? 0 : "");
	};

	const _Scene_Boot_isReady  = Scene_Boot.prototype.isReady;
	Scene_Boot.prototype.isReady = function() {
		if (_Scene_Boot_isReady.call(this)) {
			setNameKeys($dataActors);
			setNameKeys(paramKeys);
		}
		return _Scene_Boot_isReady.call(this);
	};
	const inTheWindow = mode > 0 || !useMZ;
	let enableIndent = false;
	if (inTheWindow) {
		if (useMZ) {
			Window_Message.prototype.standardFontSize = function() {
				return $gameSystem.mainFontSize();
			};
		}
		const newLineX = useMZ ? "startX" : "left";
		const _Window_Message_newPage = Window_Message.prototype.newPage;
		Window_Message.prototype.newPage = function(textState) {
			_Window_Message_newPage.call(this, textState);
			if (enableIndent) {
				const indentWidth = this.standardFontSize();
				textState[newLineX] += (textState.rtl ? -indentWidth : indentWidth);
				enableIndent = false;
			}
		};
	}
	const _Game_Interpreter_command101 = Game_Interpreter.prototype.command101;
	Game_Interpreter.prototype.command101 = function() {
		const params = this.currentCommand().parameters;
		const defaultName = params[4];
		const nextParams = this._list[this._index+1].parameters;
		const rawNextParam = nextParams[0];
		const _underbar = nextParams[0].charAt(0) === "_";
		const noName = defaultName === "_" || _underbar;
		params[4] = params[4] || "";
		if (noName) {
			params[4] =  "";
			if (_underbar) {
				nextParams[0] = nextParams[0].slice(1);
			}
 		} else {
 			const key = [params[0], params[1]];
 			const characterName = defaultName || $gameSystem.getNameKeyParam(key, "name");
			const optionString = /*defaultName ? "" : */$gameSystem.getNameKeyParam(key, "optionString");
 			if (characterName) {
 				params[4] = commonString1 + characterName + commonString2;
				nextParams[0] = optionString + nextParams[0];
	 			if (inTheWindow) {
	 				nextParams[0] = params[4] + nextParams[0];
	 				params[4] = "";
	 				enableIndent = autoIndent;
	 			}
 			}
		}
		const result = _Game_Interpreter_command101.call(this, params);
		params[4] = defaultName
		nextParams[0] = rawNextParam;
		return result;
	};
	const _Window_Base_convertEscapeCharacters = Window_Base.prototype.convertEscapeCharacters;
	Window_Base.prototype.convertEscapeCharacters = function(text) {
		text = _Window_Base_convertEscapeCharacters.call(this, text);
		text = text.replace(/\x1bEI/gi, (_, p1) =>{
			enableIndent = autoIndent;
			return "";
		});
		return text;
	};
	
}