// *****************************************************
// StatePopupCommandMZ
// Copyright (c) 2024 Tear
// This software is released under the MIT license.
// http://opensource.org/licenses/mit-license.php
// ***********************************************************************
// ver.1.00: 新規作成
// ver.1.10: コマンドタイプ選択と追加位置の指定機能を追加
//           プラグインパラメータのデフォルト値を変更、一部不具合を修正
// ver.1.11: ステータスウィンドウ表示／非表示の指定機能を追加
// ***********************************************************************
/*:ja
 * @target MZ
 * @plugindesc 戦闘コマンドにステート確認を追加するMZ専用プラグイン
 * @author Tear
 *
 * @help StatePopupCommandMZ.js
 * 戦闘コマンドにステート確認を追加します。
 * 対象者(味方、敵）を選択すると、対象者のステート状態を一覧表示します。
 *
 * ■ステートのメモ欄
 * 【ステートの説明内容】
 *   ステート一覧で、カーソルを合わせたステートの説明内容を設定します。
 *   メモ欄に記述する書式は下記の通りです。
 *   <SPCMZ_DESC:ここに表示する説明を書きます>
 *
 * 【ステート一覧の除外対象】
 *   ステート一覧から除外します。
 *   メモ欄に記述する書式は下記の通りです。
 *   <SPCMZ_HIDE:>
 *
 * プラグインコマンドはありません。
 *
 * @param command_name
 * @type string
 * @text コマンド名
 * @desc コマンド名を設定します。
 * @default ステート確認
 *
 * @param state_turns
 * @type string
 * @text ステートの残りターン
 * @desc ステートの残りターン表示を設定します。
 * %に残りターン数が表示されます。
 * @default 残り%ターン
 *
 * @param turns_fontsize
 * @type number
 * @text 残りターンのフォントサイズ
 * @desc 残りターンのフォントサイズを指定します。
 * @default 20
 *
 * @param turns_fontcolor
 * @type string
 * @text 残りターンのフォントカラー
 * @desc 残りターンのフォントカラーをR,G,B,A形式で指定します。
 * 赤255,緑255,青50,透明度80%の例：「255, 255, 50, 0.8」
 * @default 255, 255, 50, 0.8
 *
 * @param isdisp_icon_pt
 * @type boolean
 * @on 表示する
 * @off 表示しない
 * @text 味方パーティの状態アイコン
 * @desc 味方パーティの状態アイコン表示を設定します。
 * @default true
 *
 * @param isdisp_icon_em
 * @type boolean
 * @on 表示する
 * @off 表示しない
 * @text 敵グループの状態アイコン
 * @desc 敵グループの状態アイコン表示を設定します。
 * @default true
 *
 * @param turns_sort
 * @type select
 * @option ステートID順
 * @value stateId
 * @option 優先度順
 * @value statePrior
 * @text ステートのソート方法を指定
 * @desc ステートのソート方法を指定します。
 * @default stateId
 *
 * @param cmd_type
 * @type select
 * @option パーティコマンド
 * @value typePartyCmd
 * @option アクターコマンド
 * @value typeActorCmd
 * @text コマンドタイプ
 * @desc どのコマンドに追加するかを指定します。
 * @default typePartyCmd
 *
 * @param cmd_pos
 * @type number
 * @min 1
 * @max 9
 * @text コマンド追加の位置
 * @desc コマンドを追加する位置を指定します。(先頭＝1)
 * @default 2
 *
 * @param isdisp_wnd_status
 * @type boolean
 * @on 表示する
 * @off 表示しない
 * @text ステータスウィンドウ
 * @desc ステータスウィンドウ表示を設定します。
 * @default false
 *
 */

// -----------------------------------------------------
// Window_BattleTarget constructor
// -----------------------------------------------------
function Window_BattleTarget() {
    this.initialize(...arguments);
}

(() => {

    const PARAMS = PluginManager.parameters("StatePopupCommandMZ");
    const tagSPCMZ_DESC = "SPCMZ_DESC";
    const tagSPCMZ_HIDE = "SPCMZ_HIDE";
    const parCOMMAND_NAME    = PARAMS.command_name || "";
    const parSTATE_TURNS     = PARAMS.state_turns || "";
    const parTURNS_FONTSIZE  = Number(PARAMS.turns_fontsize || "20");
    const parTURNS_FONTCOLOR = PARAMS.turns_fontcolor || "255, 255, 50, 0.8";
    const parTURNS_SORT      = PARAMS.turns_sort || "";
    const parISDISP_ICON_PT  = PARAMS.isdisp_icon_pt === "true" ? true : false;
    const parISDISP_ICON_EM  = PARAMS.isdisp_icon_em === "true" ? true : false;
    const parCMD_TYPE        = PARAMS.cmd_type || "";
    const parCMD_POS         = Number(PARAMS.cmd_pos || "2");
    const parISDISP_WND_STATUS = PARAMS.isdisp_wnd_status === "true" ? true : false;
    // -----------------------------------------------------
    // Scene_Battle
    // -----------------------------------------------------
    const _Scene_Battle_createPartyCommandWindow =
        Scene_Battle.prototype.createPartyCommandWindow;
    Scene_Battle.prototype.createPartyCommandWindow = function() {
        _Scene_Battle_createPartyCommandWindow.call(this);
        if (parCMD_TYPE === "typePartyCmd") {
            this._partyCommandWindow.setHandler(
             "stpopup", this.commandSTPopup.bind(this)
            );
        }
    };

    const _Scene_Battle_createActorCommandWindow =
        Scene_Battle.prototype.createActorCommandWindow;
    Scene_Battle.prototype.createActorCommandWindow = function() {
        _Scene_Battle_createActorCommandWindow.call(this);
        if (parCMD_TYPE === "typeActorCmd") {
            this._actorCommandWindow.setHandler(
             "stpopup", this.commandSTPopup.bind(this)
            );
        }
    };

    Scene_Battle.prototype.commandSTPopup = function() {
        this.startTargetSelection();
    };

    const _Scene_Battle_createAllWindows =
        Scene_Battle.prototype.createAllWindows;
    Scene_Battle.prototype.createAllWindows = function() {
        _Scene_Battle_createAllWindows.call(this);
        this.createTargetWindow();
        this.createStateListWindow();
    };

    Scene_Battle.prototype.createTargetWindow = function() {
        const rect = this.targetWindowRect();
        this._targetWindow = new Window_BattleTarget(rect);
        this._targetWindow.setHandler("ok", this.onTargetOk.bind(this));
        this._targetWindow.setHandler("cancel", this.onTargetCancel.bind(this));
        this.addWindow(this._targetWindow);
    };

    Scene_Battle.prototype.targetWindowRect = function() {
        const wx = this._statusWindow.x;
        const ww = this._statusWindow.width;
        const wh = this.windowAreaHeight();
        const wy = Graphics.boxHeight - wh;
        return new Rectangle(wx, wy, ww, wh);
    };

    Scene_Battle.prototype.createStateListWindow = function() {
        const rect = this.stateListWindowRect();
        this._stateListWindow = new Window_BattleState(rect);
        this._stateListWindow.setHelpWindow(this._helpWindow);
        this._stateListWindow.setHandler("cancel", this.onStateCancel.bind(this));
        this.addWindow(this._stateListWindow);
        BattleManager.setStateListWindow(this._stateListWindow);
    };

    Scene_Battle.prototype.stateListWindowRect = function() {
        const ww = Graphics.boxWidth;
        const wh = this.windowAreaHeight();
        const wx = 0;
        const wy = Graphics.boxHeight - wh;
        return new Rectangle(wx, wy, ww, wh);
    };

    const _Scene_Battle_isAnyInputWindowActive =
        Scene_Battle.prototype.isAnyInputWindowActive;
    Scene_Battle.prototype.isAnyInputWindowActive = function() {
        return (
            _Scene_Battle_isAnyInputWindowActive.call(this) ||
            this._targetWindow.active ||
            this._stateListWindow.active
        );
    };

    Scene_Battle.prototype.onTargetOk = function() {
        const target = this._targetWindow.target();
        if (parISDISP_WND_STATUS === false) {
            this._statusWindow.hide();
        }
        this._stateListWindow.setTarget(target);
        this._stateListWindow.refresh();
        this._stateListWindow.show();
        this._stateListWindow.activate();
        if (parCMD_TYPE === "typePartyCmd") {
            this._partyCommandWindow.hide();
        }
        if (parCMD_TYPE === "typeActorCmd") {
        	this._actorCommandWindow.hide();
        }
    };

    Scene_Battle.prototype.onTargetCancel = function() {
       this._targetWindow.hide();
       this._statusWindow.show();
       if (parCMD_TYPE === "typePartyCmd") {
           this._partyCommandWindow.show();
           this._partyCommandWindow.activate();
       }
       if (parCMD_TYPE === "typeActorCmd") {
           this._actorCommandWindow.show();
           this._actorCommandWindow.activate();
       }
    };

    Scene_Battle.prototype.onStateCancel = function() {
        this._stateListWindow.hide();
        if (parCMD_TYPE === "typePartyCmd") {
            this._partyCommandWindow.show();
        }
        if (parCMD_TYPE === "typeActorCmd") {
            this._actorCommandWindow.show();
        }
        this._targetWindow.show();
        this._targetWindow.activate();
    };

    Scene_Battle.prototype.startTargetSelection = function() {
        this._targetWindow.refresh();
        this._targetWindow.show();
        this._targetWindow.select(0);
        this._targetWindow.activate();
        if (parISDISP_WND_STATUS === false) {
            this._statusWindow.hide();
        }
    };

    // -----------------------------------------------------
    // Window_BattleTarget
    // -----------------------------------------------------
    Window_BattleTarget.prototype = Object.create(Window_Selectable.prototype);
    Window_BattleTarget.prototype.constructor = Window_BattleTarget;

    Window_BattleTarget.prototype.initialize = function(rect) {
        this._targets = [];
        Window_Selectable.prototype.initialize.call(this, rect);
        this.refresh();
        this.hide();
        this._targetPre = null;
    };

    Window_BattleTarget.prototype.maxCols = function() {
        return 2;
    };

    Window_BattleTarget.prototype.maxItems = function() {
        return this._targets.length;
    };

    Window_BattleTarget.prototype.target = function() {
        return this._targets[this.index()];
    };

    Window_BattleTarget.prototype.targetPre = function() {
        return this._targetPre;
    };

    Window_BattleTarget.prototype.changeTargetPre = function() {
        this._targetPre = this.target();
    };

    Window_BattleTarget.prototype.targetIndex = function() {
        const target = this.target();
        return target ? target.index() : -1;
    };

    Window_BattleTarget.prototype.drawItem = function(index) {
        const target = this._targets[index];
        const rect = this.itemLineRect(index);
        if (target.isActor()) {
            this.actorTextColor();
        } else if (target.isEnemy()) {
            this.enemyTextColor();
        } else {
            this.resetTextColor();
        }
        this.drawText(target.name(), rect.x, rect.y, rect.width);
    };

    Window_BattleTarget.prototype.actorTextColor = function() {
        this.changeTextColor(ColorManager.textColor(4));
        this.changeOutlineColor(ColorManager.outlineColor());
    };

    Window_BattleTarget.prototype.enemyTextColor = function() {
        this.changeTextColor(ColorManager.textColor(2));
        this.changeOutlineColor(ColorManager.outlineColor());
    };

    Window_BattleTarget.prototype.show = function() {
        const index = this.index();
        this.refresh();
        this.forceSelect(index === -1 ? 0 : index);
        $gameTemp.clearTouchState();
        Window_Selectable.prototype.show.call(this);
    };

    Window_BattleTarget.prototype.hide = function() {
        Window_Selectable.prototype.hide.call(this);
        $gameTroop.select(null);
    };

    Window_BattleTarget.prototype.refresh = function() {
        this._targets = 
            $gameParty.aliveMembers().concat(
             $gameTroop.aliveMembers()
            );
        Window_Selectable.prototype.refresh.call(this);
    };

    Window_BattleTarget.prototype.select = function(index) {
        Window_Selectable.prototype.select.call(this, index);
        const target = this.target();
        if (target) {
            this.unitChange();
            this.changeTargetPre();
            if (target.isActor()) {
                $gameParty.select(target);
            } else if (target.isEnemy()) {
                $gameTroop.select(target);
            }
        }
    };

    Window_BattleTarget.prototype.unitChange = function() {
        const target = this.target();
        const targetPre = this.targetPre() || target;
        if (target.isActor() && targetPre.isEnemy()) {
            targetPre.deselect();
            return "party";
        } else if (target.isEnemy() && targetPre.isActor()) {
            targetPre.deselect();
            return "troop";
        } else {
            return "";
        }
    };

    Window_BattleTarget.prototype.processTouch = function() {
        Window_Selectable.prototype.processTouch.call(this);
        if (this.isOpenAndActive()) {
            const target = $gameTemp.touchTarget();
            if (target) {
                if (this._targets.includes(target)) {
                    this.select(this._targets.indexOf(target));
                    if ($gameTemp.touchState() === "click") {
                        this.processOk();
                    }
                }
                $gameTemp.clearTouchState();
            }
        }
    };

    // -----------------------------------------------------
    // Window_StateList
    // -----------------------------------------------------
    function Window_StateList() {
        this.initialize(...arguments);
    }

    Window_StateList.prototype = Object.create(Window_Selectable.prototype);
    Window_StateList.prototype.constructor = Window_StateList;

    Window_StateList.prototype.initialize = function(rect) {
        Window_Selectable.prototype.initialize.call(this, rect);
        this._target = null;
        this._data = [];
    };

    Window_StateList.prototype.setTarget = function(target) {
        if (this._target !== target) {
            this._target = target;
            this.refresh();
            this.scrollTo(0, 0);
        }
    };

    Window_StateList.prototype.target = function() {
        return this._target;
    };

    Window_StateList.prototype.maxCols = function() {
        return 2;
    };

    Window_StateList.prototype.colSpacing = function() {
        return 16;
    };

    Window_StateList.prototype.maxItems = function() {
        return this._data ? this._data.length : 1;
    };

    Window_StateList.prototype.item = function() {
        return this.itemAt(this.index());
    };

    Window_StateList.prototype.itemAt = function(index) {
        return (
            this._data && index >= 0 ? 
            this._data[index] : null
        );
    };

    Window_StateList.prototype.includes = function(item) {
        return (
            item && 
            item.meta[tagSPCMZ_HIDE] === undefined
        );
    };

    Window_StateList.prototype.makeItemList = function() {
        if (this._target) {
            this._data = this._target.states().filter(
                item => this.includes(item)
            );
            this.makeItemListSort();
        } else {
            this._data = [];
        }
    };

    Window_StateList.prototype.makeItemListSort = function() {
        if (parTURNS_SORT === "stateId") {
            this._data = this._data.sort(
                (a, b) => a.id - b.id
            );
        }
    };

    Window_StateList.prototype.selectLast = function() {
        const newIndex = this._data.length - 1;
        const index = Math.max(this.index(), 0);
        this.forceSelect(
            this._data.length > 0 ?
            (newIndex < index ? newIndex : index) : -1
        );
    };

    Window_StateList.prototype.drawItem = function(index) {
        const state = this.itemAt(index);
        if (state) {
            const turnsWidth = this.turnsWidth();
            const rect = this.itemLineRect(index);
            this.changePaintOpacity(true);
            this.drawItemName(state, rect.x, rect.y, rect.width - turnsWidth);
            this.drawStateTurns(state, rect.x, rect.y, rect.width);
            this.changePaintOpacity(1);
        }
    };

    Window_StateList.prototype.turnsWidth = function() {
        const testText = parSTATE_TURNS.replace("%", "00")
        return this.textWidth(testText);
    };

    Window_StateList.prototype.drawStateTurns = function(state, x, y, width) {
        const turns = this._target._stateTurns[state.id];
        const turnsText = parSTATE_TURNS.replace("%", turns);
        const orgFontSize = this.contents.fontSize;
        if (state.autoRemovalTiming === 2) {
            this.changeTextColor("rgba(" + parTURNS_FONTCOLOR + ")");
            this.contents.fontSize = parTURNS_FONTSIZE;
            this.drawText(turnsText, x, y, width, "right");
            this.contents.fontSize = orgFontSize;
        }
    };

    Window_StateList.prototype.updateHelp = function() {
        const item = this.item() || new Game_Item();
        item.description = this.makeDescription(item);
        this.setHelpWindowItem(item);
    };

    Window_StateList.prototype.refresh = function() {
        this.makeItemList();
        Window_Selectable.prototype.refresh.call(this);
        this.selectLast();
    };

    Window_StateList.prototype.makeDescription = function(item) {
        if (this._data.length > 0) {
            return item.meta[tagSPCMZ_DESC] || "";
        } else {
            return "";
        }
    };

    // -----------------------------------------------------
    // Window_BattleState
    // -----------------------------------------------------
    function Window_BattleState() {
        this.initialize(...arguments);
    }

    Window_BattleState.prototype = Object.create(Window_StateList.prototype);
    Window_BattleState.prototype.constructor = Window_BattleState;

    Window_BattleState.prototype.initialize = function(rect) {
        Window_StateList.prototype.initialize.call(this, rect);
        this.hide();
    };

    Window_BattleState.prototype.show = function() {
        this.selectLast();
        this.showHelpWindow();
        Window_StateList.prototype.show.call(this);
    };

    Window_BattleState.prototype.hide = function() {
        this.hideHelpWindow();
        Window_StateList.prototype.hide.call(this);
    };

    // -----------------------------------------------------
    // Window_PartyCommand
    // -----------------------------------------------------
    function addStatePopupCommand(list, length) {
        const cmdPos = parCMD_POS.clamp(1, length);
        if (cmdPos < length) {
            list.splice(cmdPos - 1, 0, list[length - 1]);
            list.pop();
        }
    }

    const _Window_PartyCommand_makeCommandList =
        Window_PartyCommand.prototype.makeCommandList;
    Window_PartyCommand.prototype.makeCommandList = function() {
        _Window_PartyCommand_makeCommandList.call(this);
       if (parCMD_TYPE === "typePartyCmd") {
           this.addStatePopupCommand();
       }
    };

    Window_PartyCommand.prototype.addStatePopupCommand = function() {
        this.addCommand(parCOMMAND_NAME, "stpopup", true);
        addStatePopupCommand(this._list, this._list.length);
    };

    // -----------------------------------------------------
    // Window_ActorCommand
    // -----------------------------------------------------
    const _Window_ActorCommand_makeCommandList =
        Window_ActorCommand.prototype.makeCommandList;
    Window_ActorCommand.prototype.makeCommandList = function() {
        _Window_ActorCommand_makeCommandList.call(this);
        if (parCMD_TYPE === "typeActorCmd") {
            if (this._actor) {
                this.addStatePopupCommand();
            }
        }
    };

    Window_ActorCommand.prototype.addStatePopupCommand = function() {
        this.addCommand(parCOMMAND_NAME, "stpopup", true);
        addStatePopupCommand(this._list, this._list.length);
    };

    // -----------------------------------------------------
    // BattleManager
    // -----------------------------------------------------
    BattleManager.setStateListWindow = function(stateListWindow) {
        this._stateListWindow = stateListWindow
    }

    BattleManager.getStateListWindow = function() {
        return this._stateListWindow;
    }

    // -----------------------------------------------------
    // Game_Battler
    // -----------------------------------------------------
    function getBattler(clsBattler) {
        if (clsBattler instanceof Game_Actor) {
            return $gameActors.actor(clsBattler.actorId());
        } else {
            return clsBattler;
        }
   }

    Game_Battler.prototype.refreshStateListWindow = function() {
        const stateListWindow = BattleManager.getStateListWindow();
        if (stateListWindow && stateListWindow.visible) {
            if (getBattler(this) === stateListWindow.target()) {
                stateListWindow.refresh();
            }
        }
    };

    const _Game_Battler_onTurnEnd = Game_Battler.prototype.onTurnEnd;
    Game_Battler.prototype.onTurnEnd = function() {
        _Game_Battler_onTurnEnd.call(this);
        this.refreshStateListWindow();
    };

    const _Game_Battler_addState = Game_Battler.prototype.addState;
    Game_Battler.prototype.addState = function(stateId) {
        _Game_Battler_addState.call(this, stateId);
        this.refreshStateListWindow();
    };

    const _Game_Battler_removeState = Game_Battler.prototype.removeState;
    Game_Battler.prototype.removeState = function(stateId) {
        _Game_Battler_removeState.call(this, stateId);
        this.refreshStateListWindow();
    };

    // -----------------------------------------------------
    // Window_StatusBase
    // -----------------------------------------------------
    const _Window_StatusBase_placeStateIcon =
        Window_StatusBase.prototype.placeStateIcon;
    Window_StatusBase.prototype.placeStateIcon = function(actor, x, y) {
        if (parISDISP_ICON_PT === false) {
            return;
        }
        _Window_StatusBase_placeStateIcon.call(this, actor, x, y);
    };

    // -----------------------------------------------------
    // Sprite_Enemy
    // -----------------------------------------------------
    const _Sprite_Enemy_createStateIconSprite =
        Sprite_Enemy.prototype.createStateIconSprite;
    Sprite_Enemy.prototype.createStateIconSprite = function() {
        _Sprite_Enemy_createStateIconSprite.call(this);
        if (parISDISP_ICON_EM === false) {
            this.children.pop();
        }
    };
    
})();