/*:
 * @target MZ
 * @plugindesc 画面ブラーのオン/オフ＆強さ変更をプラグインコマンドで制御（RPGツクールMZ）
 * @author nino
 *
 * @help
 * 画面全体にブラー（ぼかし）をかけた状態を維持し、オン/オフを切り替えできます。
 * マップ・バトルの双方に対応。フェードイン/アウトやズームとも干渉しません。
 *
 * 【使い方（イベントのプラグインコマンド）】
 *  - BlurOn  : ブラーを有効化（強さを指定）
 *  - BlurOff : ブラーを無効化（完全オフ）
 *  - BlurSet : 有効化したまま強さだけ変更
 *
 * 典型例：
 *  1) 「匂い」シーン開始時：BlurOn（強さ 6）
 *  2) シーン中に段階的に強めたい：BlurSet（強さ 8）
 *  3) 終了時：BlurOff
 *
 * 数値の目安：
 *   3〜4 = ほんのり / 5〜6 = 明確 / 7〜8 = 強め / 10+ = 霧のように強い
 *
 * このプラグインは画面の Spriteset に PIXI の BlurFilter を追加し、
 * シーン遷移（マップ⇄バトル）時も状態を引き継ぐようにしています。
 */

(() => {
  const pluginName = "Nino_BlurToggle";

  // ---- Game_System に状態を保持 ----
  const sysKey = "_ninoBlurState";
  const defaultState = () => ({ enabled: false, strength: 6 });

  const getState = () => {
    if (!$gameSystem[sysKey]) $gameSystem[sysKey] = defaultState();
    return $gameSystem[sysKey];
  };

  // ---- プラグインコマンド ----
  PluginManager.registerCommand(pluginName, "BlurOn", args => {
    const s = getState();
    const val = Number(args.strength ?? 6);
    s.enabled = true;
    s.strength = Math.max(0, val);
  });

  PluginManager.registerCommand(pluginName, "BlurOff", _args => {
    const s = getState();
    s.enabled = false;
  });

  PluginManager.registerCommand(pluginName, "BlurSet", args => {
    const s = getState();
    const val = Number(args.strength ?? s.strength ?? 6);
    s.strength = Math.max(0, val);
  });

  /*~struct~none: hidden */

  /*:
   * @command BlurOn
   * @text BlurOn（有効化）
   * @desc 画面ブラーを有効化します。強さを指定（例：6）
   * @arg strength
   * @type number
   * @min 0
   * @default 6
   *
   * @command BlurOff
   * @text BlurOff（無効化）
   * @desc 画面ブラーを無効化します。
   *
   * @command BlurSet
   * @text BlurSet（強さ変更）
   * @desc ブラーを有効化したまま強さだけ変更します。
   * @arg strength
   * @type number
   * @min 0
   * @default 6
   */

  // ---- Spriteset にブラーを適用・同期するユーティリティ ----
  const ensureBlurFilter = spriteset => {
    if (!spriteset) return null;
    if (!spriteset._ninoBlurFilter) {
      spriteset._ninoBlurFilter = new PIXI.filters.BlurFilter();
      spriteset._ninoBlurFilter.blur = 0;
    }
    return spriteset._ninoBlurFilter;
  };

  const removeBlurFilter = spriteset => {
    if (!spriteset) return;
    if (!spriteset.filters) return; // 既に無い
    // _ninoBlurFilter 以外は維持
    const keep = spriteset.filters.filter(f => f !== spriteset._ninoBlurFilter);
    spriteset.filters = keep.length ? keep : null;
  };

  const applyBlurEnabled = spriteset => {
    const state = getState();
    if (!spriteset) return;

    // キャッシュ値と比較して変化した時だけ触る
    spriteset._ninoPrevEnabled ??= false;
    spriteset._ninoPrevStrength ??= 0;

    if (state.enabled) {
      const filter = ensureBlurFilter(spriteset);
      // 既存 filters に含まれていなければ追加
      const arr = spriteset.filters ? spriteset.filters.slice() : [];
      if (!arr.includes(filter)) {
        arr.push(filter);
        spriteset.filters = arr;
      }
      // 強さを反映（毎フレームでも軽い）
      filter.blur = Number(state.strength || 0);
    } else {
      // 無効化：除去
      removeBlurFilter(spriteset);
    }

    spriteset._ninoPrevEnabled = state.enabled;
    spriteset._ninoPrevStrength = state.strength;
  };

  // ---- 全ての Spriteset（Map/Battle）で同期 ----
  const _Spriteset_Base_update = Spriteset_Base.prototype.update;
  Spriteset_Base.prototype.update = function() {
    _Spriteset_Base_update.call(this);
    applyBlurEnabled(this);
  };

  // シーン作成時にも一度同期（表示直後に反映させる）
  const _Scene_Map_createSpriteset = Scene_Map.prototype.createSpriteset;
  Scene_Map.prototype.createSpriteset = function() {
    _Scene_Map_createSpriteset.call(this);
    applyBlurEnabled(this._spriteset);
  };

  const _Scene_Battle_createSpriteset = Scene_Battle.prototype.createSpriteset;
  Scene_Battle.prototype.createSpriteset = function() {
    _Scene_Battle_createSpriteset.call(this);
    applyBlurEnabled(this._spriteset);
  };

})();