/*:
 * @target MZ
 * @plugindesc Nino_HazeToggle 向け 安全遅延ガード v1.0（マップ転送/遷移直後の数フレームだけ呼び出しを保留して実行）
 * @author nino
 * @help
 * 目的：
 *  - マップ転送直後など、PIXI のテクスチャ/renderer が未準備の瞬間に
 *    Nino_HazeToggle の HazeOn/HazePulse 等が走ってクラッシュするのを回避。
 * 方法：
 *  - シーン開始直後に「クールダウン枠（既定5F）」を設け、その間に来た
 *    Haze系コマンド/関数呼び出しをキューに溜め、枠が明けた瞬間にまとめて実行。
 * 既存イベントやスクリプトは変更不要です。これ単体で安全化します。
 *
 * 設定（コード先頭の定数を必要に応じて変更）:
 *  - DEFER_FRAMES = 5  // 遅延フレーム数（2〜6あたりが推奨）
 */

(() => {
  const DEFER_FRAMES = 5; // ←必要なら 3〜6 に調整

  // 内部状態
  let _deferFrames = 0;
  const _queue = [];

  // renderer/sceneがだいたい揃っているかの軽いチェック
  function rendererReady() {
    return !!(Graphics?.app?.renderer && SceneManager._scene && SceneManager._scene._spriteset);
  }

  function enqueue(fn) { _queue.push(fn); }

  function callOrDefer(fn) {
    if (_deferFrames > 0 || !rendererReady()) {
      enqueue(fn);
    } else {
      try { fn(); } catch(e) { console.warn("[Nino_HazeDeferGuard] call failed:", e); }
    }
  }

  function flushQueue() {
    while (_queue.length) {
      const fn = _queue.shift();
      try { fn && fn(); } catch(e) { console.warn("[Nino_HazeDeferGuard] flush failed:", e); }
    }
  }

  // --- シーン開始ごとにクールダウン枠をセット ---
  const _Scene_Map_start = Scene_Map.prototype.start;
  Scene_Map.prototype.start = function() {
    _Scene_Map_start.call(this);
    _deferFrames = Math.max(_deferFrames, DEFER_FRAMES);
  };
  const _Scene_Battle_start = Scene_Battle.prototype.start;
  if (_Scene_Battle_start) {
    Scene_Battle.prototype.start = function() {
      _Scene_Battle_start.call(this);
      _deferFrames = Math.max(_deferFrames, DEFER_FRAMES);
    };
  }

  // --- フレームごとにクールダウン消化＆準備OKならキュー実行 ---
  const _Spriteset_Base_update = Spriteset_Base.prototype.update;
  Spriteset_Base.prototype.update = function() {
    _Spriteset_Base_update.call(this);
    if (_deferFrames > 0) {
      _deferFrames--;
      if (_deferFrames === 0 && rendererReady()) flushQueue();
    }
  };

  // --- PluginManager.registerCommand をフックして、Nino_HazeToggle のコマンドを自動遅延 ---
  const _origRegister = PluginManager.registerCommand;
  PluginManager.registerCommand = function(pluginName, commandName, func) {
    if (String(pluginName) === "Nino_HazeToggle") {
      // HazeOn/HazeOff/HazeStrength/HazeSpeed/HazeAffectPictures/HazePulse 全部を対象
      const wrapped = function(args) {
        callOrDefer(() => func.call(this, args));
      };
      _origRegister.call(this, pluginName, commandName, wrapped);
    } else {
      _origRegister.call(this, pluginName, commandName, func);
    }
  };

  // --- すでに定義済みの window.* 呼び出しも安全に包む（後読み/先読みどちらでも有効） ---
  function wrapGlobal(name) {
    const g = window[name];
    if (typeof g === "function" && !g._ninoWrapped) {
      const orig = g;
      const wrapped = function(...a) { callOrDefer(() => orig.apply(this, a)); };
      wrapped._ninoWrapped = true;
      window[name] = wrapped;
    }
  }

  // 代表的な公開APIをラップ（存在するものだけ巻き取る）
  const names = ["HazeOn","HazeOff","HazeStrength","HazeSpeed","HazeAffectPictures","HazePulse"];
  // 1) いま存在するなら即ラップ
  names.forEach(wrapGlobal);
  // 2) 後からロードされるケースに備え、少しの間ポーリングでラップ（軽量）
  let _poll = 60; // 1秒程度（60fps想定）
  const _int = setInterval(() => {
    names.forEach(wrapGlobal);
    if (--_poll <= 0) clearInterval(_int);
  }, 16);

  // --- デバッグ用（必要ならコメントアウト） ---
  // window._NinoHazeDeferInfo = () => ({deferFrames:_deferFrames, queued:_queue.length});
})();
