/*:
 * @target MZ
 * @plugindesc 画像不要の“もや揺らぎ”トグル＆パルス v1.3.1（メニュー遷移安定化）
 * @author nino
 * @help
 * v1.3 からの変更点：
 *  - メニュー開閉などシーン遷移中の null/invalid テクスチャを検知して自動再生成
 *  - Spriteset 破棄時にフィルタ・参照を確実にクリーンアップ
 *
 * 既存の使い方（HazeOn/Off/Strength/Speed/AffectPictures/HazePulse）はそのまま使えます。
 */

(() => {
  const pluginName = "Nino_HazeToggle";
  const KEY = "_ninoHazeState";
  const defaults = () => ({
    enabled: false,
    strength: 8,
    speed: 0.6,
    affectPictures: false,
    pulseActive: false,
    pulseTotal: 0,
    pulseTime: 0,
    pulseStrength: 8,
    pulseAffectPictures: false
  });
  const S = () => { if (!$gameSystem[KEY]) $gameSystem[KEY] = defaults(); return $gameSystem[KEY]; };

  // ---- ノイズテクスチャ生成（画像不要）----
  let _sharedNoiseTex = null;
  function makeNoiseTexture() {
    const size = 128;
    const cvs = document.createElement("canvas");
    cvs.width = cvs.height = size;
    const ctx = cvs.getContext("2d");
    const img = ctx.createImageData(size, size);
    const d = img.data, f1 = 0.15, f2 = 0.07;
    for (let y = 0; y < size; y++) for (let x = 0; x < size; x++) {
      const i = (y * size + x) * 4;
      const nx = Math.sin(x * f1) * 0.5 + Math.cos(y * f2) * 0.5;
      const ny = Math.cos(y * f1) * 0.5 + Math.sin(x * f2) * 0.5;
      d[i+0] = 128 + Math.floor(nx * 127);
      d[i+1] = 128 + Math.floor(ny * 127);
      d[i+2] = 128; d[i+3] = 255;
    }
    ctx.putImageData(img, 0, 0);
    const tex = PIXI.Texture.from(cvs);
    tex.baseTexture.wrapMode = PIXI.WRAP_MODES.REPEAT;
    return tex;
  }

  // ---- 安全性チェック（invalid時は再生成）----
  function ensureSharedTextureValid() {
    if (!_sharedNoiseTex) { _sharedNoiseTex = makeNoiseTexture(); return; }
    const bt = _sharedNoiseTex.baseTexture;
    if (!bt || bt.destroyed || !bt.valid) {
      _sharedNoiseTex.destroy(true);
      _sharedNoiseTex = makeNoiseTexture();
    }
  }
  function isTexValid(tex) {
    try { return !!(tex && tex.baseTexture && tex.baseTexture.valid && !tex.baseTexture.destroyed); }
    catch { return false; }
  }

  function ensureArtifacts(spriteset) {
    if (!spriteset) return null;
    ensureSharedTextureValid();

    // TilingSprite
    if (!spriteset._ninoHazeMap) {
      const tiler = new PIXI.TilingSprite(_sharedNoiseTex, Graphics.width, Graphics.height);
      tiler.alpha = 0; tiler.tileScale.set(1,1);
      spriteset._ninoHazeMap = tiler;
      spriteset.addChild(tiler);
    } else if (!isTexValid(spriteset._ninoHazeMap.texture)) {
      // テクスチャが死んでいたら差し替え
      spriteset._ninoHazeMap.texture = _sharedNoiseTex;
    }

    // Filter
    if (!spriteset._ninoHazeFilter) {
      spriteset._ninoHazeFilter = new PIXI.filters.DisplacementFilter(spriteset._ninoHazeMap);
      spriteset._ninoHazeFilter.scale.set(0,0);
    } else if (spriteset._ninoHazeFilter.maskSprite !== spriteset._ninoHazeMap) {
      // マップが作り直された場合に紐付け直す
      spriteset._ninoHazeFilter.maskSprite = spriteset._ninoHazeMap;
    }

    return { map: spriteset._ninoHazeMap, filter: spriteset._ninoHazeFilter };
  }

  const targetNode = (spriteset, affectPictures) =>
    (!affectPictures && spriteset._baseSprite) ? spriteset._baseSprite : spriteset;

  const attachFilter = (node, filter) => {
    if (!node) return;
    const arr = node.filters ? node.filters.slice() : [];
    if (!arr.includes(filter)) { arr.push(filter); node.filters = arr; }
  };
  const detachFilter = (node, filter) => {
    if (!node || !node.filters) return;
    const keep = node.filters.filter(f => f !== filter);
    node.filters = keep.length ? keep : null;
  };

  function syncOnce(spriteset) {
    if (!spriteset) return;
    const st = S();
    const art = ensureArtifacts(spriteset);
    if (!art) return;

    // まず両候補から外す（適用先切替の残留対策）
    detachFilter(spriteset, art.filter);
    if (spriteset._baseSprite) detachFilter(spriteset._baseSprite, art.filter);

    const active = st.enabled || st.pulseActive;
    if (active) {
      const affect = st.pulseActive ? st.pulseAffectPictures : st.affectPictures;
      attachFilter(targetNode(spriteset, affect), art.filter);
      // 強度は update 側で毎フレ反映
    } else {
      art.filter.scale.set(0,0);
    }
  }

  function tickUpdate(spriteset) {
    if (!spriteset) return;
    const map = spriteset._ninoHazeMap, filter = spriteset._ninoHazeFilter;
    if (!map || !filter) return;

    // 破棄や invalid を検知したら安全に抜ける＆次フレで再構築
    if (!isTexValid(map.texture)) return;

    const st = S();
    // パルス
    let scale;
    if (st.pulseActive && st.pulseTotal > 0) {
      st.pulseTime++;
      const p = Math.min(1, st.pulseTime / st.pulseTotal); // 0→1
      const up = p <= 0.5
        ? (0.5 * (1 - Math.cos(Math.PI * (p * 2))))         // 上昇
        : (0.5 * (1 - Math.cos(Math.PI * (2 - p * 2))));     // 下降
      scale = st.pulseStrength * up;

      map.tilePosition.x += Math.max(0.2, st.speed) * 1.2;
      map.tilePosition.y += Math.max(0.2, st.speed) * 0.8;

      if (st.pulseTime >= st.pulseTotal) {
        st.pulseActive = false; st.pulseTime = 0;
        if (!st.enabled) filter.scale.set(0,0);
        syncNow();
      }
    } else {
      scale = st.enabled ? Number(st.strength||0) : 0;
      map.tilePosition.x += st.speed || 0;
      map.tilePosition.y += (st.speed || 0) * 0.6;
    }

    const v = Math.max(0, Number(scale||0));
    filter.scale.set(v, v);
  }

  function syncNow() {
    const scene = SceneManager._scene;
    if (scene && scene._spriteset) syncOnce(scene._spriteset);
  }

  // ---- プラグインコマンド（v1.3 と同じ）----
  PluginManager.registerCommand(pluginName, "HazeOn", args => {
    const st = S();
    st.enabled = true;
    if (args.strength != null) st.strength = Math.max(0, Number(args.strength));
    if (args.speed != null)    st.speed    = Number(args.speed);
    if (args.affectPictures != null) st.affectPictures = String(args.affectPictures).toLowerCase() === "true";
    syncNow();
  });
  PluginManager.registerCommand(pluginName, "HazeOff", () => { const st=S(); st.enabled=false; syncNow(); });
  PluginManager.registerCommand(pluginName, "HazeStrength", args => { const st=S(); st.strength=Math.max(0,Number(args.strength ?? st.strength)); syncNow(); });
  PluginManager.registerCommand(pluginName, "HazeSpeed", args => { const st=S(); st.speed=Number(args.speed ?? st.speed); });
  PluginManager.registerCommand(pluginName, "HazeAffectPictures", args => {
    const st=S(); st.affectPictures = String(args.value ?? args.affectPictures ?? "false").toLowerCase() === "true"; syncNow();
  });
  PluginManager.registerCommand(pluginName, "HazePulse", args => {
    const dur = Math.max(1, Number(args.duration || 60));
    const str = Math.max(0, Number(args.strength || 8));
    const aff = String(args.affectPictures ?? "false").toLowerCase() === "true";
    const wait = String(args.wait ?? "false").toLowerCase() === "true";
    const st=S();
    st.pulseActive=true; st.pulseTotal=dur; st.pulseTime=0; st.pulseStrength=str; st.pulseAffectPictures=aff;
    syncNow();
    if (wait && $gameMap && $gameMap._interpreter) $gameMap._interpreter.wait(dur);
  });

  /* (コマンド定義は v1.3 と同一のため省略しても動作します。必要なら前版の @command ブロックをそのまま流用してください) */

  // ---- Spriteset ライフサイクル対応 ----
  const _Spriteset_Base_update = Spriteset_Base.prototype.update;
  Spriteset_Base.prototype.update = function() {
    _Spriteset_Base_update.call(this);
    try {
      syncOnce(this);
      const st = S();
      if (st.enabled || st.pulseActive) tickUpdate(this);
    } catch (e) {
      // 破棄タイミング等のレースを念のため握りつぶし（次フレで再構築）
      // console.warn("[Nino_HazeToggle] update skipped:", e);
    }
  };

  // 破棄時にフィルタと参照をクリア
  const _Spriteset_Base_destroy = Spriteset_Base.prototype.destroy;
  Spriteset_Base.prototype.destroy = function(options) {
    try {
      if (this._ninoHazeFilter) {
        detachFilter(this, this._ninoHazeFilter);
        if (this._baseSprite) detachFilter(this._baseSprite, this._ninoHazeFilter);
      }
      // 子は PIXI 側で破棄されるが、参照をクリアしておく
      this._ninoHazeFilter = null;
      this._ninoHazeMap = null;
    } finally {
      _Spriteset_Base_destroy.call(this, options);
    }
  };

  // Map/Battle 作成時に一度同期
  const _Scene_Map_createSpriteset = Scene_Map.prototype.createSpriteset;
  Scene_Map.prototype.createSpriteset = function() { _Scene_Map_createSpriteset.call(this); syncOnce(this._spriteset); };
  const _Scene_Battle_createSpriteset = Scene_Battle.prototype.createSpriteset;
  Scene_Battle.prototype.createSpriteset = function() { _Scene_Battle_createSpriteset.call(this); syncOnce(this._spriteset); };

  // スクリプト呼び出し用
  window.HazeOn  = (strength=8, speed=0.6, affectPictures=false)=>{ const st=S(); st.enabled=true; st.strength=Math.max(0,Number(strength)); st.speed=Number(speed); st.affectPictures=!!affectPictures; syncNow(); };
  window.HazeOff = ()=>{ S().enabled=false; syncNow(); };
  window.HazeStrength = (n)=>{ S().strength=Math.max(0,Number(n)); syncNow(); };
  window.HazeSpeed = (v)=>{ S().speed=Number(v); };
  window.HazeAffectPictures = (b)=>{ S().affectPictures=!!b; syncNow(); };
  window.HazePulse = (strength=8, duration=60, affectPictures=false, wait=false)=>{
    const st=S();
    st.pulseActive=true; st.pulseTotal=Math.max(1,Number(duration)); st.pulseTime=0;
    st.pulseStrength=Math.max(0,Number(strength)); st.pulseAffectPictures=!!affectPictures;
    syncNow();
    if (wait && $gameMap && $gameMap._interpreter) $gameMap._interpreter.wait(st.pulseTotal);
  };
})();
