/*:
 * @target MZ
 * @plugindesc Picture animation with variable-controlled speed, loop & switch (RPG Maker MZ)
 * @author nino
 *
 * @help Nino_PictureAnimPlus.js
 * 画像命名: base_01.png, base_02.png, ... （img/pictures/）
 *
 * プラグインコマンド:
 *  - StartAnim:  指定ピクチャに連番アニメを開始
 *  - SwitchAnim: 連番のベース名やフレーム数、間隔を切替
 *  - SetSpeedVar: 変数でフレーム間隔を動的に管理（min~maxmsに線形マップ）
 *  - StopAnim:   アニメを停止（現在フレームの画像で静止）
 *
 * 典型手順:
 *  1) Show Picture 1: 任意画像（最初はダミーでOK）
 *  2) StartAnim [1, ani, 8, 80, true]  // 80msごとに 1..8 をループ
 *  3) SetSpeedVar [1, 11, 40, 140]      // 変数11の値に応じて40~140msに可変
 *  4) SwitchAnim   でベース名や枚数を差し替え可能
 *
 * @command StartAnim
 * @text StartAnim
 * @desc 指定ピクチャで連番アニメ開始
 * @arg pictureId
 * @type number
 * @min 1
 * @desc ピクチャ番号
 * @arg baseName
 * @type string
 * @desc 連番のベース名（例: ani → ani_01.png）
 * @arg frames
 * @type number
 * @min 1
 * @desc フレーム数
 * @arg intervalMs
 * @type number
 * @min 1
 * @desc フレーム間隔(ミリ秒)
 * @arg loop
 * @type boolean
 * @default true
 * @desc ループするか
 *
 * @command SwitchAnim
 * @text SwitchAnim
 * @desc 再生中アニメのパラメータを差し替え
 * @arg pictureId
 * @type number
 * @min 1
 * @arg baseName
 * @type string
 * @arg frames
 * @type number
 * @min 1
 * @arg intervalMs
 * @type number
 * @min 1
 *
 * @command SetSpeedVar
 * @text SetSpeedVar
 * @desc 変数値を intervalMs にマップ（線形）
 * @arg pictureId
 * @type number
 * @min 1
 * @arg variableId
 * @type variable
 * @desc 監視する変数
 * @arg minMs
 * @type number
 * @min 1
 * @desc 最小間隔（ms）
 * @arg maxMs
 * @type number
 * @min 1
 * @desc 最大間隔（ms）
 *
 * @command StopAnim
 * @text StopAnim
 * @desc アニメ停止
 * @arg pictureId
 * @type number
 * @min 1
 */

(() => {
  const pluginName = "Nino_PictureAnimPlus";

  const pad2 = n => ("0" + n).slice(-2);

  // ---- Game_Picture 拡張: アニメ状態を保持 ----
  const _Game_Picture_initBasic = Game_Picture.prototype.initBasic;
  Game_Picture.prototype.initBasic = function() {
    _Game_Picture_initBasic.call(this);
    this._anim = null;        // {base, frames, intervalMs, loop, t, frame}
    this._speedVar = null;    // {varId, minMs, maxMs}
  };

  Game_Picture.prototype.startAnim = function(base, frames, intervalMs, loop) {
    this._anim = {
      base,
      frames: Math.max(1, frames|0),
      intervalMs: Math.max(1, intervalMs|0),
      loop: !!loop,
      t: 0,
      frame: 1
    };
    this._name = `${base}_${pad2(1)}`; // 即時反映用
  };

  Game_Picture.prototype.switchAnim = function(base, frames, intervalMs) {
    if (!this._anim) return;
    this._anim.base = base;
    this._anim.frames = Math.max(1, frames|0);
    this._anim.intervalMs = Math.max(1, intervalMs|0);
    // フレームは維持しつつ名前を更新
    const f = Math.min(this._anim.frame, this._anim.frames);
    this._name = `${this._anim.base}_${pad2(f)}`;
  };

  Game_Picture.prototype.setSpeedVar = function(varId, minMs, maxMs) {
    this._speedVar = { varId: Number(varId), minMs: Number(minMs), maxMs: Number(maxMs) };
  };

  Game_Picture.prototype.stopAnim = function() {
    this._anim = null;
    this._speedVar = null;
  };

  // Sprite_Pictureが毎フレーム呼ぶ更新処理
  Game_Picture.prototype.updateAnim = function(deltaMs) {
    const A = this._anim;
    if (!A) return;
    // 変数でスピード制御
    if (this._speedVar) {
      const v = $gameVariables.value(this._speedVar.varId) || 0;
      // v: 0..100 を想定（必要なら適宜調整）
      const t = Math.max(0, Math.min(100, Number(v)));
      const im = this._speedVar.minMs;
      const ax = this._speedVar.maxMs;
      A.intervalMs = Math.round(im + (ax - im) * (t / 100));
    }
    A.t += deltaMs;
    if (A.t >= A.intervalMs) {
      A.t -= A.intervalMs;
      A.frame++;
      if (A.frame > A.frames) {
        if (A.loop) A.frame = 1;
        else { A.frame = A.frames; this.stopAnim(); } // ループ無しなら最後で停止
      }
      if (this._anim) {
        this._name = `${A.base}_${pad2(A.frame)}`;
      }
    }
  };

  // ---- Sprite_Picture 更新にフック ----
  const _Sprite_Picture_update = Sprite_Picture.prototype.update;
  Sprite_Picture.prototype.update = function() {
    const beforeName = this._pictureName;
    // dtをmsで渡す（1frame=16.7ms相当だが、実測deltaの方が素直）
    const deltaMs = (Graphics.frameTime || (1000 / 60));
    if (this.picture()) this.picture().updateAnim(deltaMs);
    _Sprite_Picture_update.call(this);
    // 画像名が変わっていればBitmapを更新
    if (this.picture()) {
      const name = this.picture().name();
      if (name && name !== beforeName) {
        this.updateBitmap();
      }
    }
  };

  // ---- プラグインコマンド ----
  PluginManager.registerCommand(pluginName, "StartAnim", args => {
    const pid = Number(args.pictureId);
    const p = $gameScreen.picture(pid);
    if (!p) return;
    p.startAnim(String(args.baseName), Number(args.frames), Number(args.intervalMs), args.loop === "true");
  });

  PluginManager.registerCommand(pluginName, "SwitchAnim", args => {
    const pid = Number(args.pictureId);
    const p = $gameScreen.picture(pid);
    if (!p || !p._anim) return;
    p.switchAnim(String(args.baseName), Number(args.frames), Number(args.intervalMs));
  });

  PluginManager.registerCommand(pluginName, "SetSpeedVar", args => {
    const pid = Number(args.pictureId);
    const p = $gameScreen.picture(pid);
    if (!p) return;
    p.setSpeedVar(Number(args.variableId), Number(args.minMs), Number(args.maxMs));
  });

  PluginManager.registerCommand(pluginName, "StopAnim", args => {
    const pid = Number(args.pictureId);
    const p = $gameScreen.picture(pid);
    if (!p) return;
    p.stopAnim();
  });

})();