
/*:
 * @target MZ
 * @plugindesc 2人固定の恋愛パラメータ管理システム（数値管理＋スイッチ連動＋メニュー表示＋イラスト表示）
 * @author You
 *
 * @param MenuCommandName
 * @text メニューコマンド名
 * @type string
 * @default 恋愛パラメータ
 * @desc メニューに表示されるコマンド名
 *
 * @param ParamNames
 * @text パラメータ名リスト
 * @type string[]
 * @default ["親密度","信頼度","愛情度","嫉妬度","協力度","依存度","理解度","尊敬度","安心感","緊張感"]
 * @desc 固定で表示される10個のパラメータ名
 *
 * @param Actor1Id
 * @text アクター1ID
 * @type actor
 * @default 1
 * @desc 主人公のアクターID
 *
 * @param Actor2Id
 * @text アクター2ID
 * @type actor
 * @default 2
 * @desc ヒロインのアクターID
 *
 * @param Actor1Image
 * @text アクター1イラスト
 * @type file
 * @dir img/pictures
 * @default
 * @desc 主人公の立ち絵画像（img/pictures 内）
 *
 * @param Actor2Image
 * @text アクター2イラスト
 * @type file
 * @dir img/pictures
 * @default
 * @desc ヒロインの立ち絵画像（img/pictures 内）
 *
 * @param Thresholds
 * @text しきい値設定
 * @type struct<Threshold>[]
 * @default []
 * @desc パラメータごとの条件とスイッチ連動設定
 *
 * @command addParam
 * @text パラメータを増減
 * @desc 指定アクターのパラメータを増減する
 *
 * @arg actorId
 * @text アクターID
 * @type actor
 * @default 1
 *
 * @arg paramIndex
 * @text パラメータ番号（0～9）
 * @type number
 * @min 0
 * @max 9
 * @default 0
 *
 * @arg value
 * @text 増減値
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 */
/*~struct~Threshold:
 * @param paramIndex
 * @text パラメータ番号（0～9）
 * @type number
 * @min 0
 * @max 9
 *
 * @param value
 * @text しきい値
 * @type number
 *
 * @param switchId
 * @text 連動スイッチID
 * @type switch
 */

(() => {
    "use strict";

    // プラグイン名の取得を安全に（currentScriptがnullでも動くフォールバック）
    const pluginName =
        (document.currentScript &&
            decodeURIComponent(document.currentScript.src).match(/([^\/]+)\.js$/)?.[1]) ||
        "LoveParams";

    const P = PluginManager.parameters(pluginName);

    const MenuCommandName = String(P["MenuCommandName"] ?? "恋愛パラメータ");
    const ParamNames = JSON.parse(P["ParamNames"] || "[]").map(String);
    const Actor1Id = Number(P["Actor1Id"] ?? 1);
    const Actor2Id = Number(P["Actor2Id"] ?? 2);
    const Actor1Image = String(P["Actor1Image"] || "");
    const Actor2Image = String(P["Actor2Image"] || "");
    const Thresholds = JSON.parse(P["Thresholds"] || "[]").map(t => JSON.parse(t));

    // -------------------------
    // 恋愛パラメータ管理
    // -------------------------
    Game_Actor.prototype.initLoveParams = function () {
        if (!this._loveParams) {
            this._loveParams = new Array(10).fill(0);
        }
    };

    Game_Actor.prototype.loveParam = function (index) {
        this.initLoveParams();
        const i = Number(index) | 0;
        return this._loveParams[i] || 0;
    };

    Game_Actor.prototype.addLoveParam = function (index, value) {
        this.initLoveParams();
        const i = Number(index) | 0;
        const v = Number(value) || 0;
        if (i >= 0 && i < 10) {
            this._loveParams[i] = (this._loveParams[i] || 0) + v;
        }
    };

    // -------------------------
    // プラグインコマンド
    // -------------------------
    PluginManager.registerCommand(pluginName, "addParam", args => {
        const actorId = Number(args.actorId);
        const paramIndex = Number(args.paramIndex);
        const value = Number(args.value);
        const actor = $gameActors.actor(actorId);
        if (actor) {
            actor.addLoveParam(paramIndex, value);
        }
    });

    // -------------------------
    // しきい値チェック（常時監視）
    // -------------------------
    function checkThresholds() {
        for (const t of Thresholds) {
            const paramIndex = Number(t.paramIndex);
            const value = Number(t.value);
            const switchId = Number(t.switchId);
            for (const actorId of [Actor1Id, Actor2Id]) {
                const actor = $gameActors.actor(actorId);
                if (actor && actor.loveParam(paramIndex) >= value) {
                    $gameSwitches.setValue(switchId, true);
                }
            }
        }
    }

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function () {
        _Scene_Map_update.call(this);
        checkThresholds();
    };

    const _Scene_Battle_update = Scene_Battle.prototype.update;
    Scene_Battle.prototype.update = function () {
        _Scene_Battle_update.call(this);
        checkThresholds();
    };

    // -------------------------
    // メニューコマンド追加
    // -------------------------
    const _Window_MenuCommand_addOriginalCommands =
        Window_MenuCommand.prototype.addOriginalCommands;
    Window_MenuCommand.prototype.addOriginalCommands = function () {
        _Window_MenuCommand_addOriginalCommands.call(this);
        this.addCommand(MenuCommandName, "loveParams");
    };

    const _Scene_Menu_createCommandWindow = Scene_Menu.prototype.createCommandWindow;
    Scene_Menu.prototype.createCommandWindow = function () {
        _Scene_Menu_createCommandWindow.call(this);
        this._commandWindow.setHandler("loveParams", () => {
            SceneManager.push(Scene_LoveMenu);
        });
    };

    // -------------------------
    // 恋愛パラメータ選択画面
    // -------------------------
    class Scene_LoveMenu extends Scene_MenuBase {
        create() {
            super.create();
            this.createPictures();
            this.createActorWindow();
        }

        createPictures() {
            // 読み込み完了後に座標を決める（原寸表示）
            if (Actor1Image) {
                const bmp1 = ImageManager.loadPicture(Actor1Image);
                this._spriteA1 = new Sprite(bmp1);
                bmp1.addLoadListener(() => {
                    this._spriteA1.x = Graphics.boxWidth / 2 - 400;
                    this._spriteA1.y = 50;
                    this._spriteA1.scale.x = 1.5;   // ★ 横を1.5倍に拡大
                    this._spriteA1.scale.y = 1.5;   // ★ 縦を1.5倍に拡大
                });
                this.addChild(this._spriteA1); }
            if (Actor2Image) {
                const bmp2 = ImageManager.loadPicture(Actor2Image);
                this._spriteA2 = new Sprite(bmp2);
                bmp2.addLoadListener(() => {
                    this._spriteA2.x = Graphics.boxWidth / 2 + 100;
                    this._spriteA2.y = 50;
                    this._spriteA2.scale.x = 1.5;   // ★ 横を1.5倍に拡大
                    this._spriteA2.scale.y = 1.5;   // ★ 縦を1.5倍に拡大

                });
                this.addChild(this._spriteA2);
            }
        }

        createActorWindow() {
            const rect = this.actorWindowRect();
            this._actorWindow = new Window_LoveActorSelect(rect);
            this._actorWindow.setHandler("ok", this.onActorOk.bind(this));
            this._actorWindow.setHandler("cancel", this.popScene.bind(this));
            this.addWindow(this._actorWindow);
        }

        actorWindowRect() {
            const ww = Graphics.boxWidth;
            const wh = 100;
            const wx = 0;
            const wy = Graphics.boxHeight - wh;
            return new Rectangle(wx, wy, ww, wh);
        }

        onActorOk() {
            const actor = this._actorWindow.actor();
            SceneManager.push(Scene_LoveParamDetail);
            // ★ 正しい引数受け渡し：次シーンの prepare(...) を呼んでくれる
            SceneManager.prepareNextScene(actor);
        }
    }

    class Window_LoveActorSelect extends Window_Command {
        initialize(rect) {
            super.initialize(rect);
            this.select(0);
        }

        makeCommandList() {
            const actor1 = $gameActors.actor(Actor1Id);
            const actor2 = $gameActors.actor(Actor2Id);
            if (actor1) this.addCommand(actor1.name(), "actor1");
            if (actor2) this.addCommand(actor2.name(), "actor2");
        }

        actor() {
            return this.currentSymbol() === "actor1"
                ? $gameActors.actor(Actor1Id)
                : $gameActors.actor(Actor2Id);
        }
    }

   
 // -------------------------
// 恋愛パラメータ詳細画面
// -------------------------
class Scene_LoveParamDetail extends Scene_MenuBase {
    prepare(actor) {
        this._actor = actor;
    }

    create() {
        super.create();
        this.createPicture();
        this.createParamWindow();
        this.createBackButton();   // ★ 戻るボタン追加
    }

    createPicture() {
        let filename = "";
        if (this._actor && this._actor.actorId() === Actor1Id && Actor1Image) {
            filename = Actor1Image;
        } else if (this._actor && this._actor.actorId() === Actor2Id && Actor2Image) {
            filename = Actor2Image;
        }
        if (filename) {
            const bmp = ImageManager.loadPicture(filename);
            this._sprite = new Sprite(bmp);
            bmp.addLoadListener(() => {
                this._sprite.x = (Graphics.boxWidth - this._sprite.width) / 2;
                this._sprite.y = 50;
            });
            this.addChild(this._sprite);
        }
    }

    createParamWindow() {
        const rect = this.paramWindowRect();
        this._paramWindow = new Window_LoveParamList(rect, this._actor);
        this.addWindow(this._paramWindow);
    }

    paramWindowRect() {
        const ww = Graphics.boxWidth;
        const wh = Math.max(100, Graphics.boxHeight - 400);
        const wx = 0;
        const wy = Graphics.boxHeight - wh - 60; // ★ 戻るボタンのスペース
        return new Rectangle(wx, wy, ww, wh);
    }

    // ★ 戻るボタンを作成
    createBackButton() {
        const rect = this.backButtonRect();
        this._backButton = new Window_LoveBackCommand(rect);
        this._backButton.setHandler("cancel", this.popScene.bind(this));
        this.addWindow(this._backButton);
        this._backButton.activate();
    }

    backButtonRect() {
        const ww = 200;
        const wh = 60;
        const wx = (Graphics.boxWidth - ww) / 2;
        const wy = Graphics.boxHeight - wh;
        return new Rectangle(wx, wy, ww, wh);
    }
}

// ★ 戻るボタン用のコマンドウィンドウ
class Window_LoveBackCommand extends Window_HorzCommand {
    maxCols() {
        return 1;
    }

    makeCommandList() {
        this.addCommand("戻る", "cancel");
    }
}
   
    class Window_LoveParamList extends Window_Selectable {
        initialize(rect, actor) {
            this._actor = actor;
            super.initialize(rect);
            this.refresh();
        }

        maxItems() {
            return ParamNames.length;
        }

        drawItem(index) {
            const name = ParamNames[index] || ("Param" + index);
            const value = this._actor ? this._actor.loveParam(index) : 0;
            const r = this.itemLineRect(index);
            this.drawText(name, r.x, r.y, 200);
            this.drawText(String(value), r.x + 220, r.y, 60, "right");
        }
    }

    // （！）元コードにあった SceneManager.prepareNextScene/goto の上書きは不要＆危険なので削除
})();
