//=============================================================================
// 100CursorControl.js
//=============================================================================

/*:
 * @plugindesc 
 *
 * @param Rules
 * @text カーソル切替ルール
 * @type struct<CursorRule>[]
 * @default []
 * @desc 条件を満たしたときにマウスカーソルを変更するルールを定義します。
 *
 * @target MZ
 * @plugindesc 保存データ
 * @help
 * 
 * 
 */

/*~struct~CursorRule:
 * @param SwitchId
 * @text 条件スイッチ
 * @type switch
 * @default 0
 * @desc このスイッチが ON なら変更
 *
 * @param VariableId
 * @text 条件変数
 * @type variable
 * @default 0
 * @desc この変数を条件判定がtrueなら変更
 *
 * @param Compare
 * @text 変数判定方法
 * @type select
 * @option 無視
 * @value none
 * @option = 指定値と等しい
 * @value equal
 * @option >= 指定値以上
 * @value greaterEqual
 * @option <= 指定値以下
 * @value lessEqual
 * @default none
 *
 * @param Value
 * @text 判定値
 * @type number
 * @default 0
 * @desc 変数判定に使う値
 *
 * @param CursorImage
 * @text カーソル画像
 * @desc img/pictures/ に置いたファイル
 * @default mouse
 */

(() => {
    const pluginName = "100CursorControl";
    const parameters = PluginManager.parameters(pluginName);
    const rules = JSON.parse(parameters["Rules"] || "[]").map(r => JSON.parse(r));

    const updateCursor = () => {
        let applied = false;

        for (const rule of rules) {
            let ok = true;

            const switchId = Number(rule.SwitchId || 0);
            if (switchId > 0 && !$gameSwitches.value(switchId)) {
                ok = false;
            }

            const variableId = Number(rule.VariableId || 0);
            const compare = rule.Compare || "none";
            const value = Number(rule.Value || 0);
            if (variableId > 0 && compare !== "none") {
                const v = $gameVariables.value(variableId);
                if (compare === "equal" && v !== value) ok = false;
                if (compare === "greaterEqual" && v < value) ok = false;
                if (compare === "lessEqual" && v > value) ok = false;
            }

            if (ok) {
                const path = `img/pictures/${rule.CursorImage}.png`;
                document.body.style.cursor = `url(${path}) 0 0, default`;
                applied = true;
                break;
            }
        }

        if (!applied) {
            document.body.style.cursor = "default";
        }
    };

    const _SceneManager_updateScene = SceneManager.updateScene;
    SceneManager.updateScene = function() {
        _SceneManager_updateScene.apply(this, arguments);
        if (this._scene && $gameSwitches) {
            updateCursor();
        }
    };
})();
