//=============================================================================
// PresetTextFile.js
//=============================================================================

/*:
 * @target MZ
 * @plugindesc 外部テキスト読み込み + プリセット対応 Ver1.0.0
 * @author You
 *
 * @param PresetList
 * @type struct<Preset>[]
 * @default []
 * @desc プリセット設定リスト
 *
 * @param ConditionPrefixList
 * @type struct<ConditionPrefix>[]
 * @default []
 * @desc 名前に応じてテキストの頭に追加するリスト
 *
 * @param GlobalPrefixList
 * @type struct<GlobalPrefix>[]
 * @default []
 * @desc すべてのテキストの頭に追加するリスト
 *
 * @help
 * PresetTextFile.command("ファイル名");
 *
 * 外部テキスト例:
 *   :1:こんにちは       ← プリセットID=1適用
 *   \N[1]:2:やあ        ← 名前が\N[1]なら条件追加してプリセット2適用
 */

/*~struct~Preset:
 * @param Id
 * @type number
 *
 * @param Align
 * @type select
 * @option 左
 * @value 0
 * @option 中央
 * @value 1
 * @option 右
 * @value 2
 * @default 0
 *
 * @param FontSize
 * @type number
 * @default 28
 *
 * @param Color
 * @type string
 * @default #ffffff
 *
 * @param OutlineColor
 * @type string
 * @default #000000
 *
 * @param OutlineSize
 * @type number
 * @default 4
 *
 * @param WindowSkin
 * @type file
 * @dir img/system
 *
 * @param WindowOpacity
 * @type number
 * @max 255
 * @default 255
 *
 * @param TailImage
 * @type file
 * @dir img/system
 *
 * @param TailPos
 * @type select
 * @option 右上
 * @value 0
 * @option 右下
 * @value 1
 * @option 中央上
 * @value 2
 * @option 中央下
 * @value 3
 * @option 左上
 * @value 4
 * @option 左下
 * @value 5
 * @default 2
 *
 * @param TailOffsetX
 * @type number
 * @default 0
 *
 * @param TailOffsetY
 * @type number
 * @default 0
 *
 * @param WaitInput
 * @type boolean
 * @default true
 */

/*~struct~ConditionPrefix:
 * @param Name
 * @type string
 * @param Prefix
 * @type string
 */

/*~struct~GlobalPrefix:
 * @param Prefix
 * @type string
 */

(() => {
    const pluginName = "PresetTextFile";
    const parameters = PluginManager.parameters(pluginName);

    const presetList = JSON.parse(parameters["PresetList"] || "[]").map(e => JSON.parse(e));
    const conditionPrefixList = JSON.parse(parameters["ConditionPrefixList"] || "[]").map(e => JSON.parse(e));
    const globalPrefixList = JSON.parse(parameters["GlobalPrefixList"] || "[]").map(e => JSON.parse(e));

    function findPreset(id) {
        return presetList.find(p => Number(p.Id) === Number(id));
    }

    //====================================================
    // Sequential Processor (NrTextFile準拠 + プリセット拡張)
    //====================================================
    class PresetSequentialTextProcessor {
        constructor(lines) {
            this._lines = lines;
            this._index = 0;
            this._bg = 0;
            this._pos = 2;
        }

        async start() { await this.nextLine(); }

        async nextLine() {
            if (this._index >= this._lines.length) return;
            const line = this._lines[this._index++].trim();
            if (!line) return this.nextLine();
            if (line.startsWith('@')) {
                await this.processCommand(line);
            } else {
                await this.showMessage(line);
            }
            await this.nextLine();
        }

        async showMessage(line) {
            let name = null, presetId = null, text = "";

            let m = line.match(/^([^:]+):(\d+):(.*)$/);
            if (m) { name = m[1]; presetId = Number(m[2]); text = m[3]; }
            else if ((m = line.match(/^:(\d+):(.*)$/))) { presetId = Number(m[1]); text = m[2]; }
            else { text = line; }

            // 条件付きプレフィックス
            if (name) {
                const cond = conditionPrefixList.find(c => c.Name === name);
                if (cond) text = cond.Prefix + text;
            }

            // グローバルプレフィックス
            globalPrefixList.forEach(p => text = p.Prefix + text);

            // プリセット適用
            const preset = findPreset(presetId);
            if (preset) this.applyPreset(preset);

            // メッセージ表示
            $gameMessage.clear();
            if (name) $gameMessage.setSpeakerName(name);
            $gameMessage.add(text);
            $gameMessage.setBackground(this._bg);
            $gameMessage.setPositionType(this._pos);
                // デバッグ: テキスト内容確認
                console.log("[PresetSequentialTextProcessor] showMessage text:", text);
                if (!text || text.trim() === "") {
                    console.warn("[PresetSequentialTextProcessor] テキストが空です。プリセットID:", presetId, "name:", name);
                }

            await new Promise(resolve => {
                const check = () => {
                    if (!$gameMessage.isBusy()) resolve();
                    else setTimeout(check, 30);
                };
                check();
            });
        }

applyPreset(p) {
    const win = SceneManager._scene._messageWindow;
    if (!win) return;

    // フォント
    win.contents.fontSize = Number(p.FontSize || 28);
    win.contents.outlineWidth = Number(p.OutlineSize || 4);
    win.contents.outlineColor = p.OutlineColor || "#000000";
    win.contents.textColor = p.Color || "#ffffff";

    // ウィンドウ
    win.opacity = Number(p.WindowOpacity || 255);
    if (p.WindowSkin) {
        win.windowskin = ImageManager.loadSystem(p.WindowSkin);
    }

    // 既存のテール削除
    if (win._presetTailSprite) {
        win.removeChild(win._presetTailSprite);
        win._presetTailSprite = null;
    }

    // テール画像設定
    if (p.TailImage) {
        const sprite = new Sprite(ImageManager.loadSystem(p.TailImage));
        sprite.anchor.x = 0.5;
        sprite.anchor.y = 0.5;

        // 基準位置
        let x = win.width / 2;
        let y = win.height / 2;

        switch (Number(p.TailPos)) {
            case 0: x = win.width; y = 0; break;   // 右上
            case 1: x = win.width; y = win.height; break; // 右下
            case 2: x = win.width / 2; y = 0; break;      // 中央上
            case 3: x = win.width / 2; y = win.height; break; // 中央下
            case 4: x = 0; y = 0; break;          // 左上
            case 5: x = 0; y = win.height; break; // 左下
        }

        // 補正
        x += Number(p.TailOffsetX || 0);
        y += Number(p.TailOffsetY || 0);

        sprite.x = x;
        sprite.y = y;

        win.addChild(sprite);
        win._presetTailSprite = sprite;
    }
}


        //====================================================
        // @コマンド処理 (NrTextFileから完全移植)
        //====================================================
        async processCommand(line) {
            const parts = line.trim().split(/\s+/);
            const cmd = parts[0].toLowerCase();
            try {
                switch (cmd) {
                    case '@wait': {
                        const dur = Number(parts[1] || 60);
                        await this.sleep(dur);
                        break;
                    }
                    case '@bg': { this._bg = Number(parts[1] || 0); break; }
                    case '@pos': { this._pos = Number(parts[1] || 2); break; }
                    case '@fadeout':
                    case '@fadein': {
                        const dur = Number(parts[1] || 30);
                        const fn = cmd === '@fadeout' ? 'startFadeOut' : 'startFadeIn';
                        $gameScreen[fn](dur);
                        await this.sleep(dur);
                        break;
                    }
                    case '@bgm':
                    case '@bgs':
                    case '@me':
                    case '@se': {
                        const [_, name, vol, pitch, pan] = parts;
                        const sound = {
                            name: name || "",
                            volume: Number(vol || 90),
                            pitch: Number(pitch || 100),
                            pan: Number(pan || 0)
                        };
                        if (cmd === '@bgm') AudioManager.playBgm(sound);
                        if (cmd === '@bgs') AudioManager.playBgs(sound);
                        if (cmd === '@me') AudioManager.playMe(sound);
                        if (cmd === '@se') AudioManager.playSe(sound);
                        break;
                    }
                    case '@fadebgm': AudioManager.fadeOutBgm(Number(parts[1] || 30)); break;
                    case '@fadebgs': AudioManager.fadeOutBgs(Number(parts[1] || 30)); break;
                    case '@stopbgm': AudioManager.stopBgm(); break;
                    case '@stopbgs': AudioManager.stopBgs(); break;
                    case '@stopme': AudioManager.stopMe(); break;
                    case '@stopse': AudioManager.stopSe(); break;
                    case '@var': {
                        const varId = Number(parts[1]);
                        const op = Number(parts[2]);
                        const valType = parts[3];
                        const raw = parts.slice(4).join(" ");
                        let value = 0;
                        switch (valType) {
                            case '0': value = Number(raw); break;
                            case '1': value = $gameVariables.value(Number(raw)); break;
                            case '2': {
                                const [min, max] = raw.split('-').map(Number);
                                value = Math.floor(Math.random() * (max - min + 1)) + min;
                                break;
                            }
                            case '3': value = eval(raw); break;
                        }
                        const cur = $gameVariables.value(varId);
                        const res = op === 1 ? cur + value : op === 2 ? cur - value : value;
                        $gameVariables.setValue(varId, res);
                        break;
                    }
                    case '@switch':
                    case '@sw': {
                        const switchId = Number(parts[1]);
                        const val = parts[2]?.toLowerCase() === 'true' || parts[2] === '1';
                        $gameSwitches.setValue(switchId, val);
                        break;
                    }
                    case '@load': {
                        const nextFile = parts[1];
                        if (!nextFile) break;
                        const text = await fetch(`data/${nextFile.replace(/\.txt$/i, "")}.txt`).then(r => r.text());
                        const lines = text.split(/\r?\n/).map(s => s.replace(/^\uFEFF/, ""));
                        const subProcessor = new PresetSequentialTextProcessor(lines);
                        await subProcessor.start();
                        break;
                    }
                    case '@script':
                    case '@sc': {
                        const lines = [];
                        while (this._index < this._lines.length) {
                            const next = this._lines[this._index++].trim();
                            if (next.toLowerCase() === '@end') break;
                            lines.push(next);
                        }
                        const code = lines.join("\n");
                        try { eval(code); } catch (e) { console.error("[PresetTextFile] @script エラー:", e); }
                        break;
                    }
                    case '@fadeoutc': {
                        const r = Number(parts[1] || 0);
                        const g = Number(parts[2] || 0);
                        const b = Number(parts[3] || 0);
                        const dur = Number(parts[4] || 30);
                        const scene = SceneManager._scene;
                        if (scene) {
                            if (scene._presetFadeSprite) {
                                try { scene.removeChild(scene._presetFadeSprite); } catch (e) {}
                                scene._presetFadeSprite = null;
                            }
                            const fadeSprite = new PIXI.Graphics();
                            fadeSprite.beginFill(PIXI.utils.rgb2hex([
                                Math.max(0, Math.min(1, r / 255)),
                                Math.max(0, Math.min(1, g / 255)),
                                Math.max(0, Math.min(1, b / 255))
                            ]));
                            fadeSprite.drawRect(0, 0, Graphics.width, Graphics.height);
                            fadeSprite.endFill();
                            fadeSprite.alpha = 0;
                            scene.addChild(fadeSprite);
                            scene._presetFadeSprite = fadeSprite;
                            let count = 0;
                            await new Promise(resolve => {
                                const step = () => {
                                    count++;
                                    fadeSprite.alpha = Math.min(1, count / Math.max(1, dur));
                                    if (count >= dur) resolve();
                                    else requestAnimationFrame(step);
                                };
                                step();
                            });
                        } else await this.sleep(dur);
                        break;
                    }
                    case '@fadeinc': {
                        const dur = Number(parts[4] || 30);
                        const scene = SceneManager._scene;
                        if (scene && scene._presetFadeSprite) {
                            const fadeSprite = scene._presetFadeSprite;
                            let count = 0;
                            await new Promise(resolve => {
                                const step = () => {
                                    count++;
                                    fadeSprite.alpha = Math.max(0, 1 - (count / Math.max(1, dur)));
                                    if (count >= dur) {
                                        try { scene.removeChild(fadeSprite); } catch (e) {}
                                        scene._presetFadeSprite = null;
                                        resolve();
                                    } else requestAnimationFrame(step);
                                };
                                step();
                            });
                        } else await this.sleep(dur);
                        break;
                    }
                    case '@shake': {
                        const dir = Number(parts[1] || 0);
                        const power = Number(parts[2] || 5);
                        const speed = Number(parts[3] || 5);
                        const duration = Number(parts[4] || 60);
                        if (dir === 0) {
                            $gameScreen.startShake(power, speed, duration);
                        } else {
                            const spriteset = SceneManager._scene._spriteset;
                            if (spriteset) {
                                let count = 0;
                                let frame = 0;
                                const originY = spriteset.y;
                                const interval = setInterval(() => {
                                    const offset = Math.round(Math.sin(frame / speed * Math.PI) * power * 2);
                                    spriteset.y = originY + offset;
                                    frame++;
                                    if (++count >= duration) {
                                        clearInterval(interval);
                                        spriteset.y = originY;
                                    }
                                }, 1000 / 60);
                            }
                        }
                        await this.sleep(duration);
                        break;
                    }
                    case '@flash': {
                        const r = Number(parts[1] || 255);
                        const g = Number(parts[2] || 255);
                        const b = Number(parts[3] || 255);
                        const a = Number(parts[4] || 160);
                        const dur = Number(parts[5] || 60);
                        const wait = (parts[6]?.toLowerCase() === 'true');
                        $gameScreen.startFlash([r, g, b, a], dur);
                        if (wait) await this.sleep(dur);
                        break;
                    }
                    case '@if': {
                        const expr = parts.slice(1).join(" ");
                        let result = false;
                        try { result = !!eval(expr); } catch (e) { console.error("[PresetTextFile] @if 式エラー:", expr, e); }
                        if (!result) {
                            while (this._index < this._lines.length) {
                                const peek = this._lines[this._index].trim().toLowerCase();
                                if (peek.startsWith("@else") || peek.startsWith("@endif")) break;
                                this._index++;
                            }
                        }
                        break;
                    }
                    case '@else': {
                        while (this._index < this._lines.length) {
                            const peek = this._lines[this._index].trim().toLowerCase();
                            if (peek.startsWith("@endif")) break;
                            this._index++;
                        }
                        break;
                    }
                    case '@endif': break;
                    default:
                        console.warn("[PresetTextFile] 未知のコマンド:", cmd);
                        break;
                }
            } catch (err) {
                console.error("[PresetTextFile] コマンド処理中エラー:", cmd, err);
            }
        }

        sleep(ms) { return new Promise(r => setTimeout(r, ms * 1000 / 60)); }
    }

    //====================================================
    // グローバル管理
    //====================================================
    window.PresetTextFile = {
        async start(fileName) {
            const path = `data/text/${fileName.replace(/\.txt$/i, "")}.txt`;
            const text = await fetch(path).then(r => r.text());
            const lines = text.split(/\r?\n/).map(s => s.replace(/^\uFEFF/, ""));
            const proc = new PresetSequentialTextProcessor(lines);
            await proc.start();
        },
        command(fileName) {
            const interpreter = $gameMap._interpreter;
            if (interpreter) {
                interpreter.setWaitMode("preset_text");
                return this.start(fileName).then(() => interpreter.setWaitMode(""));
            } else return this.start(fileName);
        }
    };

    // Interpreter拡張
    const _updateWaitMode = Game_Interpreter.prototype.updateWaitMode;
    Game_Interpreter.prototype.updateWaitMode = function() {
        if (this._waitMode === "preset_text") return $gameMessage.isBusy();
        return _updateWaitMode.call(this);
    };
})();
