//=============================================================================
// DebugMonitor.js
// PUBLIC DOMAIN
// ----------------------------------------------------------------------------
// 2016/10/25 リアルタイム変更モードの起動キーをCtrlからTabに変更しました
// 2017/03/25 プラグインパラメータに0を指定できないバグを修正しました
//=============================================================================

/*:
 * @plugindesc スイッチ・変数の状態を画面上にリアルタイム表示します。
 * @author くらむぼん
 *
 * @param monitorX
 * @desc モニターのX座標
 * @default 0
 *
 * @param monitorY
 * @desc モニターのY座標
 * @default 0
 *
 * @param monitorW
 * @desc モニターの幅。「変数名が長くて潰れるんだけど！」という方は大きくしましょう
 * @default 240
 *
 * @help
 * デバッグメニューで指定したスイッチ・変数をモニターします。
 * テストプレイ中のみ機能します。
 * 
 * 使い方：
 * １．テストプレイ中にF9キーを押してデバッグメニューを表示
 * ２．監視したいスイッチ・変数の上でShiftキーを押す
 * 　　（監視対象に設定すると右端に*マークが表示されます）
 * ３．デバッグメニューを閉じると指定座標にモニターが現れ、監視対象が表示されます
 * 　　（通常のマップ上か、バトル中のみ表示されます。メニューや店などは非対応）
 * ４．Tabキーを押すと、監視対象の値をリアルタイムに変更できるモードになります
 * 　　（操作方法は普通のデバッグメニューと同じです。再度Tabで解除）
 * 
 * ライセンス：
 * このプラグインの利用法に制限はありません。お好きなようにどうぞ。
 */

(() => {
    'use strict';
    if (!Utils.isOptionValid('test') || !Utils.isNwjs()) return;

    const parameters = PluginManager.parameters('DebugMonitor');
    const monitorX = Number(parameters['monitorX'] || 0);
    const monitorY = Number(parameters['monitorY'] || 0);
    const monitorW = Number(parameters['monitorW'] || 240);

    let active = false;
    const switches = {};
    const variables = {};

    class Window_Monitor extends Window_Command {
    constructor(x, y) {
        const rect = new Rectangle(x, y, monitorW, 200); // 高さを仮に固定
        super(rect);
        this.active = active;
    }

        windowWidth() {
            return monitorW;
        }

        windowHeight() {
            return this.fittingHeight(this.maxItems());
        }

        makeCommandList() {
            for (const switchId in switches) {
                if (switches[switchId]) {
                    this.addCommand('', 'S', true, Number(switchId));
                }
            }
            for (const variableId in variables) {
                if (variables[variableId]) {
                    this.addCommand('', 'V', true, Number(variableId));
                }
            }
            if (!this._list.length) {
                this.hide();
                active = this.active = false;
            }
        }

        isOkEnabled() {
            return false;
        }

        update() {
            super.update();
            if (Input.isTriggered('tab') && this.visible) active = this.active = !this.active;

            let needsRedraw = false;

            for (let i = 0; i < this._list.length; i++) {
                const symbol = this._list[i].symbol;
                const id = this._list[i].ext;
                let name;

                if (active && this.index() === i) {
                    if (symbol === 'S') {
                        if (Input.isRepeated('ok') || Input.isRepeated('left') || Input.isRepeated('right')) {
                            SoundManager.playCursor();
                            $gameSwitches.setValue(id, !$gameSwitches.value(id));
                        }
                    } else {
                        let value = $gameVariables.value(id);
                        if (Input.isRepeated('right')) value++;
                        if (Input.isRepeated('left')) value--;
                        if (Input.isRepeated('pagedown')) value += 10;
                        if (Input.isRepeated('pageup')) value -= 10;
                        if ($gameVariables.value(id) !== value) {
                            $gameVariables.setValue(id, value);
                            SoundManager.playCursor();
                        }
                    }
                }

                if (symbol === 'S') {
                    const label = $dataSystem.switches[id] || `S[${String(id).padStart(4, '0')}]`;
                    name = `${label}:${$gameSwitches.value(id) ? '[ON]' : '[OFF]'}`;
                } else {
                    const label = $dataSystem.variables[id] || `V[${String(id).padStart(4, '0')}]`;
                    name = `${label}:${$gameVariables.value(id)}`;
                }

                if (this._list[i].name !== name) {
                    this._list[i].name = name;
                    needsRedraw = true;
                }
            }

            if (needsRedraw) {
                this.contents.clear();
                this.drawAllItems();
            }
        }

        isOpenAndActive() {
            return this.isOpen() && this.active;
        }
    }

    const _Window_DebugEdit_update = Window_DebugEdit.prototype.update;
    Window_DebugEdit.prototype.update = function () {
        _Window_DebugEdit_update.apply(this, arguments);
        this.updateMonitor();
    };

    Window_DebugEdit.prototype.updateMonitor = function () {
        if (this.active && Input.isRepeated('shift')) {
            const id = this.currentId();
            if (this._mode === 'switch') {
                switches[id] = !switches[id];
            } else {
                variables[id] = !variables[id];
            }
            SoundManager.playCursor();
            this.redrawCurrentItem();
        }
    };

    Window_DebugEdit.prototype.itemStatus = function (dataId) {
        if (this._mode === 'switch') {
            return `${$gameSwitches.value(dataId) ? '[ON]' : '[OFF]'}${switches[dataId] ? '*' : ''}`;
        } else {
            return `${$gameVariables.value(dataId)}${variables[dataId] ? '*' : ''}`;
        }
    };

    const _Window_Selectable_isOpenAndActive = Window_Selectable.prototype.isOpenAndActive;
    Window_Selectable.prototype.isOpenAndActive = function () {
        if ((SceneManager._scene instanceof Scene_Map || SceneManager._scene instanceof Scene_Battle) && active) {
            return false;
        }
        return _Window_Selectable_isOpenAndActive.apply(this, arguments);
    };

    const _Game_Player_canMove = Game_Player.prototype.canMove;
    Game_Player.prototype.canMove = function () {
        return !active && _Game_Player_canMove.apply(this, arguments);
    };

    const _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
    Scene_Map.prototype.createAllWindows = function () {
        _Scene_Map_createAllWindows.apply(this, arguments);
        this.createMonitorWindow();
    };

    const _Scene_Battle_createAllWindows = Scene_Battle.prototype.createAllWindows;
    Scene_Battle.prototype.createAllWindows = function () {
        _Scene_Battle_createAllWindows.apply(this, arguments);
        this.createMonitorWindow();
    };

    function createMonitorWindow() {
        this._monitorWindow = new Window_Monitor(monitorX, monitorY);
        this.addWindow(this._monitorWindow);
    }

    Scene_Map.prototype.createMonitorWindow = Scene_Battle.prototype.createMonitorWindow = createMonitorWindow;
})();