﻿/*:
 * @target MZ
 * @plugindesc This plugin enables plugin commands to delete save data. It includes features to delete specific save data or randomly delete one.
 * @author Malomi
 * @url https://gendaigoyaku.com/
 * 
 * @command deleteAllSaveData
 * @text Delete all save data
 * @desc It will delete all save data.
 * 
 * @command deleteSaveData
 * @text Delete specific save data
 * @desc It will delete the specified save file.
 * 
 * @arg savefileId
 * @type number
 * @default 1
 * @min 0
 * @max 19
 * @text Save file number
 * @desc Specify the number of the save file to delete.
 * 
 * @command deleteRandomSaveData
 * @text Delete one at random
 * @desc It will delete one save file at random.
 * 
 * @help
 * ----------------------------------------------------------------------------
 * DeleteSaveData.js
 * ----------------------------------------------------------------------------
 * This plugin enables the use of plugin commands to delete save data.
 * 
 * There are three types of plugin commands.
 * - deleteAllSaveData
 * - deleteSaveData
 * - deleteRandomSaveData
 * 
 * No specific settings are required.
 * Simply enable the plugin and you're good to go.
 * ----------------------------------------------------------------------------
 * License:
 * This is the MIT License. 
 * It can be used for both commercial and non-commercial purposes.
 * ----------------------------------------------------------------------------
 */

/*:ja
 * @target MZ
 * @plugindesc セーブデータを消去するプラグインです。指定したセーブデータを消去したり、ランダムに1つだけ消去する機能を含みます。
 * @author まろみ
 * @url https://gendaigoyaku.com/
 * 
 * @command deleteAllSaveData
 * @text 全てのセーブデータを削除
 * @desc 全てのセーブデータを削除します。
 * 
 * @command deleteSaveData
 * @text 指定したセーブデータを削除
 * @desc 指定したセーブファイルを削除します。
 * 
 * @arg savefileId
 * @type number
 * @default 1
 * @min 0
 * @max 19
 * @text セーブファイル番号
 * @desc 削除するセーブファイルの番号を指定します。
 * 
 * @command deleteRandomSaveData
 * @text ランダムに1つ削除
 * @desc ランダムに1つのセーブファイルを削除します。
 * 
 * @help
 * ----------------------------------------------------------------------------
 * DeleteSaveData.js
 * ----------------------------------------------------------------------------
 * このプラグインは、セーブデータを消去するプラグインコマンドが
 * 使えるようになるよ。
 * 
 * プラグインコマンドは
 * 全部消す
 * 指定したデータを消す
 * ランダムにひとつ消す
 * の3種類。
 * 
 * 特に設定もいらないよ。プラグインを有効化するだけでOKです。
 * ----------------------------------------------------------------------------
 * 利用規約:
 * 商用・非商用問わず利用可能です。
 * ----------------------------------------------------------------------------
 */

(() => {
    'use strict';

    const pluginName = "DeleteSaveData";

    const removeSavefile = (savefileId) => {
        const saveName = DataManager.makeSavename(savefileId);
        if (StorageManager.exists(saveName)) {
            try {
                StorageManager.remove(saveName);

            } catch (error) {

            }
        }
    };

    PluginManager.registerCommand(pluginName, "deleteAllSaveData", () => {
        const maxSavefiles = DataManager.maxSavefiles();
        let deletedFilesCount = 0;

        for (let i = 1; i <= maxSavefiles; i++) {
            const saveName = DataManager.makeSavename(i);
            if (StorageManager.exists(saveName)) {
                removeSavefile(i);
                deletedFilesCount++;
            }
        }

        DataManager.loadGlobalInfo();
    });

    PluginManager.registerCommand(pluginName, "deleteSaveData", args => {
        const savefileId = Number(args.savefileId);
        removeSavefile(savefileId);
        DataManager.loadGlobalInfo();
    });

    PluginManager.registerCommand(pluginName, "deleteRandomSaveData", () => {
        const maxSavefiles = DataManager.maxSavefiles();
        const existingSaveFiles = [];

        for (let i = 1; i <= maxSavefiles; i++) {
            if (StorageManager.exists(DataManager.makeSavename(i))) {
                existingSaveFiles.push(i);
            }
        }

        if (existingSaveFiles.length > 0) {
            const randomIndex = Math.floor(Math.random() * existingSaveFiles.length);
            const randomSavefileId = existingSaveFiles[randomIndex];
            removeSavefile(randomSavefileId);
            DataManager.loadGlobalInfo();

        }
    });
})();
