//=============================================================================
// MOG_TouchAnimation.js
//=============================================================================

/*:
 * @target MZ
 * @plugindesc (v1.1) 条件付きタッチアニメーション
 * @author Moghunter (Modified)
 * @url https://mogplugins.wordpress.com
 * 

 *
 * @param Conditional Patterns
 * @text 条件付きパターン
 * @desc 条件付きアニメーションパターンのリスト
 * @type struct<ConditionalPattern>[]
 * @default []
 *
 * @help
 * =============================================================================
 * ♦♦♦ MOG - Touch Animation (条件付き拡張版) ♦♦♦
 * Author   -   Moghunter (Modified)
 * Version  -   1.1
 * Updated  -   2024/09/30
 * https://mogplugins.wordpress.com
 * =============================================================================
 * 条件に応じて異なるタッチアニメーションを表示します。
 * 
 * 【使用方法】
 * 1. "条件付きパターン"で各パターンを設定
 * 3. 各パターンには以下を設定可能：
 *    - 条件（スイッチのON/OFF AND 変数の値）
 *    - 画像ファイル名
 *    - アニメーションタイプ
 *    - パーティクル数
 *    - 座標オフセット
 *    - その他のパラメータ
 * 
 * 【条件の評価方法】
 * 各パターンでスイッチ条件 AND 変数条件を評価します。
 * 両方の条件が満たされたパターンが使用されます。
 * 
 * 【条件の評価順序】
 * リストの上から順番に条件をチェックし、最初に両方の条件を
 * 満たしたパターンが使用されます。どの条件も満たさない場合は
 * アニメーションが無効化されます。
 * 
 * 画像ファイルは /img/system/ フォルダに配置してください。
 *
 */

/*~struct~ConditionalPattern:
 * @param switch_id
 * @text スイッチID
 * @desc 条件に使用するスイッチのID
 * @type switch
 * @default 1
 * 
 * @param switch_value
 * @text スイッチの値
 * @desc スイッチがこの値の時に条件成立
 * @type boolean
 * @default true
 * 
 * @param variable_id
 * @text 変数ID
 * @desc 条件に使用する変数のID
 * @type variable
 * @default 1
 * 
 * @param variable_value
 * @text 変数の値
 * @desc 変数がこの値の時に条件成立
 * @type number
 * @default 1
 * 
 * @param file_name
 * @text ファイル名
 * @desc 使用する画像ファイル名
 * @type file
 * @dir img/system/
 * @default TouchParticles
 * 
 * @param animation_type
 * @text アニメーションタイプ
 * @desc アニメーションの種類
 * @type select
 * @default Fireworks
 * @option Fireworks
 * @value Fireworks
 * @option Random
 * @value Random
 * @option Zoom Out
 * @value Zoom Out
 * @option Zoom In
 * @value Zoom In
 * 
 * @param particle_count
 * @text パーティクル数
 * @desc パーティクルの数 (1 - 999)
 * @type number
 * @default 10
 * @min 1
 * @max 999
 * 
 * @param x_offset
 * @text X軸オフセット
 * @desc X軸のオフセット値
 * @type number
 * @default 0
 * @min -9999
 * @max 9999
 * 
 * @param y_offset
 * @text Y軸オフセット
 * @desc Y軸のオフセット値
 * @type number
 * @default 0
 * @min -9999
 * @max 9999
 * 
 * @param blend_mode
 * @text ブレンドモード
 * @desc ブレンドモードの設定
 * @type select
 * @default Normal
 * @option Additive
 * @value Additive
 * @option Normal
 * @value Normal
 * @option Multiply
 * @value Multiply
 * 
 * @param duration
 * @text 持続時間
 * @desc アニメーションの持続時間 (1 - 999)
 * @type number
 * @default 10
 * @min 1
 * @max 999
 * 
 * @param fade_speed
 * @text フェード速度
 * @desc フェードアウトの速度 (2 - 125)
 * @type number
 * @default 20
 * @min 2
 * @max 125
 * 
 * @param random_tone
 * @text ランダムトーン
 * @desc ランダムカラーを使用するか
 * @type boolean
 * @default true
 * 
 * @param enable_movement
 * @text 座標移動を有効化
 * @desc アニメーションを指定座標へ移動させるか
 * @type boolean
 * @default false
 * 
 * @param target_x
 * @text 移動先X座標
 * @desc アニメーションの移動先X座標
 * @type number
 * @default 400
 * @min 0
 * @max 9999
 * 
 * @param target_y
 * @text 移動先Y座標
 * @desc アニメーションの移動先Y座標
 * @type number
 * @default 300
 * @min 0
 * @max 9999
 * 
 * @param movement_speed
 * @text 移動速度
 * @desc 座標移動の速度 (1-20)
 * @type number
 * @default 5
 * @min 1
 * @max 20
 * 
 * @param curve_movement
 * @text 曲線移動
 * @desc 曲線を描いて移動するか
 * @type boolean
 * @default false
 * 
 * @param curve_height
 * @text 曲線の高さ
 * @desc 曲線の頂点の高さ (-500 ~ 500)
 * @type number
 * @default -100
 * @min -500
 * @max 500
 * 
 * @param enable_convergence
 * @text 移動先で収束
 * @desc 移動先で散らばったパーティクルを収束させるか
 * @type boolean
 * @default true
 * 
 * @param convergence_speed
 * @text 収束速度
 * @desc 収束の速度 (1-10)
 * @type number
 * @default 3
 * @min 1
 * @max 10
 * 
 * @param convergence_delay
 * @text 収束開始遅延
 * @desc 移動完了後、収束開始までの遅延フレーム数
 * @type number
 * @default 15
 * @min 0
 * @max 60
 */
 
(() => {
	
//=============================================================================
// ** PLUGIN PARAMETERS
//=============================================================================
　　var Imported = Imported || {};
　　Imported.MOG_TouchAnimation = true;
　　var Moghunter = Moghunter || {}; 

  　Moghunter.parameters = PluginManager.parameters('MOG_TouchAnimation');
    
    // 条件付きパターンの読み込み
    Moghunter.touchAnime_patterns = [];
    
    // パラメータ名の候補をチェック
    const possibleKeys = [
        'Conditional Patterns',
        '条件付きパターン',
        'conditionalPatterns',
        'ConditionalPatterns'
    ];
    
    let foundKey = null;
    for (const key of possibleKeys) {
        if (Moghunter.parameters[key] !== undefined) {
            foundKey = key;
            break;
        }
    }
    
    if (!foundKey) {
        return;
    }
    
    try {
        const rawPatternsData = Moghunter.parameters[foundKey] || '[]';
        const patternsData = JSON.parse(rawPatternsData);
        
        for (let i = 0; i < patternsData.length; i++) {
            const patternData = patternsData[i];
            const pattern = JSON.parse(patternData);
            
            const processedPattern = {
                switchId: Number(pattern.switch_id || 1),
                switchValue: String(pattern.switch_value || 'true') === 'true',
                variableId: Number(pattern.variable_id || 1),
                variableValue: Number(pattern.variable_value || 1),
                fileName: String(pattern.file_name || 'TouchParticles'),
                animationType: String(pattern.animation_type || 'Fireworks'),
                particleCount: Number(pattern.particle_count || 10),
                xOffset: Number(pattern.x_offset || 0),
                yOffset: Number(pattern.y_offset || 0),
                blendMode: String(pattern.blend_mode || 'Normal'),
                duration: Number(pattern.duration || 10),
                fadeSpeed: Number(pattern.fade_speed || 20),
                randomTone: String(pattern.random_tone || 'true') === 'true',
                enableMovement: String(pattern.enable_movement || 'false') === 'true',
                targetX: Number(pattern.target_x || 400),
                targetY: Number(pattern.target_y || 300),
                movementSpeed: Math.min(Math.max(Number(pattern.movement_speed || 5), 1), 20),
                curveMovement: String(pattern.curve_movement || 'false') === 'true',
                curveHeight: Math.min(Math.max(Number(pattern.curve_height || -100), -500), 500),
                enableConvergence: String(pattern.enable_convergence || 'true') === 'true',
                convergenceSpeed: Math.min(Math.max(Number(pattern.convergence_speed || 4), 1), 10),
                convergenceDelay: Math.min(Math.max(Number(pattern.convergence_delay || 15), 0), 60)
            };
            
            Moghunter.touchAnime_patterns.push(processedPattern);
        }
        
    } catch (e) {
        console.warn('TouchAnimation: 条件付きパターンの読み込みに失敗しました:', e);
    }
	
	DataManager._touchPart = {};	
	DataManager._touchPart.x = 0;
	DataManager._touchPart.y = 0;
	DataManager._touchPart.needRefresh = false;
	DataManager._touchPart.skip = false;
	DataManager._touchPart.wait = 0;
	DataManager._touchPart.wait2 = 0;
    DataManager._touchPart.currentPattern = null;

//=============================================================================
// ** 条件システム関数
//=============================================================================

//==============================
// * getCurrentPattern - 現在の条件に合致するパターンを取得
//==============================
DataManager.getTouchAnimationPattern = function() {
    if (Moghunter.touchAnime_patterns.length === 0) {
        return null;
    }
    
    // 条件を上から順番にチェック
    for (const pattern of Moghunter.touchAnime_patterns) {
        if (this.checkPatternCondition(pattern)) {
            return pattern;
        }
    }
    
    // 条件に合致するものがない場合はnullを返す（アニメーション無効化）
    return null;
};

//==============================
// * checkPatternCondition - パターンの条件をチェック (AND条件)
//==============================
DataManager.checkPatternCondition = function(pattern) {
    // スイッチ条件をチェック
    const switchCondition = $gameSwitches.value(pattern.switchId) === pattern.switchValue;
    
    // 変数条件をチェック
    const variableCondition = $gameVariables.value(pattern.variableId) === pattern.variableValue;
    
    // 両方の条件が満たされた場合のtrue (AND条件)
    return switchCondition && variableCondition;
};


	
//=============================================================================
// ■■■ Scene Base ■■■
//=============================================================================

//==============================
// * create Sprite Field
//==============================
Scene_Base.prototype.createSpriteField3 = function() {
   this._spriteField3 = new Sprite();
   this._spriteField3.z = 100;
   this.addChild(this._spriteField3);
};

//==============================
// * sort Sprite Field
//==============================
Scene_Base.prototype.sortSpriteField = function() {
	if (this._spriteField1) {this._spriteField1.children.sort((a, b) => a.z - b.z)};
	if (this._spriteField2) {this._spriteField2.children.sort((a, b) => a.z - b.z)};
    if (this._spriteField3) {this._spriteField3.children.sort((a, b) => a.z - b.z)};
};

//==============================
// * createTouchParticles
//==============================
Scene_Base.prototype.createTouchParticles = function() {
	 DataManager._touchPart.wait = 0;
	 DataManager._touchPart.wait2 = 5;
	 DataManager._touchPart.x = 0;
	 DataManager._touchPart.y = 0;	 
	 DataManager._touchPart.needRefresh = false;
	 DataManager._touchPart.skip = false;
	 DataManager._touchPart.map = false;
     this._touchPar = new TouchParticles();
	 this._touchPar.z = 450;
	 this._spriteField3.addChild(this._touchPar);
};

//==============================
// * update Touch Animation Base
//==============================
Scene_Base.prototype.updateTouchAnimationBase = function() {
	 if (!this._spriteField3) {this.createSpriteField3()};
	 if (!this._touchPar) {this.createTouchParticles()};
	 if (DataManager._touchPart.wait2 > 0) {DataManager._touchPart.wait2--};
	 if (DataManager._touchPart.wait2  == 0) {this.updateTouchAnimation()};
};

//==============================
// * update Touch Particles
//==============================
Scene_Base.prototype.updateTouchAnimation = function() {	 
	 if (TouchInput.isTriggered()) {
		 if (DataManager._touchPart.wait == 0) {
			 DataManager._touchPart.wait = 2};
	    	 this.setTouchAnimePos();
	 };
	 if (DataManager._touchPart.wait > 0) {
		 DataManager._touchPart.wait--;
		 if (DataManager._touchPart.wait == 0 && this.isTouchAnimeEnabled()) {
			 DataManager._touchPart.needRefresh = true;
		 }; 
	 ;}
};

//==============================
// * setTouchAnimePos
//==============================
Scene_Base.prototype.setTouchAnimePos = function() {
   DataManager._touchPart.map = false;
   DataManager._touchPart.x = TouchInput.x;
   DataManager._touchPart.y = TouchInput.y;
};

//==============================
// * isTouchAnimeEnabled
//==============================
Scene_Base.prototype.isTouchAnimeEnabled = function() {
	if (DataManager._touchPart.skip) {return false};
	// 有効なパターンがない場合はアニメーション無効
	if (DataManager.getTouchAnimationPattern() === null) {
		return false;
	}
	return true;
};

//==============================
// ♦ ALIAS ♦  Initialize
//==============================
const _mog_ScnBase_touchPar_update = Scene_Base.prototype.update;
Scene_Base.prototype.update = function() {
	 _mog_ScnBase_touchPar_update.call(this);
     this.updateTouchAnimationBase();
};

//=============================================================================
// ■■■ Scene Map ■■■
//=============================================================================

//==============================
// * setTouchAnimePos
//==============================
Scene_Map.prototype.setTouchAnimePos = function() {
   DataManager._touchPart.map = true;
   const mx = $gameMap.displayX() * $gameMap.tileWidth();
   const my = $gameMap.displayY() * $gameMap.tileHeight();
   DataManager._touchPart.x = TouchInput.x + mx;
   DataManager._touchPart.y = TouchInput.y + my;
};

//=============================================================================
// ■■■ Sprite Button ■■■
//=============================================================================

//==============================
// ♦ ALIAS ♦  on Click
//==============================
const _mog_touchAnime_sprtButton_onClick = Sprite_Button.prototype.onClick;
Sprite_Button.prototype.onClick = function() {
	_mog_touchAnime_sprtButton_onClick.call(this);
   	 DataManager._touchPart.wait = 10;
};

//=============================================================================
// ■■■ TouchParticles ■■■
//=============================================================================
function TouchParticles() {
    this.initialize.apply(this, arguments);
};

TouchParticles.prototype = Object.create(Sprite.prototype);
TouchParticles.prototype.constructor = TouchParticles;

//==============================
// ♦♦ Initialize
//==============================
TouchParticles.prototype.initialize = function() {
	Sprite.prototype.initialize.call(this);
    this.updateCurrentPattern();
    this.createParticles();  
};

//==============================
// * updateCurrentPattern - 現在のパターンを更新
//==============================
TouchParticles.prototype.updateCurrentPattern = function() {
    this._currentPattern = DataManager.getTouchAnimationPattern();
    
    // パターンがnullの場合（条件に合致しない場合）は無効パターンを設定
    if (this._currentPattern === null) {
        this._mode = 0;
        this._power = 0;
        this._blendType = 0;
        this._fileName = '';
        return;
    }
    
    this._mode = this.getMode(this._currentPattern.animationType);
    this._power = Math.min(Math.max(this._currentPattern.particleCount, 1), 999);
    this._blendType = this.getBlend(this._currentPattern.blendMode);
    this._fileName = this._currentPattern.fileName;
};

//==============================
// * get Mode
//==============================
TouchParticles.prototype.getMode = function(mode) {
	if (mode == "Random") {return 1;
	} else if (mode == "Zoom Out") {return 2;
	} else if (mode == "Zoom In") {return 3};
	return 0;
}; 

//==============================
// * weather get Blend
//==============================
TouchParticles.prototype.getBlend = function(blend) {
	if (blend == "Additive") {return 1;
	} else if (blend == "Multiply") {return 2};
	return 0;
}; 


//==============================
// * file Name
//==============================
TouchParticles.prototype.fileNAme = function() {
    return this._fileName || '';
};

//==============================
// * create Particles
//==============================
TouchParticles.prototype.createParticles = function() {
     this._particlesSprites = [];
     
     // 有効なパターンがない場合はパーティクルを作成しない
     if (this._currentPattern === null) {
         return;
     }
     
     // ファイル名が空の場合もパーティクルを作成しない
     const fileName = this.fileNAme();
     if (!fileName || fileName === '') {
         return;
     }
     
	 for (var i = 0; i < this.power(); i++) {
		 this._particlesSprites[i] = new Sprite(ImageManager.loadSystem(fileName));
		 this.setBaseData(this._particlesSprites[i],i);
		 this.addChild(this._particlesSprites[i]);
	};
};

//==============================
// * mode
//==============================
TouchParticles.prototype.mode = function() {
    return this._mode;
};

//==============================
// * x Pos
//==============================
TouchParticles.prototype.xPos = function() {
    return DataManager._touchPart.x;
};

//==============================
// * y Pos
//==============================
TouchParticles.prototype.yPos = function() {
    return DataManager._touchPart.y;
};

//==============================
// * x Offset
//==============================
TouchParticles.prototype.xOffset = function() {
    return this._currentPattern ? this._currentPattern.xOffset : 0;
};

//==============================
// * y Offset
//==============================
TouchParticles.prototype.yOffset = function() {
    return this._currentPattern ? this._currentPattern.yOffset : 0;
};

//==============================
// * power
//==============================
TouchParticles.prototype.power = function() {
    // 有効なパターンがない場合は0を返す
    if (this._currentPattern === null) {
        return 0;
    }
    
	if (this.mode() == 2) {return 1};
	if (this.mode() == 3) {return 1};
    return this._power;
};

//==============================
// * force Hide
//==============================
TouchParticles.prototype.forcedHide = function(sprite) {
   if (DataManager._touchPart.skip) {return true};
   if (DataManager._touchPart.wait > 0) {return true};
   return false;
};

//==============================
// * set Base Data
//==============================
TouchParticles.prototype.setBaseData = function(sprite,index) {
     sprite.index = index;
	 sprite.opacity = 0;
	 sprite.anchor.x = 0.5;
	 sprite.anchor.y = 0.5;
	 sprite.sx = [0,1];
	 sprite.sy = [0,1];
	 sprite.rt = [0,0.08];
	 sprite.sc = 0;
	 sprite.wait = 0;
	 sprite.blendType = this._blendType;
     
     const currentPattern = this._currentPattern;
	 sprite.fadeSpeed = Math.min(Math.max(currentPattern ? currentPattern.fadeSpeed : 20, 2), 125);
	 sprite.duration = 0;
	 sprite.duration2 = Math.min(Math.max(currentPattern ? currentPattern.duration : 10, 1), 999);
     
     // 座標移動用パラメータ
     sprite.enableMovement = currentPattern ? currentPattern.enableMovement : false;
     sprite.targetX = currentPattern ? currentPattern.targetX : 400;
     sprite.targetY = currentPattern ? currentPattern.targetY : 300;
     sprite.movementSpeed = currentPattern ? currentPattern.movementSpeed : 5;
     sprite.curveMovement = currentPattern ? currentPattern.curveMovement : false;
     sprite.curveHeight = currentPattern ? currentPattern.curveHeight : -100;
     sprite.moveStartX = 0;
     sprite.moveStartY = 0;
     sprite.moveProgress = 0;
     sprite.isMoving = false;
     sprite.moveCompleted = false; // 移動完了フラグ
     // 個別の目標地点（散らばり効果用）
     sprite.individualTargetX = sprite.targetX;
     sprite.individualTargetY = sprite.targetY;
     // 収束関連パラメータ
     sprite.enableConvergence = currentPattern ? currentPattern.enableConvergence : true;
     sprite.convergenceSpeed = currentPattern ? currentPattern.convergenceSpeed : 4;
     sprite.convergenceDelay = 0; // 遅延を無効化
     sprite.convergenceWait = 0;
     sprite.isConverging = false;
     sprite.convergenceProgress = 0;
     sprite.convergenceStartX = 0;
     sprite.convergenceStartY = 0;
     // 個別フェード管理
     sprite.individualFadeStart = false;
     sprite.reachedTarget = false; // 目標到達フラグ
     sprite.targetReachDistance = 8; // 目標到達判定距離
     
     const useRandomTone = currentPattern ? currentPattern.randomTone : true;
	 if (useRandomTone) {this.setToneRandom(sprite,index)};
};

//==============================
// * set Tone Random
//==============================
TouchParticles.prototype.setToneRandom = function(sprite,index) {
     const r = Math.randomInt(255);
	 const g = Math.randomInt(255);
	 const b = Math.randomInt(255);
     const colorTone = [r,g,b,255]
     sprite.setColorTone(colorTone);
};

//==============================
// * setIndividualTarget - 個別目標地点を設定
//==============================
TouchParticles.prototype.setIndividualTarget = function(sprite, index) {
    const mode = this.mode();
    const baseTargetX = sprite.targetX;
    const baseTargetY = sprite.targetY;
    
    if (mode === 0) { // Fireworks
        // 花火の散らばりパターンを目標地点に適用
        const spreadRange = 80; // 散らばり範囲
        const offsetX = -spreadRange + Math.random() * (spreadRange * 2);
        const offsetY = -spreadRange + Math.random() * (spreadRange * 2);
        
        sprite.individualTargetX = baseTargetX + offsetX;
        sprite.individualTargetY = baseTargetY + offsetY;
        
    } else if (mode === 1) { // Random
        // ランダムな散らばり
        const spreadRange = 60;
        const offsetX = -spreadRange + Math.random() * (spreadRange * 2);
        const offsetY = -spreadRange + Math.random() * (spreadRange * 2);
        
        sprite.individualTargetX = baseTargetX + offsetX;
        sprite.individualTargetY = baseTargetY + offsetY;
        
    } else {
        // Zoom Out/Zoom In は中央に集約
        sprite.individualTargetX = baseTargetX;
        sprite.individualTargetY = baseTargetY;
    }
};

//==============================
// * refresh particles
//==============================
TouchParticles.prototype.refreshParticles = function(sprite,index) {
	 sprite.x = this.xPos();
	 sprite.y = this.yPos();
     
     // 座標移動の初期化
     if (sprite.enableMovement) {
         sprite.moveStartX = sprite.x;
         sprite.moveStartY = sprite.y;
         sprite.moveProgress = 0;
         sprite.isMoving = true;
         sprite.moveCompleted = false; // 移動完了フラグをリセット
         
         // アニメーションタイプに応じた個別目標地点を設定
         this.setIndividualTarget(sprite, index);
     }
     
	 if (this.mode() == 0) {
		 this.setAnimation1(sprite,index) ;
	 } else if (this.mode() == 1) {
		 this.setAnimation2(sprite,index);
	 } else if (this.mode() == 2) {
		 this.setAnimation3(sprite,index);
	 } else if (this.mode() == 3) {
		 this.setAnimation4(sprite,index);		 
	 };
};

//==============================
// * set Animation 1
//==============================
TouchParticles.prototype.setAnimation1 = function(sprite,index) { 
     sprite.duration = sprite.duration2 + Math.randomInt(20);
	 sprite.wait = 0;
	 var r = 0.7 + Math.abs(Math.random() * sprite.sx[1]);
	 var d = Math.randomInt(100);
	 sprite.sx[0] = d > 40 ? r : -r;
	 sprite.sx[0] = d > 90 ? 0 : sprite.sx[0];
	 var r = 0.7 + Math.abs(Math.random() * sprite.sy[1]);
	 var d = Math.randomInt(100);
     sprite.sy[0] = d > 40 ? r : -r;
	 sprite.sy[0] = d > 90 ? 0 : sprite.sy[0];
	 var r = 0.01 + Math.abs(Math.random() * sprite.rt[1]);
     sprite.rt[0] = sprite.rt[1] > 0 ? r : -r;
	 var pz = ((Math.random() * 0.5) * 1);
	 sprite.scale = new PIXI.Point(0.5 + Number(pz), 0.5 + Number(pz));	 
};

//==============================
// * set Animation 2
//==============================
TouchParticles.prototype.setAnimation2 = function(sprite,index) { 
     sprite.duration = sprite.duration2 + Math.randomInt(20);
	 sprite.wait = Math.randomInt(20);
	 sprite.x += -30 + Math.randomInt(60);
	 sprite.y += -30 + Math.randomInt(60);
     sprite.sc = 0.02;
	 var pz = ((Math.random() * 0.5) * 1);
	 sprite.scale = new PIXI.Point(0.5 + Number(pz), 0.5 + Number(pz));	 
};

//==============================
// * set Animation 3
//==============================
TouchParticles.prototype.setAnimation3 = function(sprite,index) { 
     sprite.duration = sprite.duration2 + 10;
	 sprite.wait = 0;
     sprite.sc = 0.04;
	 var pz = 0.1;
	 sprite.scale = new PIXI.Point(0.5 + Number(pz), 0.5 + Number(pz));	 
};

//==============================
// * set Animation 4
//==============================
TouchParticles.prototype.setAnimation4 = function(sprite,index) { 
     sprite.duration = sprite.duration2 + 10;
	 sprite.wait = 0;
     sprite.sc = -0.1;
	 var pz = 0.1;
	 sprite.scale = new PIXI.Point(3.0, 3.0);	 
};

//==============================
// * update particles
//==============================
TouchParticles.prototype.updateParticles = function(sprite,index) {
	 if (this.forcedHide(sprite,index)) {
		 this.updateHide(sprite,index);
	 } else {
		 if (sprite.wait > 0) {
			 sprite.wait--;
		 } else {
			 this.updateParticlesData(sprite,index);
		 };
	 };
};

//==============================
// * updateParticlesData
//==============================
TouchParticles.prototype.updateParticlesData = function(sprite,index) {
	 sprite.visible = true;
     
     // 座標移動処理
     if (sprite.enableMovement && sprite.isMoving) {
         // 座標移動中は正確な移動のみ実行
         this.updateMovement(sprite);
     } else if (!sprite.enableMovement) {
         // 座標移動が無効な場合は通常のアニメーション移動
         sprite.x += sprite.sx[0];
         sprite.y += sprite.sy[0];
     }
     // 座標移動有効かつ移動完了後は位置を変更しない
     
	 sprite.rotation += sprite.rt[0];
	 if (sprite.scale.x < 4.00) {sprite.scale.x += sprite.sc};
	 if (sprite.scale.x < 0.00) {sprite.scale.x = 0.00};
	 sprite.scale.y = sprite.scale.x;
     this.updateFade(sprite,index)
};

//==============================
// * update Hide
//==============================
TouchParticles.prototype.updateHide = function(sprite,index) {
	sprite.visible = false;
	sprite.duration = 0;
	sprite.opacity = 0;
};

//==============================
// * updateConvergence - 収束更新
//==============================
TouchParticles.prototype.updateConvergence = function(sprite) {
    if (!sprite.isConverging) return;
    
    sprite.convergenceProgress += sprite.convergenceSpeed / 100.0;
    
    if (sprite.convergenceProgress >= 1.0) {
        sprite.convergenceProgress = 1.0;
        sprite.isConverging = false;
        sprite.individualFadeStart = true;
        // 最終位置を目標地点に固定
        sprite.x = sprite.targetX;
        sprite.y = sprite.targetY;
        return;
    }
    
    // イージングで滑らかな収束
    const easeProgress = this.easeInOutSmooth(sprite.convergenceProgress);
    
    // 元の目標地点に向かって収束
    sprite.x = sprite.convergenceStartX + (sprite.targetX - sprite.convergenceStartX) * easeProgress;
    sprite.y = sprite.convergenceStartY + (sprite.targetY - sprite.convergenceStartY) * easeProgress;
};

//==============================
// * update Movement - 座標移動更新
//==============================
TouchParticles.prototype.updateMovement = function(sprite) {
    sprite.moveProgress += sprite.movementSpeed / 100.0;
    
    if (sprite.moveProgress >= 1.0) {
        sprite.moveProgress = 1.0;
        sprite.isMoving = false;
        sprite.moveCompleted = true;
    }
    
    // イージング処理（滑らかな移動）
    const easeProgress = this.easeInOut(sprite.moveProgress);
    
    if (sprite.curveMovement) {
        // 曲線移動（個別目標地点使用）
        const baseX = sprite.moveStartX + (sprite.individualTargetX - sprite.moveStartX) * easeProgress;
        const baseY = sprite.moveStartY + (sprite.individualTargetY - sprite.moveStartY) * easeProgress;
        
        // 曲線の頂点を計算（sin曲線で弧を描く）
        const curveOffset = Math.sin(sprite.moveProgress * Math.PI) * sprite.curveHeight;
        
        sprite.x = baseX;
        sprite.y = baseY + curveOffset;
    } else {
        // 直線移動（個別目標地点使用）
        sprite.x = sprite.moveStartX + (sprite.individualTargetX - sprite.moveStartX) * easeProgress;
        sprite.y = sprite.moveStartY + (sprite.individualTargetY - sprite.moveStartY) * easeProgress;
    }
    
    // 移動完了時の処理
    if (sprite.moveCompleted && !sprite.reachedTarget) {
        sprite.reachedTarget = true;
        
        if (sprite.enableConvergence) {
            // 収束有効の場合は収束開始
            sprite.isConverging = true;
            sprite.convergenceProgress = 0;
            sprite.convergenceStartX = sprite.x;
            sprite.convergenceStartY = sprite.y;
        } else {
            // 収束無効の場合は即座フェード開始
            sprite.individualFadeStart = true;
        }
    }
};

//==============================
// * easeInOut - イージング関数
//==============================
TouchParticles.prototype.easeInOut = function(t) {
    return t < 0.5 ? 2 * t * t : -1 + (4 - 2 * t) * t;
};

//==============================
// * easeInOutSmooth - より滑らかなイージング関数
//==============================
TouchParticles.prototype.easeInOutSmooth = function(t) {
    // より滑らかなスタートと終了を実現
    return t * t * (3.0 - 2.0 * t);
};

//==============================
// * update Fade
//==============================
TouchParticles.prototype.updateFade = function(sprite,index) {
     // 移動有効の場合のフェード制御
     if (sprite.enableMovement) {
         if (!sprite.moveCompleted) {
             // 移動中は不透明度を上げる
             if (sprite.duration > 0) {
                 sprite.duration--;
                 sprite.opacity += 20;
                 if (sprite.opacity > 255) sprite.opacity = 255;
             }
             return;
         } else if (!sprite.individualFadeStart) {
             // 移動中または収束中は不透明度を維持
             if (sprite.duration > 0) {
                 sprite.duration--;
                 sprite.opacity += 20;
                 if (sprite.opacity > 255) sprite.opacity = 255;
             }
             return;
         }
         // sprite.individualFadeStartがtrueの場合は下の通常フェード処理へ
     }
     
     // 通常のフェード処理（移動無効または個別フェード開始後）
     if (sprite.duration > 0) {
		 sprite.duration--;
		 sprite.opacity += 20;
	 } else {
         sprite.opacity -= sprite.fadeSpeed;
	 };
};

//==============================
// * refresh Base
//==============================
TouchParticles.prototype.refreshBase = function() {
    DataManager._touchPart.needRefresh = false;
    
    // パターンが変更されているかチェック
    const newPattern = DataManager.getTouchAnimationPattern();
    if (this.hasPatternChanged(newPattern)) {
        this.updateCurrentPattern();
        this.recreateParticles();
    }
    
    // パターンがnullの場合（条件に合致しない）はアニメーションを実行しない
    if (newPattern === null) {
        return;
    }
    
	for (var i = 0; i < this._particlesSprites.length; i++) {
	     this.refreshParticles(this._particlesSprites[i],i);
	};
};

//==============================
// * hasPatternChanged - パターンが変更されたかチェック
//==============================
TouchParticles.prototype.hasPatternChanged = function(newPattern) {
    if (!this._currentPattern && newPattern === null) return false;
    if (!this._currentPattern || newPattern === null) return true;
    
    return (this._currentPattern.fileName !== newPattern.fileName ||
            this._currentPattern.animationType !== newPattern.animationType ||
            this._currentPattern.particleCount !== newPattern.particleCount ||
            this._currentPattern.blendMode !== newPattern.blendMode ||
            this._currentPattern.enableMovement !== newPattern.enableMovement ||
            this._currentPattern.targetX !== newPattern.targetX ||
            this._currentPattern.targetY !== newPattern.targetY ||
            this._currentPattern.movementSpeed !== newPattern.movementSpeed ||
            this._currentPattern.curveMovement !== newPattern.curveMovement ||
            this._currentPattern.curveHeight !== newPattern.curveHeight ||
            this._currentPattern.enableConvergence !== newPattern.enableConvergence ||
            this._currentPattern.convergenceSpeed !== newPattern.convergenceSpeed ||
            this._currentPattern.convergenceDelay !== newPattern.convergenceDelay);
};

//==============================
// * recreateParticles - パーティクルを再生成
//==============================
TouchParticles.prototype.recreateParticles = function() {
    // 既存のパーティクルを削除
    if (this._particlesSprites) {
        for (var i = 0; i < this._particlesSprites.length; i++) {
            if (this._particlesSprites[i]) {
                this.removeChild(this._particlesSprites[i]);
            }
        }
        this._particlesSprites = [];
    }
    
    // 新しい設定でパーティクルを作成
    this.createParticles();
};

//==============================
// * need Refresh
//==============================
TouchParticles.prototype.needRefresh = function() {
    return DataManager._touchPart.needRefresh;
};

//==============================
// * update Position Real
//==============================
TouchParticles.prototype.updatePositionReal = function() {
    if (DataManager._touchPart.map) {
		this.updatePositionMap()
	} else {
        this.updatePositionNormal()
	};
};

//==============================
// * update Position Map
//==============================
TouchParticles.prototype.updatePositionMap = function() {
	const mx = $gameMap.displayX() * $gameMap.tileWidth();
	const my = $gameMap.displayY() * $gameMap.tileHeight();
	this.x = this.xOffset() - mx;
	this.y = this.yOffset() - my;
};

//==============================
// * update Position Normal
//==============================
TouchParticles.prototype.updatePositionNormal = function() {
	this.x = this.xOffset();
	this.y = this.yOffset();
};

//==============================
// ♦♦ Update
//==============================
TouchParticles.prototype.update = function() {
	Sprite.prototype.update.call(this);
    if (this.needRefresh()) {this.refreshBase()};
    
    // 現在のパターンがnullの場合またはパーティクルがない場合は更新しない
    const currentPattern = DataManager.getTouchAnimationPattern();
    if (currentPattern === null || !this._particlesSprites || this._particlesSprites.length === 0) {
        return;
    }
    
	for (var i = 0; i < this._particlesSprites.length; i++) {
		 if (this._particlesSprites[i]) {
		     this.updateParticles(this._particlesSprites[i],i);
		 }
	};
	this.updatePositionReal();
};

})();