//=============================================================================
// NrAutoBreak.js
//=============================================================================

/*:
 * @target MZ
 * @plugindesc 指定文字数または指定幅で自動改行 V1.0.0
 * @author NJ
 *
 * @param BreakMode
 * @text 改行モード
 * @desc 自動改行の基準を選択します。
 * @type select
 * @option 文字数
 * @value count
 * @option 指定幅
 * @value width
 * @default count
 *
 * @param BreakLength
 * @text 改行文字数
 * @desc 「文字数」モード時、この文字数に達すると改行されます。
 * @type number
 * @default 30
 *
 * @param BreakPixel
 * @text 改行幅（ピクセル）
 * @desc 「指定幅」モード時、この幅に達すると改行されます。
 * @type number
 * @default 480
 *
 * @param IndentSpace
 * @text 行頭に全角スペース
 * @desc 改行後の行頭に全角スペースを追加します。
 * @type boolean
 * @default true
 *
 * @help
 * ・「文字数」モード：指定文字数で改行。
 * ・「指定幅」モード：描画幅が設定したピクセル数に達すると改行。
 * ・行頭に全角スペース（　）を入れるかを選択可能
 *
 * ・NrAutoBreak.setBreakLength(n);  改行をn文字目に行うように変更
 * ・NrAutoBreak.setBreakPixel(n); 改行のピクセル数をnに変更
 * 
 * バージョン:
 *  1.0.0 初回
 * 
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可です。
 */

(() => {
    const params = PluginManager.parameters("NrAutoBreak");
    const BreakMode   = params["BreakMode"] || "count";
    const BreakLength = Number(params["BreakLength"] || 30);
    const BreakPixel  = Number(params["BreakPixel"] || 480);
    const IndentSpace = params["IndentSpace"] === "true";

    window.NrAutoBreak = {
        breakMode: BreakMode,
        breakLength: BreakLength,
        breakPixel: BreakPixel,
        setBreakLength(n) {
            const v = Number(n);
            if (!isNaN(v) && v > 0) this.breakLength = v;
        },
        setBreakPixel(px) {
            const v = Number(px);
            if (!isNaN(v) && v > 0) this.breakPixel = v;
        },
        getBreakLength() {
            return this.breakLength;
        },
        getBreakPixel() {
            return this.breakPixel;
        },
        setBreakMode(mode) {
            if (["count", "width"].includes(mode)) this.breakMode = mode;
        },
        getBreakMode() {
            return this.breakMode;
        }
    };

    const _convertEscapeCharacters = Window_Base.prototype.convertEscapeCharacters;
    Window_Base.prototype.convertEscapeCharacters = function(text) {
        text = _convertEscapeCharacters.call(this, text);

        let result = "";
        let visibleCount = 0;
        let currentWidth = 0;
        const indent = (IndentSpace && $gameMessage?.speakerName()) ? "　" : "";
        const mode = NrAutoBreak.breakMode;
        const limitWidth = NrAutoBreak.breakPixel;
        const margin = this.textWidth("あ");

        let i = 0;
        while (i < text.length) {
            if (text[i] === "\x1b") {
                const matched = text.slice(i).match(/^\x1b([A-Z]+)\[([^\]]*)\]/i)
                             || text.slice(i).match(/^\x1b([A-Z]+)/i);
                if (matched) {
                    const code = matched[1].toUpperCase();
                    const arg  = matched[2];
                    let visual = "";

                    if (code === "V" && arg !== undefined) {
                        visual = String($gameVariables.value(Number(arg)));
                    } else if (code === "N" && arg !== undefined) {
                        const actor = $gameActors.actor(Number(arg));
                        visual = actor ? actor.name() : "";
                    } else if (code === "P" && arg !== undefined) {
                        const member = $gameParty.members()[Number(arg)];
                        visual = member ? member.name() : "";
                    }

                    result += matched[0];
                    visibleCount += [...visual].length;
                    currentWidth += this.textWidth(visual);
                    i += matched[0].length;
                    continue;
                }
            }

            const char = text[i];
            result += char;

            if (char === "\n") {
                visibleCount = 0;
                currentWidth = 0;
            } else if (char !== " " && char !== "　") {
                visibleCount++;
                currentWidth += this.textWidth(char);
            }

            if (mode === "count" && visibleCount >= NrAutoBreak.breakLength && i < text.length - 1) {
                result += "\n" + (IndentSpace && $gameMessage?.speakerName() ? "　" : "");
                visibleCount = 0;
                currentWidth = 0;
            } else if (mode === "width" && currentWidth >= limitWidth - margin && i < text.length - 1) {
                result += "\n" + (IndentSpace && $gameMessage?.speakerName() ? "　" : "");
                visibleCount = 0;
                currentWidth = 0;
            }

            i++;
        }

        return result;
    };
})();
