//=============================================================================
// NrCustomMessage.js
//=============================================================================

/*:
 * @target MZ
 * @plugindesc 独立メッセージ表示プラグイン v1.0.0
 * @author NJ
 *
 * @param FontSize
 * @text フォントサイズ
 * @desc 表示文字のフォントサイズ
 * @type number
 * @default 28
 *
 * @param x
 * @text X座標
 * @type number
 * @default 100
 *
 * @param y
 * @text Y座標
 * @type number
 * @default 50
 *
 * @param width
 * @text 幅
 * @type number
 * @default 600
 *
 * @param height
 * @text 高さ
 * @type number
 * @default 120
 *
 * @param backgroundType
 * @text 背景タイプ（0:通常 1:暗く 2:透明）
 * @type select
 * @option 通常
 * @value 0
 * @option 暗く
 * @value 1
 * @option 透明
 * @value 2
 * @default 0
 *
 * @param autoSwitchId
 * @text オートスイッチID
 * @type switch
 * @default 0
 *
 * @param autoWaitFormula
 * @text オート待機フレーム式
 * @desc text.lengthを使って待機フレーム数を計算する式
 * @default 100 + text.length * 10
 * @type string
 *
 * @param SpeedVariableId
 * @text 表示速度変数ID
 * @type variable
 * @default 1
 *
 * @help
 *   概要]:
 *   画面上の任意位置に独立メッセージを1文字ずつ描画します。
 *   プラグインコマンドはありません。イベントの「スクリプト」から以下APIを呼び出してください。
 *
 *   スクリプトコマンド:
 *    NrCustomMessage.create("文字列")
 *      - ウィンドウを生成（初回）／既存に差し替え（2回目以降）して表示開始。
 *
 *    NrCustomMessage.setText("文字列")
 *      - 表示中のテキストを差し替え、先頭から描画し直します。
 *
 *    NrCustomMessage.hide()
 *      - ウィンドウを非表示。
 *
 *    NrCustomMessage.show()
 *      - ウィンドウを再表示。
 *
 *    NrCustomMessage.destroy()
 *      - ウィンドウを破棄（次回は create で再生成）。
 *
 *    NrCustomMessage.updateSpeed()
 *      - 表示途中でも「表示速度変数ID」の現在値を即時反映。
 *
 *    NrCustomMessage.flush()
 *      - 残りの文字を即時描画（全文表示）します。
 *
 */
(() => {
    'use strict';

    const PLUGIN_NAME = "NrCustomMessage";
    const params = PluginManager.parameters(PLUGIN_NAME);
    const posX = Number(params.x || 100);
    const posY = Number(params.y || 50);
    const winW = Number(params.width || 600);
    const winH = Number(params.height || 120);
    const bgType = Number(params.backgroundType || 0);
    const autoSw = Number(params.autoSwitchId || 0);
    const autoWaitFormula = String(params.autoWaitFormula || "$gameVariables.value(492) + text.length * 10");
    const speedVarId = Number(params.SpeedVariableId || 1);
	const fontSize = Number(params.FontSize || 28);

    class Window_IndMsg extends Window_Base {
        constructor(rect) {
            super(rect);
            this._text = "";
            this._textIndex = 0;
            this._waitCount = 0;
            this._autoWait = 0;
            this._autoSwitchId = autoSw;
            this._messageSpeed = 1;
            this._drawX = 0;
            this._drawY = 0;
            this.hide();
			this.contents.fontSize = fontSize;
        }

        setText(text) {
            this._text = String(text ?? "");
            this._textIndex = 0;
            this._waitCount = 0;
            this._autoWait = this.calcAutoWait(this._text);
            this._messageSpeed = $gameVariables.value(speedVarId);
            this.contents.clear();
            this.refresh();
            this.show();
            this.open();
        }

        refresh() {
            const rect = this.baseTextRect();
            this._drawX = rect.x;
            this._drawY = rect.y;
        }

        calcAutoWait(text) {
            try {
                return Math.max(0, Number(Function("text", `return ${autoWaitFormula}`)(text)) || 0);
            } catch (e) {
                console.error("NrCustomMessage: autoWaitFormula 評価エラー", e);
                return 60;
            }
        }

        update() {
            super.update();
            if (!this.visible) return;

            this._messageSpeed = $gameVariables.value(speedVarId);

            if (this._waitCount > 0) {
                this._waitCount--;
                return;
            }

            if (this._textIndex < this._text.length) {
                const speed = Number(this._messageSpeed) || 0;

                if (speed <= 0) {
                    this.flushNow();
                    return;
                }

                const c = this._text[this._textIndex++];
                this.drawOneChar(c);
                this._waitCount = (speed <= 1) ? 0 : speed;
            } else if (autoSw > 0 && $gameSwitches.value(autoSw)) {
                this._autoWait--;
                if (this._autoWait <= 0) {
                    this._textIndex = 0;
                    this.contents.clear();
                    this._autoWait = this.calcAutoWait(this._text);
                    this.refresh();
                }
            }
        }

        drawOneChar(c) {
            const rect = this.baseTextRect();
            const remaining = Math.max(1, rect.width - (this._drawX - rect.x));
            this.contents.drawText(c, this._drawX, this._drawY, remaining, this.lineHeight());
            this._drawX += this.textWidth(c);
        }

        flushNow() {
            while (this._textIndex < this._text.length) {
                const c = this._text[this._textIndex++];
                this.drawOneChar(c);
            }
            this._waitCount = 0;
        }
    }

    window.NrCustomMessage = {
        _window: null,

        create(text) {
            const scene = SceneManager._scene;
            if (!scene) return;
            if (!this._window) {
                const rect = new Rectangle(posX, posY, winW, winH);
                const win = new Window_IndMsg(rect);
                if (win.setBackgroundType) win.setBackgroundType(bgType);
                scene.addChild(win);
                this._window = win;
            }
            this._window.setText(text);
        },

        setText(text) {
            if (this._window) this._window.setText(text);
        },

        hide() {
            if (this._window) this._window.hide();
        },

        show() {
            if (this._window) this._window.show();
        },

        destroy() {
            if (this._window) {
                const scene = SceneManager._scene;
                if (scene) scene.removeChild(this._window);
                this._window = null;
            }
        },

        updateSpeed() {
            if (this._window) {
                this._window._messageSpeed = $gameVariables.value(speedVarId);
            }
        },

        flush() {
            if (this._window) this._window.flushNow();
        }
    };
})();
