//=============================================================================
// NrCustomNumber.js
//=============================================================================

/*:
 * @target MZ
 * @plugindesc 変数の数字を画像で表示。v1.0.0
 *
 * @param configs
 * @text 表示設定リスト
 * @type struct<Config>[]
 * @desc 数字表示の設定リスト
 * @default []
 *
 * @command move
 * @text 表示位置をズラす
 * @desc 指定した名前の数字表示を指定量だけ移動
 *
 * @arg name
 * @text 表示名
 * @desc 移動させたい表示設定の名前
 * @default
 *
 * @arg dx
 * @text X方向の移動量
 * @type number
 * @default 0
 *
 * @arg dy
 * @text Y方向の移動量
 * @type number
 * @default 0
 *
 * @command setImage
 * @text 使用画像を変更
 * @desc 指定した名前の数字表示に使用する画像を変更します。
 *
 * @arg name
 * @text 表示名
 * @desc 変更したい表示設定の名前
 * @default
 *
 * @arg image
 * @text 画像ファイル名
 * @type file
 * @dir img/system
 * @default
 *
 * @help
 * プラグインコマンド:
 * - move
 *    指定名の数字表示を相対移動します。
 *    例：name:"Coin" dx:20 dy:-10
 *
 * - setImage
 *    指定名の数字表示で使用する画像を変更します。
 *    例：name:"Coin" image:"NumberStyle2"
 *
 * スクリプトコマンド:
 *  - NrCustomNumber.move("Coin", 10, 0);             // 表示を右へ移動
 *  - NrCustomNumber.setPosition("Coin", 500, 120);   // 絶対位置指定
 *  - NrCustomNumber.getPosition("Coin");             // 現在位置取得
 *  - NrCustomNumber.setImage("Coin", "NumberStyle2");// 使用画像を変更
 * 
 * バージョン:
 * v1.0.0 初回
 *
 * 利用規約:
 * - プラグイン作者に無断で使用、改変、再配布は不可です。
 */


/*~struct~Config:
 * @param name
 * @text 名前
 * @desc 識別用の名前
 * @default
 *
 * @param image
 * @text 画像ファイル名
 * @type file
 * @dir img/system
 * @default
 *
 * @param x
 * @text X座標
 * @type number
 * @default 0
 *
 * @param y
 * @text Y座標
 * @type number
 * @default 0
 *
 * @param switchId
 * @text スイッチID
 * @type switch
 * @default 1
 *
 * @param hideSwitchIds
 * @text 非表示スイッチIDリスト
 * @type string
 * @desc コンマ区切りのスイッチIDリスト（例: "3,5,7"）。いずれかがONなら非表示
 * @default
 * 
 * @param variableId
 * @text 変数ID
 * @type variable
 * @default 1
 *
 * @param align
 * @text 揃え
 * @type select
 * @option 左揃え
 * @value left
 * @option 右揃え
 * @value right
 * @option 中央揃え
 * @value center
 * @default left
 *
 * @param digits
 * @text 桁数
 * @type number
 * @min 0
 * @default 0
 *
 * @param zeroPad
 * @text 0埋め
 * @type boolean
 * @default false
 *
 * @param spacing
 * @text 数字間の幅
 * @type number
 * @min -99
 * @default 0
 */

(() => {
    const pluginName = 'NrCustomNumber';
    const paramRaw = PluginManager.parameters(pluginName)['configs'];
    let configs = [];
    try {
        configs = JSON.parse(paramRaw || '[]').map(item => JSON.parse(item));
    } catch (e) {
        configs = [];
    }

    class Sprite_NumberDisplay extends Sprite {
        constructor(config) {
            super();
            this.alpha = 1;
            this.scale.x = 1;
            this.scale.y = 1;
            this.visible = true;
            this._baseX = Number(config.x || 0);
            this._baseY = Number(config.y || 0);
            this.x = this._baseX;
            this.y = this._baseY;
            this.z = 0;
            this._name = config.name || '';
            this._switchId = Number(config.switchId || 0);
            this._hideSwitchIds = (config.hideSwitchIds || '').split(',').map(function(s){ return Number(s.trim()||0); }).filter(function(n){ return n > 0; });
            this._variableId = Number(config.variableId || 0);
            this._imageName = config.image || '';
            this._align = config.align || 'left';
            this._digits = Number(config.digits || 1);
            this._zeroPad = (config.zeroPad === true || config.zeroPad === 'true');
            this._spacing = Number(config.spacing || 0);
            this._baseBitmap = null;
            this._lastValue = null;
            this.bitmap = null;

            if (this._imageName) {
                this._baseBitmap = ImageManager.loadSystem(this._imageName);
                this._baseBitmap.addLoadListener(() => {
                    try {
                        const v = $gameVariables ? ($gameVariables.value(this._variableId) || 0) : 0;
                        this._lastValue = null;
                        this._refreshBitmap(v);
                    } catch (e) {
                    }
                });
            }

            this.visible = false;
        }

        update() {
            super.update();

            let shouldVisible = true;
            if (this._switchId > 0) {
                shouldVisible = !!$gameSwitches.value(this._switchId);
            }
            if (this._hideSwitchIds && this._hideSwitchIds.length) {
                for (var i = 0; i < this._hideSwitchIds.length; i++) {
                    if ($gameSwitches.value(this._hideSwitchIds[i])) {
                        shouldVisible = false;
                        break;
                    }
                }
            }
            this.visible = shouldVisible;

            if (!this._baseBitmap || !this._baseBitmap.isReady()) {
                return;
            }

            const value = $gameVariables.value(this._variableId) || 0;
            if (this._lastValue !== value || !this.bitmap) {
                this._refreshBitmap(value);
                this._lastValue = value;
            }
        }

        _refreshBitmap(value) {
            if (!this._baseBitmap || !this._baseBitmap.isReady()) return;
            const img = this._baseBitmap;
            const digitW = Math.floor(img.width / 10);
            const digitH = img.height;

            let digitsArr;
            if (this._digits > 0) {
                if (String(Math.floor(value)).length > this._digits) {
                    digitsArr = Array(this._digits).fill('9');
                } else {
                    digitsArr = String(Math.max(0, Math.floor(value))).split('');
                    while (digitsArr.length < this._digits) {
                        if (this._zeroPad) digitsArr.unshift('0'); else digitsArr.unshift(' ');
                    }
                }
            } else {
                digitsArr = String(Math.max(0, Math.floor(value))).split('');
            }

            const drawnDigits = digitsArr.map((d, i) => ({char: d, index: i})).filter(d => d.char !== ' ');
            const totalW = drawnDigits.length > 0 ? (digitW * drawnDigits.length + this._spacing * (drawnDigits.length - 1)) : 1;
            const resultBitmap = new Bitmap(totalW, digitH);

            drawnDigits.forEach((d, drawIndex) => {
                const n = Number(d.char);
                const x = drawIndex * (digitW + this._spacing);
                resultBitmap.blt(img, n * digitW, 0, digitW, digitH, x, 0);
            });

            this.bitmap = resultBitmap;
            this.setFrame(0, 0, totalW, digitH);

            this.anchor.x = 0;
            if (this._align === 'left') {
                this.x = this._baseX;
            } else if (this._align === 'center') {
                this.x = this._baseX - Math.floor(totalW / 2);
            } else if (this._align === 'right') {
                this.x = this._baseX - totalW;
            }
            this.y = this._baseY;
        }
    }

    const _Scene_Map_start = Scene_Map.prototype.start;
    Scene_Map.prototype.start = function() {
        _Scene_Map_start.call(this);
        this._numberSprites = [];
        configs.forEach(config => {
            const s = new Sprite_NumberDisplay(config);
            this._numberSprites.push(s);
            const ss = this._spriteset;
            let added = false;
            if (ss) {
                if (ss._pictureContainer) {
                    try { ss._pictureContainer.addChild(s); added = true; } catch(e){ added = false; }
                }
                if (!added && ss._baseSprite) {
                    try { ss._baseSprite.addChild(s); added = true; } catch(e){ added = false; }
                }
                if (!added && ss._tilemap) {
                    try { ss._tilemap.addChild(s); added = true; } catch(e){ added = false; }
                }
            }
            if (!added) {
                try { this.addChild(s); added = true; } catch(e){ added = false; }
            }
            if (!added) {
                console.warn('[NrCustomNumber] failed to add sprite to any container:', s);
            }
        });
    };

    PluginManager.registerCommand(pluginName, 'move', args => {
        const name = args.name || '';
        const dx = Number(args.dx || 0);
        const dy = Number(args.dy || 0);
        const scene = SceneManager._scene;
        if (!scene || !scene._numberSprites) return;
        const sprite = scene._numberSprites.find(s => s._name === name);
        if (sprite) {
            sprite._baseX += dx;
            sprite._baseY += dy;
            sprite._refreshBitmap(sprite._lastValue);
        }
    });

    PluginManager.registerCommand(pluginName, 'setImage', args => {
        const name = args.name || '';
        const imageName = args.image || '';
        const scene = SceneManager._scene;
        if (!scene || !scene._numberSprites) return;
        const sprite = scene._numberSprites.find(s => s._name === name);
        if (sprite && imageName) {
            sprite._imageName = imageName;
            sprite._baseBitmap = ImageManager.loadSystem(imageName);
            sprite._baseBitmap.addLoadListener(() => {
                sprite._refreshBitmap($gameVariables.value(sprite._variableId));
            });
        }
    });

    window.NrCustomNumber = {
        move(name, dx = 0, dy = 0) {
            const scene = SceneManager._scene;
            if (!scene || !scene._numberSprites) return;
            const sprite = scene._numberSprites.find(s => s._name === name);
            if (sprite) {
                sprite._baseX += dx;
                sprite._baseY += dy;
                sprite._refreshBitmap(sprite._lastValue);
            }
        },
        setPosition(name, x, y) {
            const scene = SceneManager._scene;
            if (!scene || !scene._numberSprites) return;
            const sprite = scene._numberSprites.find(s => s._name === name);
            if (sprite) {
                sprite._baseX = x;
                sprite._baseY = y;
                sprite._refreshBitmap(sprite._lastValue);
            }
        },
        getPosition(name) {
            const scene = SceneManager._scene;
            if (!scene || !scene._numberSprites) return null;
            const sprite = scene._numberSprites.find(s => s._name === name);
            if (sprite) {
                return { x: sprite._baseX, y: sprite._baseY };
            }
            return null;
        },
        setImage(name, imageName) {
            const scene = SceneManager._scene;
            if (!scene || !scene._numberSprites) return;
            const sprite = scene._numberSprites.find(s => s._name === name);
            if (sprite && imageName) {
                sprite._imageName = imageName;
                sprite._baseBitmap = ImageManager.loadSystem(imageName);
                sprite._baseBitmap.addLoadListener(() => {
                    sprite._refreshBitmap($gameVariables.value(sprite._variableId));
                });
            }
        }
    };


})();
