//=============================================================================
// NrCustomPictureMove.js
//=============================================================================

/*~struct~Group:
 * @param name
 * @text 呼び出し名
 * @type string
 * @desc グループを識別する名前（例: groupA）
 *
 * @param parent
 * @text 親ピクチャID
 * @type number
 * @desc 親ピクチャのID
 *
 * @param children
 * @text 子ピクチャID群
 * @type string
 * @desc 子ピクチャIDのリスト（例: 6,8,10-12）
 */
/*:
 * @target MZ
 * @plugindesc 複数ピクチャをグループ化して移動・フェードを同期させるプラグイン
 * @author トリアコンタン MarkⅡ
 *
 * @param groups
 * @text グループ設定
 * @type struct<Group>[]
 * @default []
 *
 * @command MoveGroup
 * @text グループ移動
 * @desc グループを移動させます（空欄引数は現在値維持）
 * @arg name
 * @text 呼び出し名
 * @type string
 * @arg x
 * @text X座標
 * @type string
 * @default ""
 * @arg y
 * @text Y座標
 * @type string
 * @default ""
 * @arg scale
 * @text 拡大率
 * @type string
 * @default ""
 * @arg duration
 * @text フレーム数
 * @type string
 * @default ""
 *
 * @command FadeGroup
 * @text グループフェード
 * @desc グループをフェードさせます（空欄引数は現在値維持）
 * @arg name
 * @text 呼び出し名
 * @type string
 * @arg opacity
 * @text 透明度
 * @type string
 * @default ""
 * @arg duration
 * @text フレーム数
 * @type string
 * @default ""
 *
 * @command AddGroup
 * @text グループ追加
 * @desc 実行時にグループを追加します
 * @arg name
 * @text 呼び出し名
 * @type string
 * @arg parent
 * @text 親ピクチャID
 * @type number
 * @arg children
 * @text 子ピクチャID群
 * @type string
 * @default ""
 *
 * @help
 * スクリプト関数:
 * movePictureGroup(name, x, y, scale, duration)
 * fadePictureGroup(name, opacity, duration)
 */

(() => {
    const pluginName = "NrCustomPictureMove";
    const params = PluginManager.parameters(pluginName);
    const groupsParam = JSON.parse(params.groups || "[]");
    const DEFAULT_DURATION = 60;
    function parseChildIds(str) {
        if (!str) return [];
        return str.split(",").map(s => s.trim()).filter(Boolean).flatMap(part => {
            if (part.includes("-")) {
                const [a,b] = part.split("-").map(t => Number(t.trim()));
                if (Number.isNaN(a) || Number.isNaN(b)) return [];
                const lo = Math.min(a,b), hi = Math.max(a,b);
                const arr = [];
                for (let i = lo; i <= hi; i++) arr.push(i);
                return arr;
            } else {
                const n = Number(part);
                return Number.isNaN(n) ? [] : [n];
            }
        });
    }
    const groups = {};
    function createGroupObj(parent, children) {
        return { parent: Number(parent), children: children, offsets: [], _move: null, _fade: null };
    }
    for (const gStr of groupsParam) {
        try {
            const g = JSON.parse(gStr);
            const name = g.name;
            const parent = Number(g.parent);
            if (!name || Number.isNaN(parent)) continue;
            groups[name] = createGroupObj(parent, parseChildIds(g.children || ""));
        } catch (e) {
        }
    }
    PluginManager.registerCommand(pluginName, "MoveGroup", args => {
        const name = args.name;
        const x = args.x !== "" ? Number(args.x) : undefined;
        const y = args.y !== "" ? Number(args.y) : undefined;
        const scale = args.scale !== "" ? Number(args.scale) : undefined;
        const duration = args.duration !== "" ? Number(args.duration) : undefined;
        moveGroup(name, x, y, scale, duration);
    });
    PluginManager.registerCommand(pluginName, "FadeGroup", args => {
        const name = args.name;
        const opacity = args.opacity !== "" ? Number(args.opacity) : undefined;
        const duration = args.duration !== "" ? Number(args.duration) : undefined;
        fadeGroup(name, opacity, duration);
    });
    PluginManager.registerCommand(pluginName, "AddGroup", args => {
        const name = args.name;
        const parent = Number(args.parent);
        const children = parseChildIds(args.children || "");
        if (!name || Number.isNaN(parent)) return;
        groups[name] = createGroupObj(parent, children);
        recordOffsetsForGroup(groups[name]);
    });
    function recordOffsetsForGroup(grp) {
        const parentPic = $gameScreen.picture(grp.parent);
        const baseX = parentPic ? parentPic.x() : 0;
        const baseY = parentPic ? parentPic.y() : 0;
        grp.offsets = grp.children.map(cid => {
            const pic = $gameScreen.picture(cid);
            return pic ? { x: pic.x() - baseX, y: pic.y() - baseY } : { x: 0, y: 0 };
        });
    }
    function ensureOffsets(grp) {
        if (!grp) return;
        if (!grp.offsets || grp.offsets.length !== grp.children.length) recordOffsetsForGroup(grp);
    }
    function moveGroup(name, x, y, scale, duration) {
        const grp = groups[name];
        if (!grp) {
            return;
        }
        const parentPic = $gameScreen.picture(grp.parent);
        if (!parentPic) {
            return;
        }
        const curX = parentPic.x();
        const curY = parentPic.y();
        const curScale = (parentPic.scaleX() + parentPic.scaleY()) / 2;
        const targetX = x !== undefined ? x : curX;
        const targetY = y !== undefined ? y : curY;
        const targetScale = scale !== undefined ? scale : curScale;
        const dur = duration !== undefined ? duration : DEFAULT_DURATION;
        $gameScreen.movePicture(grp.parent, parentPic._origin, targetX, targetY, targetScale, targetScale, parentPic.opacity(), 0, dur);
        ensureOffsets(grp);
        grp._move = {
            startX: curX,
            startY: curY,
            startScale: curScale,
            targetX: targetX,
            targetY: targetY,
            targetScale: targetScale,
            duration: Math.max(1, Math.floor(dur)),
            frame: 0
        };
    }
    function fadeGroup(name, opacity, duration) {
        const grp = groups[name];
        if (!grp) {
            return;
        }
        const parentPic = $gameScreen.picture(grp.parent);
        if (!parentPic) {
            return;
        }
        const curOp = parentPic.opacity();
        const targetOp = opacity !== undefined ? opacity : curOp;
        const dur = duration !== undefined ? duration : DEFAULT_DURATION;
        
        $gameScreen.movePicture(grp.parent, parentPic._origin, parentPic.x(), parentPic.y(), parentPic.scaleX(), parentPic.scaleY(), targetOp, 0, dur);
        grp._fade = {
            startOpacity: curOp,
            targetOpacity: targetOp,
            duration: Math.max(1, Math.floor(dur)),
            frame: 0
        };
    }
    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        _Scene_Map_update.call(this);
        for (const name in groups) {
            const grp = groups[name];
            const parentPic = $gameScreen.picture(grp.parent);
            if (!parentPic) continue;
            ensureOffsets(grp);
            const curX = parentPic.x();
            const curY = parentPic.y();
            const curScale = (parentPic.scaleX() + parentPic.scaleY()) / 2;
            if (grp._move) {
                const m = grp._move;
                m.frame++;
                const t = Math.min(1, m.frame / m.duration);
                const currentX = m.startX + (m.targetX - m.startX) * t;
                const currentY = m.startY + (m.targetY - m.startY) * t;
                const currentScale = m.startScale + (m.targetScale - m.startScale) * t;
                grp.children.forEach((cid, idx) => {
                    const pic = $gameScreen.picture(cid);
                    if (!pic) {
                        return;
                    }
                    const off = grp.offsets[idx] || { x: 0, y: 0 };
                    const nx = currentX + off.x * currentScale / 100;
                    const ny = currentY + off.y * currentScale / 100;
                    
                    pic._x = nx;
                    pic._y = ny;
                    pic._targetX = nx;
                    pic._targetY = ny;
                    pic._scaleX = currentScale;
                    pic._scaleY = currentScale;
                    pic._targetScaleX = currentScale;
                    pic._targetScaleY = currentScale;
                });
                if (m.frame >= m.duration) grp._move = null;
            }
            if (grp._fade) {
                const f = grp._fade;
                f.frame++;
                const t = Math.min(1, f.frame / f.duration);
                const curOp = Math.round(f.startOpacity + (f.targetOpacity - f.startOpacity) * t);
                
                grp.children.forEach(cid => {
                    const pic = $gameScreen.picture(cid);
                    if (!pic) {
                        return;
                    }
                    pic._opacity = curOp;
                    pic._targetOpacity = curOp;
                });
                
                if (f.frame >= f.duration) {
                    grp._fade = null;
                }
            }
        }
    };
    window.movePictureGroup = function(name, x, y, scale, duration) {
        moveGroup(String(name), x == null ? undefined : Number(x), y == null ? undefined : Number(y), scale == null ? undefined : Number(scale), duration == null ? undefined : Number(duration));
    };
    window.fadePictureGroup = function(name, opacity, duration) {
        fadeGroup(String(name), opacity == null ? undefined : Number(opacity), duration == null ? undefined : Number(duration));
    };
})();
