//=============================================================================
// NrFakeLoading_MZ.js
//=============================================================================

/*:
 * @target MZ
 * @plugindesc タイトル前にフェイクロード画面を表示するプラグイン v1.0.0
 * @author NJ
 *
 * @param gaugeFrameImage
 * @text ゲージ枠画像
 * @desc ゲージの枠画像ファイル名。空なら枠なし。
 * @default 
 *
 * @param gaugeBarImage
 * @text ゲージ本体画像
 * @desc ゲージバー画像ファイル名。空なら白矩形。
 * @default 
 *
 * @param gaugeDirection
 * @text ゲージ方向
 * @desc up/down/left/right
 * @default right
 *
 * @param gaugeX
 * @text ゲージX座標
 * @default 100
 *
 * @param gaugeY
 * @text ゲージY座標
 * @default 300
 *
 * @param text
 * @text 表示テキスト
 * @default 読み込み中...
 *
 * @param textSize
 * @text テキストサイズ(px)
 * @default 20
 *
 * @param textColor
 * @text テキストカラー
 * @default #ffffff
 *
 * @param textX
 * @text テキストX座標
 * @default 100
 *
 * @param textY
 * @text テキストY座標
 * @default 270
 *
 * @param waitTime
 * @text フェイク時間(秒) or "min-max"
 * @desc 数値単一 or "1-3"形式
 * @default 2
 *
 * @param useAnimation
 * @text アニメーション使用
 * @type boolean
 * @default true
 *
 * @param animationId
 * @text アニメーションID
 * @type number
 * @default 3
 *
 * @param animationX
 * @text アニメーションX座標
 * @default 80
 *
 * @param animationY
 * @text アニメーションY座標
 * @default 320
 *
 * @param showPercent
 * @text パーセント表示
 * @type boolean
 * @default true
 *
 * @param percentSize
 * @text パーセントサイズ(px)
 * @default 20
 *
 * @param percentX
 * @text パーセントX座標
 * @default 200
 *
 * @param percentY
 * @text パーセントY座標
 * @default 300
 *
 * @param preloadImages
 * @text プリロード画像
 * @type struct<PreloadSet>[]
 * @desc ローディング中に読み込む画像\n※量が多い時は秒数を長く調整してください。
 * @default []
 *
 * @help
 * タイトル画面へ移行する際、指定のフェイクローディング演出を挿入します。
 * また、演出中にプリロードに登録した画像を読み込みます。
 *
 * 利用規約：
 * プラグイン作者に無断で使用、改変、再配布は不可です。
 */

/*~struct~PreloadSet:
 * @param folder
 * @text フォルダ（img/以降）
 * @desc 例: pictures, faces, characters, system
 * @default pictures
 *
 * @param files
 * @text 画像ファイル名配列
 * @type string[]
 * @desc 拡張子不要。例: ["Actor1", "Enemy1"]
 * @default []
 */

(() => {
    const pluginName = 'NrFakeLoading_MZ';
    const params = PluginManager.parameters(pluginName);

    function parsePreloadImages(json) {
        try {
            const arr = JSON.parse(json);
            return arr.map(e => JSON.parse(e));
        } catch {
            return [];
        }
    }

    const cfg = {
        gaugeFrameImage: params['gaugeFrameImage'] || '',
        gaugeBarImage: params['gaugeBarImage'] || '',
        gaugeDirection: params['gaugeDirection'] || 'right',
        gaugeX: Number(params['gaugeX'] || 100),
        gaugeY: Number(params['gaugeY'] || 300),
        text: params['text'] || '読み込み中...',
        textSize: Number(params['textSize'] || 20),
        textColor: params['textColor'] || '#ffffff',
        textX: Number(params['textX'] || 100),
        textY: Number(params['textY'] || 270),
        waitTime: params['waitTime'] || '2',
        useAnimation: params['useAnimation'] === 'true',
        animationId: Number(params['animationId'] || 3),
        animationX: Number(params['animationX'] || 80),
        animationY: Number(params['animationY'] || 320),
        showPercent: params['showPercent'] === 'true',
        percentSize: Number(params['percentSize'] || 20),
        percentX: Number(params['percentX'] || 200),
        percentY: Number(params['percentY'] || 300),
        preloadImages: parsePreloadImages(params['preloadImages'] || '[]')
    };

    class Scene_NrFakeLoading extends Scene_Base {
        constructor() {
            super();
            const wt = cfg.waitTime;
            this._waitMax = /^\d+$/.test(wt)
                ? Number(wt) * 60
                : (Number(wt.split('-')[0]) + Math.random() * (Number(wt.split('-')[1]) - Number(wt.split('-')[0]))) * 60;
            this._counter = 0;
            this._fakeProgress = 0;
            this._preloadDone = false;
            this._animationStarted = false;
        }

        create() {
            super.create();
            this._gaugeGroup = new Sprite();
            this._gaugeGroup.x = cfg.gaugeX;
            this._gaugeGroup.y = cfg.gaugeY;
            this.addChild(this._gaugeGroup);

            if (cfg.gaugeBarImage) {
                const barBitmap = ImageManager.loadPicture(cfg.gaugeBarImage);
                this._bar = new Sprite(barBitmap);
            } else {
                this._bar = new Sprite(new Bitmap(200, 15));
            }
            this._gaugeGroup.addChild(this._bar);

            if (cfg.gaugeFrameImage) {
                const frameBitmap = ImageManager.loadPicture(cfg.gaugeFrameImage);
                this._frame = new Sprite(frameBitmap);
                this._gaugeGroup.addChild(this._frame);
            }

            this._text = new Sprite(new Bitmap(400, cfg.textSize + 10));
            this._text.bitmap.fontSize = cfg.textSize;
            this._text.bitmap.textColor = cfg.textColor;
            this._text.x = cfg.textX;
            this._text.y = cfg.textY;
            this.addChild(this._text);

            this._percent = new Sprite(new Bitmap(300, cfg.percentSize + 10));
            this._percent.bitmap.fontSize = cfg.percentSize;
            this._percent.bitmap.textColor = cfg.textColor;
            this._percent.x = cfg.percentX;
            this._percent.y = cfg.percentY;
            this.addChild(this._percent);

            if (cfg.useAnimation) {
                const animation = $dataAnimations && $dataAnimations[cfg.animationId];
                if (animation && (animation.effectName || Array.isArray(animation.timings))) {
                    const dummy = new Sprite();
                    dummy.x = cfg.animationX;
                    dummy.y = cfg.animationY;
                    this.addChild(dummy);

                    const animationSprite = new Sprite_Animation();
                    animationSprite.setup([dummy], animation, false, 0);
                    animationSprite.targetObjects = [dummy];
                    this.addChild(animationSprite);

                    this._animTarget = dummy;
                    this._animationStarted = true;
                } else {
                    console.warn('Animation not found or invalid timings:', cfg.animationId);
                    this._animTarget = null;
                }
            }
        }

        update() {
            super.update();
            this._counter++;

            if (!this._preloadDone) {
                this._preloadDone = true;
                for (const set of cfg.preloadImages) {
                    if (set.files && Array.isArray(set.files)) {
                        for (const name of set.files) {
                            ImageManager.loadBitmap(`img/${set.folder}/`, name);
                        }
                    }
                }
            }

            if (this._fakeProgress < 1) {
                const remain = Math.max(this._waitMax - this._counter, 1);
                const rest = 1 - this._fakeProgress;
                const minSpeed = rest / remain;
                this._fakeProgress += minSpeed + Math.random() * minSpeed;
                this._fakeProgress = Math.min(this._fakeProgress, 1);
            }

            const rate = this._fakeProgress;
            if (!cfg.gaugeBarImage) {
                this._bar.bitmap.clear();
                const w = this._bar.bitmap.width;
                const h = this._bar.bitmap.height;
                switch (cfg.gaugeDirection) {
                    case 'right': this._bar.bitmap.fillRect(0, 0, w * rate, h, '#ffffff'); break;
                    case 'left': this._bar.bitmap.fillRect(w * (1 - rate), 0, w * rate, h, '#ffffff'); break;
                    case 'down': this._bar.bitmap.fillRect(0, 0, w, h * rate, '#ffffff'); break;
                    case 'up': this._bar.bitmap.fillRect(0, h * (1 - rate), w, h * rate, '#ffffff'); break;
                }
            } else {
                const w = this._bar.bitmap.width;
                const h = this._bar.bitmap.height;
                this._bar.setFrame(0, 0, w * rate, h);
            }

            this._text.bitmap.clear();
            this._text.bitmap.drawText(cfg.text, 0, 0, 400, cfg.textSize + 10, 'left');

            if (cfg.showPercent) {
                this._percent.bitmap.clear();
                this._percent.bitmap.drawText(`${Math.floor(rate * 100)}%`, 0, 0, 300, cfg.percentSize + 10, 'right');
            }

            if (this._counter >= this._waitMax && rate >= 1) {
                SceneManager.goto(Scene_Title);
            }
        }
    }

    const _SceneManager_goto = SceneManager.goto;
    SceneManager._fakeLoadingDone = false;
    SceneManager.goto = function(sceneClass) {
        if (sceneClass === Scene_Title && !this._fakeLoadingDone) {
            this._fakeLoadingDone = true;
            _SceneManager_goto.call(this, Scene_NrFakeLoading);
            return;
        }
        _SceneManager_goto.call(this, sceneClass);
    };

    const removeSpinner = () => {
        const spinner = document.getElementById('loadingSpinner');
        if (spinner && spinner.parentNode) {
            spinner.parentNode.removeChild(spinner);
        }
    };

    document.addEventListener('DOMContentLoaded', () => {
        removeSpinner();
        setTimeout(removeSpinner, 100);
        setTimeout(removeSpinner, 300);
    });

    window.addEventListener('load', removeSpinner);
})();
