//=============================================================================
// NrNovelGamePlus.js
//=============================================================================

/*:
 * @target MZ
 * @plugindesc ログ追加プラグイン V1.0.0
 * @author NJ
 *
 * @param LogWindowBackground
 * @text ログ画面背景画像
 * @desc ログ全体の背景画像（img/pictures/ に配置）
 * @default
 *
 * @param NameColor
 * @text 名前の色コード
 * @desc 名前表示部分の文字色（例：#FF0000）
 * @default #000000
 *
 * @param NameFontSize
 * @text 名前のフォントサイズ
 * @desc 名前表示のフォントサイズ（例：28）
 * @type number
 * @default 28
 *
 * @param LogColor
 * @text ログ本文の色コード
 * @desc ログ本文の文字色（例：#333333）
 * @default #000000
 *
 * @param LogFontSize
 * @text ログ本文のフォントサイズ
 * @desc ログ本文のフォントサイズ（例：22）
 * @type number
 * @default 22
 *
 * @param CloseButtonImage
 * @text 閉じるボタン画像
 * @desc img/pictures/ に配置する閉じるボタン画像ファイル名（拡張子除く）
 * @default
 * 
 * @param CloseButtonOffsetX
 * @text 閉じるボタンX座標オフセット
 * @type number
 * @default 0
 *
 * @param CloseButtonOffsetY
 * @text 閉じるボタンY座標オフセット
 * @type number
 * @default 0
 *
 * @param PlayButtonImage
 * @text 再生ボタン画像
 * @desc img/pictures/ に配置する再生ボタン画像ファイル名（拡張子除く）
 * @default
 * 
 * @param VoiceButtonOffsetX
 * @text 再生ボタンX座標オフセット
 * @type number
 * @default 0
 *
 * @param VoiceButtonOffsetY
 * @text 再生ボタンY座標オフセット
 * @type number
 * @default 0
 * 
 * @param NamedBlockBackground
 * @text 名前ありログ背景画像
 * @desc img/pictures/ に配置する名前付きログ用背景画像ファイル名（拡張子除く）
 * @default
 * 
 * @param NormalBlockBackground
 * @text 名前なしログ背景画像
 * @desc img/pictures/ に配置する通常ログ用背景画像ファイル名（拡張子除く）
 * @default
 *
 * @param FaceIcons
 * @text 顔画像登録リスト
 * @type struct<FaceIcon>[]
 * @desc \NrN[名前]で指定できる顔画像の登録（例：名前→画像ファイル）
 * @default []
 *
 * @param IconWidth
 * @text アイコン横幅
 * @type number
 * @default 96
 *
 * @param IconHeight
 * @text アイコン高さ
 * @type number
 * @default 96
 *
 * @param IconOffsetX
 * @text アイコンX座標オフセット
 * @type number
 * @default 30
 *
 * @param IconOffsetY
 * @text アイコンY座標オフセット
 * @type number
 * @default 10
 *
 * @param LineSpacing
 * @text 行間スペース
 * @type number
 * @default 4
 *
 * @param BlockSpacing
 * @text ブロック間スペース
 * @type number
 * @default 16
 *
 * @command LogStart
 * @text ログ記録開始
 * @desc ログの記録を開始します。
 *
 * @command LogStop
 * @text ログ記録停止
 * @desc ログの記録を停止します。
 *
 * @command LogClear
 * @text ログ削除
 * @desc ログを全て削除します。
 *
 * @command ShowLog
 * @text ログ表示
 * @desc ログウィンドウを表示します。
 *
 * @command SetLogName
 * @text 名前設定
 * @desc メッセージブロックの名前を設定します。
 *
 * @arg name
 * @type string
 * @text 名前
 * @desc 表示するキャラクター名など
 *
 * @help
 *  このプラグインはログ機能を追加します。
 *
 * 使い方:
 *  「ログ記録開始」をしたタイミングから「ログ記録停止」をするまでに文章の表示で表示された
 *  テキストをログに保存します。
 *
 *  保存されたテキストは「ログ表示」で表示ができます。
 *
 *  保存されたテキストを消去したい時は「ログ削除」で消してください。
 *  動作が重くなるのを防ぎたい場合は一つのイベントが終わる度に削除を推奨します。
 *
 * ボイス登録:
 *  テキスト内に「\SV[n]」があればログからnに登録されているものを再生します。
 *  再生されるものは「audio/voice/」に入っているものが再生されます。
 *
 * アイコン登録:
 *  テキスト内に「\NrN[n]」があればnに入っている文字と同じ登録名のものが
 *  呼び出されます。
 *
 *
 * 注意:
 *  ウィンドウの表示はUIエリアの幅と広さを参照しているため、画面サイズとUIエリアの数値は
 *  同じようにしてください。
 *
 *  NrSimpleVoiceControlと併用する場合は、このプラグインをNrSimpleVoiceControlより
 *  下においてください。
 *
 *  「名前設定」は確認用なので使用しないでください。
 *
 * バージョン
 *   1.0.0 - 初回
 * 
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可です。
 */

/*~struct~FaceIcon:
 * @param name
 * @text 登録名
 * @desc \NrN[登録名] で指定する名前
 *
 * @param file
 * @text 顔画像ファイル名
 * @desc img/pictures/ に置いた画像ファイル名（拡張子なし）
 */


(() => {
	'use strict';
    const params = PluginManager.parameters("NrNovelGamePlus");
    const CloseButtonImage      = params["CloseButtonImage"]      || "";
    const PlayButtonImage       = params["PlayButtonImage"]       || "";
    const NamedBlockBackground  = params["NamedBlockBackground"]  || "";
    const NormalBlockBackground = params["NormalBlockBackground"] || "";
    const NameColor             = params["NameColor"]             || "#000000";
    const NameFontSize          = Number(params["NameFontSize"]   || 28);
    const LogColor              = params["LogColor"]              || "#000000";
    const LogFontSize           = Number(params["LogFontSize"]    || 22);
    const FaceIcons = JSON.parse(params["FaceIcons"] || "[]").map(JSON.parse);
    const faceIconMap = {};
    for (const icon of FaceIcons) {
        faceIconMap[icon.name] = icon.file;
    }
    const IconWidth    = Number(params["IconWidth"]    || 96);
    const IconHeight   = Number(params["IconHeight"]   || 96);
    const IconOffsetX  = Number(params["IconOffsetX"]  || 30);
    const IconOffsetY  = Number(params["IconOffsetY"]  || 10);
    const LogWindowBackground = params["LogWindowBackground"] || "";
    const LineSpacing = Number(params["LineSpacing"] || 4);
    const BlockSpacing = Number(params["BlockSpacing"] || 16);
    const VoiceButtonOffsetX = Number(params["VoiceButtonOffsetX"] || 0);
    const VoiceButtonOffsetY = Number(params["VoiceButtonOffsetY"] || 0);
    const CloseButtonOffsetX = Number(params["CloseButtonOffsetX"] || 0);
    const CloseButtonOffsetY = Number(params["CloseButtonOffsetY"] || 0);

    let isLogging = false;
    let messageLogBuffer = [];
    let currentMessageBlock = { name: "", messages: [], voice: null, iconName: null };
    window.lastLogVoice = null;

    PluginManager.registerCommand("NrNovelGamePlus", "LogStart", () => {
        isLogging = true;

        if (CloseButtonImage)      ImageManager.loadPicture(CloseButtonImage);
        if (PlayButtonImage)       ImageManager.loadPicture(PlayButtonImage);
        if (NamedBlockBackground)  ImageManager.loadPicture(NamedBlockBackground);
        if (NormalBlockBackground) ImageManager.loadPicture(NormalBlockBackground);
    });

    PluginManager.registerCommand("NrNovelGamePlus", "LogStop", () => {
        isLogging = false;
    });

    PluginManager.registerCommand("NrNovelGamePlus", "LogClear", () => {
        $gameMessage.clearMessageLog();
    });

    PluginManager.registerCommand("NrNovelGamePlus", "ShowLog", () => {
        SceneManager.push(Scene_MessageLog);
    });

    PluginManager.registerCommand("NrNovelGamePlus", "SetLogName", args => {
        if (isLogging) {
            currentMessageBlock.name = `【${args.name}】`;
        }
    });


    const _Window_Message_terminateMessage = Window_Message.prototype.terminateMessage;
    Window_Message.prototype.terminateMessage = function() {
        if (isLogging && (currentMessageBlock.messages.length > 0 || currentMessageBlock.name || currentMessageBlock.voice)) {
            messageLogBuffer.push(currentMessageBlock);
        }
        currentMessageBlock = { name: "", messages: [], voice: null };
        _Window_Message_terminateMessage.call(this);
    };

    const _Game_Message_add = Game_Message.prototype.add;
    Game_Message.prototype.add = function(text) {
        const originalText = text;
        let displayText = originalText;

        if (isLogging) {
            if (!PluginManager._scripts.includes("NrSimpleVoiceControl")) {
                const m = originalText.match(/\\SV\[\s*(?:\\v\[(\d+)\]|(\d+))\s*\]/i);
                if (m) {
                    const vid = m[1] || m[2];
                    const voiceName = String($gameVariables.value(Number(vid)));
                    if (!currentMessageBlock.voice) {
                        currentMessageBlock.voice = voiceName;
                    }
                }
            }

            const processed = processControlCharacters(originalText);
            const autoBroken = Window_Base.prototype.convertEscapeCharacters.call(
                new Window_Base(new Rectangle()), processed
            );
            currentMessageBlock.messages.push(autoBroken);

            if (!currentMessageBlock.name && this._speakerName) {
                const nm = processControlCharacters(this._speakerName);
                currentMessageBlock.name = `【${nm}】`;
            }
        }

        if (!PluginManager._scripts.includes("NrSimpleVoiceControl")) {
            displayText = originalText.replace(/\\SV\[((?:\\v\[[^\]]+\]|[^\]])*)\]/gi, "");
        }

        _Game_Message_add.call(this, displayText);
    };

    Game_Message.prototype.getMessageLog = function() {
        return messageLogBuffer;
    };
    Game_Message.prototype.clearMessageLog = function() {
        messageLogBuffer = [];
    };

    function processControlCharacters(text) {
        text = text.replace(
            /\\SV\[\s*(?:\\v\[(\d+)\]|(\d+))\s*\]/gi,
            (_, v1, v2) => {
                const varId = v1 || v2;
                const resolved = varId ? String($gameVariables.value(Number(varId))) : "";
                if (isLogging && resolved) currentMessageBlock.voice = resolved;
                return "";
            }
        );

        text = text.replace(/\\[nN]\[(\d+)\]/g, (_, id) => {
            const actor = $gameActors.actor(Number(id));
            return actor ? actor.name() : "";
        });

        text = text.replace(/\\NW\[(.+?)\]/g, (_, name) => {
            if (isLogging) currentMessageBlock.name = `【${name.trim()}】`;
            return "";
        });

        text = text.replace(
            /\\NrN\[\s*(?:\\v\[(\d+)\]|(\S+?))\s*\]/gi,
            (_, v1, str) => {
                const resolved = v1 ? String($gameVariables.value(Number(v1))) : str;
                if (isLogging && resolved) currentMessageBlock.iconName = resolved;
                return "";
            }
        );

        text = text.replace(/\\SN\[(\d+)\]/gi, (_, id) => {
            const skill = $dataSkills[Number(id)];
            return skill ? skill.name : '';
        });

        text = text.replace(/\\IN\[(\d+)\]/gi, (_, id) => {
            const item = $dataItems[Number(id)];
            return item ? item.name : '';
        });

        text = text.replace(/\\WN\[(\d+)\]/gi, (_, id) => {
            const weapon = $dataWeapons[Number(id)];
            return weapon ? weapon.name : '';
        });

        text = text.replace(/\\AN\[(\d+)\]/gi, (_, id) => {
            const armor = $dataArmors[Number(id)];
            return armor ? armor.name : '';
        });

        text = text.replace(/\\RB\[(.+?),(.+?)\]/gi, (_, base, ruby) => {
            return base;
        });

        const unusedPatterns = [
            /\\SP\[[^\]]*\]/gi,
            /\\AT\[[^\]]*\]/gi,
            /\\SET\[[^\]]*\]/gi,
            /\\CO\[[^\]]*\]/gi,
            /\\MX\[[^\]]*\]/gi,
            /\\MY\[[^\]]*\]/gi,
            /\\PX\[[^\]]*\]/gi,
            /\\PY\[[^\]]*\]/gi,
            /\\SW\[[^\]]*\]/gi,
            /\\FO\[[^\]]*\]/gi,
            /\\OC\[[^\]]*\]/gi,
            /\\OP\[[^\]]*\]/gi,
            /\\OW\[[^\]]*\]/gi,
            /\\RC\[[^\]]*\]/gi
        ];

        const unusedNoArg = [
            /\\FB/gi,
            /\\FI/gi,
            /\\DF/gi,
            /\\SV/gi,
            /\\LD/gi,
            /\\A/gi,
            /\\ES/gi,
            /\\WE/gi
        ];

        unusedPatterns.forEach(pattern => {
            text = text.replace(pattern, '');
        });

        unusedNoArg.forEach(pattern => {
            text = text.replace(pattern, '');
        });

        return text;
    }

    function Scene_MessageLog() { 
	    this.initialize(...arguments);
    }

    Scene_MessageLog.prototype = Object.create(Scene_MenuBase.prototype);
    Scene_MessageLog.prototype.constructor = Scene_MessageLog;

    Scene_MessageLog.prototype.initialize = function() {
        Scene_Base.prototype.initialize.call(this);
    };

    Scene_MessageLog.prototype.create = function() {
        Scene_Base.prototype.create.call(this);
        this.createBackground();
        this.createLogWindow();
        this.createReturnButton();
        this._titleWindow = new Window_MessageLogTitle();
        this._logWindow.scrollToBottom();
        if (LogWindowBackground) this._titleWindow.visible = false;
        this.addChild(this._titleWindow);
    };

    Scene_MessageLog.prototype.createBackground = function() {
        if (LogWindowBackground) {
            const bitmap = ImageManager.loadPicture(LogWindowBackground);
            this._backgroundSprite = new Sprite(bitmap);
        } else {
            this._backgroundSprite = new Sprite();
            this._backgroundSprite.bitmap = SceneManager.backgroundBitmap();
            this._backgroundSprite.opacity = 192;
        }
        this.addChild(this._backgroundSprite);
    };

    Scene_MessageLog.prototype.createLogWindow = function() {
        this._logWindow = new Window_MessageLog();
		this.addChild(this._logWindow);
    };

    Scene_MessageLog.prototype.createReturnButton = function() {
        if (CloseButtonImage) {
            const sprite = new Sprite(ImageManager.loadPicture(CloseButtonImage));
            sprite.update = function() {
                Sprite.prototype.update.call(this);
                if (TouchInput.isTriggered()) {
                    const x = TouchInput.x;
                    const y = TouchInput.y;
                    if (x >= this.x && x <= this.x + this.width && y >= this.y && y <= this.y + this.height) {
                        if (this._click) this._click();
                    }
                }
            };
            sprite.setClickHandler = function(cb) {
                this._click = cb;
            };
            sprite.setClickHandler(() => {
                if (AudioManager.stopVoice && window.lastLogVoice) {
                    AudioManager.stopVoice(window.lastLogVoice, null);
                    window.lastLogVoice = null;
                }
                SceneManager.pop();
            });

            this._returnButton = sprite;
            this.addChild(sprite);

            sprite.bitmap.addLoadListener(() => {
                sprite.x = (Graphics.boxWidth - sprite.width) / 2 + CloseButtonOffsetX;
                sprite.y = Graphics.boxHeight - sprite.height - 20 + CloseButtonOffsetY;
            });
        } else {
            const btnW = Math.floor(Graphics.boxWidth * 0.25);
            const btnH = 50;
            const btn = new Sprite(new Bitmap(btnW, btnH));
            btn.bitmap.fillRect(0, 0, btnW, btnH, 'rgba(0,0,0,0.6)');
            btn.bitmap.drawText('閉じる', 0, 0, btnW, btnH, 'center');

            btn.x = (Graphics.boxWidth - btnW) / 2 + CloseButtonOffsetX;
            btn.y = Graphics.boxHeight - btnH - 20 + CloseButtonOffsetY;

            btn.update = function() {
                Sprite.prototype.update.call(this);
                if (TouchInput.isTriggered()) {
                    const x = TouchInput.x;
                    const y = TouchInput.y;
                    if (x >= this.x && x <= this.x + btnW && y >= this.y && y <= this.y + btnH) {
                        if (AudioManager.stopVoice && window.lastLogVoice) {
                            AudioManager.stopVoice(window.lastLogVoice, null);
                            window.lastLogVoice = null;
                        }
                        SceneManager.pop();
                    }
                }
            };

            this._returnButton = btn;
            this.addChild(btn);
        }
    };

    function Window_MessageLog() { 
	    this.initialize(...arguments);
    }

    Window_MessageLog.prototype = Object.create(Window_Scrollable.prototype);
    Window_MessageLog.prototype.constructor = Window_MessageLog;
    Window_MessageLog.prototype.initialize = function() {
        const marginX = Math.floor(Graphics.boxWidth * 0.05);
        const marginY = Math.floor(Graphics.boxHeight * 0.08);
        const width   = Graphics.boxWidth - marginX * 2;
        const height  = Graphics.boxHeight - marginY * 2;
        const x       = marginX;
        const y       = marginY;

        const rect = new Rectangle(x, y, width, height);
        Window_Scrollable.prototype.initialize.call(this, rect);

        this._scrollY = 0;
        this._blockIndex = 0;
        this._blockOffsets = [];

        this._scrollDragging = false;
        this._dragStartY = 0;
        this._dragStartScrollY = 0;

        this.createScrollBar();

        if (LogWindowBackground) {
            this.opacity = 0;
            this.backOpacity = 0;
            this.contentsOpacity = 255;
        }

        this.refresh();
    };

    Window_MessageLog.prototype.createScrollBar = function () {
        const barW = 8;
        const trackH = this.height - 16;
        const offsetX = -10;
        const offsetY = 8;

        this._scrollBarBack = new Sprite(new Bitmap(barW, trackH));
        this._scrollBarBack.bitmap.fillAll('rgba(0,0,0,0.2)');
        this._scrollBarBack.x = this.width - barW + offsetX;
        this._scrollBarBack.y = offsetY;
        this.addChild(this._scrollBarBack);

        const knobH = 40;
        this._scrollBar = new Sprite(new Bitmap(barW, knobH));
        this._scrollBar.bitmap.fillAll('rgba(255,255,255,0.9)');
        this._scrollBar.x = this._scrollBarBack.x;
        this._scrollBar.y = offsetY;
        this.addChild(this._scrollBar);

        this._scrollBarTrackHeight = trackH;
        this._scrollBarOffsetY = offsetY;
    };

    Window_MessageLog.prototype.drawLogBlockBackground = function(y, h, hasName) {
        const img = hasName ? NamedBlockBackground : NormalBlockBackground;
        if (img) {
            const bmp = ImageManager.loadPicture(img);
            if (bmp.isReady()) {
                const ctx = this.contents.context;

                const dw = this.contents.width;
                const dh = h;

                ctx.save();
                ctx.imageSmoothingEnabled = true;
                ctx.drawImage(
                    bmp.canvas,
                    0, 0, bmp.width, bmp.height,
                    0, y, dw, dh
                );
                ctx.restore();
                this.contents._baseTexture.update();
            } else {
                bmp.addLoadListener(() => this.refresh());
            }
        } else {
            this.contents.fillRect(0, y, this.contents.width, h, '#ffffff');
        }
    };

    Window_MessageLog.prototype.drawLogBlock = function(block, y, hblock) {
        const hasName  = !!block.name;
        const hasVoice = !!block.voice;
        const lh       = this.lineHeight();
        const bp       = 10;
        const cw       = this.contents.width;
        const mx = Math.floor(this.contents.width * 0.2);

        this.drawLogBlockBackground(y, hblock - BlockSpacing, hasName);

        if (block.iconName && faceIconMap[block.iconName]) {
            const path = faceIconMap[block.iconName];
            const folder = path.includes("/") ? path.substring(0, path.lastIndexOf("/") + 1) : "";
            const filename = path.includes("/") ? path.substring(path.lastIndexOf("/") + 1) : path;
            const bitmap = ImageManager.loadBitmap("img/pictures/" + folder, filename);


            if (bitmap.isReady()) {
                const ctx = this.contents.context;
                ctx.save();
                ctx.imageSmoothingEnabled = false;
                ctx.drawImage(
                    bitmap.canvas,
                    0, 0, bitmap.width, bitmap.height,
                    IconOffsetX, y + IconOffsetY,
                    IconWidth, IconHeight
                );
                ctx.restore();
                this.contents._baseTexture.update();
            } else {
                bitmap.addLoadListener(() => {
                    block._cacheBitmap = null;
                    this.refresh();
                });
            }
        }

            let iy = y + bp;
            if (hasName) {
                this.contents.fontSize = NameFontSize;
                const textState = this.createTextState(block.name, mx, iy, cw - mx * 2);
                textState.drawing = true;
                this.processAllText(textState);
                iy += lh + LineSpacing;
                iy += lh + LineSpacing / 2;
            } else {
                iy += lh / 2;
            }

            this.contents.fontSize = LogFontSize;
            this.changeTextColor(LogColor);
            const linesArr = block.messages.slice(0, 4);
            for (let j = 0; j < linesArr.length; j++) {
                const line = processControlCharacters(linesArr[j]);
                const ts = this.createTextState(line, mx, iy, cw - mx * 2);
                ts.drawing = true;
                this.processAllText(ts);
                iy += lh + LineSpacing;
            }

        if (hasVoice && block.voice) {
            const btnText = "▶ 再生";
            const bmp     = PlayButtonImage ? ImageManager.loadPicture(PlayButtonImage) : null;
            const btnW    = bmp && bmp.isReady() ? bmp.width  : this.textWidth(btnText);
            const btnH    = bmp && bmp.isReady() ? bmp.height : lh;
            const btnX    = mx - btnW - 10 + VoiceButtonOffsetX;
            const btnY    = y + bp + Math.floor((NameFontSize - btnH) / 2) + VoiceButtonOffsetY;

            if (bmp && bmp.isReady()) {
                this.contents.blt(bmp, 0, 0, btnW, btnH, btnX, btnY);
            } else if (bmp) {
                bmp.addLoadListener(() => this.refresh());
            } else {
                this.contents.fontSize = LogFontSize;
                this.changeTextColor(LogColor);
                this.contents.drawText(btnText, btnX, btnY, btnW, btnH, "left");
            }

            this._interactiveButtons.push({
                rect: {
                    x: btnX,
                    y: btnY,
                    width: btnW,
                    height: btnH
                },
                callback: () => {
                    AudioManager.playVoice({
                        name: block.voice,
                        volume: ConfigManager.voiceVolume || 90,
                        pitch: 100,
                        pan: 0
                    }, false, 0);
					window.lastLogVoice = block.voice;
                }
            });
        }
    };

    Window_MessageLog.prototype.refresh = function () {
        this.children.filter(c => c._isPlayButton).forEach(c => this.removeChild(c));
        this._interactiveButtons = [];
        this.contents.clear();

        const messages = $gameMessage.getMessageLog();
        const lh = this.lineHeight();
        const bm = 16;
        const bp = 10;
        const cw = this.contents.width;
        const mx = Math.floor(this.contents.width * 0.3); // 幅の30%を左余白に


        let totalH = 0;
        const blockHeights = [];

        for (let i = 0; i < messages.length; i++) {
            const block = messages[i];
            const lines = block.name ? 6 : 4;
            const hblock = lines * (lh + LineSpacing) + bp * 2 + BlockSpacing;
            blockHeights[i] = hblock;
            totalH += hblock;
        }
        this._contentHeight = totalH;

        const viewTop = this._scrollY;
        const viewBottom = viewTop + this.contentsHeight();

        let y = -this._scrollY;
        for (let i = 0; i < messages.length; i++) {
            const block = messages[i];
            const hblock = blockHeights[i];
            const blockTop = y + this._scrollY;
            const blockBottom = blockTop + hblock;

            if (blockBottom > viewTop && blockTop < viewBottom) {
                this.drawLogBlock(block, y, hblock);
            }

            y += hblock;
        }

        this.updateScrollBar();

    };

    Window_MessageLog.prototype.updateScrollBar = function () {
        const contentHeight = this._contentHeight;
        const viewHeight = this.contentsHeight();
        const trackH = this._scrollBarTrackHeight;
        const offsetY = this._scrollBarOffsetY;
        const barW = 8;

        if (contentHeight <= viewHeight) {
            this._scrollBar.visible = false;
            this._scrollBarBack.visible = false;
            return;
        }

        const ratio = viewHeight / contentHeight;
        const knobH = Math.max(32, trackH * ratio);
        const maxScroll = contentHeight - viewHeight;
        const scrollRatio = this._scrollY / maxScroll;
        const y = offsetY + scrollRatio * (trackH - knobH);

        if (this._scrollBar.height !== knobH) {
            this._scrollBar.bitmap = new Bitmap(barW, knobH);
            this._scrollBar.bitmap.fillAll('rgba(255,255,255,0.9)');
        }

        this._scrollBar.y = y;

        this._scrollBar.visible = true;
        this._scrollBarBack.visible = true;
    };

    Window_MessageLog.prototype.scrollToBottom = function() {
        const maxScroll = Math.max(0, this._contentHeight - this.contentsHeight());
        this._scrollY = maxScroll;
        this.refresh();
    };

    function Window_MessageLogTitle() { this.initialize(...arguments); }
    Window_MessageLogTitle.prototype=Object.create(Window_Scrollable.prototype);
    Window_MessageLogTitle.prototype.constructor=Window_MessageLogTitle;
    Window_MessageLogTitle.prototype.initialize = function() {
        const w = Math.floor(Graphics.boxWidth * 0.4);
        const h = 64;
        const x = (Graphics.boxWidth - w) / 2;
        const y = 0;

        Window_Scrollable.prototype.initialize.call(this, new Rectangle(x, y, w, h));
        this.opacity = 255;
        this.backOpacity = 192;
        this.contentsOpacity = 255;
        this.refresh();
    };

    Window_MessageLogTitle.prototype.refresh=function(){
        this.contents.clear(); this.contents.fontSize=30;
        this.changeTextColor(ColorManager.normalColor());
        const t="バックログ", tw=this.textWidth(t+"　");
        const ox=(this.width-this.padding-tw)/2;
        this.drawText(t,ox,3,tw,'left'); this.resetTextColor();
    };

    Window_MessageLog.prototype.scrollDown = function (amount) {
        const maxScroll = Math.max(0, this._contentHeight - this.contentsHeight());
        this._scrollY = Math.min(this._scrollY + amount, maxScroll);
        this.refresh();
    };

    Window_MessageLog.prototype.scrollUp = function (amount) {
        this._scrollY = Math.max(0, this._scrollY - amount);
        this.refresh();
    };

    Window_MessageLog.prototype.update = function () {
        Window_Base.prototype.update.call(this);

        const scrollAmount = 40;
        if (TouchInput.wheelY > 0) this.scrollDown(scrollAmount);
        else if (TouchInput.wheelY < 0) this.scrollUp(scrollAmount);

        const knob = this._scrollBar;
        const localWinX = TouchInput.x - this.x;
        const localWinY = TouchInput.y - this.y;

        if (TouchInput.isTriggered()) {
            const hit =
                localWinX >= knob.x && localWinX <= knob.x + knob.width &&
                localWinY >= knob.y && localWinY <= knob.y + knob.height;
            if (hit) {
                this._scrollDragging = true;
                this._dragStartY = localWinY;
                this._dragStartScrollY = this._scrollY;
            }
        }

        if (this._scrollDragging && TouchInput.isPressed()) {
            const contentHeight = this._contentHeight;
            const viewHeight = this.contentsHeight();
            const maxScroll = Math.max(0, contentHeight - viewHeight);

            const trackH = this.height;
            const knobH = this._scrollBar.height;
            const denom = Math.max(1, (trackH - knobH));
            const ratio = maxScroll / denom;

            const delta = localWinY - this._dragStartY;
            this._scrollY = Math.max(0, Math.min(this._dragStartScrollY + delta * ratio, maxScroll));
            this.refresh();
        }

        if (!TouchInput.isPressed()) {
            this._scrollDragging = false;
        }

        if (TouchInput.isTriggered()) {
            const localX = TouchInput.x - this.x - this.padding;
            const localY = TouchInput.y - this.y - this.padding;

            for (const btn of this._interactiveButtons) {
                const r = btn.rect;
                if (localX >= r.x && localX <= r.x + r.width &&
                    localY >= r.y && localY <= r.y + r.height) {
                    if (btn.callback) btn.callback();
                    break;
                }
            }
        }
    };

    window.NovelGameLog = {
        start: function() {
            isLogging = true;
            if (CloseButtonImage)      ImageManager.loadPicture(CloseButtonImage);
            if (PlayButtonImage)       ImageManager.loadPicture(PlayButtonImage);
            if (NamedBlockBackground)  ImageManager.loadPicture(NamedBlockBackground);
            if (NormalBlockBackground) ImageManager.loadPicture(NormalBlockBackground);
        },
        stop: function() {
            isLogging = false;
        },
        clear: function() {
            $gameMessage.clearMessageLog();
        },
        show: function() {
            SceneManager.push(Scene_MessageLog);
        },
        setName: function(name) {
            if (isLogging) {
                currentMessageBlock.name = `【${name}】`;
            }
        }
    };

    const _Window_Message_convertEscapeCharacters = Window_Message.prototype.convertEscapeCharacters;
    Window_Message.prototype.convertEscapeCharacters = function(text) {
        text = text.replace(
            /\\NrN\[\s*(?:\\v\[(\d+)\]|([^\]]+))\s*\]/gi,
            (_, v1, str) => {
                const resolved = v1 ? String($gameVariables.value(Number(v1))) : str;
                if (isLogging && resolved) currentMessageBlock.iconName = resolved.trim();
                return "";
            }
        );

        return _Window_Message_convertEscapeCharacters.call(this, text);
    };

})();
