//=============================================================================
// NrOptionNumber.js
//=============================================================================

/*:
 * @target MZ
 * @plugindesc ピクチャに手入力可能な数字を表示 v1.0.0
 * @author NJ
 *
 * @param lists
 * @text フィールドリスト設定
 * @type struct<FieldList>[]
 * @desc 複数のフィールドリストを定義します。スクリプトで呼び出せます。
 *
 * @help
 * $gameSystem.showNrOptionNumber("リスト名"); 表示
 * $gameSystem.hideNrOptionNumber();          非表示
 * 
 * バージョン
 *   1.0.0 - 初回
 *
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可です。
*/

/*~struct~FieldList:
 * @param name
 * @text リスト名
 * @type string
 *
 * @param fields
 * @text フィールド定義
 * @type struct<Field>[]
 */

/*~struct~Field:
 * @param pictureId
 * @text ピクチャ番号
 * @type number
 *
 * @param variableId
 * @text 変数ID
 * @type variable
 *
 * @param min
 * @text 最小値
 * @type number
 * @default 0
 *
 * @param max
 * @text 最大値
 * @type number
 * @default 999999
 *
 * @param x
 * @text 表示X座標
 * @type number
 *
 * @param y
 * @text 表示Y座標
 * @type number
 *
 * @param align
 * @text 揃え
 * @type select
 * @option left
 * @option center
 * @option right
 * @default left
 *
 * @param fontSize
 * @text 文字サイズ
 * @type number
 * @default 28
 *
 * @param fontColor
 * @text 文字色
 * @type string
 * @default #FFFFFF
 *
 * @param outlineColor
 * @text アウトライン色
 * @type string
 * @default #000000
 *
 * @param outlineWidth
 * @text アウトライン太さ
 * @type number
 * @default 2
 *
 * @param fontBold
 * @text 太文字
 * @type boolean
 * @default false
 * @desc trueにすると文字を太文字（ボールド）で表示します
 *
 * @param script
 * @text 実行スクリプト
 * @type note
 */

(() => {
    const pluginName = "NrOptionNumber";
    const parameters = PluginManager.parameters(pluginName);

    function parseStructList(list) {
        try { return JSON.parse(list).map(s => JSON.parse(s)); } catch { return []; }
    }

    const rawLists = parseStructList(parameters["lists"] || "[]");
    const lists = {};
    rawLists.forEach(list => { lists[list.name] = parseStructList(list.fields || "[]"); });

    let _currentFields = [];
    let _keyboardInput = "";
    
    document.addEventListener("keydown", function(event) {
        if (_currentFields.length === 0) return;
        const editing = _currentFields.find(f => f._editing);
        if (!editing) return;
        
        event.preventDefault();
        
        if (event.key >= "0" && event.key <= "9") {
            _keyboardInput = event.key;
        }
        else if (event.key === "Enter") {
            _keyboardInput = "enter";
        }
        else if (event.key === "Backspace") {
            _keyboardInput = "backspace";
        }
        else if (event.key === "Escape") {
            _keyboardInput = "escape";
        }
    });

    class NrOptionField {
        constructor(cfg) {
            this.cfg = cfg;
            this.pictureId = Number(cfg.pictureId);
            this.variableId = Number(cfg.variableId);
            this.min = Number(cfg.min);
            this.max = Number(cfg.max);
            this.x = Number(cfg.x);
            this.y = Number(cfg.y);
            this.align = cfg.align;
            this.fontSize = Number(cfg.fontSize);
            this.fontColor = cfg.fontColor;
            this.outlineColor = cfg.outlineColor;
            this.outlineWidth = Number(cfg.outlineWidth);
            this.fontBold = cfg.fontBold === "true" || cfg.fontBold === true;
            this.script = cfg.script ? JSON.parse(cfg.script) : "";
            this._editing = false;
            this._input = "";
            this._lastValue = null;
            this._scriptExecuted = false;

            const width = 200;
            const height = this.fontSize + this.outlineWidth * 2 + 4;
            this._bitmap = new Bitmap(width, height);
            this._bitmap.clear();
            $gameScreen.showPicture(this.pictureId, "", 0, this.x, this.y, 100, 100, 255, 0);
            
            const picture = $gameScreen.picture(this.pictureId);
            if (picture) {
                picture._customBitmap = this._bitmap;
            }
            
            this.refreshBitmap();
            this.setupSprite();
        }

        setupSprite() {
            const container = this.getPictureContainer();
            if (!container) return;
            
            const sprite = container.children.find(s => s._pictureId === this.pictureId);
            if (sprite) {
                sprite.bitmap = this._bitmap;
                sprite.visible = true;
                sprite.opacity = 255;
                sprite.x = this.x;
                sprite.y = this.y;
            }
        }

        getPictureContainer() {
            const scene = SceneManager._scene;
            if (!scene) return null;
            
            if (scene._spriteset && scene._spriteset._pictureContainer) {
                return scene._spriteset._pictureContainer;
            }
            
            if (scene._mapFusion && scene._mapFusion._spriteset && scene._mapFusion._spriteset._pictureContainer) {
                return scene._mapFusion._spriteset._pictureContainer;
            }
            
            return null;
        }

        refreshBitmap() {
            this._bitmap.clear();
            const val = $gameVariables.value(this.variableId);
            const text = String(val);
            this._bitmap.fontSize = this.fontSize;
            this._bitmap.fontBold = this.fontBold;
            this._bitmap.outlineWidth = this.outlineWidth;
            this._bitmap.outlineColor = this.outlineColor;
            this._bitmap.textColor = this.fontColor;
            const tw = this._bitmap.measureTextWidth(text);
            let dx = 0;
            if (this.align === "center") dx = (this._bitmap.width - tw) / 2;
            if (this.align === "right") dx = this._bitmap.width - tw;
            this._bitmap.drawText(text, dx, 0, tw, this.fontSize, this.align);
            
            this._textWidth = tw;
            this._textX = dx;
            
            const picture = $gameScreen.picture(this.pictureId);
            if (picture) {
                picture._customBitmap = this._bitmap;
            }
        }

        update() {
            if (!this._editing) {
                const cur = $gameVariables.value(this.variableId);
                if (this._lastValue !== cur) {
                    this._lastValue = cur;
                    this.refreshBitmap();
                }
            } else {
                this.updateInput();
                this.checkClickOutside();
            }
            
            this.ensureSprite();
        }
        
        checkClickOutside() {
            if (!TouchInput.isTriggered()) return;
            
            const px = TouchInput.x;
            const py = TouchInput.y;
            const tw = this._textWidth || this._bitmap.width;
            const tx = this._textX || 0;
            const h = this.fontSize;
            
            const isInside = px >= this.x + tx && px <= this.x + tx + tw && py >= this.y && py <= this.y + h;
            
            if (isInside) {
                this.finishEditing();
                TouchInput.clear();
            }
        }
        
        finishEditing() {
            if (this._input.length > 0) {
                let num = Number(this._input);
                if (num < this.min) num = this.min;
                if (num > this.max) num = this.max;
                $gameVariables.setValue(this.variableId, num);
                
                if (this.script) {
                    try { 
                        eval(this.script); 
                    } catch (e) { 
                        console.error("Script error:", e); 
                    }
                }
            }
            
            this._editing = false;
            this._input = "";
            this.refreshBitmap();
        }
        
        ensureSprite() {
            const container = this.getPictureContainer();
            if (!container) return;
            
            const sprite = container.children.find(s => s._pictureId === this.pictureId);
            if (sprite && sprite.bitmap !== this._bitmap) {
                sprite.bitmap = this._bitmap;
                sprite.visible = true;
            }
        }

        isTouched() {
            const px = TouchInput.x;
            const py = TouchInput.y;
            const tw = this._textWidth || this._bitmap.width;
            const tx = this._textX || 0;
            const h = this.fontSize;
            return px >= this.x + tx && px <= this.x + tx + tw && py >= this.y && py <= this.y + h;
        }

        startEditing() {
            this._editing = true;
            this._input = "";
            this.drawInput();
        }

        cancelEditing() {
            this._editing = false;
            this._input = "";
            this.refreshBitmap();
        }

        updateInput() {
            if (!_keyboardInput) return;
            
            const input = _keyboardInput;
            _keyboardInput = "";
            
            if (input >= "0" && input <= "9") {
                this._input += input;
                this.drawInput();
                return;
            }
            
            if (input === "backspace") {
                this._input = this._input.slice(0, -1);
                this.drawInput();
                return;
            }
            
            if (input === "enter") {
                this.finishEditing();
                return;
            }
            
            if (input === "escape") {
                this._editing = false;
                this._input = "";
                this.refreshBitmap();
                return;
            }
        }

        drawInput() {
            this._bitmap.clear();
            this._bitmap.fontSize = this.fontSize;
            this._bitmap.fontBold = this.fontBold;
            this._bitmap.outlineWidth = this.outlineWidth;
            this._bitmap.outlineColor = this.outlineColor;
            this._bitmap.textColor = this.fontColor;
            
            const disp = this._input.length > 0 ? this._input + "_" : "_";
            
            const tw = this._bitmap.measureTextWidth(disp);
            let dx = 0;
            if (this.align === "center") dx = (this._bitmap.width - tw) / 2;
            if (this.align === "right") dx = this._bitmap.width - tw;
            
            this._bitmap.drawText(disp, dx, 0, tw, this.fontSize, this.align);
            
            this._textWidth = tw;
            this._textX = dx;
            
            const picture = $gameScreen.picture(this.pictureId);
            if (picture) {
                picture._customBitmap = this._bitmap;
            }
        }
    }

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        _Scene_Map_update.call(this);
        updateNrOptionFields();
    };

    const _Scene_Title_update = Scene_Title.prototype.update;
    Scene_Title.prototype.update = function() {
        _Scene_Title_update.call(this);
        updateNrOptionFields();
    };

    const _Scene_Base_update = Scene_Base.prototype.update;
    Scene_Base.prototype.update = function() {
        _Scene_Base_update.call(this);
        if (!(this instanceof Scene_Map) && !(this instanceof Scene_Title)) {
            updateNrOptionFields();
        }
    };

    function updateNrOptionFields() {
        _currentFields.forEach(f => {
            f.update();
        });
        
        if (TouchInput.isTriggered()) {
            let touchedField = null;
            
            for (const f of _currentFields) {
                if (f.isTouched()) {
                    touchedField = f;
                    break;
                }
            }
            
            _currentFields.forEach(f => {
                if (f._editing && f !== touchedField) {
                    f.cancelEditing();
                }
            });
            
            if (touchedField && !touchedField._editing) {
                touchedField.startEditing();
                TouchInput.clear();
            }
        }
    }

    const _Sprite_Picture_loadBitmap = Sprite_Picture.prototype.loadBitmap;
    Sprite_Picture.prototype.loadBitmap = function() {
        const pic = this.picture();
        if (pic && pic._name === "" && pic._customBitmap) {
            this.bitmap = pic._customBitmap;
            return;
        }
        _Sprite_Picture_loadBitmap.call(this);
    };
    
    const _Sprite_Picture_updateBitmap = Sprite_Picture.prototype.updateBitmap;
    Sprite_Picture.prototype.updateBitmap = function() {
        const pic = this.picture();
        if (pic && pic._name === "" && pic._customBitmap) {
            if (this.bitmap !== pic._customBitmap) {
                this.bitmap = pic._customBitmap;
            }
            return;
        }
        _Sprite_Picture_updateBitmap.call(this);
    };

    Game_System.prototype.showNrOptionNumber = function(listName) {
        const fieldDefs = lists[listName];
        if (!fieldDefs) return;
        this.hideNrOptionNumber();
        fieldDefs.forEach(cfg => {
            const fld = new NrOptionField(cfg);
            _currentFields.push(fld);
        });
    };

    Game_System.prototype.hideNrOptionNumber = function() {
        _currentFields.forEach(f => {
            const picture = $gameScreen.picture(f.pictureId);
            if (picture) {
                picture._customBitmap = null;
            }
            $gameScreen.erasePicture(f.pictureId);
        });
        _currentFields = [];
    };
})();
