//=============================================================================
// NrParameterCalculation_MZ.js
//=============================================================================

/*:
 * @plugindesc 計算補助システム v1.0.0
 * @author NJ
 *
 * @param statusList
 * @text ステータス定義リスト
 * @type struct<StatusItem>[]
 * @default []
 *
 * @command gainExp
 * @text 経験値加算
 * @desc 指定したステータス名に経験値を加算します
 * @arg name
 * @text ステータス名
 * @type string
 * @arg value
 * @text 経験値（数値 or \v[n]）
 * @type string
 *
 * @help
 * ステータス名と変数IDを指定することで、レベル・経験値・最大経験値の管理を自動化します。
 * レベルアップまたはレベルダウン時に、設定されたスクリプトを一度だけ実行できます。
 *
 * プラグインコマンド：
 *   プラグインコマンド gainExp ステータス名 経験値
 *
 * スクリプトでの利用：
 *   NrGAIN_EXP("剣術", 100)
 *
 * バージョン:
 *   v1.0.0 初回
 */

 /*~struct~StatusItem:
  * @param name
  * @text ステータス名
  * @type string
  *
  * @param levelVar
  * @text レベル変数ID
  * @type variable
  *
  * @param expVar
  * @text 現在EXP変数ID
  * @type variable
  *
  * @param expMaxVar
  * @text 最大EXP変数ID
  * @type variable
  *
  * @param expMaxBase
  * @text 基準レベル時の最大EXP
  * @type number
  * @default 0
  *
  * @param expCalcMode
  * @text 計算方法
  * @type select
  * @option +
  * @option -
  * @option *
  * @option /
  * @default +
  *
  * @param expCalcValue
  * @text 計算式または定数
  * @type string
  * @default 1
  *
  * @param manualExpList
  * @text 手動指定リスト
  * @type struct<ManualExp>[]
  * @default []
  *
  * @param levelStart
  * @text 開始レベル
  * @type number
  * @default 1
  *
  * @param levelBase
  * @text 計算基準レベル
  * @type number
  * @default 1
  *
  * @param levelMax
  * @text レベル上限
  * @type number
  * @default 99
  *
  * @param onLevelUpScript
  * @text レベルアップ時スクリプト
  * @type note
  * @default
  *
  * @param onLevelDownScript
  * @text レベルダウン時スクリプト
  * @type note
  * @default
  */

/*~struct~ManualExp:
 * @param level
 * @text レベル
 * @type number
 *
 * @param value
 * @text 最大EXP
 * @type number
 */

(() => {
    const pluginName = "NrParameterCalculation_MZ";
    const parameters = PluginManager.parameters(pluginName);

    let statusList = [];
    try {
        statusList = JSON.parse(parameters.statusList || "[]").map(str => {
            const item = JSON.parse(str);
            item.manualExpList = JSON.parse(item.manualExpList || "[]").map(e => JSON.parse(e));
            return item;
        });
    } catch (e) {
        console.error(`[${pluginName}] パラメータ読み込み失敗`, e);
    }

    function convertVar(str) {
        return String(str).replace(/\\v\[(\d+)]/gi, (_, n) => $gameVariables.value(Number(n)));
    }

    function calcExpMax(item, levelRaw) {
        const level = Number(levelRaw);
        const baseLevel = Number(item.levelBase || 1);
        const baseExp = Number(item.expMaxBase);
        const mode = item.expCalcMode;
        let formula = String(item.expCalcValue || "1").trim();

        for (const entry of item.manualExpList) {
            if (Number(entry.level) === level) return Number(entry.value);
        }

        if (/^\d+(\.\d+)?$/.test(formula)) {
            formula = `(level - baseLevel) * ${formula}`;
        }

        let delta = 0;
        try {
            delta = new Function("level", "baseLevel", `return (${formula})`)(level, baseLevel);
        } catch (e) {
            console.error(`[${pluginName}] 計算式エラー (${item.name})`, e);
        }

        switch (mode) {
            case '+': return baseExp + delta;
            case '-': return baseExp - delta;
            case '*': return baseExp * delta;
            case '/': return delta !== 0 ? baseExp / delta : baseExp;
            default: return baseExp;
        }
    }

    function updateOne(item) {
        const level = $gameVariables.value(Number(item.levelVar));
        const max = calcExpMax(item, level);
        $gameVariables.setValue(Number(item.expMaxVar), max);
    }

    function updateAll() {
        statusList.forEach(updateOne);
    }

    function initializeLevels() {
        statusList.forEach(item => {
            const id = Number(item.levelVar);
            const start = Number(item.levelStart || 1);
            if ($gameVariables.value(id) === 0) {
                $gameVariables.setValue(id, start);
            }
        });
    }

    function runScript(code) {
        if (!code) return;
        try {
            let cleaned = String(code)
                .trim()
                .replace(/^"|"$/g, "")
                .replace(/^'|'$/g, "")
                .replace(/\\n/g, ";")
                .replace(/\\r/g, "")
                .replace(/\\t/g, "")
                .replace(/\\"/g, '"')
                .replace(/\\'/g, "'")
                .replace(/[\r\n]+/g, ";")
                .replace(/；/g, ";");
            new Function(cleaned)();
        } catch (e) {
            console.error(`[${pluginName}] スクリプトエラー:`, e, code);
        }
    }

    function gainExp(nameRaw, valueRaw) {
        const name = convertVar(nameRaw);
        const gain = Number(convertVar(valueRaw));
        const item = statusList.find(i => i.name === name);
        if (!item) return;

        const expId = Number(item.expVar);
        const lvId = Number(item.levelVar);
        const maxLv = Number(item.levelMax);
        const baseLv = Number(item.levelBase);

        let exp = $gameVariables.value(expId) + gain;
        let level = $gameVariables.value(lvId);
        const oldLevel = level;

        if (gain >= 0) {
            while (level < maxLv) {
                const maxExp = calcExpMax(item, level);
                if (exp < maxExp) break;
                exp -= maxExp;
                level++;
            }
            if (level >= maxLv) {
                const maxExp = calcExpMax(item, level);
                if (exp >= maxExp) exp = maxExp - 1;
            }
        } else {
            while (level > baseLv) {
                const prev = level - 1;
                const prevExp = calcExpMax(item, prev);
                if (exp >= 0) break;
                level--;
                exp += prevExp;
            }
            if (exp < 0) exp = 0;
        }

        $gameVariables.setValue(expId, exp);
        $gameVariables.setValue(lvId, level);
        updateOne(item);

        if (level > oldLevel && item.onLevelUpScript) {
            const code = String(item.onLevelUpScript).trim().replace(/^"|"$/g, "");
            if (code) runScript(code);
        } else if (level < oldLevel && item.onLevelDownScript) {
            const code = String(item.onLevelDownScript).trim().replace(/^"|"$/g, "");
            if (code) runScript(code);
        }
    }

    PluginManager.registerCommand(pluginName, "gainExp", args => {
        gainExp(args.name, args.value);
    });

    const _Scene_Map_start = Scene_Map.prototype.start;
    Scene_Map.prototype.start = function() {
        _Scene_Map_start.apply(this, arguments);
        initializeLevels();
        updateAll();
    };

    const _Game_Variables_setValue = Game_Variables.prototype.setValue;
    Game_Variables.prototype.setValue = function(variableId, value) {
        _Game_Variables_setValue.apply(this, arguments);
        statusList.forEach(item => {
            if (Number(item.levelVar) === variableId || Number(item.expVar) === variableId) {
                updateOne(item);
            }
        });
    };

    window.NrGAIN_EXP = function(nameRaw, valueRaw) {
        gainExp(nameRaw, valueRaw);
    };

})();
