//=============================================================================
// NrPictureEffect.js
//=============================================================================

/*:
 * @target MZ
 * @plugindesc パーティクルエフェクトシステム v1.0.0
 * @author 
 * 
 * @command startEffect
 * @text エフェクト開始
 * @desc 指定した呼び出しIDと座標でエフェクトを開始します
 * 
 * @arg callId
 * @text 呼び出しID
 * @desc 使用するエフェクトパターンの呼び出しID
 * @type number
 * @default 1
 * @min 1
 * 
 * @arg startX
 * @text 開始座標X
 * @desc エフェクトの開始X座標
 * @type number
 * @default 400
 * @min 0
 * @max 9999
 * 
 * @arg startY
 * @text 開始座標Y
 * @desc エフェクトの開始Y座標
 * @type number
 * @default 300
 * @min 0
 * @max 9999
 *
 * @command startLoopEffect
 * @text ループエフェクト開始
 * @desc 指定した呼び出しIDと座標でループエフェクトを開始します
 * 
 * @arg callId
 * @text 呼び出しID
 * @desc 使用するループエフェクトパターンの呼び出しID
 * @type number
 * @default 1
 * @min 1
 * 
 * @arg startX
 * @text 開始座標X
 * @desc エフェクトの開始X座標
 * @type number
 * @default 400
 * @min 0
 * @max 9999
 * 
 * @arg startY
 * @text 開始座標Y
 * @desc エフェクトの開始Y座標
 * @type number
 * @default 300
 * @min 0
 * @max 9999
 *
 * @command stopLoopEffect
 * @text ループエフェクト停止
 * @desc すべてのループエフェクトを停止します
 *
 * @command stopEffectById
 * @text エフェクト停止（ID指定）
 * @desc 指定したIDのエフェクトを停止します
 * 
 * @arg effectId
 * @text エフェクトID
 * @desc 停止するエフェクトのID
 * @type number
 * @default 1
 * @min 1
 *
 * @command startEffectDirect
 * @text エフェクト開始（直接指定）
 * @desc パターンを使わずに直接設定してエフェクトを開始
 * 
 * @arg effectId
 * @text エフェクトID
 * @desc エフェクトの識別ID（停止時に使用）
 * @type number
 * @default 1
 * @min 1
 * 
 * @arg startX
 * @text 開始座標X
 * @desc エフェクトの開始X座標
 * @type number
 * @default 400
 * @min 0
 * @max 9999
 * 
 * @arg startY
 * @text 開始座標Y
 * @desc エフェクトの開始Y座標
 * @type number
 * @default 300
 * @min 0
 * @max 9999
 * 
 * @arg fileName
 * @text ファイル名
 * @desc 使用する画像ファイル名
 * @type file
 * @dir img/system/
 * @default TouchParticles
 * 
 * @arg animationType
 * @text アニメーションタイプ
 * @desc アニメーションの種類
 * @type select
 * @default Fireworks
 * @option Fireworks
 * @value Fireworks
 * @option Random
 * @value Random
 * @option Zoom Out
 * @value Zoom Out
 * @option Zoom In
 * @value Zoom In
 * 
 * @arg particleCount
 * @text パーティクル数
 * @desc パーティクルの数 (1 - 999)
 * @type number
 * @default 10
 * @min 1
 * @max 999
 * 
 * @arg xOffset
 * @text X軸オフセット
 * @desc X軸のオフセット値
 * @type number
 * @default 0
 * @min -9999
 * @max 9999
 * 
 * @arg yOffset
 * @text Y軸オフセット
 * @desc Y軸のオフセット値
 * @type number
 * @default 0
 * @min -9999
 * @max 9999
 * 
 * @arg blendMode
 * @text ブレンドモード
 * @desc ブレンドモードの設定
 * @type select
 * @default Normal
 * @option Additive
 * @value Additive
 * @option Normal
 * @value Normal
 * @option Multiply
 * @value Multiply
 * 
 * @arg duration
 * @text 持続時間
 * @desc アニメーションの持続時間 (1 - 999)
 * @type number
 * @default 10
 * @min 1
 * @max 999
 * 
 * @arg fadeSpeed
 * @text フェード速度
 * @desc フェードアウトの速度 (2 - 125)
 * @type number
 * @default 20
 * @min 2
 * @max 125
 * 
 * @arg randomTone
 * @text ランダムトーン
 * @desc ランダムカラーを使用するか
 * @type boolean
 * @default true
 * 
 * @arg enableMovement
 * @text 座標移動を有効化
 * @desc アニメーションを指定座標へ移動させるか
 * @type boolean
 * @default false
 * 
 * @arg targetX
 * @text 移動先X座標
 * @desc アニメーションの移動先X座標
 * @type number
 * @default 400
 * @min 0
 * @max 9999
 * 
 * @arg targetY
 * @text 移動先Y座標
 * @desc アニメーションの移動先Y座標
 * @type number
 * @default 300
 * @min 0
 * @max 9999
 * 
 * @arg movementSpeed
 * @text 移動速度
 * @desc 座標移動の速度 (1-20)
 * @type number
 * @default 5
 * @min 1
 * @max 20
 * 
 * @arg curveMovement
 * @text 曲線移動
 * @desc 曲線を描いて移動するか
 * @type boolean
 * @default false
 * 
 * @arg curveHeight
 * @text 曲線の高さ
 * @desc 曲線の頂点の高さ (-500 ~ 500)
 * @type number
 * @default -100
 * @min -500
 * @max 500
 * 
 * @arg enableConvergence
 * @text 移動先で収束
 * @desc 移動先で散らばったパーティクルを収束させるか
 * @type boolean
 * @default true
 * 
 * @arg convergenceSpeed
 * @text 収束速度
 * @desc 収束の速度 (1-10)
 * @type number
 * @default 3
 * @min 1
 * @max 10
 * 
 * @arg convergenceDelay
 * @text 収束開始遅延
 * @desc 移動完了後、収束開始までの遅延フレーム数
 * @type number
 * @default 15
 * @min 0
 * @max 60
 * 
 * @arg spreadRange
 * @text 散らばり範囲
 * @desc パーティクルの散らばり範囲（0-999）移動有効時のみ適用
 * @type number
 * @default 150
 * @min 0
 * @max 999
 *
 * @command startLoopEffectDirect
 * @text ループエフェクト開始（直接指定）
 * @desc パターンを使わずに直接設定してループエフェクトを開始
 * 
 * @arg effectId
 * @text エフェクトID
 * @desc エフェクトの識別ID（停止時に使用）
 * @type number
 * @default 1
 * @min 1
 * 
 * @arg startX
 * @text 開始座標X
 * @desc エフェクトの開始X座標
 * @type number
 * @default 400
 * @min 0
 * @max 9999
 * 
 * @arg startY
 * @text 開始座標Y
 * @desc エフェクトの開始Y座標
 * @type number
 * @default 300
 * @min 0
 * @max 9999
 * 
 * @arg fileName
 * @text ファイル名
 * @desc 使用する画像ファイル名
 * @type file
 * @dir img/system/
 * @default TouchParticles
 * 
 * @arg animationType
 * @text アニメーションタイプ
 * @desc アニメーションの種類
 * @type select
 * @default Fireworks
 * @option Fireworks
 * @value Fireworks
 * @option Random
 * @value Random
 * @option Zoom Out
 * @value Zoom Out
 * @option Zoom In
 * @value Zoom In
 * 
 * @arg particleCount
 * @text パーティクル数
 * @desc パーティクルの数 (1 - 999)
 * @type number
 * @default 10
 * @min 1
 * @max 999
 * 
 * @arg xOffset
 * @text X軸オフセット
 * @desc X軸のオフセット値
 * @type number
 * @default 0
 * @min -9999
 * @max 9999
 * 
 * @arg yOffset
 * @text Y軸オフセット
 * @desc Y軸のオフセット値
 * @type number
 * @default 0
 * @min -9999
 * @max 9999
 * 
 * @arg blendMode
 * @text ブレンドモード
 * @desc ブレンドモードの設定
 * @type select
 * @default Normal
 * @option Additive
 * @value Additive
 * @option Normal
 * @value Normal
 * @option Multiply
 * @value Multiply
 * 
 * @arg duration
 * @text 持続時間
 * @desc アニメーションの持続時間 (1 - 999)
 * @type number
 * @default 10
 * @min 1
 * @max 999
 * 
 * @arg fadeSpeed
 * @text フェード速度
 * @desc フェードアウトの速度 (2 - 125)
 * @type number
 * @default 20
 * @min 2
 * @max 125
 * 
 * @arg randomTone
 * @text ランダムトーン
 * @desc ランダムカラーを使用するか
 * @type boolean
 * @default true
 * 
 * @arg enableMovement
 * @text 座標移動を有効化
 * @desc アニメーションを指定座標へ移動させるか
 * @type boolean
 * @default false
 * 
 * @arg targetX
 * @text 移動先X座標
 * @desc アニメーションの移動先X座標
 * @type number
 * @default 400
 * @min 0
 * @max 9999
 * 
 * @arg targetY
 * @text 移動先Y座標
 * @desc アニメーションの移動先Y座標
 * @type number
 * @default 300
 * @min 0
 * @max 9999
 * 
 * @arg movementSpeed
 * @text 移動速度
 * @desc 座標移動の速度 (1-20)
 * @type number
 * @default 5
 * @min 1
 * @max 20
 * 
 * @arg curveMovement
 * @text 曲線移動
 * @desc 曲線を描いて移動するか
 * @type boolean
 * @default false
 * 
 * @arg curveHeight
 * @text 曲線の高さ
 * @desc 曲線の頂点の高さ (-500 ~ 500)
 * @type number
 * @default -100
 * @min -500
 * @max 500
 * 
 * @arg enableConvergence
 * @text 移動先で収束
 * @desc 移動先で散らばったパーティクルを収束させるか
 * @type boolean
 * @default true
 * 
 * @arg convergenceSpeed
 * @text 収束速度
 * @desc 収束の速度 (1-10)
 * @type number
 * @default 3
 * @min 1
 * @max 10
 * 
 * @arg convergenceDelay
 * @text 収束開始遅延
 * @desc 移動完了後、収束開始までの遅延フレーム数
 * @type number
 * @default 15
 * @min 0
 * @max 60
 * 
 * @arg spreadRange
 * @text 散らばり範囲
 * @desc パーティクルの散らばり範囲（0-999）移動有効時のみ適用
 * @type number
 * @default 150
 * @min 0
 * @max 999
 * 
 * @arg loopInterval
 * @text エフェクト再生間隔
 * @desc エフェクトの再生間隔（フレーム数）
 * @type number
 * @default 60
 * @min 1
 * @max 9999
 * 
 * @arg autoEnd
 * @text 自動終了
 * @desc 一定時間後に自動的に終了するか
 * @type boolean
 * @default false
 * 
 * @arg endWait
 * @text 終了ウェイト
 * @desc 自動終了までの時間（フレーム数）自動終了がtrueの時のみ有効
 * @type number
 * @default 600
 * @min 1
 * @max 99999
 *
 * @param Effect Patterns
 * @text エフェクトパターン
 * @desc エフェクトパターンのリスト
 * @type struct<EffectPattern>[]
 * @default []
 *
 * @param Loop Effect Patterns
 * @text ループエフェクトパターン
 * @desc ループエフェクトパターンのリスト
 * @type struct<LoopEffectPattern>[]
 * @default []
 *
 * @help
 * パーティクルエフェクトシステム
 * 呼び出しIDと座標を指定してエフェクトを表示できます。
 * 
 * 
 * ★ 種類
 * 
 * 【通常エフェクト】
 *   一度だけ再生されるエフェクトです。
 *   プラグインコマンド「エフェクト開始」で実行できます。
 * 
 * 【ループエフェクト】
 *   指定した間隔で連続してパーティクルを生成し続けるエフェクトです。
 *   プラグインコマンド「ループエフェクト開始」で実行し、
 *   「ループエフェクト停止」で停止できます。
 *   自動終了の設定も可能です。
 * 
 * 【直接指定エフェクト】
 *   パターンを事前に登録することなく、プラグインコマンドから
 *   直接エフェクトの詳細設定を指定して実行できます。
 *   「エフェクト開始（直接指定）」と「ループエフェクト開始（直接指定）」が利用できます。
 * 
 * 
 * ★ 基本的な使い方
 * 
 * 1. プラグインパラメータでエフェクトパターンを登録
 *    - 呼び出しID（1以上の数値）を設定
 *    - 画像ファイル名を指定（/img/system/に配置）
 *    - アニメーションタイプを選択（Fireworks/Random/Zoom Out/Zoom In）
 *    - パーティクル数、移動設定、フェード設定などを調整
 * 
 * 2. プラグインコマンドでエフェクトを実行
 *    - 呼び出しID: 登録したパターンのID
 *    - 開始座標X, Y: エフェクトの発生位置
 * 
 * 【直接指定での使い方】
 * 
 * 1. プラグインコマンド「エフェクト開始（直接指定）」または
 *    「ループエフェクト開始（直接指定）」を選択
 * 
 * 2. 必要な設定を直接指定
 *    - ファイル名: 使用する画像ファイル（/img/system/に配置）
 *    - アニメーションタイプ: Fireworks/Random/Zoom Out/Zoom In
 *    - パーティクル数、座標、移動設定、フェード設定など
 *    - ループエフェクトの場合は追加で間隔と自動終了設定
 * 
 * 
 * 【アニメーション設定】
 * - animation_type: パーティクルの動きのパターン
 * - particle_count: 同時に表示するパーティクルの数（1-999）
 * - duration: パーティクルが目標地点に到達後、消えるまでの待機時間
 * - fade_speed: フェードアウトの速度（2-125、大きいほど速い）
 * 
 * 【移動設定】
 * - enable_movement: 座標移動を有効化
 * - target_x, target_y: 移動先の座標
 * - movement_speed: 移動速度（1-20）
 * - curve_movement: 曲線移動を有効化
 * - enable_convergence: 収束効果（散らばったパーティクルが1点に集まる）
 * - spread_range: パーティクルの初期散らばり範囲（0-999）
 * 
 * 【ループエフェクト専用設定】
 * - loop_interval: パーティクル生成間隔（フレーム数）
 * - auto_end: 自動終了を有効化
 * - end_wait: 自動終了までの待機時間（フレーム数）
 * 
 *   注意事項
 * - 画像ファイルは /img/system/ フォルダに配置してください
 * - ループエフェクトは自動的に高速フェードが適用されます
 * - 同時に大量のパーティクルを表示すると処理が重くなる場合があります
 * 
 * ★ バージョン
 *   1.0.0 初回
 * 
 */

/*~struct~EffectPattern:
 * @param call_id
 * @text 呼び出しID
 * @desc このパターンの呼び出しID
 * @type number
 * @default 1
 * @min 1
 * 
 * @param file_name
 * @text ファイル名
 * @desc 使用する画像ファイル名
 * @type file
 * @dir img/system/
 * @default TouchParticles
 * 
 * @param animation_type
 * @text アニメーションタイプ
 * @desc アニメーションの種類
 * @type select
 * @default Fireworks
 * @option Fireworks
 * @value Fireworks
 * @option Random
 * @value Random
 * @option Zoom Out
 * @value Zoom Out
 * @option Zoom In
 * @value Zoom In
 * 
 * @param particle_count
 * @text パーティクル数
 * @desc パーティクルの数 (1 - 999)
 * @type number
 * @default 10
 * @min 1
 * @max 999
 * 
 * @param x_offset
 * @text X軸オフセット
 * @desc X軸のオフセット値
 * @type number
 * @default 0
 * @min -9999
 * @max 9999
 * 
 * @param y_offset
 * @text Y軸オフセット
 * @desc Y軸のオフセット値
 * @type number
 * @default 0
 * @min -9999
 * @max 9999
 * 
 * @param blend_mode
 * @text ブレンドモード
 * @desc ブレンドモードの設定
 * @type select
 * @default Normal
 * @option Additive
 * @value Additive
 * @option Normal
 * @value Normal
 * @option Multiply
 * @value Multiply
 * 
 * @param duration
 * @text 持続時間
 * @desc アニメーションの持続時間 (1 - 999)
 * @type number
 * @default 10
 * @min 1
 * @max 999
 * 
 * @param fade_speed
 * @text フェード速度
 * @desc フェードアウトの速度 (2 - 125)
 * @type number
 * @default 20
 * @min 2
 * @max 125
 * 
 * @param random_tone
 * @text ランダムトーン
 * @desc ランダムカラーを使用するか
 * @type boolean
 * @default true
 * 
 * @param enable_movement
 * @text 座標移動を有効化
 * @desc アニメーションを指定座標へ移動させるか
 * @type boolean
 * @default false
 * 
 * @param target_x
 * @text 移動先X座標
 * @desc アニメーションの移動先X座標
 * @type number
 * @default 400
 * @min 0
 * @max 9999
 * 
 * @param target_y
 * @text 移動先Y座標
 * @desc アニメーションの移動先Y座標
 * @type number
 * @default 300
 * @min 0
 * @max 9999
 * 
 * @param movement_speed
 * @text 移動速度
 * @desc 座標移動の速度 (1-20)
 * @type number
 * @default 5
 * @min 1
 * @max 20
 * 
 * @param curve_movement
 * @text 曲線移動
 * @desc 曲線を描いて移動するか
 * @type boolean
 * @default false
 * 
 * @param curve_height
 * @text 曲線の高さ
 * @desc 曲線の頂点の高さ (-500 ~ 500)
 * @type number
 * @default -100
 * @min -500
 * @max 500
 * 
 * @param enable_convergence
 * @text 移動先で収束
 * @desc 移動先で散らばったパーティクルを収束させるか
 * @type boolean
 * @default true
 * 
 * @param convergence_speed
 * @text 収束速度
 * @desc 収束の速度 (1-10)
 * @type number
 * @default 3
 * @min 1
 * @max 10
 * 
 * @param convergence_delay
 * @text 収束開始遅延
 * @desc 移動完了後、収束開始までの遅延フレーム数
 * @type number
 * @default 15
 * @min 0
 * @max 60
 * 
 * @param spread_range
 * @text 散らばり範囲
 * @desc パーティクルの散らばり範囲（0-999）移動有効時のみ適用
 * @type number
 * @default 150
 * @min 0
 * @max 999
 */

/*~struct~LoopEffectPattern:
 * @param call_id
 * @text 呼び出しID
 * @desc このパターンの呼び出しID
 * @type number
 * @default 1
 * @min 1
 * 
 * @param file_name
 * @text ファイル名
 * @desc 使用する画像ファイル名
 * @type file
 * @dir img/system/
 * @default TouchParticles
 * 
 * @param animation_type
 * @text アニメーションタイプ
 * @desc アニメーションの種類
 * @type select
 * @default Fireworks
 * @option Fireworks
 * @value Fireworks
 * @option Random
 * @value Random
 * @option Zoom Out
 * @value Zoom Out
 * @option Zoom In
 * @value Zoom In
 * 
 * @param particle_count
 * @text パーティクル数
 * @desc パーティクルの数 (1 - 999)
 * @type number
 * @default 10
 * @min 1
 * @max 999
 * 
 * @param x_offset
 * @text X軸オフセット
 * @desc X軸のオフセット値
 * @type number
 * @default 0
 * @min -9999
 * @max 9999
 * 
 * @param y_offset
 * @text Y軸オフセット
 * @desc Y軸のオフセット値
 * @type number
 * @default 0
 * @min -9999
 * @max 9999
 * 
 * @param blend_mode
 * @text ブレンドモード
 * @desc ブレンドモードの設定
 * @type select
 * @default Normal
 * @option Additive
 * @value Additive
 * @option Normal
 * @value Normal
 * @option Multiply
 * @value Multiply
 * 
 * @param duration
 * @text 持続時間
 * @desc アニメーションの持続時間 (1 - 999)
 * @type number
 * @default 10
 * @min 1
 * @max 999
 * 
 * @param fade_speed
 * @text フェード速度
 * @desc フェードアウトの速度 (2 - 125)
 * @type number
 * @default 20
 * @min 2
 * @max 125
 * 
 * @param random_tone
 * @text ランダムトーン
 * @desc ランダムカラーを使用するか
 * @type boolean
 * @default true
 * 
 * @param enable_movement
 * @text 座標移動を有効化
 * @desc アニメーションを指定座標へ移動させるか
 * @type boolean
 * @default false
 * 
 * @param target_x
 * @text 移動先X座標
 * @desc アニメーションの移動先X座標
 * @type number
 * @default 400
 * @min 0
 * @max 9999
 * 
 * @param target_y
 * @text 移動先Y座標
 * @desc アニメーションの移動先Y座標
 * @type number
 * @default 300
 * @min 0
 * @max 9999
 * 
 * @param movement_speed
 * @text 移動速度
 * @desc 座標移動の速度 (1-20)
 * @type number
 * @default 5
 * @min 1
 * @max 20
 * 
 * @param curve_movement
 * @text 曲線移動
 * @desc 曲線を描いて移動するか
 * @type boolean
 * @default false
 * 
 * @param curve_height
 * @text 曲線の高さ
 * @desc 曲線の頂点の高さ (-500 ~ 500)
 * @type number
 * @default -100
 * @min -500
 * @max 500
 * 
 * @param enable_convergence
 * @text 移動先で収束
 * @desc 移動先で散らばったパーティクルを収束させるか
 * @type boolean
 * @default true
 * 
 * @param convergence_speed
 * @text 収束速度
 * @desc 収束の速度 (1-10)
 * @type number
 * @default 3
 * @min 1
 * @max 10
 * 
 * @param convergence_delay
 * @text 収束開始遅延
 * @desc 移動完了後、収束開始までの遅延フレーム数
 * @type number
 * @default 15
 * @min 0
 * @max 60
 * 
 * @param spread_range
 * @text 散らばり範囲
 * @desc パーティクルの散らばり範囲（0-999）移動有効時のみ適用
 * @type number
 * @default 150
 * @min 0
 * @max 999
 * 
 * @param loop_interval
 * @text エフェクト再生間隔
 * @desc エフェクトの再生間隔（フレーム数）
 * @type number
 * @default 60
 * @min 1
 * @max 9999
 * 
 * @param auto_end
 * @text 自動終了
 * @desc 一定時間後に自動的に終了するか
 * @type boolean
 * @default false
 * 
 * @param end_wait
 * @text 終了ウェイト
 * @desc 自動終了までの時間（フレーム数）自動終了がtrueの時のみ有効
 * @type number
 * @default 600
 * @min 1
 * @max 99999
 */
 
(() => {
	
    var Imported = Imported || {};
    Imported.MOG_PictureEffect = true;
    var Moghunter = Moghunter || {}; 

    Moghunter.parameters = PluginManager.parameters('NrPictureEffect');
    Moghunter.effectPatterns = [];

    const possibleKeys = [
        'Effect Patterns',
        'エフェクトパターン',
        'effectPatterns',
        'EffectPatterns'
    ];

    let foundKey = null;
    for (const key of possibleKeys) {
        if (Moghunter.parameters[key] !== undefined) {
            foundKey = key;
            break;
        }
    }

    if (foundKey) {
        try {
            const rawPatternsData = Moghunter.parameters[foundKey] || '[]';
            const patternsData = JSON.parse(rawPatternsData);
            for (let i = 0; i < patternsData.length; i++) {
                const patternData = patternsData[i];
                const pattern = JSON.parse(patternData);
                const processedPattern = {
                    callId: Number(pattern.call_id || 1),
                    fileName: String(pattern.file_name || 'TouchParticles'),
                    animationType: String(pattern.animation_type || 'Fireworks'),
                    particleCount: Number(pattern.particle_count || 10),
                    xOffset: Number(pattern.x_offset || 0),
                    yOffset: Number(pattern.y_offset || 0),
                    blendMode: String(pattern.blend_mode || 'Normal'),
                    duration: Number(pattern.duration || 10),
                    fadeSpeed: Number(pattern.fade_speed || 20),
                    randomTone: String(pattern.random_tone || 'true') === 'true',
                    enableMovement: String(pattern.enable_movement || 'false') === 'true',
                    targetX: Number(pattern.target_x || 400),
                    targetY: Number(pattern.target_y || 300),
                    movementSpeed: Math.min(Math.max(Number(pattern.movement_speed || 5), 1), 20),
                    curveMovement: String(pattern.curve_movement || 'false') === 'true',
                    curveHeight: Math.min(Math.max(Number(pattern.curve_height || -100), -500), 500),
                    enableConvergence: String(pattern.enable_convergence || 'true') === 'true',
                    convergenceSpeed: Math.min(Math.max(Number(pattern.convergence_speed || 4), 1), 10),
                    convergenceDelay: Math.min(Math.max(Number(pattern.convergence_delay || 15), 0), 60),
                    spreadRange: Math.min(Math.max(Number(pattern.spread_range || 150), 0), 999)
                };
                Moghunter.effectPatterns.push(processedPattern);
            }
        } catch (e) {
            console.warn('PictureEffect: エフェクトパターンの読み込みに失敗しました:', e);
        }
    }

    Moghunter.loopEffectPatterns = [];

    const loopPossibleKeys = [
        'Loop Effect Patterns',
        'ループエフェクトパターン',
        'loopEffectPatterns',
        'LoopEffectPatterns'
    ];

    let loopFoundKey = null;
    for (const key of loopPossibleKeys) {
        if (Moghunter.parameters[key] !== undefined) {
            loopFoundKey = key;
            break;
        }
    }

    if (loopFoundKey) {
        try {
            const rawPatternsData = Moghunter.parameters[loopFoundKey] || '[]';
            const patternsData = JSON.parse(rawPatternsData);
            for (let i = 0; i < patternsData.length; i++) {
                const patternData = patternsData[i];
                const pattern = JSON.parse(patternData);
                const processedPattern = {
                    callId: Number(pattern.call_id || 1),
                    fileName: String(pattern.file_name || 'TouchParticles'),
                    animationType: String(pattern.animation_type || 'Fireworks'),
                    particleCount: Number(pattern.particle_count || 10),
                    xOffset: Number(pattern.x_offset || 0),
                    yOffset: Number(pattern.y_offset || 0),
                    blendMode: String(pattern.blend_mode || 'Normal'),
                    duration: Number(pattern.duration || 10),
                    fadeSpeed: Number(pattern.fade_speed || 20),
                    randomTone: String(pattern.random_tone || 'true') === 'true',
                    enableMovement: String(pattern.enable_movement || 'false') === 'true',
                    targetX: Number(pattern.target_x || 400),
                    targetY: Number(pattern.target_y || 300),
                    movementSpeed: Math.min(Math.max(Number(pattern.movement_speed || 5), 1), 20),
                    curveMovement: String(pattern.curve_movement || 'false') === 'true',
                    curveHeight: Math.min(Math.max(Number(pattern.curve_height || -100), -500), 500),
                    enableConvergence: String(pattern.enable_convergence || 'true') === 'true',
                    convergenceSpeed: Math.min(Math.max(Number(pattern.convergence_speed || 4), 1), 10),
                    convergenceDelay: Math.min(Math.max(Number(pattern.convergence_delay || 15), 0), 60),
                    spreadRange: Math.min(Math.max(Number(pattern.spread_range || 150), 0), 999),
                    loopInterval: Math.min(Math.max(Number(pattern.loop_interval || 60), 1), 9999),
                    autoEnd: String(pattern.auto_end || 'false') === 'true',
                    endWait: Math.min(Math.max(Number(pattern.end_wait || 600), 1), 99999)
                };
                Moghunter.loopEffectPatterns.push(processedPattern);
            }
        } catch (e) {
            console.warn('PictureEffect: ループエフェクトパターンの読み込みに失敗しました:', e);
        }
    }

    // 旧形式の互換性維持
    DataManager._pictureEffect = {};	
    DataManager._pictureEffect.x = 0;
    DataManager._pictureEffect.y = 0;
    DataManager._pictureEffect.needRefresh = false;
    DataManager._pictureEffect.currentPattern = null;
    DataManager._pictureEffect.callId = 0;
    DataManager._pictureEffect.directPattern = null;

    DataManager._loopEffect = {};
    DataManager._loopEffect.active = false;
    DataManager._loopEffect.callId = 0;
    DataManager._loopEffect.x = 0;
    DataManager._loopEffect.y = 0;
    DataManager._loopEffect.interval = 0;
    DataManager._loopEffect.intervalCounter = 0;
    DataManager._loopEffect.autoEnd = false;
    DataManager._loopEffect.endWait = 0;
    DataManager._loopEffect.endCounter = 0;
    DataManager._loopEffect.needFirstEmit = false;
    DataManager._loopEffect.directPattern = null;

    // 新しいID管理システム
    DataManager._effectsById = {};
    DataManager._loopEffectsById = {};

    PluginManager.registerCommand('NrPictureEffect', 'startEffect', args => {
        const callId = Number(args.callId || 1);
        const startX = Number(args.startX || 400);
        const startY = Number(args.startY || 300);
        DataManager._pictureEffect.callId = callId;
        DataManager._pictureEffect.x = startX;
        DataManager._pictureEffect.y = startY;
        DataManager._pictureEffect.needRefresh = true;
    });

    PluginManager.registerCommand('NrPictureEffect', 'startLoopEffect', args => {
        const callId = Number(args.callId || 1);
        const startX = Number(args.startX || 400);
        const startY = Number(args.startY || 300);
        const pattern = DataManager.getLoopEffectPatternByCallId(callId);
        if (pattern) {
            DataManager._loopEffect.active = true;
            DataManager._loopEffect.callId = callId;
            DataManager._loopEffect.x = startX;
            DataManager._loopEffect.y = startY;
            DataManager._loopEffect.interval = pattern.loopInterval;
            DataManager._loopEffect.intervalCounter = 0;
            DataManager._loopEffect.autoEnd = pattern.autoEnd;
            DataManager._loopEffect.endWait = pattern.endWait;
            DataManager._loopEffect.endCounter = 0;
            DataManager._loopEffect.needFirstEmit = true;
        }
    });

    PluginManager.registerCommand('NrPictureEffect', 'stopLoopEffect', args => {
        // 全てのループエフェクトを停止
        DataManager._loopEffect.active = false;
        DataManager._loopEffect.callId = 0;
        DataManager._loopEffect.intervalCounter = 0;
        DataManager._loopEffect.endCounter = 0;
        DataManager._loopEffect.directPattern = null;
        
        // ID管理システムからも全て削除
        DataManager._loopEffectsById = {};
    });

    PluginManager.registerCommand('NrPictureEffect', 'stopEffectById', args => {
        const effectId = Number(args.effectId || 1);
        
        // ループエフェクトの停止
        if (DataManager._loopEffectsById[effectId]) {
            delete DataManager._loopEffectsById[effectId];
            
            // 旧システムとの互換性維持
            if (DataManager._loopEffect.callId === effectId) {
                DataManager._loopEffect.active = false;
                DataManager._loopEffect.callId = 0;
                DataManager._loopEffect.intervalCounter = 0;
                DataManager._loopEffect.endCounter = 0;
                DataManager._loopEffect.directPattern = null;
            }
        }
        
        // 通常エフェクトの停止（パーティクルの強制フェード）
        if (DataManager._effectsById[effectId]) {
            DataManager._effectsById[effectId].active = false;
            delete DataManager._effectsById[effectId];
        }
    });

    PluginManager.registerCommand('NrPictureEffect', 'startEffectDirect', args => {
        const effectId = Number(args.effectId || 1);
        const startX = Number(args.startX || 400);
        const startY = Number(args.startY || 300);
        
        const directPattern = {
            callId: effectId,
            fileName: String(args.fileName || 'TouchParticles'),
            animationType: String(args.animationType || 'Fireworks'),
            particleCount: Number(args.particleCount || 10),
            xOffset: Number(args.xOffset || 0),
            yOffset: Number(args.yOffset || 0),
            blendMode: String(args.blendMode || 'Normal'),
            duration: Number(args.duration || 10),
            fadeSpeed: Number(args.fadeSpeed || 20),
            randomTone: String(args.randomTone || 'true') === 'true',
            enableMovement: String(args.enableMovement || 'false') === 'true',
            targetX: Number(args.targetX || 400),
            targetY: Number(args.targetY || 300),
            movementSpeed: Math.min(Math.max(Number(args.movementSpeed || 5), 1), 20),
            curveMovement: String(args.curveMovement || 'false') === 'true',
            curveHeight: Math.min(Math.max(Number(args.curveHeight || -100), -500), 500),
            enableConvergence: String(args.enableConvergence || 'true') === 'true',
            convergenceSpeed: Math.min(Math.max(Number(args.convergenceSpeed || 3), 1), 10),
            convergenceDelay: Math.min(Math.max(Number(args.convergenceDelay || 15), 0), 60),
            spreadRange: Math.min(Math.max(Number(args.spreadRange || 150), 0), 999)
        };

        // ID管理システムに登録
        DataManager._effectsById[effectId] = {
            pattern: directPattern,
            x: startX,
            y: startY,
            active: true
        };

        // 旧システムとの互換性維持
        DataManager._pictureEffect.callId = effectId;
        DataManager._pictureEffect.x = startX;
        DataManager._pictureEffect.y = startY;
        DataManager._pictureEffect.directPattern = directPattern;
        DataManager._pictureEffect.needRefresh = true;
    });

    PluginManager.registerCommand('NrPictureEffect', 'startLoopEffectDirect', args => {
        const effectId = Number(args.effectId || 1);
        const startX = Number(args.startX || 400);
        const startY = Number(args.startY || 300);
        
        const directPattern = {
            callId: effectId,
            fileName: String(args.fileName || 'TouchParticles'),
            animationType: String(args.animationType || 'Fireworks'),
            particleCount: Number(args.particleCount || 10),
            xOffset: Number(args.xOffset || 0),
            yOffset: Number(args.yOffset || 0),
            blendMode: String(args.blendMode || 'Normal'),
            duration: Number(args.duration || 10),
            fadeSpeed: Number(args.fadeSpeed || 20),
            randomTone: String(args.randomTone || 'true') === 'true',
            enableMovement: String(args.enableMovement || 'false') === 'true',
            targetX: Number(args.targetX || 400),
            targetY: Number(args.targetY || 300),
            movementSpeed: Math.min(Math.max(Number(args.movementSpeed || 5), 1), 20),
            curveMovement: String(args.curveMovement || 'false') === 'true',
            curveHeight: Math.min(Math.max(Number(args.curveHeight || -100), -500), 500),
            enableConvergence: String(args.enableConvergence || 'true') === 'true',
            convergenceSpeed: Math.min(Math.max(Number(args.convergenceSpeed || 3), 1), 10),
            convergenceDelay: Math.min(Math.max(Number(args.convergenceDelay || 15), 0), 60),
            spreadRange: Math.min(Math.max(Number(args.spreadRange || 150), 0), 999),
            loopInterval: Math.min(Math.max(Number(args.loopInterval || 60), 1), 9999),
            autoEnd: String(args.autoEnd || 'false') === 'true',
            endWait: Math.min(Math.max(Number(args.endWait || 600), 1), 99999)
        };

        // ID管理システムに登録
        DataManager._loopEffectsById[effectId] = {
            pattern: directPattern,
            x: startX,
            y: startY,
            active: true,
            interval: directPattern.loopInterval,
            intervalCounter: 0,
            autoEnd: directPattern.autoEnd,
            endWait: directPattern.endWait,
            endCounter: 0,
            needFirstEmit: true
        };

        // 旧システムとの互換性維持（最後に開始されたループエフェクト）
        DataManager._loopEffect.active = true;
        DataManager._loopEffect.callId = effectId;
        DataManager._loopEffect.x = startX;
        DataManager._loopEffect.y = startY;
        DataManager._loopEffect.interval = directPattern.loopInterval;
        DataManager._loopEffect.intervalCounter = 0;
        DataManager._loopEffect.autoEnd = directPattern.autoEnd;
        DataManager._loopEffect.endWait = directPattern.endWait;
        DataManager._loopEffect.endCounter = 0;
        DataManager._loopEffect.needFirstEmit = true;
        DataManager._loopEffect.directPattern = directPattern;
    });

    DataManager.getEffectPatternByCallId = function(callId) {
        // 登録済みパターンから検索（優先）
        if (Moghunter.effectPatterns.length > 0) {
            for (const pattern of Moghunter.effectPatterns) {
                if (pattern.callId === callId) {
                    return pattern;
                }
            }
        }
        
        // ID管理システムから検索
        const effectById = DataManager._effectsById[callId];
        if (effectById && effectById.pattern) {
            return effectById.pattern;
        }
        
        // 旧システムとの互換性維持
        if (callId === -1 || callId === DataManager._pictureEffect.callId) {
            return DataManager._pictureEffect.directPattern;
        }
        
        return null;
    };

    DataManager.getLoopEffectPatternByCallId = function(callId) {
        // 登録済みパターンから検索（優先）
        if (Moghunter.loopEffectPatterns.length > 0) {
            for (const pattern of Moghunter.loopEffectPatterns) {
                if (pattern.callId === callId) {
                    return pattern;
                }
            }
        }
        
        // ID管理システムから検索
        const loopEffectById = DataManager._loopEffectsById[callId];
        if (loopEffectById && loopEffectById.pattern) {
            return loopEffectById.pattern;
        }
        
        // 旧システムとの互換性維持
        if (callId === -1 || callId === DataManager._loopEffect.callId) {
            return DataManager._loopEffect.directPattern;
        }
        
        return null;
    };

    DataManager.getDirectEffectPattern = function() {
        return DataManager._pictureEffect.directPattern;
    };

    DataManager.getEffectById = function(effectId) {
        return DataManager._effectsById[effectId] || null;
    };

    DataManager.getLoopEffectById = function(effectId) {
        return DataManager._loopEffectsById[effectId] || null;
    };

    DataManager.isEffectActive = function(effectId) {
        return !!(DataManager._effectsById[effectId] && DataManager._effectsById[effectId].active);
    };

    DataManager.isLoopEffectActive = function(effectId) {
        return !!(DataManager._loopEffectsById[effectId] && DataManager._loopEffectsById[effectId].active);
    };

    Scene_Base.prototype.createSpriteField3 = function() {
    this._spriteField3 = new Sprite();
    this._spriteField3.z = 100;
    this.addChild(this._spriteField3);
    };

    Scene_Base.prototype.sortSpriteField = function() {
        if (this._spriteField1) {this._spriteField1.children.sort((a, b) => a.z - b.z)};
        if (this._spriteField2) {this._spriteField2.children.sort((a, b) => a.z - b.z)};
        if (this._spriteField3) {this._spriteField3.children.sort((a, b) => a.z - b.z)};
    };

    Scene_Base.prototype.createEffectParticles = function() {
        DataManager._pictureEffect.x = 0;
        DataManager._pictureEffect.y = 0;	 
        DataManager._pictureEffect.needRefresh = false;
        DataManager._pictureEffect.callId = 0;
        this._effectPar = new EffectParticles();
        this._effectPar.z = 450;
        this._spriteField3.addChild(this._effectPar);
    };

    Scene_Base.prototype.updateEffectAnimationBase = function() {
        if (!this._spriteField3) {this.createSpriteField3()};
        if (!this._effectPar) {this.createEffectParticles()};
        this.updateLoopEffect();
    };

    Scene_Base.prototype.updateLoopEffect = function() {
        // 新しいID管理システムの更新
        for (const effectId in DataManager._loopEffectsById) {
            const loopEffect = DataManager._loopEffectsById[effectId];
            if (!loopEffect.active) continue;
            
            if (loopEffect.needFirstEmit && this._effectPar) {
                loopEffect.needFirstEmit = false;
                this._effectPar.addLoopParticles(
                    Number(effectId),
                    loopEffect.x,
                    loopEffect.y
                );
            }
            
            if (loopEffect.autoEnd) {
                loopEffect.endCounter++;
                if (loopEffect.endCounter >= loopEffect.endWait) {
                    loopEffect.active = false;
                    delete DataManager._loopEffectsById[effectId];
                    continue;
                }
            }
            
            loopEffect.intervalCounter++;
            if (loopEffect.intervalCounter >= loopEffect.interval) {
                loopEffect.intervalCounter = 0;
                if (this._effectPar) {
                    this._effectPar.addLoopParticles(
                        Number(effectId),
                        loopEffect.x,
                        loopEffect.y
                    );
                }
            }
        }
        
        // 旧システムとの互換性維持
        if (!DataManager._loopEffect.active) {return};
        if (DataManager._loopEffect.needFirstEmit && this._effectPar) {
            DataManager._loopEffect.needFirstEmit = false;
            this._effectPar.addLoopParticles(
                DataManager._loopEffect.callId,
                DataManager._loopEffect.x,
                DataManager._loopEffect.y
            );
        }
        if (DataManager._loopEffect.autoEnd) {
            DataManager._loopEffect.endCounter++;
            if (DataManager._loopEffect.endCounter >= DataManager._loopEffect.endWait) {
                DataManager._loopEffect.active = false;
                DataManager._loopEffect.callId = 0;
                DataManager._loopEffect.intervalCounter = 0;
                DataManager._loopEffect.endCounter = 0;
                return;
            }
        }
        DataManager._loopEffect.intervalCounter++;
        if (DataManager._loopEffect.intervalCounter >= DataManager._loopEffect.interval) {
            DataManager._loopEffect.intervalCounter = 0;
            if (this._effectPar) {
                this._effectPar.addLoopParticles(
                    DataManager._loopEffect.callId,
                    DataManager._loopEffect.x,
                    DataManager._loopEffect.y
                );
            }
        }
    };

    const _mog_ScnBase_effectPar_update = Scene_Base.prototype.update;
    Scene_Base.prototype.update = function() {
        _mog_ScnBase_effectPar_update.call(this);
        this.updateEffectAnimationBase();
    };

    function EffectParticles() {
        this.initialize.apply(this, arguments);
    };

    EffectParticles.prototype = Object.create(Sprite.prototype);
    EffectParticles.prototype.constructor = EffectParticles;

    EffectParticles.prototype.initialize = function() {
        Sprite.prototype.initialize.call(this);
        this.updateCurrentPattern();
        this.createParticles();  
    };

    EffectParticles.prototype.updateCurrentPattern = function() {
        const callId = DataManager._pictureEffect.callId;
        let pattern = DataManager.getEffectPatternByCallId(callId);
        if (!pattern) {
            pattern = DataManager.getLoopEffectPatternByCallId(callId);
        }
        this._currentPattern = pattern;
        if (this._currentPattern === null) {
            this._mode = 0;
            this._power = 0;
            this._blendType = 0;
            this._fileName = '';
            return;
        }
        this._mode = this.getMode(this._currentPattern.animationType);
        this._power = Math.min(Math.max(this._currentPattern.particleCount, 1), 999);
        this._blendType = this.getBlend(this._currentPattern.blendMode);
        this._fileName = this._currentPattern.fileName;
    };

    EffectParticles.prototype.getMode = function(mode) {
        if (mode == "Random") {return 1;
        } else if (mode == "Zoom Out") {return 2;
        } else if (mode == "Zoom In") {return 3};
        return 0;
    };

    EffectParticles.prototype.getBlend = function(blend) {
        if (blend == "Additive") {return 1;
        } else if (blend == "Multiply") {return 2};
        return 0;
    };

    EffectParticles.prototype.fileNAme = function() {
        return this._fileName || '';
    };

    EffectParticles.prototype.createParticles = function() {
        this._particlesSprites = [];
        if (this._currentPattern === null) {
            return;
        }
        const fileName = this.fileNAme();
        if (!fileName || fileName === '') {
            return;
        }
        for (var i = 0; i < this.power(); i++) {
            this._particlesSprites[i] = new Sprite(ImageManager.loadSystem(fileName));
            this.setBaseData(this._particlesSprites[i],i);
            this.addChild(this._particlesSprites[i]);
        };
    };

    EffectParticles.prototype.addLoopParticles = function(callId, x, y) {
        let pattern = DataManager.getEffectPatternByCallId(callId);
        if (!pattern) {
            pattern = DataManager.getLoopEffectPatternByCallId(callId);
        }
        if (!pattern) return;
        const mode = this.getMode(pattern.animationType);
        let particleCount = pattern.particleCount;
        if (mode == 2 || mode == 3) {
            particleCount = 1;
        }
        const fileName = pattern.fileName;
        if (!fileName || fileName === '') {
            return;
        }
        const blendType = this.getBlend(pattern.blendMode);
        for (var i = 0; i < particleCount; i++) {
            const sprite = new Sprite(ImageManager.loadSystem(fileName));
            sprite.index = i;
            sprite.opacity = 0;
            sprite.anchor.x = 0.5;
            sprite.anchor.y = 0.5;
            sprite.sx = [0,1];
            sprite.sy = [0,1];
            sprite.rt = [0,0.08];
            sprite.sc = 0;
            sprite.wait = 0;
            sprite.blendType = blendType;
            sprite.fadeSpeed = 85;
            sprite.duration = 0;
            sprite.duration2 = 0;
            sprite.isLoopEffect = true;
            sprite.enableMovement = pattern.enableMovement;
            sprite.targetX = pattern.targetX;
            sprite.targetY = pattern.targetY;
            sprite.movementSpeed = pattern.movementSpeed;
            sprite.curveMovement = pattern.curveMovement;
            sprite.curveHeight = pattern.curveHeight;
            sprite.spreadRange = pattern.spreadRange;
            sprite.moveStartX = 0;
            sprite.moveStartY = 0;
            sprite.moveProgress = 0;
            sprite.isMoving = false;
            sprite.moveCompleted = false;
            sprite.individualTargetX = sprite.targetX;
            sprite.individualTargetY = sprite.targetY;
            sprite.enableConvergence = pattern.enableConvergence;
            sprite.convergenceSpeed = pattern.convergenceSpeed;
            sprite.convergenceDelay = 0;
            sprite.convergenceWait = 0;
            sprite.isConverging = false;
            sprite.convergenceProgress = 0;
            sprite.convergenceStartX = 0;
            sprite.convergenceStartY = 0;
            sprite.individualFadeStart = false;
            sprite.reachedTarget = false;
            sprite.targetReachDistance = 8;
            const useRandomTone = pattern.randomTone;
            if (useRandomTone) {
                const r = Math.randomInt(255);
                const g = Math.randomInt(255);
                const b = Math.randomInt(255);
                const colorTone = [r,g,b,255];
                sprite.setColorTone(colorTone);
            }
            sprite.x = x;
            sprite.y = y;
            if (sprite.enableMovement) {
                sprite.moveStartX = sprite.x;
                sprite.moveStartY = sprite.y;
                sprite.moveProgress = 0;
                sprite.isMoving = true;
                sprite.moveCompleted = false;
                const baseTargetX = sprite.targetX;
                const baseTargetY = sprite.targetY;
                if (mode === 0) {
                    const spreadRange = sprite.enableMovement ? sprite.spreadRange : 80;
                    const offsetX = -spreadRange + Math.random() * (spreadRange * 2);
                    const offsetY = -spreadRange + Math.random() * (spreadRange * 2);
                    sprite.individualTargetX = baseTargetX + offsetX;
                    sprite.individualTargetY = baseTargetY + offsetY;
                } else if (mode === 1) {
                    const baseSpreadRange = sprite.enableMovement ? sprite.spreadRange : 60;
                    const spreadRange = baseSpreadRange * 0.8;
                    const offsetX = -spreadRange + Math.random() * (spreadRange * 2);
                    const offsetY = -spreadRange + Math.random() * (spreadRange * 2);
                    sprite.individualTargetX = baseTargetX + offsetX;
                    sprite.individualTargetY = baseTargetY + offsetY;
                } else {
                    sprite.individualTargetX = baseTargetX;
                    sprite.individualTargetY = baseTargetY;
                }
            }
            if (mode == 0) {
                sprite.duration = sprite.duration2 + Math.randomInt(20);
                sprite.wait = 0;
                var r = 0.7 + Math.abs(Math.random() * sprite.sx[1]);
                var d = Math.randomInt(100);
                sprite.sx[0] = d > 40 ? r : -r;
                sprite.sx[0] = d > 90 ? 0 : sprite.sx[0];
                var r = 0.7 + Math.abs(Math.random() * sprite.sy[1]);
                var d = Math.randomInt(100);
                sprite.sy[0] = d > 40 ? r : -r;
                sprite.sy[0] = d > 90 ? 0 : sprite.sy[0];
                var r = 0.01 + Math.abs(Math.random() * sprite.rt[1]);
                sprite.rt[0] = sprite.rt[1] > 0 ? r : -r;
                var pz = ((Math.random() * 0.5) * 1);
                sprite.scale = new PIXI.Point(0.5 + Number(pz), 0.5 + Number(pz));
            } else if (mode == 1) {
                sprite.duration = sprite.duration2 + Math.randomInt(20);
                sprite.wait = Math.randomInt(20);
                sprite.x += -30 + Math.randomInt(60);
                sprite.y += -30 + Math.randomInt(60);
                sprite.sc = 0.02;
                var pz = ((Math.random() * 0.5) * 1);
                sprite.scale = new PIXI.Point(0.5 + Number(pz), 0.5 + Number(pz));
            } else if (mode == 2) {
                sprite.duration = sprite.duration2 + 10;
                sprite.wait = 0;
                sprite.sc = 0.04;
                var pz = 0.1;
                sprite.scale = new PIXI.Point(0.5 + Number(pz), 0.5 + Number(pz));
            } else if (mode == 3) {
                sprite.duration = sprite.duration2 + 10;
                sprite.wait = 0;
                sprite.sc = -0.1;
                var pz = 0.1;
                sprite.scale = new PIXI.Point(3.0, 3.0);
            }
            this._particlesSprites.push(sprite);
            this.addChild(sprite);
        }
    };

    EffectParticles.prototype.mode = function() {
        return this._mode;
    };

    EffectParticles.prototype.xPos = function() {
        return DataManager._pictureEffect.x;
    };

    EffectParticles.prototype.yPos = function() {
        return DataManager._pictureEffect.y;
    };

    EffectParticles.prototype.xOffset = function() {
        return this._currentPattern ? this._currentPattern.xOffset : 0;
    };

    EffectParticles.prototype.yOffset = function() {
        return this._currentPattern ? this._currentPattern.yOffset : 0;
    };

    EffectParticles.prototype.power = function() {
        if (this._currentPattern === null) {
            return 0;
        }
        if (this.mode() == 2) {return 1};
        if (this.mode() == 3) {return 1};
        return this._power;
    };

    EffectParticles.prototype.forcedHide = function(sprite) {
    return false;
    };

    EffectParticles.prototype.setBaseData = function(sprite,index) {
        sprite.index = index;
        sprite.opacity = 0;
        sprite.anchor.x = 0.5;
        sprite.anchor.y = 0.5;
        sprite.sx = [0,1];
        sprite.sy = [0,1];
        sprite.rt = [0,0.08];
        sprite.sc = 0;
        sprite.wait = 0;
        sprite.blendType = this._blendType;
        const currentPattern = this._currentPattern;
        sprite.fadeSpeed = Math.min(Math.max(currentPattern ? currentPattern.fadeSpeed : 20, 2), 125);
        sprite.duration = 0;
        sprite.duration2 = Math.min(Math.max(currentPattern ? currentPattern.duration : 10, 1), 999);
        sprite.isLoopEffect = false;
        sprite.enableMovement = currentPattern ? currentPattern.enableMovement : false;
        sprite.targetX = currentPattern ? currentPattern.targetX : 400;
        sprite.targetY = currentPattern ? currentPattern.targetY : 300;
        sprite.movementSpeed = currentPattern ? currentPattern.movementSpeed : 5;
        sprite.curveMovement = currentPattern ? currentPattern.curveMovement : false;
        sprite.curveHeight = currentPattern ? currentPattern.curveHeight : -100;
        sprite.spreadRange = currentPattern ? currentPattern.spreadRange : 150;
        sprite.moveStartX = 0;
        sprite.moveStartY = 0;
        sprite.moveProgress = 0;
        sprite.isMoving = false;
        sprite.moveCompleted = false;
        sprite.individualTargetX = sprite.targetX;
        sprite.individualTargetY = sprite.targetY;
        sprite.enableConvergence = currentPattern ? currentPattern.enableConvergence : true;
        sprite.convergenceSpeed = currentPattern ? currentPattern.convergenceSpeed : 4;
        sprite.convergenceDelay = 0;
        sprite.convergenceWait = 0;
        sprite.isConverging = false;
        sprite.convergenceProgress = 0;
        sprite.convergenceStartX = 0;
        sprite.convergenceStartY = 0;
        sprite.individualFadeStart = false;
        sprite.reachedTarget = false;
        sprite.targetReachDistance = 8;
        const useRandomTone = currentPattern ? currentPattern.randomTone : true;
        if (useRandomTone) {this.setToneRandom(sprite,index)};
    };

    EffectParticles.prototype.setToneRandom = function(sprite,index) {
        const r = Math.randomInt(255);
        const g = Math.randomInt(255);
        const b = Math.randomInt(255);
        const colorTone = [r,g,b,255]
        sprite.setColorTone(colorTone);
    };

    EffectParticles.prototype.setIndividualTarget = function(sprite, index) {
        const mode = this.mode();
        const baseTargetX = sprite.targetX;
        const baseTargetY = sprite.targetY;
        if (mode === 0) {
            const spreadRange = sprite.enableMovement ? sprite.spreadRange : 80;
            const offsetX = -spreadRange + Math.random() * (spreadRange * 2);
            const offsetY = -spreadRange + Math.random() * (spreadRange * 2);
            sprite.individualTargetX = baseTargetX + offsetX;
            sprite.individualTargetY = baseTargetY + offsetY;
        } else if (mode === 1) {
            const baseSpreadRange = sprite.enableMovement ? sprite.spreadRange : 60;
            const spreadRange = baseSpreadRange * 0.8;
            const offsetX = -spreadRange + Math.random() * (spreadRange * 2);
            const offsetY = -spreadRange + Math.random() * (spreadRange * 2);
            sprite.individualTargetX = baseTargetX + offsetX;
            sprite.individualTargetY = baseTargetY + offsetY;
        } else {
            sprite.individualTargetX = baseTargetX;
            sprite.individualTargetY = baseTargetY;
        }
    };

    EffectParticles.prototype.refreshParticles = function(sprite,index) {
        sprite.x = this.xPos();
        sprite.y = this.yPos();
        if (sprite.enableMovement) {
            sprite.moveStartX = sprite.x;
            sprite.moveStartY = sprite.y;
            sprite.moveProgress = 0;
            sprite.isMoving = true;
            sprite.moveCompleted = false;
            this.setIndividualTarget(sprite, index);
        }
        if (this.mode() == 0) {
            this.setAnimation1(sprite,index) ;
        } else if (this.mode() == 1) {
            this.setAnimation2(sprite,index);
        } else if (this.mode() == 2) {
            this.setAnimation3(sprite,index);
        } else if (this.mode() == 3) {
            this.setAnimation4(sprite,index);		 
        };
    };

    EffectParticles.prototype.setAnimation1 = function(sprite,index) { 
        sprite.duration = sprite.duration2 + Math.randomInt(20);
        sprite.wait = 0;
        var r = 0.7 + Math.abs(Math.random() * sprite.sx[1]);
        var d = Math.randomInt(100);
        sprite.sx[0] = d > 40 ? r : -r;
        sprite.sx[0] = d > 90 ? 0 : sprite.sx[0];
        var r = 0.7 + Math.abs(Math.random() * sprite.sy[1]);
        var d = Math.randomInt(100);
        sprite.sy[0] = d > 40 ? r : -r;
        sprite.sy[0] = d > 90 ? 0 : sprite.sy[0];
        var r = 0.01 + Math.abs(Math.random() * sprite.rt[1]);
        sprite.rt[0] = sprite.rt[1] > 0 ? r : -r;
        var pz = ((Math.random() * 0.5) * 1);
        sprite.scale = new PIXI.Point(0.5 + Number(pz), 0.5 + Number(pz));	 
    };

    EffectParticles.prototype.setAnimation2 = function(sprite,index) { 
        sprite.duration = sprite.duration2 + Math.randomInt(20);
        sprite.wait = Math.randomInt(20);
        sprite.x += -30 + Math.randomInt(60);
        sprite.y += -30 + Math.randomInt(60);
        sprite.sc = 0.02;
        var pz = ((Math.random() * 0.5) * 1);
        sprite.scale = new PIXI.Point(0.5 + Number(pz), 0.5 + Number(pz));	 
    };

    EffectParticles.prototype.setAnimation3 = function(sprite,index) { 
        sprite.duration = sprite.duration2 + 10;
        sprite.wait = 0;
        sprite.sc = 0.04;
        var pz = 0.1;
        sprite.scale = new PIXI.Point(0.5 + Number(pz), 0.5 + Number(pz));	 
    };

    EffectParticles.prototype.setAnimation4 = function(sprite,index) { 
        sprite.duration = sprite.duration2 + 10;
        sprite.wait = 0;
        sprite.sc = -0.1;
        var pz = 0.1;
        sprite.scale = new PIXI.Point(3.0, 3.0);	 
    };

    EffectParticles.prototype.updateParticles = function(sprite,index) {
        if (this.forcedHide(sprite,index)) {
            this.updateHide(sprite,index);
        } else {
            if (sprite.wait > 0) {
                sprite.wait--;
            } else {
                this.updateParticlesData(sprite,index);
            };
        };
    };

    EffectParticles.prototype.updateParticlesData = function(sprite,index) {
        sprite.visible = true;
        if (sprite.enableMovement && sprite.isMoving) {
            this.updateMovement(sprite);
        } else if (!sprite.enableMovement) {
            sprite.x += sprite.sx[0];
            sprite.y += sprite.sy[0];
        }
        sprite.rotation += sprite.rt[0];
        if (sprite.scale.x < 4.00) {sprite.scale.x += sprite.sc};
        if (sprite.scale.x < 0.00) {sprite.scale.x = 0.00};
        sprite.scale.y = sprite.scale.x;
        this.updateFade(sprite,index)
    };

    EffectParticles.prototype.updateHide = function(sprite,index) {
        sprite.visible = false;
        sprite.duration = 0;
        sprite.opacity = 0;
    };

    EffectParticles.prototype.updateConvergence = function(sprite) {
        if (!sprite.isConverging) return;
        sprite.convergenceProgress += sprite.convergenceSpeed / 100.0;
        if (sprite.convergenceProgress >= 1.0) {
            sprite.convergenceProgress = 1.0;
            sprite.isConverging = false;
            sprite.individualFadeStart = true;
            sprite.x = sprite.targetX;
            sprite.y = sprite.targetY;
            return;
        }
        const easeProgress = this.easeInOutSmooth(sprite.convergenceProgress);
        sprite.x = sprite.convergenceStartX + (sprite.targetX - sprite.convergenceStartX) * easeProgress;
        sprite.y = sprite.convergenceStartY + (sprite.targetY - sprite.convergenceStartY) * easeProgress;
    };

    EffectParticles.prototype.updateMovement = function(sprite) {
        sprite.moveProgress += sprite.movementSpeed / 100.0;
        if (sprite.moveProgress >= 1.0) {
            sprite.moveProgress = 1.0;
            sprite.isMoving = false;
            sprite.moveCompleted = true;
        }
        const easeProgress = this.easeInOut(sprite.moveProgress);
        let convergenceProgress = 0;
        if (sprite.enableConvergence) {
            const convergenceStart = 0.5;
            if (sprite.moveProgress > convergenceStart) {
                const adjustedProgress = (sprite.moveProgress - convergenceStart) / (1.0 - convergenceStart);
                convergenceProgress = this.easeInOutSmooth(adjustedProgress);
            }
        }
        if (sprite.curveMovement) {
            const baseX = sprite.moveStartX + (sprite.individualTargetX - sprite.moveStartX) * easeProgress;
            const baseY = sprite.moveStartY + (sprite.individualTargetY - sprite.moveStartY) * easeProgress;
            const curveOffset = Math.sin(sprite.moveProgress * Math.PI) * sprite.curveHeight;
            if (sprite.enableConvergence) {
                const targetX = sprite.individualTargetX + (sprite.targetX - sprite.individualTargetX) * convergenceProgress;
                const targetY = sprite.individualTargetY + (sprite.targetY - sprite.individualTargetY) * convergenceProgress;
                sprite.x = sprite.moveStartX + (targetX - sprite.moveStartX) * easeProgress;
                sprite.y = sprite.moveStartY + (targetY - sprite.moveStartY) * easeProgress + curveOffset;
            } else {
                sprite.x = baseX;
                sprite.y = baseY + curveOffset;
            }
        } else {
            if (sprite.enableConvergence) {
                const targetX = sprite.individualTargetX + (sprite.targetX - sprite.individualTargetX) * convergenceProgress;
                const targetY = sprite.individualTargetY + (sprite.targetY - sprite.individualTargetY) * convergenceProgress;
                sprite.x = sprite.moveStartX + (targetX - sprite.moveStartX) * easeProgress;
                sprite.y = sprite.moveStartY + (targetY - sprite.moveStartY) * easeProgress;
            } else {
                sprite.x = sprite.moveStartX + (sprite.individualTargetX - sprite.moveStartX) * easeProgress;
                sprite.y = sprite.moveStartY + (sprite.individualTargetY - sprite.moveStartY) * easeProgress;
            }
        }
        if (sprite.moveCompleted && !sprite.reachedTarget) {
            sprite.reachedTarget = true;
            sprite.individualFadeStart = true;
            sprite.duration = sprite.duration2;
            sprite.x = sprite.targetX;
            sprite.y = sprite.targetY;
        }
    };

    EffectParticles.prototype.easeInOut = function(t) {
        return t < 0.5 ? 2 * t * t : -1 + (4 - 2 * t) * t;
    };

    EffectParticles.prototype.easeInOutSmooth = function(t) {
        return t * t * (3.0 - 2.0 * t);
    };

    EffectParticles.prototype.updateFade = function(sprite,index) {
        if (sprite.enableMovement) {
            if (!sprite.moveCompleted) {
                if (sprite.duration > 0) {
                    sprite.duration--;
                    sprite.opacity += 20;
                    if (sprite.opacity > 255) sprite.opacity = 255;
                }
                return;
            }
            if (sprite.duration > 0) {
                sprite.duration--;
                return;
            }
            sprite.opacity -= sprite.fadeSpeed;
            return;
        }
        if (sprite.duration > 0) {
            sprite.duration--;
            sprite.opacity += 20;
            if (sprite.opacity > 255) sprite.opacity = 255;
        } else {
            sprite.opacity -= sprite.fadeSpeed;
        };
    };

    EffectParticles.prototype.refreshBase = function() {
        DataManager._pictureEffect.needRefresh = false;
        const callId = DataManager._pictureEffect.callId;
        let newPattern = DataManager.getEffectPatternByCallId(callId);
        if (!newPattern) {
            newPattern = DataManager.getLoopEffectPatternByCallId(callId);
        }
        if (this.hasPatternChanged(newPattern)) {
            this.updateCurrentPattern();
            this.recreateParticles();
        }
        if (newPattern === null) {
            return;
        }
        for (var i = 0; i < this._particlesSprites.length; i++) {
            this.refreshParticles(this._particlesSprites[i],i);
        };
    };

    EffectParticles.prototype.hasPatternChanged = function(newPattern) {
        if (!this._currentPattern && newPattern === null) return false;
        if (!this._currentPattern || newPattern === null) return true;
        return (this._currentPattern.callId !== newPattern.callId ||
                this._currentPattern.fileName !== newPattern.fileName ||
                this._currentPattern.animationType !== newPattern.animationType ||
                this._currentPattern.particleCount !== newPattern.particleCount ||
                this._currentPattern.blendMode !== newPattern.blendMode ||
                this._currentPattern.enableMovement !== newPattern.enableMovement ||
                this._currentPattern.targetX !== newPattern.targetX ||
                this._currentPattern.targetY !== newPattern.targetY ||
                this._currentPattern.movementSpeed !== newPattern.movementSpeed ||
                this._currentPattern.curveMovement !== newPattern.curveMovement ||
                this._currentPattern.curveHeight !== newPattern.curveHeight ||
                this._currentPattern.enableConvergence !== newPattern.enableConvergence ||
                this._currentPattern.convergenceSpeed !== newPattern.convergenceSpeed ||
                this._currentPattern.convergenceDelay !== newPattern.convergenceDelay ||
                this._currentPattern.spreadRange !== newPattern.spreadRange);
    };

    EffectParticles.prototype.recreateParticles = function() {
        if (this._particlesSprites) {
            for (var i = 0; i < this._particlesSprites.length; i++) {
                if (this._particlesSprites[i]) {
                    this.removeChild(this._particlesSprites[i]);
                }
            }
            this._particlesSprites = [];
        }
        this.createParticles();
    };

    EffectParticles.prototype.needRefresh = function() {
        return DataManager._pictureEffect.needRefresh;
    };

    EffectParticles.prototype.updatePositionReal = function() {
        this.updatePositionNormal();
    };

    EffectParticles.prototype.updatePositionNormal = function() {
        this.x = this.xOffset();
        this.y = this.yOffset();
    };

    EffectParticles.prototype.update = function() {
        Sprite.prototype.update.call(this);
        if (this.needRefresh()) {this.refreshBase()};
        if (!this._particlesSprites || this._particlesSprites.length === 0) {
            this.updatePositionReal();
            return;
        }
        for (var i = 0; i < this._particlesSprites.length; i++) {
            if (this._particlesSprites[i]) {
                this.updateParticles(this._particlesSprites[i],i);
            }
        };
        this.removeDeadParticles();
        this.updatePositionReal();
    };

    EffectParticles.prototype.removeDeadParticles = function() {
        for (var i = this._particlesSprites.length - 1; i >= 0; i--) {
            const sprite = this._particlesSprites[i];
            if (sprite && sprite.opacity <= 0) {
                this.removeChild(sprite);
                this._particlesSprites.splice(i, 1);
            }
        }
    };

    window.NrPictureEffect = window.NrPictureEffect || {};

    NrPictureEffect.startEffect = function(callId, startX, startY) {
        const args = { callId: String(callId), startX: String(startX), startY: String(startY) };
        const scene = SceneManager._scene;
        if (scene) {
            PluginManager.callCommand({ eventId: () => 0 }, 'NrPictureEffect', 'startEffect', args);
        }
    };

    NrPictureEffect.startLoopEffect = function(callId, startX, startY) {
        const args = { callId: String(callId), startX: String(startX), startY: String(startY) };
        const scene = SceneManager._scene;
        if (scene) {
            PluginManager.callCommand({ eventId: () => 0 }, 'NrPictureEffect', 'startLoopEffect', args);
        }
    };

    NrPictureEffect.stopLoopEffect = function() {
        const scene = SceneManager._scene;
        if (scene) {
            PluginManager.callCommand({ eventId: () => 0 }, 'NrPictureEffect', 'stopLoopEffect', {});
        }
    };

    NrPictureEffect.startEffectDirect = function(options) {
        const args = {
            effectId: String(options.effectId || 1),
            startX: String(options.startX || 400),
            startY: String(options.startY || 300),
            fileName: String(options.fileName || 'TouchParticles'),
            animationType: String(options.animationType || 'Fireworks'),
            particleCount: String(options.particleCount || 10),
            xOffset: String(options.xOffset || 0),
            yOffset: String(options.yOffset || 0),
            blendMode: String(options.blendMode || 'Normal'),
            duration: String(options.duration || 10),
            fadeSpeed: String(options.fadeSpeed || 20),
            randomTone: String(options.randomTone || true),
            enableMovement: String(options.enableMovement || false),
            targetX: String(options.targetX || 400),
            targetY: String(options.targetY || 300),
            movementSpeed: String(options.movementSpeed || 5),
            curveMovement: String(options.curveMovement || false),
            curveHeight: String(options.curveHeight || -100),
            enableConvergence: String(options.enableConvergence || true),
            convergenceSpeed: String(options.convergenceSpeed || 3),
            convergenceDelay: String(options.convergenceDelay || 15),
            spreadRange: String(options.spreadRange || 150)
        };
        const scene = SceneManager._scene;
        if (scene) {
            PluginManager.callCommand({ eventId: () => 0 }, 'NrPictureEffect', 'startEffectDirect', args);
        }
    };

    NrPictureEffect.startLoopEffectDirect = function(options) {
        const args = {
            effectId: String(options.effectId || 1),
            startX: String(options.startX || 400),
            startY: String(options.startY || 300),
            fileName: String(options.fileName || 'TouchParticles'),
            animationType: String(options.animationType || 'Fireworks'),
            particleCount: String(options.particleCount || 10),
            xOffset: String(options.xOffset || 0),
            yOffset: String(options.yOffset || 0),
            blendMode: String(options.blendMode || 'Normal'),
            duration: String(options.duration || 10),
            fadeSpeed: String(options.fadeSpeed || 20),
            randomTone: String(options.randomTone || true),
            enableMovement: String(options.enableMovement || false),
            targetX: String(options.targetX || 400),
            targetY: String(options.targetY || 300),
            movementSpeed: String(options.movementSpeed || 5),
            curveMovement: String(options.curveMovement || false),
            curveHeight: String(options.curveHeight || -100),
            enableConvergence: String(options.enableConvergence || true),
            convergenceSpeed: String(options.convergenceSpeed || 3),
            convergenceDelay: String(options.convergenceDelay || 15),
            spreadRange: String(options.spreadRange || 150),
            loopInterval: String(options.loopInterval || 60),
            autoEnd: String(options.autoEnd || false),
            endWait: String(options.endWait || 600)
        };
        const scene = SceneManager._scene;
        if (scene) {
            PluginManager.callCommand({ eventId: () => 0 }, 'NrPictureEffect', 'startLoopEffectDirect', args);
        }
    };

    NrPictureEffect.stopEffectById = function(effectId) {
        const args = { effectId: String(effectId || 1) };
        const scene = SceneManager._scene;
        if (scene) {
            PluginManager.callCommand({ eventId: () => 0 }, 'NrPictureEffect', 'stopEffectById', args);
        }
    };

    NrPictureEffect.isEffectActive = function(effectId) {
        return DataManager.isEffectActive(effectId);
    };

    NrPictureEffect.isLoopEffectActive = function(effectId) {
        return DataManager.isLoopEffectActive(effectId);
    };


})();