//=============================================================================
// NrSimpleVoiceControl.js
//=============================================================================

/*:
 * @target MZ
 * @plugindesc SimpleVoice拡張 v1.2.0
 * @author NJ
 * @base PluginCommonBase
 * @orderAfter SimpleVoice
 *
 * @param VoiceCutSwitch
 * @text ボイスカットスイッチ
 * @type switch
 * @default 1
 *
 * @param EnableVoicePlayback
 * @text ボイス再生を有効にする
 * @type boolean
 * @default true
 *
 * @param Characters
 * @text キャラクター音量定義
 * @type struct<CharacterVolume>[]
 * @default []
 *
 * @command NrPlayVoice
 * @text ボイスを再生
 * @desc 指定したボイスファイルを再生します
 *
 * @arg voiceName
 * @text ボイスファイル名
 * @type string
 * @desc audio/voice/フォルダ内のファイル名（拡張子なし）
 * @default
 *
 * @command NrStopVoice
 * @text ボイスを停止
 * @desc 現在再生中のボイスを停止します
 *
 * @help
 * 概要:
 * - \SV[ファイル名] または \SV[\v[n] を含む式] で audio/voice/ から再生
 * - メッセージ終了時、指定スイッチONなら直近のボイスを停止
 *
 * 外部から直接ボイスを再生する方法:
 * 1. プラグインコマンド「NrPlayVoice」を使用
 * 2. スクリプトで NrSimpleVoiceControl.playVoice("ファイル名") を呼び出し
 * 3. スクリプトで NrPlayVoice("ファイル名") を呼び出し
 *
 * キャラ別ボイス音量:
 * ・プラグインパラメータ「キャラクター音量定義」で以下を設定：
 *    - 登録名 … オプションに表示される名称（例：主人公）
 *    - ラベル … 判定基準。ファイル名（最後の / 以降、拡張子なし）が
 *                「ラベル」そのもの、または「ラベル + 数字のみ」のとき一致
 *                例：ラベル=M → M, M0, M110 は一致／MX, M_A は不一致
 *    - タグ   … Config保存用キーの基部。例：hero → heroVolume として保存
 *             ※ここは念のため、なるべく英語で登録してください。また以下の名前は設定しないようにしてください。
 *             [voice,bgm,bgs,me,se]
 *    - 初期値 … 0～100（％） 既定：100
 * ・一致した場合、そのキャラ音量が適用。未一致はマスターのみ。
 *
 * 使い方:
 *  \v[1]などの変数も対応しています。
 *  変数1に 110 → \SV[M\v[1]] は M110 を再生。ラベル=M が一致し、Mの音量が適用されます。
 *  プラグインコマンドからの呼び出しであっても、ファイル名にラベル名が入っていれば自動的に適応されます。
 *
 * バージョン
 *   1.2.0 - 外部から直接呼び出せるAPI追加（プラグインコマンド、グローバル関数）
 *   1.1.2 - 制御文字との衝突を回避
 *   1.1.1 - \SV[] の競合修正、複数対応
 *   1.0.0 - 初回
 * 
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可です。
 */

/*~struct~CharacterVolume:
 * @param name
 * @text 登録名（オプション表示名）
 * @type string
 *
 * @param label
 * @text ラベル（判定用）
 * @type string
 *
 * @param tag
 * @text タグ（Configキー接尾辞）
 * @type string
 *
 * @param defaultValue
 * @text 初期値(％)
 * @type number
 * @min 0
 * @max 100
 * @default 100
 */

(() => {
    'use strict';
    const script = document.currentScript;
    const params = PluginManagerEx.createParameter(script);

    const voiceCutSwitchId    = Number(params['VoiceCutSwitch'] || 1);
    const enableVoicePlayback = !!params['EnableVoicePlayback'];
    let lastPlayedVoiceName   = null;

    //========================
    // キャラ音量定義
    //========================
    const characterDefs = (params.Characters || []).map(c => {
        const name         = String(c.name  || '').trim();
        const label        = String(c.label || '').trim();
        const tag          = String(c.tag   || '').trim();
        const defaultValue = Math.max(0, Math.min(100, Number(c.defaultValue ?? 100)));
        const key          = tag ? `${tag}Volume` : '';
        return { name, label, tag, defaultValue, key };
    }).filter(c => c.label && c.key);

    const escapeRegExp = s => String(s).replace(/[.*+?^${}()|[\]\\]/g, '\\$&');

    function getBaseName(fullName) {
        const part = String(fullName || '').split('/').pop();
        return part.replace(/\.[^/.]+$/, '');
    }

    function findCharacterByFilename(voiceName) {
        const base = getBaseName(voiceName);
        for (const def of characterDefs) {
            const re = new RegExp(`^${escapeRegExp(def.label)}(?:\\d+)?$`);
            if (re.test(base)) return def;
        }
        return null;
    }

    //========================
    // ConfigManager 拡張
    //========================
    const _CM_makeData  = ConfigManager.makeData;
    const _CM_applyData = ConfigManager.applyData;
    const _CM_load      = ConfigManager.load;

    characterDefs.forEach(def => {
        if (!(def.key in ConfigManager)) {
            Object.defineProperty(ConfigManager, def.key, {
                configurable: true,
                enumerable: true,
                get() {
                    if (!Object.prototype.hasOwnProperty.call(this, `_${def.key}`)) {
                        this[`_${def.key}`] = def.defaultValue;
                    }
                    return this[`_${def.key}`];
                },
                set(v) {
                    const n = Math.max(0, Math.min(100, Number(v)));
                    this[`_${def.key}`] = isNaN(n) ? def.defaultValue : n;
                }
            });
        }
    });

    ConfigManager.load = function() {
        characterDefs.forEach(def => {
            ConfigManager[def.key] = def.defaultValue;
        });
        _CM_load.apply(this, arguments);
    };

    ConfigManager.makeData = function() {
        const config = _CM_makeData.apply(this, arguments);
        characterDefs.forEach(def => {
            config[def.key] = ConfigManager[def.key];
        });
        return config;
    };

    ConfigManager.applyData = function(config) {
        _CM_applyData.apply(this, arguments);
        characterDefs.forEach(def => {
            const symbol = def.key;
            if (config && Object.prototype.hasOwnProperty.call(config, symbol)) {
                const v = config[symbol];
                const n = Math.max(0, Math.min(100, Number(v)));
                ConfigManager[symbol] = isNaN(n) ? def.defaultValue : n;
            } else {
                ConfigManager[symbol] = def.defaultValue;
            }
        });
    };

    //========================
    // オプション画面
    //========================
    const _WO_addVolumeOptions = Window_Options.prototype.addVolumeOptions;
    Window_Options.prototype.addVolumeOptions = function() {
        _WO_addVolumeOptions.apply(this, arguments);
        characterDefs.forEach(def => {
            this.addCommand(def.name, def.key);
        });
    };

    //========================
    // Voice再生処理（内部用：メッセージ表示から呼ばれる）
    //========================
    function playVoiceInternal(name) {
        if (!enableVoicePlayback || !name) return;

        lastPlayedVoiceName = name;
        const def = findCharacterByFilename(name);
        const indivVolume = def ? Number(ConfigManager[def.key] ?? def.defaultValue) : 100;
        const args = { name, volume: indivVolume, pitch: 100, pan: 0, loop: false };

        if (AudioManager.playVoice) {
            AudioManager.playVoice(args, false, 0);
        } else {
            const buffer = AudioManager.createBuffer('voice/', name);
            buffer.volume = (ConfigManager.voiceVolume ?? 100) * indivVolume / 10000;
            buffer.play(false, 0);
            buffer.path = name;
        }
    }

    //========================
    // Voice再生処理（外部用：直接呼び出し専用）
    //========================
    function playVoiceExternal(name) {
        if (!enableVoicePlayback || !name) return;

        const voiceName = String(name).trim();
        if (!voiceName) return;

        lastPlayedVoiceName = voiceName;
        const def = findCharacterByFilename(voiceName);
        const indivVolume = def ? Number(ConfigManager[def.key] ?? def.defaultValue) : 100;

        // SimpleVoiceのプラグインコマンドと同じ形式で呼び出し
        if (AudioManager.playVoice) {
            const voiceArgs = {
                name: voiceName,
                volume: indivVolume,
                pitch: 100,
                pan: 0
            };
            AudioManager.playVoice(voiceArgs, false, 0);
        } else {
            const buffer = AudioManager.createBuffer('voice/', voiceName);
            buffer.volume = (ConfigManager.voiceVolume ?? 100) * indivVolume / 10000;
            buffer.play(false, 0);
            buffer.path = voiceName;
        }
    }

    function stopVoice() {
        if (lastPlayedVoiceName) {
            if (AudioManager.stopVoice) {
                AudioManager.stopVoice(lastPlayedVoiceName, null);
            }
            lastPlayedVoiceName = null;
        }
    }

    // グローバルに公開（Nrプレフィックス付き）
    window.NrPlayVoice = playVoiceExternal;
    window.NrStopVoice = stopVoice;

    // 名前空間を使った公開（推奨）
    window.NrSimpleVoiceControl = {
        playVoice: playVoiceExternal,
        stopVoice: stopVoice,
        getLastPlayedVoiceName: () => lastPlayedVoiceName,
        isEnabled: () => enableVoicePlayback
    };

    //========================
    // プラグインコマンド
    //========================
    PluginManagerEx.registerCommand(script, "NrPlayVoice", args => {
        const voiceName = String(args.voiceName || '').trim();
        if (voiceName) {
            playVoiceExternal(voiceName);
        }
    });

    PluginManagerEx.registerCommand(script, "NrStopVoice", args => {
        stopVoice();
    });

    //========================
    // \SV[] 展開
    //========================
    const _Window_Base_convertEscapeCharacters = Window_Base.prototype.convertEscapeCharacters;
    Window_Base.prototype.convertEscapeCharacters = function(text) {
        text = text.replace(/\\SV\[((?:\\v\[\d+\]|[^\]])*)\]/gi, (_, raw) => {
            const expanded = raw.replace(/\\v\[(\d+)\]/gi, (_, n) =>
                String($gameVariables.value(Number(n)))
            );
            playVoiceInternal(expanded.trim());
            return '';
        });
        return _Window_Base_convertEscapeCharacters.call(this, text);
    };

    //========================
    // メッセージ終了時のカット
    //========================
    const _Window_Message_terminateMessage = Window_Message.prototype.terminateMessage;
    Window_Message.prototype.terminateMessage = function() {
        if ($gameSwitches.value(voiceCutSwitchId)) {
            stopVoice();
        }
        _Window_Message_terminateMessage.call(this);
    };

    // 旧形式のパラメータ公開（後方互換性）
    window.NrSimpleVoiceControlParams = {
        enableVoicePlayback: enableVoicePlayback,
        voiceCutSwitchId: voiceCutSwitchId
    };

})();
