//=============================================================================
// NrSpineChoice.js
//=============================================================================

/*:
 * @plugindesc Spine選択肢の補助プラグイン v1.0.0
 * @author NJ
 *
 * @param JudgeVariable
 * @text 判定用変数
 * @type variable
 * @desc Spine選択中はこの変数を0にします。選択が行われたら変数に選択番号（1〜）を代入してください。
 * @default 0
 *
 * @param SpineChoices
 * @text Spine選択肢リスト
 * @type struct<SpineChoice>[]
 * @default []
 *
 * @help
 * 使用方法:
 *   SpineChoiceManager.show("呼び出しID");
 *
 * バージョン:
 * v1.0.0 初回
 *
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可です。
 */

/*~struct~SpineChoice:
 * @param id
 * @text 呼び出しID
 * @desc show() で呼び出すための一意なIDを指定します。
 * @type string
 *
 * @param skeleton
 * @text スケルトン名
 * @desc 表示するSpineのスケルトン名。
 * @type string
 *
 * @param animationTrack
 * @text アニメーション配列再生トラック
 * @desc animation（配列）で再生するトラック番号。未指定時は 0。
 * @type number
 * @default 0
 *
 * @param animation
 * @text アニメーション（配列）
 * @desc Spineに再生させるアニメーションのリスト。複数指定可能。
 * @type string[]
 * @default []
 *
 * @param skins
 * @text スキン名リスト（最大5要素）
 * @desc Spineに適用するスキン名リスト。最大5つ。
 * @type string[]
 * @default []
 *
 * @param animationTracks
 * @text アニメーション（トラック付き）
 * @desc トラック番号付きアニメーションリスト。{"track": 0, "name": "idle"} の形式で指定。
 * @type struct<SpineAnimation>[]
 * @default []
 *
 * @param isNewSpine
 * @text 新規Spine作成
 * @desc trueなら新規Spine作成、falseなら既存にアニメーションのみ再適用。
 * @type boolean
 * @default true
 *
 * @param postSelectionTracks
 * @text 選択後アニメーション（トラック付き）
 * @desc 選択後に再生されるアニメーションリスト。
 * @type struct<SpineAnimation>[]
 * @default []
 *
 * @param deleteOnSelect
 * @text 選択後Spine削除
 * @desc trueなら選択後にSpineを削除。falseなら残す。
 * @type boolean
 * @default true
 *
 * @param applySwitches
 * @text 実行時にONにするスイッチID
 * @desc この選択肢が表示されたときにONにするスイッチのリスト。
 * @type switch[]
 * @default []
 *
 * @param applyVariable
 * @text 実行時に代入する変数ID
 * @desc この選択肢が表示されたときに代入される変数ID。
 * @type variable
 * @default 0
 *
 * @param applyValue
 * @text 実行時に代入する値
 * @desc 上記の変数に代入される数値。
 * @type number
 * @default 0
 *
 * @param pictureId
 * @text 表示ピクチャID
 * @desc Spineを表示するピクチャのID。
 * @type number
 * @default 1
 *
 * @param pictureX
 * @text ピクチャX座標
 * @desc ピクチャのX座標。
 * @type number
 * @default 400
 *
 * @param pictureY
 * @text ピクチャY座標
 * @desc ピクチャのY座標。
 * @type number
 * @default 300
 *
 * @param pictureScale
 * @text ピクチャスケール（%）
 * @desc ピクチャの拡大率（％単位）。
 * @type number
 * @default 100
 *
 * @param showOnceScript
 * @text 表示時一度だけ実行スクリプト
 * @desc この選択肢が呼び出された時に一度だけ実行されます。
 * @type note
 * @default
 *
 * @param allowRepeatInput
 * @text 連続入力の可否
 * @desc trueで選択後、一定時間で再入力を許可する。
 * @type boolean
 * @default false
 *
 * @param repeatDelay
 * @text 入力制御時間(ms)
 * @desc allowRepeatInputがtrueの時、再入力を許可するまでの時間（ミリ秒）。
 * @type number
 * @default 1000
 *
 * @param hitSelections
 * @text ヒットボックス選択肢
 * @desc ヒットボックスによる個別の選択肢定義。
 * @type struct<SpineChoiceHit>[]
 * @default []
 */

/*~struct~SpineChoiceHit:
 * @param name
 * @text 選択対象ヒットボックス名
 * @desc マウスがこのスロット名に一致したときに対象とする選択肢として認識されます。
 * @type string
 *
 * @param onEnterScript
 * @text ヒット時スクリプト
 * @desc ヒットボックスにマウスが初めて入った時に一度だけ実行されるJavaScriptスクリプト。
 * @type note
 * @default
 *
 * @param onTriggerScript
 * @text トリガースクリプト
 * @desc トリガー操作が行われたときに一度だけ実行されるJavaScriptスクリプト。
 * @type note
 * @default
 *
 * @param triggerJudgeValue
 * @text 判定変数代入値
 * @desc トリガー成立時に代入される数値。選択肢として扱われる番号（1以上）。
 * @type number
 * @default 0
 *
 * @param choiceTriggerType
 * @text トリガー種別
 * @desc どの操作がトリガーとして反応するかを選びます。
 * @type select
 * @option none
 * @option leftClick
 * @option TriggerleftClick
 * @option leftClickReleased
 * @option rightClick
 * @option keyboard
 * @default none
 *
 * @param choiceTriggerKey
 * @text トリガーキー（keyboard時）
 * @desc トリガー種別が "keyboard" のときに押されるキーの名称（例: ok, cancel, shiftなど）。
 * @type string
 * @default
 */

/*~struct~SpineAnimation:
 * @param track
 * @text トラック番号
 * @desc 再生トラック番号（整数）。
 * @type number
 * @default 0
 *
 * @param name
 * @text アニメーション名
 * @desc 再生するアニメーション名。
 * @type string
 * @default
 */

(function() {
    var parameters = PluginManager.parameters("NrSpineChoice");
    var rawList = JSON.parse(parameters["SpineChoices"] || "[]");
    var judgeVariable = Number(parameters["JudgeVariable"] || 0);

    var SpineChoiceManager = {
        _list: {},
        _active: false,
        _readyForInput: false,
        _buttons: [],
        _sprite: null,
        _interpreter: null,
        _currentConfig: null,
        _enteredBox: "",
        _triggeredFlags: [],
        _lastLoggedBox: "",
        _windowFocused: true,
        _lastSpineCheck: 0,

        initialize: function() {
            for (var i = 0; i < rawList.length; i++) {
                var cfg = JSON.parse(rawList[i]);
                this._list[cfg.id] = {
                    skeleton: cfg.skeleton,
                    animation: JSON.parse(cfg.animation || "[]"),
                    animationTrack: Number(cfg.animationTrack || 0),
                    animationTracks: JSON.parse(cfg.animationTracks || "[]").map(JSON.parse),
                    skins: JSON.parse(cfg.skins || "[]"),
                    applySwitches: JSON.parse(cfg.applySwitches || "[]").map(Number),
                    applyVariable: Number(cfg.applyVariable || 0),
                    applyValue: Number(cfg.applyValue || 0),
                    pictureId: Number(cfg.pictureId),
                    x: Number(cfg.pictureX),
                    y: Number(cfg.pictureY),
                    scale: Number(cfg.pictureScale),
                    hitSelections: JSON.parse(cfg.hitSelections || "[]").map(JSON.parse),
                    isNewSpine: String(cfg.isNewSpine) === "false" ? false : true,
                    postSelectionTracks: JSON.parse(cfg.postSelectionTracks || "[]").map(JSON.parse),
                    deleteOnSelect: String(cfg.deleteOnSelect) === "false" ? false : true,
                    showOnceScript: cfg.showOnceScript,
                    repeatDelay: Number(cfg.repeatDelay || 1000),
                    allowRepeatInput: String(cfg.allowRepeatInput) === "true" ? true : false,
                    _lastTriggerTime: 0,
                    _shownOnce: false
                };
            }
        },

        runScriptSafely: function(script) {
            if (script && script.trim()) {
                try {
                    var decoded = script;
                    while (decoded.charAt(0) === "\"" && decoded.charAt(decoded.length - 1) === "\"") {
                        decoded = JSON.parse(decoded);
                    }
                    decoded = decoded.replace(/CommonEvent\s*:\s*(\d+)/g, 'CommonEvent($1)');
                    var wrapped = new Function(
                        "var CommonEvent = function(id) {" +
                        "if ($dataCommonEvents[id]) {$gameTemp.reserveCommonEvent(id);}" +
                        "};" + decoded
                    );
                    wrapped();
                } catch (e) {
                    console.error("スクリプト実行エラー:", e);
                }
            }
        },

        show: function(id) {
            if (this._active || !this._list[id]) return;

            var cfg = this._list[id];
            this._active = true;
            this._readyForInput = false;
            this._currentConfig = cfg;
            this._enteredBox = "";
            this._triggeredFlags = cfg.hitSelections.map(function() { return false; });
            this._triggerTimestamps = cfg.hitSelections.map(() => 0);


            if (judgeVariable > 0) {
                $gameVariables.setValue(judgeVariable, 0);
            }

            if (SpineChoiceManager._lastInterpreter) {
                SpineChoiceManager._lastInterpreter.setWaitMode("spineChoice");
                this._interpreter = SpineChoiceManager._lastInterpreter;
                SpineChoiceManager._lastInterpreter = null;
            }

            for (var i = 0; i < cfg.applySwitches.length; i++) {
                $gameSwitches.setValue(cfg.applySwitches[i], true);
            }

            if (cfg.applyVariable > 0) {
                $gameVariables.setValue(cfg.applyVariable, cfg.applyValue);
            }

            var picId = cfg.pictureId;

            if (cfg.isNewSpine) {
                //$gameScreen.erasePicture(picId);
                $gameScreen.showPicture(picId, "", 0, cfg.x, cfg.y, cfg.scale, cfg.scale, 255, 0);
            }

            setTimeout(() => {
                let spine = $gameScreen.spine(picId);
                if (!spine) {
                    console.warn(`Failed to get spine for picture ${picId}`);
                    return;
                }

                spine.setSkeleton(cfg.skeleton);

                if (cfg.skins.length > 0) {
                    spine.setSkin.apply(spine, cfg.skins);
                }

                if (cfg.animationTracks && cfg.animationTracks.length > 0) {
                    for (const a of cfg.animationTracks) {
                        spine.setAnimation(Number(a.track), a.name);
                    }
                } else if (cfg.animation.length >= 2) {
                    spine.setAnimation(cfg.animationTrack || 0, cfg.animation, 'sequential', 'continue', true);
                } else if (cfg.animation.length === 1) {
                    spine.setAnimation(cfg.animationTrack || 0, cfg.animation[0]);
                }

                if (cfg.showOnceScript) {
                    SpineChoiceManager.runScriptSafely(cfg.showOnceScript);
                }

                // 30ms後に入力判定を有効化
                setTimeout(() => {
                    SpineChoiceManager._readyForInput = true;
                }, 30);
            }, 30);

        },

        select: function(index) {
        
            if (judgeVariable > 0) {
                $gameVariables.setValue(judgeVariable, index);
            }

            this._enteredBox = "";
            this._active = false;
            this._readyForInput = false;

            if (this._interpreter) {
                this._interpreter._branch[this._interpreter._indent] = index - 1;
                this._interpreter.setWaitMode(null);
            }

            var cfg = this._currentConfig;
            var deleteSpine = cfg ? cfg.deleteOnSelect !== false : true;

            if (!deleteSpine && cfg && cfg.pictureId > 0 && typeof $gameScreen.spine === "function") {
                var spine = $gameScreen.spine(cfg.pictureId);
                if (spine) {
                    var tracks = cfg.postSelectionTracks;
                    if (tracks && tracks.length > 0) {
                        for (var i = 0; i < tracks.length; i++) {
                            var entry = tracks[i];
                            var track = Number(entry.track);
                            var name = entry.name;
                            if (track >= 0 && name) {
                                spine.setAnimation(track, name);
                            }
                        }
                    }
                }
            }

            if (deleteSpine && cfg && cfg.pictureId > 0) {
                $gameScreen.erasePicture(cfg.pictureId);
            }

        },


        _checkChoiceTriggerInput: function(entry) {
            switch (entry.choiceTriggerType) {
                case "leftClick": return TouchInput.isPressed();
                case "TriggerleftClick": return TouchInput.isTriggered();
                case "leftClickReleased": return TouchInput.isReleased();
                case "rightClick": return TouchInput.isCancelled();
                case "keyboard": return entry.choiceTriggerKey ? Input.isTriggered(entry.choiceTriggerKey) : false;
            }
            return false;
        },

        update: function() {
            if (!this._active || !this._readyForInput || !this._currentConfig) return;

            var cfg = this._currentConfig;
            var spine = null;

            // フォーカス復帰時にSpineの状態を再検証
            const now = Date.now();
            if (now - this._lastSpineCheck > 100) {
                this._lastSpineCheck = now;
                const picture = $gameScreen.picture(cfg.pictureId);
                if (!picture) {
                    console.warn(`Picture ${cfg.pictureId} lost, stopping choice manager`);
                    this._active = false;
                    this._readyForInput = false;
                    return;
                }
            }

            if (typeof $gameScreen.spine === "function" && cfg.pictureId > 0) {
                spine = $gameScreen.spine(cfg.pictureId);
            }

            if (!spine || !spine._skeleton || typeof spine.hitTest !== "function") return;

            var mouseX = TouchInput.x;
            var mouseY = TouchInput.y;
            var hitList = spine.hitTest(mouseX, mouseY, true) || [];
            var frontHitBox = "noname";

            if (hitList.length > 0) {
                var hit = hitList[0];
                if (hit.slotName) {
                    frontHitBox = hit.slotName;
                } else if (hit.attachment && hit.attachment.name) {
                    frontHitBox = hit.attachment.name;
                }
            }

            if (this._lastLoggedBox !== frontHitBox) {
                this._lastLoggedBox = frontHitBox;
            }

            for (var i = 0; i < cfg.hitSelections.length; i++) {
                var entry = cfg.hitSelections[i];
                var type = entry.choiceTriggerType;
                var key = entry.choiceTriggerKey || "";
                var isKeyboard = type === "keyboard";
                var isMouse = type === "TriggerleftClick";
                var isOkKey = isKeyboard && key === "ok";

                var isHitByMouse = entry.name === frontHitBox;
                var isRepeatableKey = isKeyboard && ["up", "down", "left", "right"].indexOf(key) >= 0;
                var allowCheck =
                    (isKeyboard && (isOkKey || isRepeatableKey)) ||
                    (isMouse && isHitByMouse);

                if (!allowCheck) continue;

                if (isHitByMouse && this._enteredBox !== frontHitBox) {
                    this._enteredBox = frontHitBox;
                    this.runScriptSafely(entry.onEnterScript);
                }

                var alreadyTriggered = this._triggeredFlags[i];
                var allowTrigger = isRepeatableKey || !alreadyTriggered;

                if (allowTrigger && this._checkChoiceTriggerInput(entry)) {
                    const now = Date.now();
                    const lastTime = this._triggerTimestamps[i] || 0;
                    const elapsed = now - lastTime;

                    if (cfg.allowRepeatInput && elapsed < cfg.repeatDelay) continue;

                    if (cfg.allowRepeatInput) {
                        this._triggerTimestamps[i] = now;
                    } else {
                        this._triggeredFlags[i] = true;
                    }

                    var val = Number(entry.triggerJudgeValue || 0);

                    if (val > 0 && judgeVariable > 0) {
                        $gameVariables.setValue(judgeVariable, val);
                    }

                    this.runScriptSafely(entry.onTriggerScript);

                    if (val > 0) {
                        this._readyForInput = false;
                        this.select(val);
                        return;
                    }
                }
            }
        },
    };

    SpineChoiceManager.initialize();
    window.SpineChoiceManager = SpineChoiceManager;

    // ウィンドウフォーカスの監視
    window.addEventListener('blur', function() {
        SpineChoiceManager._windowFocused = false;
    });

    window.addEventListener('focus', function() {
        SpineChoiceManager._windowFocused = true;
        // フォーカス復帰時、次のSpineチェックを即座に実行
        SpineChoiceManager._lastSpineCheck = 0;
    });

    var _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        _Scene_Map_update.call(this);
        if (SpineChoiceManager && SpineChoiceManager.update) {
            SpineChoiceManager.update();
        }
    };

    var _Game_Interpreter_updateWaitMode = Game_Interpreter.prototype.updateWaitMode;
    Game_Interpreter.prototype.updateWaitMode = function() {
        if (this._waitMode === "spineChoice") {
            if (judgeVariable > 0 && $gameVariables.value(judgeVariable) !== 0) {
                for (var key in SpineChoiceManager._list) {
                    var cfg = SpineChoiceManager._list[key];
                    if (cfg.pictureId) {
                        $gameScreen.erasePicture(cfg.pictureId);
                    }
                }
                this._waitMode = "";
                return false;
            }
            return true;
        }
        return _Game_Interpreter_updateWaitMode.call(this);
    };

    SpineChoiceManager.cancel = function(value = 99) {
        if (!this._active) return;

        if (judgeVariable > 0) {
            $gameVariables.setValue(judgeVariable, value);
        }

        if (this._currentConfig && this._currentConfig.pictureId > 0) {
            $gameScreen.erasePicture(this._currentConfig.pictureId);
        }

        this._enteredBox = "";
        this._active = false;

        if (this._interpreter) {
            this._interpreter.setWaitMode(null);
        }
    };

})();

(function() {
    const pluginName = "NrSpineChoice";
    const _Game_Interpreter_command355 = Game_Interpreter.prototype.command355;

    Game_Interpreter.prototype.command355 = function() {
        const index = this._index;
        const result = _Game_Interpreter_command355.apply(this, arguments);

        try {
            let script = this._list[index].parameters[0] + "\n";
            let i = index + 1;
            while (this._list[i] && this._list[i].code === 655) {
                script += this._list[i].parameters[0] + "\n";
                i++;
            }

            if (/SpineChoiceManager\.show\s*\(/.test(script)) {
                if (!this._waitMode) this.setWaitMode("spineChoice");
            }
        } catch (e) {
            console.warn(`[${pluginName}] スクリプト検出中に例外:`, e);
        }

        return result;
    };
})();
