//=============================================================================
// NrTextCollar.js 
//=============================================================================

/*:
 * @plugindesc テキストカラープラグイン v1.0.0
 * @target MZ
 * 
 * @command Enable
 * @text NrText 開始
 * @desc テキスト自動装飾を開始します。指定した制御文字列がすべての文章に先頭追加されます。
 *
 * @arg prefix
 * @type string
 * @desc 先頭に自動追加する制御文字列（例：\NRCG[#FF0000,#0000FF|horizontal]）
 * @default 
 *
 * @arg applyToChoices
 * @type boolean
 * @text 選択肢にも適用
 * @desc trueで選択肢にもグラデーションなどを反映します
 * @default false
 *
 * @arg applyToName
 * @type boolean
 * @text 名前にも適用
 * @desc trueで名前（Window_NameBox）にもグラデーションなどを反映します
 * @default false
 *
 * @command Disable
 * @text NrText 停止
 * @desc テキスト自動装飾を停止します。
 * 
 * @help
 * 機能概要:
 * このプラグインは、テキストに以下の機能を追加します
 * - カラーコードによる色変更
 * - グラデーション文字
 * - 文字影
 * - カスタマイズ可能なアウトライン
 * 
 * - カラーコード設定
 *  \NRC[#RRGGBB]
 *   テキスト色をカラーコードで設定します。
 *   例：\NRC[#FF0000] → 赤色テキスト
 * 
 * - グラデーション設定
 *  \NRCG[#色1,#色2,...|方向|割合1,割合2,...]
 *   テキストにグラデーションを適用します。
 *   方向：horizontal（横）, vertical（縦）, diagonal1（斜め）, diagonal2（斜め）
 *   割合：各色の分布割合（0.0〜1.0の範囲で指定可能）。省略時は均等に分布。
 *   例：
 *     \NRCG[#FF0000,#0000FF|horizontal] → 赤から青の横グラデーション（均等分布）
 *     \NRCG[#FF0000,#00FF00,#0000FF|vertical|0.3,0.5,1.0] → 赤、緑、青の縦グラデーション（割合指定）
 * 
 * - 影設定
 *  \NRSH[色,サイズ,透明度,角度,距離,ぼかし]
 *   テキストに影を追加します。
 *   - 色：影の色（#RRGGBB形式）
 *   - サイズ：影の濃さ（1〜、数値が大きいほど濃い）
 *   - 透明度：0〜255（0=透明、255=不透明）
 *   - 角度：0〜360度（0=右、90=下、180=左、270=上）
 *   - 距離：影の距離（ピクセル）
 *   - ぼかし：ぼかしの強さ（0=ぼかしなし）
 *   例：\NRSH[#000000,3,255,45,5,0] → 黒い影（右下方向）
 * 
 * - アウトライン設定
 *  \NrOw[透明度]
 *   アウトラインの透明度を設定（0〜255）
 *   例：\NrOw[128] → 半透明アウトライン
 * 
 *  \NrOws[サイズ]
 *   アウトラインの太さを設定
 *   例：\NrOws[5] → 太いアウトライン
 * 
 *  \NrOwc[#RRGGBB]
 *   アウトラインの色を設定
 *   例：\NrOwc[#FF0000] → 赤いアウトライン
 * 
 * - 設定リセット
 *  \NrR
 *   すべてのカスタム設定をデフォルトに戻します。
 *   ※文章の途中で変えたいときに使用。
 * 
 * プラグインコマンド:
 * 
 *  NrText 開始
 *    テキスト自動装飾を開始します。指定した制御文字列が以降のすべての文章に先頭追加されます。
 *    
 *    パラメータ：
 *    - prefix: 先頭に自動追加する制御文字列（例：\NRCG[#FF0000,#0000FF|horizontal]）
 *    - applyToChoices: 選択肢にも適用するか（true/false）
 *    - applyToName: 名前ウィンドウにも適用するか（true/false）
 *    
 *    例：すべての文章を赤から青のグラデーションにする
 *    プラグインコマンド > NrText 開始
 *    prefix: \NRCG[#FF0000,#0000FF|horizontal]
 *    applyToChoices: false
 *    applyToName: false
 * 
 *  NrText 停止
 *    テキスト自動装飾を停止します。
 *    
 *    例：自動装飾を解除する
 *    プラグインコマンド > NrText 停止
 * 
 * 使用例:
 * 
 * 1. 赤いグラデーション文字：
 *    \NRCG[#FF0000,#990000|vertical]グラデーションテキスト
 * 
 * 2. 影付き青文字：
 *    \NRC[#0000FF]\NRSH[#000000,2,200,45,3,2]影付きテキスト
 * 
 * 3. カスタムアウトライン：
 *    \NrOws[4]\NrOwc[#FFFF00]黄色アウトラインテキスト
 * 
 * 4. 複合効果：
 *    \NRCG[#FF0000,#0000FF|horizontal]\NRSH[#000000,3,255,45,5,0]複合効果
 * 
 * パラメータ説明:
 *  影の角度の目安：
 *   0度   → 右
 *   45度  → 右下
 *   90度  → 下
 *   135度 → 左下
 *   180度 → 左
 *   225度 → 左上
 *   270度 → 上
 *   315度 → 右上
 * 
 * グラデーション方向:
 *   horizontal → 左から右
 *   vertical   → 上から下
 *   diagonal1  → 左上から右下
 *   diagonal2  → 左下から右上
 * 
 * バージョン:
 * v1.0.0 初回
 *
 * 利用規約:
 * - プラグイン作者に無断で使用、改変、再配布は不可です。
 */

(() => {
    let _currentGradient = null;
    let _gradientActive = false;
    let _shadowSettings = null;
    let _shadowActive = false;
    let _textColor = null;
    let _colorActive = false;
    let _outlineAlpha = 255;
    let _outlineWidth = null;
    let _outlineColor = null;
    let _NrTextCollarAutoPrefix = null;
    let _NrTextCollarApplyToChoices = false;
    let _NrTextCollarApplyToName = false;

    const resetNrTextStates = function() {
        _currentGradient = null;
        _gradientActive = false;
        _shadowSettings = null;
        _shadowActive = false;
        _textColor = null;
        _colorActive = false;
        _outlineAlpha = 255;
        _outlineWidth = null;
        _outlineColor = null;
    };

    const _Window_Base_processEscapeCharacter = Window_Base.prototype.processEscapeCharacter;
    Window_Base.prototype.processEscapeCharacter = function(code, textState) {
        switch (code) {
            case 'NRC':
                _textColor = this.obtainEscapeParamEx(textState);
                _colorActive = true;
                _currentGradient = null;
                _gradientActive = false;
                break;
            case 'NRCG':
                this.processNrGradientChange(this.obtainEscapeParamEx(textState));
                _gradientActive = true;
                _textColor = null;
                _colorActive = false;
                break;
            case 'NRSH':
                this.processNrShadowChange(this.obtainEscapeParamEx(textState));
                _shadowActive = true;
                break;
            case 'NROW':
                this.processNrOutlineAlpha(this.obtainEscapeParamEx(textState));
                break;
            case 'NROWS':
                this.processNrOutlineSize(this.obtainEscapeParamEx(textState));
                break;
            case 'NROWC':
                this.processNrOutlineColor(this.obtainEscapeParamEx(textState));
                break;
            default:
                _Window_Base_processEscapeCharacter.apply(this, arguments);
                if (code === 'NRR') {
                    _textColor = null;
                    _colorActive = false;
                    _currentGradient = null;
                    _gradientActive = false;
                    _shadowSettings = null;
                    _shadowActive = false;
                    _outlineAlpha = 255;
                    _outlineWidth = null;
                    _outlineColor = null;
                }
                break;
        }
    };

    Window_Base.prototype.obtainEscapeParamEx = function(textState) {
        const regExp = /^\[([^\]]+)\]/;
        const arr = regExp.exec(textState.text.slice(textState.index));
        if (arr) {
            textState.index += arr[0].length;
            return arr[1];
        } else {
            return '';
        }
    };

    Window_Base.prototype.processNrGradientChange = function(param) {
        const parts = param.split('|');
        let colorsParam = parts[0];
        let direction = 'vertical';
        let ratios = null;

        if (parts.length > 1) {
            const dir = parts[1].trim().toLowerCase();
            if (['horizontal', 'vertical', 'diagonal1', 'diagonal2'].includes(dir)) {
                direction = dir;
            }
        }

        if (parts.length > 2) {
            ratios = parts[2].split(',').map(r => parseFloat(r.trim())).filter(r => !isNaN(r) && r >= 0 && r <= 1);
        }

        const colors = colorsParam.split(',')
            .map(s => s.trim().replace(/['"]/g, ''))
            .filter(s => /^#[0-9a-fA-F]{6}$/.test(s));

        if (colors.length >= 2) {
            _currentGradient = { colors: colors, direction: direction, ratios: ratios };
        } else {
            _currentGradient = null;
            _gradientActive = false;
        }
    };

    Window_Base.prototype.processNrShadowChange = function(param) {
        const settings = param.split(',')
            .map(s => s.trim().replace(/['"]/g, ''));
        
        if (settings.length >= 1 && settings[0]) {
            _shadowSettings = {
                color: settings[0] || '#000000',
                size: Math.max(0, parseInt(settings[1]) || 3),
                alpha: Math.max(0, Math.min(255, parseInt(settings[2]) || 255)) / 255,
                angle: Math.max(0, Math.min(360, parseInt(settings[3]) || 45)),
                distance: Math.max(0, parseInt(settings[4]) || 3),
                blur: Math.max(0, parseInt(settings[5]) || 5)
            };
            
            const rad = _shadowSettings.angle * Math.PI / 180;
            _shadowSettings.offsetX = Math.cos(rad) * _shadowSettings.distance;
            _shadowSettings.offsetY = Math.sin(rad) * _shadowSettings.distance;
        } else {
            _shadowSettings = null;
            _shadowActive = false;
        }
    };

    Window_Base.prototype.processNrOutlineAlpha = function(param) {
        const alpha = parseInt(param);
        if (!isNaN(alpha)) {
            _outlineAlpha = Math.max(0, Math.min(255, alpha));
        }
    };

    Window_Base.prototype.processNrOutlineSize = function(param) {
        const size = parseInt(param);
        if (!isNaN(size)) {
            _outlineWidth = Math.max(0, size);
        } else {
            _outlineWidth = null;
        }
    };

    Window_Base.prototype.processNrOutlineColor = function(param) {
        if (param && /^#[0-9a-fA-F]{6}$/.test(param)) {
            _outlineColor = param;
        } else {
            _outlineColor = null;
        }
    };

    const _Window_Message_startMessage = Window_Message.prototype.startMessage;
    Window_Message.prototype.startMessage = function() {
        _Window_Message_startMessage.call(this);
        resetNrTextStates();
    };

    const _Sprite_Picture_makeDynamicBitmap = Sprite_Picture.prototype.makeDynamicBitmap;
    Sprite_Picture.prototype.makeDynamicBitmap = function() {
        resetNrTextStates();
        return _Sprite_Picture_makeDynamicBitmap.apply(this, arguments);
    };

    const _Scene_Base_start = Scene_Base.prototype.start;
    Scene_Base.prototype.start = function() {
        _Scene_Base_start.apply(this, arguments);
        resetNrTextStates();
    };

    const _Window_Base_changeTextColor = Window_Base.prototype.changeTextColor;
    Window_Base.prototype.changeTextColor = function(color) {
        if (_colorActive && _textColor && /^#[0-9a-fA-F]{6}$/.test(_textColor)) {
            _Window_Base_changeTextColor.call(this, _textColor);
        } else {
            _Window_Base_changeTextColor.call(this, color);
        }
    };

    Bitmap.prototype.drawOutlineText = function(text, x, y, maxWidth, lineHeight, align) {
        const context = this._context;
        if (!context) return;
        
        context.save();
        context.font = this._makeFontNameText();
        context.textAlign = align || "left";
        context.textBaseline = "alphabetic";
        
        const textWidth = context.measureText(text).width;
        const textMetrics = context.measureText(text);
        const textHeight = textMetrics.actualBoundingBoxAscent + textMetrics.actualBoundingBoxDescent;
        let drawX = x;
        
        if (align === "center") {
            drawX = x + (maxWidth - textWidth) / 2;
        } else if (align === "right") {
            drawX = x + maxWidth - textWidth;
        }
        
        const drawY = y + lineHeight / 2 + this.fontSize / 2 - 4;
        
        const outlineWidth = _outlineWidth !== null ? _outlineWidth : this.outlineWidth;
        if (outlineWidth > 0) {
            context.save();
            context.strokeStyle = _outlineColor || this.outlineColor;
            context.lineWidth = outlineWidth;
            context.lineJoin = "round";
            context.globalAlpha = _outlineAlpha / 255;
            context.strokeText(text, drawX, drawY);
            context.restore();
        }
        
        return { drawX, drawY, textWidth, textHeight };
    };

    const _Bitmap_drawCharacter = Bitmap.prototype.drawCharacter;
    Bitmap.prototype.drawCharacter = function(character, x, y) {
        if (_shadowActive && _shadowSettings) {
            this.drawCharacterWithShadow(character, x, y);
        } else if (_gradientActive && _currentGradient && _currentGradient.colors.length >= 2) {
            this.drawCharacterWithGradient(character, x, y);
        } else if (_colorActive && _textColor) {
            this.drawCharacterWithColor(character, x, y);
        } else {
            _Bitmap_drawCharacter.call(this, character, x, y);
        }
    };

    Bitmap.prototype.drawCharacterWithColor = function(character, x, y) {
        const context = this._context;
        if (!context) return;
        
        context.save();
        context.font = this._makeFontNameText();
        context.textAlign = 'left';
        context.textBaseline = 'alphabetic';
        
        const drawY = y + this.fontSize;
        
        const outlineWidth = _outlineWidth !== null ? _outlineWidth : this.outlineWidth;
        if (outlineWidth > 0) {
            context.save();
            context.strokeStyle = _outlineColor || this.outlineColor;
            context.lineWidth = outlineWidth;
            context.lineJoin = 'round';
            context.globalAlpha = _outlineAlpha / 255;
            context.strokeText(character, x, drawY);
            context.restore();
        }
        
        context.fillStyle = _textColor;
        context.fillText(character, x, drawY);
        
        context.restore();
        this._dirty = true;
    };

    Bitmap.prototype.drawCharacterWithShadow = function(character, x, y) {
        const context = this._context;
        if (!context) return;
        
        context.save();
        context.font = this._makeFontNameText();
        context.textAlign = 'left';
        context.textBaseline = 'alphabetic';
        
        const charWidth = this.textWidth(character);
        const charHeight = this.fontSize;
        const drawY = y + charHeight;
        
        const offsetX = _shadowSettings.offsetX;
        const offsetY = _shadowSettings.offsetY;
        
        context.save();
        context.fillStyle = _shadowSettings.color;
        context.globalAlpha = _shadowSettings.alpha;
        
        if (_shadowSettings.blur > 0) {
            context.shadowColor = _shadowSettings.color;
            context.shadowBlur = _shadowSettings.blur;
            context.shadowOffsetX = offsetX;
            context.shadowOffsetY = offsetY;
            context.fillText(character, x, drawY);
        } else {
            context.fillText(character, x + offsetX, drawY + offsetY);
        }
        context.restore();
        
        const outlineWidth = _outlineWidth !== null ? _outlineWidth : this.outlineWidth;
        if (outlineWidth > 0) {
            context.save();
            context.strokeStyle = _outlineColor || this.outlineColor;
            context.lineWidth = outlineWidth;
            context.lineJoin = 'round';
            context.globalAlpha = _outlineAlpha / 255;
            context.strokeText(character, x, drawY);
            context.restore();
        }
        
        context.save();
        if (_gradientActive && _currentGradient && _currentGradient.colors.length >= 2) {
            const gradient = this.createTextGradient(context, x, y, charWidth, charHeight);
            context.fillStyle = gradient;
        } else if (_colorActive && _textColor) {
            context.fillStyle = _textColor;
        } else {
            context.fillStyle = this.textColor;
        }
        context.fillText(character, x, drawY);
        context.restore();
        
        context.restore();
        this._dirty = true;
    };

    Bitmap.prototype.drawCharacterWithGradient = function(character, x, y) {
        const context = this._context;
        if (!context) return;
        
        context.save();
        context.font = this._makeFontNameText();
        context.textAlign = 'left';
        context.textBaseline = 'alphabetic';
        
        const charWidth = this.textWidth(character);
        const charHeight = this.fontSize;
        const drawY = y + charHeight;
        
        const gradient = this.createTextGradient(context, x, y, charWidth, charHeight);
        
        const outlineWidth = _outlineWidth !== null ? _outlineWidth : this.outlineWidth;
        if (outlineWidth > 0) {
            context.save();
            context.strokeStyle = _outlineColor || this.outlineColor;
            context.lineWidth = outlineWidth;
            context.lineJoin = 'round';
            context.globalAlpha = _outlineAlpha / 255;
            context.strokeText(character, x, drawY);
            context.restore();
        }
        
        context.fillStyle = gradient;
        context.fillText(character, x, drawY);
        
        context.restore();
        this._dirty = true;
    };

    Bitmap.prototype.createTextGradient = function(context, x, y, width, height) {
        if (SceneManager._scene && SceneManager._scene.constructor.name === "Scene_MessageLog") {
            return this.textColor;
        }

        if (!isFinite(x)) x = 0;
        if (!isFinite(y)) y = 0;
        if (!isFinite(width) || width <= 0) width = this.width || 1;
        if (!isFinite(height) || height <= 0) height = this.fontSize || 16;

        if (!_currentGradient || _currentGradient.colors.length < 2) {
            return this.textColor;
        }

        const colors = _currentGradient.colors;
        const ratios = _currentGradient.ratios;
        let gradient;

        switch (_currentGradient.direction) {
            case 'horizontal':
                gradient = context.createLinearGradient(x, y, x + width, y);
                break;
            case 'diagonal1':
                gradient = context.createLinearGradient(x, y, x + width, y + height);
                break;
            case 'diagonal2':
                gradient = context.createLinearGradient(x, y + height, x + width, y);
                break;
            case 'vertical':
            default:
                gradient = context.createLinearGradient(x, y, x, y + height);
                break;
        }

        const step = 1 / (colors.length - 1);
        const totalHeight = height;
        colors.forEach((color, i) => {
            const ratio = ratios && ratios[i] !== undefined ? ratios[i] : i * step;
            const scaled = ratio * totalHeight / totalHeight;
            gradient.addColorStop(scaled, color);
        });

        return gradient;
    };

    const _Bitmap_drawText = Bitmap.prototype.drawText;
    Bitmap.prototype.drawText = function(text, x, y, maxWidth, lineHeight, align) {
        if (_shadowActive && _shadowSettings && text && text.length > 0) {
            this.drawTextWithShadow(text, x, y, maxWidth, lineHeight, align);
        } else if (_gradientActive && _currentGradient && _currentGradient.colors.length >= 2 && text && text.length > 0) {
            this.drawTextWithGradient(text, x, y, maxWidth, lineHeight, align);
        } else if (_colorActive && _textColor && text && text.length > 0) {
            this.drawTextWithColor(text, x, y, maxWidth, lineHeight, align);
        } else {
            _Bitmap_drawText.call(this, text, x, y, maxWidth, lineHeight, align);
        }
    };

    Bitmap.prototype.drawTextWithColor = function(text, x, y, maxWidth, lineHeight, align) {
        const pos = this.drawOutlineText(text, x, y, maxWidth, lineHeight, align);
        const context = this._context;
        if (!context) return;
        
        context.fillStyle = _textColor;
        context.fillText(text, pos.drawX, pos.drawY);
        context.restore();
        this._dirty = true;
    };

    
    Bitmap.prototype.drawTextWithGradient = function(text, x, y, maxWidth, lineHeight, align) {
        const pos = this.drawOutlineText(text, x, y, maxWidth, lineHeight, align);
        const context = this._context;
        if (!context) return;
        
        const gradient = this.createTextGradient(context, pos.drawX, y, pos.textWidth, pos.textHeight);
        context.fillStyle = gradient;

        context.fillText(text, pos.drawX, pos.drawY);
        context.restore();
        this._dirty = true;
    };

    Bitmap.prototype.drawTextWithShadow = function(text, x, y, maxWidth, lineHeight, align) {
        const context = this._context;
        if (!context) return;
        
        context.save();
        context.font = this._makeFontNameText();
        context.textAlign = align || "left";
        context.textBaseline = "alphabetic";
        
        const textWidth = context.measureText(text).width;
        const textHeight = this.fontSize;
        let drawX = x;
        
        if (align === "center") {
            drawX = x + (maxWidth - textWidth) / 2;
        } else if (align === "right") {
            drawX = x + maxWidth - textWidth;
        }
        
        const drawY = y + lineHeight - (lineHeight - textHeight) / 2;
        
        const offsetX = _shadowSettings.offsetX;
        const offsetY = _shadowSettings.offsetY;
        
        context.save();
        context.fillStyle = _shadowSettings.color;
        context.globalAlpha = _shadowSettings.alpha;
        
        if (_shadowSettings.blur > 0) {
            context.shadowColor = _shadowSettings.color;
            context.shadowBlur = _shadowSettings.blur;
            context.shadowOffsetX = offsetX;
            context.shadowOffsetY = offsetY;
            context.fillText(text, drawX, drawY);
        } else {
            context.fillText(text, drawX + offsetX, drawY + offsetY);
        }
        context.restore();
        
        const outlineWidth = _outlineWidth !== null ? _outlineWidth : this.outlineWidth;
        if (outlineWidth > 0) {
            context.save();
            context.strokeStyle = _outlineColor || this.outlineColor;
            context.lineWidth = outlineWidth;
            context.lineJoin = "round";
            context.globalAlpha = _outlineAlpha / 255;
            context.strokeText(text, drawX, drawY);
            context.restore();
        }
        
        context.save();
        if (_gradientActive && _currentGradient && _currentGradient.colors.length >= 2) {
            const gradient = this.createTextGradient(context, drawX, y, textWidth, textHeight);
            context.fillStyle = gradient;
        } else if (_colorActive && _textColor) {
            context.fillStyle = _textColor;
        } else {
            context.fillStyle = this.textColor;
        }
        context.fillText(text, drawX, drawY);
        context.restore();
        
        context.restore();
        this._dirty = true;
    };

    const _SceneManager_changeScene = SceneManager.changeScene;
    SceneManager.changeScene = function() {
        const next = this._nextScene;
        if (next && next instanceof Scene_MenuBase) {
            resetNrTextStates();
        }
        _SceneManager_changeScene.call(this);
    };

    const _Window_Base_initialize = Window_Base.prototype.initialize;
    Window_Base.prototype.initialize = function(rect) {
        if (SceneManager._scene instanceof Scene_MenuBase) {
            resetNrTextStates();
        }
        _Window_Base_initialize.call(this, rect);
    };

    const _Window_NameBox_start = Window_NameBox.prototype.start;
    Window_NameBox.prototype.start = function() {
        _Window_NameBox_start.call(this);
        _currentGradient = null;
        _gradientActive = false;
        _shadowSettings = null;
        _shadowActive = false;
        _textColor = null;
        _colorActive = false;
        _outlineAlpha = 255;
        _outlineWidth = null;
        _outlineColor = null;
    };

    window.resetNrTextStates = resetNrTextStates;

    window.NrTextCollar_Enable = function(prefix, applyToChoices = false, applyToName = false) {
        if (typeof prefix === "string") {
            _NrTextCollarAutoPrefix = prefix;
            _NrTextCollarApplyToChoices = applyToChoices;
            _NrTextCollarApplyToName = applyToName;
        }
    };

    window.NrTextCollar_Disable = function() {
        _NrTextCollarAutoPrefix = null;
        _NrTextCollarApplyToChoices = false;
        _NrTextCollarApplyToName = false;
    };

    const _Window_Base_convertEscapeCharacters_NRTC = Window_Base.prototype.convertEscapeCharacters;
    Window_Base.prototype.convertEscapeCharacters = function(text) {
        const isChoiceWindow = this instanceof Window_ChoiceList;
        const isNameBox = this instanceof Window_NameBox;
        if (_NrTextCollarAutoPrefix && typeof text === "string") {
            if (
                (!isChoiceWindow || _NrTextCollarApplyToChoices) &&
                (!isNameBox || _NrTextCollarApplyToName)
            ) {
                text = _NrTextCollarAutoPrefix + text;
            }
        }
        return _Window_Base_convertEscapeCharacters_NRTC.call(this, text);
    };

    PluginManager.registerCommand("NrTextCollar", "Enable", args => {
        const prefix = args.prefix || "";
        const applyToChoices = args.applyToChoices === "true";
        const applyToName = args.applyToName === "true";
        NrTextCollar_Enable(prefix, applyToChoices, applyToName);
    });

    PluginManager.registerCommand("NrTextCollar", "Disable", args => {
        NrTextCollar_Disable();
    });

    const _Bitmap_drawTextWithGradient = Bitmap.prototype.drawTextWithGradient;
    Bitmap.prototype.drawTextWithGradient = function(text, x, y, maxWidth, lineHeight, align) {
        const currentWindow = SceneManager._scene?._windowLayer?.children?.find(w => w.contents === this);
        const isNameBox = currentWindow instanceof Window_NameBox;

        if (isNameBox) {
            const context = this._context;
            if (!context) return;

            context.save();
            context.font = this._makeFontNameText();
            context.textAlign = align || "left";
            context.textBaseline = "alphabetic";

            const textWidth = context.measureText(text).width;
            let drawX = x;
            if (align === "center") {
                drawX = x + (maxWidth - textWidth) / 2;
            } else if (align === "right") {
                drawX = x + maxWidth - textWidth;
            }

            const drawY = y + this.fontSize;
            const gradient = this.createTextGradient(context, drawX, y, textWidth, this.fontSize);

            const outlineWidth = _outlineWidth !== null ? _outlineWidth : this.outlineWidth;
            if (outlineWidth > 0) {
                context.save();
                context.strokeStyle = _outlineColor || this.outlineColor;
                context.lineWidth = outlineWidth;
                context.lineJoin = "round";
                context.globalAlpha = _outlineAlpha / 255;
                context.strokeText(text, drawX, drawY);
                context.restore();
            }

            context.fillStyle = gradient;
            context.fillText(text, drawX, drawY);
            context.restore();
            this._dirty = true;
            return;
        }

        _Bitmap_drawTextWithGradient.call(this, text, x, y, maxWidth, lineHeight, align);
    };

    const _Game_Picture_show = Game_Picture.prototype.show;
    Game_Picture.prototype.show = function() {
        if (typeof resetNrTextStates === "function") {
            resetNrTextStates();
        }

        _Game_Picture_show.apply(this, arguments);

        let textPictureText = "";
        if (this._name === "" && textPictureText) {
            this.mzkp_text = textPictureText;
            this.mzkp_textChanged = true;
            textPictureText = "";
        }
    };

})();