//=============================================================================
// NrTextFile.js 
//=============================================================================

/*:
 * @plugindesc 外部テキスト読み込みプラグイン Ver1.0.0
 * @target MZ
 *
 * @param Background
 * @type select
 * @option 通常
 * @value 0
 * @option 暗くする
 * @value 1
 * @option 透明
 * @value 2
 * @default 0
 *
 * @param Position
 * @type select
 * @option 下
 * @value 0
 * @option 中
 * @value 1
 * @option 上
 * @value 2
 * @default 2
 *
 * @param InputLockSwitchId
 * @type switch
 * @default 0
 *
 * @param SpeakerIndexVariable
 * @type variable
 * @default 0
 * 
 * @param NameSuffixList
 * @type struct<NameSuffix>[]
 * @desc 特定の名前に対してテキストを自動追加する設定リスト
 * @default []
 *
 * @param NameColorList
 * @type struct<NameColor>[]
 * @desc 特定の名前に対して名前の色を変更する設定リスト
 * @default []
 *
 * @param PrefixText
 * @type string
 * @desc 常に全テキストの冒頭に追加する文字列
 * @default 
 * 
 * @help
 * NrTextFile.command('ファイル名');
 * 
 * 
 * バージョン：
 * v1.0.0 初回
 *
 * 利用規約：
 * - プラグイン作者に無断で使用、改変、再配布は不可です。
*/

/*~struct~NameColor:
 * @param Name
 * @type string
 * @desc 判定する名前（例：\N[1]）
 *
 * @param NameColorFormat
 * @type string
 * @desc 制御文字そのものを直接指定（例：\\NC[3]）
 */

/*~struct~NameSuffix:
 * @param Name
 * @type string
 * @desc 判定する名前（例：主人公、\N[1] など）
 *
 * @param Suffix
 * @type string
 * @desc 追加する文字列（例：★）
 *
 * @param Script
 * @type note
 * @desc テキストが表示が終わった後に実行するスクリプト
 */

(() => {
    const pluginName = "NrTextFile";
    const parameters = PluginManager.parameters(pluginName);
    const defaultBackground = Number(parameters['Background'] || 0);
    const defaultPosition = Number(parameters['Position'] || 2);
    const inputLockSwitchId = Number(parameters['InputLockSwitchId'] || 0);
    const speakerIndexVarId = Number(parameters['SpeakerIndexVariable'] || 0);
    const nameSuffixList = JSON.parse(parameters['NameSuffixList'] || "[]").map(e => {
        try {
            const parsed = JSON.parse(e);
            parsed.Script = parsed.Script ? JSON.parse(parsed.Script) : null;
            return parsed;
        } catch (error) {
            console.error("Failed to parse NameSuffixList entry:", e, error);
            return null;
        }
    }).filter(e => e !== null);

    const nameColorList = JSON.parse(parameters['NameColorList'] || "[]").map(e => JSON.parse(e));

    class SequentialTextProcessor {
        constructor(lines) {
            this._lines = lines;
            this._index = 0;
            this._bg = defaultBackground;
            this._pos = defaultPosition;
            this._baseWindow = null;
            this._triggeredName = null;
            this._labels = {};
            this._initializeLabels();
            window.currentSequentialTextProcessor = this;
        }

        _initializeLabels() {
            this._lines.forEach((line, index) => {
                const match = line.trim().match(/^@label\s+(\S+)/);
                if (match) {
                    this._labels[match[1]] = index;
                }
            });
        }

        async start() {
            await this.nextLine();
        }

        async nextLine() {
            if (this._index >= this._lines.length) return;
            const line = this._lines[this._index++].trim();
            if (!line) return this.nextLine();
            if (inputLockSwitchId > 0 && $gameSwitches.value(inputLockSwitchId)) {
                await this._waitWhileLocked();
            }
            if (line.startsWith('@')) {
                await this.processCommand(line);
            } else {
                await this.showMessage(line);
            }
            await this.nextLine();
        }

        async showMessage(line) {
            let name = null, index = null, text = "";
            let m = line.match(/^([^:]+):(\d+):(.*)$/);
            if (m) {
                name = m[1]; index = Number(m[2]); text = m[3];
            } else if ((m = line.match(/^([^:]+)::(.*)$/))) {
                name = m[1]; text = m[2];
            } else if ((m = line.match(/^:(\d+):(.*)$/))) {
                index = Number(m[1]); text = m[2];
            } else {
                text = line;
            }

            if (parameters['PrefixText']) {
                text = parameters['PrefixText'] + text;
            }

            if (index !== null && speakerIndexVarId > 0) {
                $gameVariables.setValue(speakerIndexVarId, index);
            }

            $gameMessage.clear();
            if (name) {
                function expandEscape(text) {
                    if (!text) return "";
                    return text.replace(/\\V\[(\d+)\]/gi, (_, v) => $gameVariables.value(Number(v)))
                            .replace(/\\N\[(\d+)\]/gi, (_, n) => {
                                const actor = $gameActors.actor(Number(n));
                                return actor ? actor.name() : "";
                            })
                            .replace(/\\SET\[(\d+)\]/gi, (_, n) => "")
                            .replace(/\\\\/g, "\\");
                }

                this._triggeredName = name;

                const foundSuffix = nameSuffixList.find(cfg => {
                    return expandEscape(cfg.Name || "") === expandEscape(name);
                });
                if (foundSuffix) {
                    text += foundSuffix.Suffix;
                }

                const foundColor = nameColorList.find(cfg => {
                    const expandedCfg = expandEscape(cfg.Name || "");
                    const expandedName = expandEscape(name || "");
                    return expandedCfg === expandedName;
                });

                if (foundColor && foundColor.NameColorFormat) {
                    const tag = expandEscape(foundColor.NameColorFormat);
                    name = `${tag}\\FB${expandEscape(name)}`;
                }

                $gameMessage.setSpeakerName(name);

            }

            $gameMessage.add(text);
            $gameMessage.setBackground(this._bg);
            $gameMessage.setPositionType(this._pos);

            const interpreter = new Game_Interpreter();
            interpreter.setup([{ code: 0, indent: 0 }], 0);
            $gameMap._interpreter = interpreter;

            await new Promise(resolve => {
                const check = () => {
                    if (!$gameMessage.isBusy()) resolve();
                    else setTimeout(check, 30);
                };
                check();
            });
            this._triggeredName = name;
        }

        async processCommand(line) {
            const parts = line.trim().split(/\s+/);
            const cmd = parts[0].toLowerCase();

            try {
                switch (cmd) {
                    case '@label': {
                        break;
                    }
                    case '@jump': {
                        const labelName = parts[1];
                        if (this._labels[labelName] !== undefined) {
                            this._index = this._labels[labelName];
                        } else {
                            console.error(`[NrTextFile] @jump: ラベル '${labelName}' が見つかりません。`);
                        }
                        break;
                    }
                    case '@wait': {
                        const dur = Number(parts[1] || 60);
                        await this.sleep(dur);
                        break;
                    }
                    case '@bg': {
                        this._bg = Number(parts[1] || defaultBackground);
                        break;
                    }
                    case '@pos': {
                        this._pos = Number(parts[1] || defaultPosition);
                        break;
                    }
                    case '@fadeout':
                    case '@fadein': {
                        const dur = Number(parts[1] || 30);
                        const fn = cmd === '@fadeout' ? 'startFadeOut' : 'startFadeIn';
                        $gameScreen[fn](dur);
                        await this.sleep(dur);
                        break;
                    }
                    case '@bgm':
                    case '@bgs':
                    case '@me':
                    case '@se': {
                        const [_, name, vol, pitch, pan] = parts;
                        const sound = {
                            name: name || "",
                            volume: Number(vol || 90),
                            pitch: Number(pitch || 100),
                            pan: Number(pan || 0)
                        };
                        if (cmd === '@bgm') AudioManager.playBgm(sound);
                        if (cmd === '@bgs') AudioManager.playBgs(sound);
                        if (cmd === '@me') AudioManager.playMe(sound);
                        if (cmd === '@se') AudioManager.playSe(sound);
                        break;
                    }
                    case '@fadebgm': AudioManager.fadeOutBgm(Number(parts[1] || 30)); break;
                    case '@fadebgs': AudioManager.fadeOutBgs(Number(parts[1] || 30)); break;
                    case '@stopbgm': AudioManager.stopBgm(); break;
                    case '@stopbgs': AudioManager.stopBgs(); break;
                    case '@stopme': AudioManager.stopMe(); break;
                    case '@stopse': AudioManager.stopSe(); break;
                    case '@var': {
                        const varId = Number(parts[1]);
                        const op = Number(parts[2]);
                        const valType = parts[3];
                        const raw = parts.slice(4).join(" ");
                        let value = 0;
                        switch (valType) {
                            case '0': value = Number(raw); break;
                            case '1': value = $gameVariables.value(Number(raw)); break;
                            case '2': {
                                const [min, max] = raw.split('-').map(Number);
                                value = Math.floor(Math.random() * (max - min + 1)) + min;
                                break;
                            }
                            case '3': value = eval(raw); break;
                        }
                        const cur = $gameVariables.value(varId);
                        const res = op === 1 ? cur + value : op === 2 ? cur - value : value;
                        $gameVariables.setValue(varId, res);
                        break;
                    }
                    case '@switch':
                    case '@sw': {
                        const switchId = Number(parts[1]);
                        const val = parts[2]?.toLowerCase() === 'true' || parts[2] === '1';
                        $gameSwitches.setValue(switchId, val);
                        break;
                    }
                    case '@load': {
                        const fileName = parts[1];
                        if (!fileName) {
                            console.error(`[NrTextFile] @load コマンド: ファイル名が指定されていません。`);
                            break;
                        }

                        try {
                            await NrTextFile.command(fileName);
                        } catch (error) {
                            console.error(`[NrTextFile] @load コマンド: ファイル ${fileName} の読み込み中にエラーが発生しました:`, error);
                        }

                        break;
                    }
                    case '@script':
                    case '@sc': {
                        const lines = [];
                        while (this._index < this._lines.length) {
                            const next = this._lines[this._index++].trim();
                            if (next.toLowerCase() === '@end') break;
                            lines.push(next);
                        }
                        const code = lines.join("\n");
                        try {
                            const scriptFunction = new Function(code);
                            scriptFunction();
                        } catch (e) {
                            console.error("Script execution error:", e);
                        }
                        break;
                    }
                    case '@fadeoutc': {
                        const r = Number(parts[1] || 0);
                        const g = Number(parts[2] || 0);
                        const b = Number(parts[3] || 0);
                        const dur = Number(parts[4] || 30);
                        const scene = SceneManager._scene;
                        if (scene) {
                            if (scene._nrFadeSprite) {
                                try { scene.removeChild(scene._nrFadeSprite); } catch (e) {}
                                scene._nrFadeSprite = null;
                            }
                            const fadeSprite = new PIXI.Graphics();
                            fadeSprite.beginFill(PIXI.utils.rgb2hex([Math.max(0, Math.min(1, r / 255)), Math.max(0, Math.min(1, g / 255)), Math.max(0, Math.min(1, b / 255))]));
                            fadeSprite.drawRect(0, 0, Graphics.width, Graphics.height);
                            fadeSprite.endFill();
                            fadeSprite.alpha = 0;
                            scene.addChild(fadeSprite);
                            scene._nrFadeSprite = fadeSprite;
                            let count = 0;
                            await new Promise(resolve => {
                                const step = () => {
                                    count++;
                                    fadeSprite.alpha = Math.min(1, count / Math.max(1, dur));
                                    if (count >= dur) resolve();
                                    else requestAnimationFrame(step);
                                };
                                step();
                            });
                        } else {
                            await this.sleep(dur);
                        }
                        break;
                    }
                    case '@fadeinc': {
                        const dur = Number(parts[4] || 30);
                        const scene = SceneManager._scene;
                        if (scene && scene._nrFadeSprite) {
                            const fadeSprite = scene._nrFadeSprite;
                            let count = 0;
                            await new Promise(resolve => {
                                const step = () => {
                                    count++;
                                    fadeSprite.alpha = Math.max(0, 1 - (count / Math.max(1, dur)));
                                    if (count >= dur) {
                                        try { scene.removeChild(fadeSprite); } catch (e) {}
                                        scene._nrFadeSprite = null;
                                        resolve();
                                    } else requestAnimationFrame(step);
                                };
                                step();
                            });
                        } else {
                            await this.sleep(dur);
                        }
                        break;
                    }
                    case '@shake': {
                        const dir = Number(parts[1] || 0);
                        const power = Number(parts[2] || 5);
                        const speed = Number(parts[3] || 5);
                        const duration = Number(parts[4] || 60);

                        if (dir === 0) {
                            $gameScreen.startShake(power, speed, duration);
                        } else {
                            const spriteset = SceneManager._scene._spriteset;
                            if (spriteset) {
                                let count = 0;
                                let frame = 0;
                                const originY = spriteset.y;
                                const interval = setInterval(() => {
                                    const offset = Math.round(Math.sin(frame / speed * Math.PI) * power * 2);
                                    spriteset.y = originY + offset;
                                    frame++;
                                    if (++count >= duration) {
                                        clearInterval(interval);
                                        spriteset.y = originY;
                                    }
                                }, 1000 / 60);
                            }
                        }

                        await this.sleep(duration);
                        break;
                    }
                    case '@flash': {
                        const r = Number(parts[1] || 255);
                        const g = Number(parts[2] || 255);
                        const b = Number(parts[3] || 255);
                        const a = Number(parts[4] || 160);
                        const dur = Number(parts[5] || 60);
                        const wait = (parts[6]?.toLowerCase() === 'true');

                        $gameScreen.startFlash([r, g, b, a], dur);

                        if (wait) {
                            await this.sleep(dur);
                        }
                        break;
                    }
                    case '@if': {
                        const expr = parts.slice(1).join(" ");
                        let result = false;
                        try {
                            result = !!eval(expr);
                        } catch (e) {
                            console.error("[NrTextFile] @if 式エラー:", expr, e);
                        }
                        if (!result) {
                            while (this._index < this._lines.length) {
                                const peek = this._lines[this._index].trim().toLowerCase();
                                if (peek.startsWith("@else") || peek.startsWith("@endif")) {
                                    break;
                                }
                                this._index++;
                            }
                        }
                        break;
                    }
                    case '@else': {
                        while (this._index < this._lines.length) {
                            const peek = this._lines[this._index].trim().toLowerCase();
                            if (peek.startsWith("@endif")) {
                                break;
                            }
                            this._index++;
                        }
                        break;
                    }
                    case '@endif':
                        break;
                    case '@nrchoice': {
                        const choiceName = parts.slice(1).join(" ");
                        if (typeof SpineChoiceManager !== 'undefined' && SpineChoiceManager) {
                            try {
                                await new Promise(resolve => {
                                    SpineChoiceManager.show(choiceName);
                                    const check = () => {
                                        if (!SpineChoiceManager._active) resolve();
                                        else setTimeout(check, 30);
                                    };
                                    check();
                                });
                            } catch (e) {
                                console.error("[NrTextFile] @NrChoice エラー:", e);
                            }
                        } else {
                            console.warn("[NrTextFile] SpineChoiceManager が存在しません。@NrChoice をスキップします。");
                        }
                        break;
                    }
                    default:
                        console.warn("[NrTextFile] 未知のコマンド:", cmd);
                        break;
                }
            } catch (err) {
                console.error("[NrTextFile] コマンド処理中エラー:", cmd, err);
            }
        }

        _waitWhileLocked() {
            return new Promise(resolve => {
                const check = () => {
                    if (!(inputLockSwitchId > 0 && $gameSwitches.value(inputLockSwitchId))) {
                        resolve();
                    } else {
                        setTimeout(check, 100);
                    }
                };
                check();
            });
        }

        sleep(ms) {
            return new Promise(resolve => setTimeout(resolve, ms * 1000 / 60));
        }
    }

    
    window.NrTextFile = {
        _finished: true,

        start: async function(fileName) {
            this._finished = false;
            const path = `data/text/${fileName.replace(/\.txt$/i, "")}.txt`;
            try {
                const text = await fetch(path).then(r => {
                    if (!r.ok) throw new Error("Failed to load " + path);
                    return r.text();
                });
                const lines = text.split(/\r?\n/).map(s => s.replace(/^\uFEFF/, ""));
                const proc = new SequentialTextProcessor(lines);
                await proc.start();
            } catch (e) {
                console.error("Error loading file:", e);
            }
            this._finished = true;
        },

        command: function(fileName) {
            const interpreter = this._getInterpreter();
            if (interpreter) {
                interpreter._nrTextFileFinished = false;
                interpreter.setWaitMode("nr_text");
                return this.start(fileName).then(() => {
                    interpreter._nrTextFileFinished = true;
                });
            } else {
                return this.start(fileName);
            }
        },
        _getInterpreter: function() {
            return $gameMap._interpreter && $gameMap._interpreter.isRunning()
                ? $gameMap._interpreter
                : null;
        }
    };

    if (!Game_Interpreter.prototype._nrTextFilePatched) {
        const _updateWaitMode = Game_Interpreter.prototype.updateWaitMode;
        Game_Interpreter.prototype.updateWaitMode = function() {
            if (this._waitMode === "nr_text") {
                return !this._nrTextFileFinished;
            }
            return _updateWaitMode.call(this);
        };
        Game_Interpreter.prototype._nrTextFilePatched = true;
    }

    Window_Message.prototype.close = function() { 
        this._opening = false;
        this._closing = false;
        this.openness = 0;
    };

    const _Window_Message_isTriggered = Window_Message.prototype.isTriggered;
    Window_Message.prototype.isTriggered = function() {
        if (window.NrTextFile && $gameSwitches.value(inputLockSwitchId)) {
            return false;
        }
        return _Window_Message_isTriggered.call(this);
    };

    const _Window_Message_lineHeight = Window_Message.prototype.lineHeight;
    Window_Message.prototype.lineHeight = function() {
        const baseFontSize = this.contents.fontSize || $gameSystem.mainFontSize();
        return Math.max(_Window_Message_lineHeight.call(this), baseFontSize + 12);
    };

    const _Window_Message_terminateMessage = Window_Message.prototype.terminateMessage;
    Window_Message.prototype.terminateMessage = function() {
        _Window_Message_terminateMessage.call(this);

        if (SceneManager._scene instanceof Scene_Map) {
            const processor = window.currentSequentialTextProcessor;
            if (processor && processor._triggeredName) {
                const foundSuffix = nameSuffixList.find(cfg => cfg.Name === processor._triggeredName);
                if (foundSuffix && foundSuffix.Script) {
                    try {
                        const scriptFunction = new Function('$gameVariables', foundSuffix.Script);
                        scriptFunction($gameVariables);
                    } catch (e) {
                        console.error("Error while executing script:", e);
                    }
                }
                processor._triggeredName = null;
            }
        }
    };

    window.SequentialTextProcessor = SequentialTextProcessor;
})();
