//=============================================================================
// NrTextFile_EX.js
//=============================================================================
/*:
 * @plugindesc NrTextFile 拡張：既読装飾＋スキップ制御 v1.1.0
 * @target MZ
 *
 * @param UseCustomPrefix
 * @text 自動制御文字使用
 * @type boolean
 * @desc true: カスタム制御文字で色を付ける / false: 通常の\C[n]で色付け
 * @default true
 *
 * @param NormalColorIndex
 * @text 通常カラー番号
 * @type number
 * @desc UseCustomPrefixがfalseのときに使う\C[n]のnの値
 * @default 7
 *
 * @param CustomPrefixText
 * @text カスタム制御文字
 * @type string
 * @desc UseCustomPrefixがtrueのときに先頭に追加される制御文字列（例：\\NRC[#888888]）
 * @default \\NRC[#888888]
 *
 * @param BlockSwitchIds
 * @text 禁止スイッチIDリスト
 * @type switch[]
 * @desc いずれかのスイッチがONなら既読強調を適用しない
 * @default []
 *
 * @param SkipControlSwitchId
 * @text 既読スキップ制御スイッチID
 * @type switch
 * @desc このスイッチがONの時だけ既読スキップを許可。未設定時は常に許可。
 * @default 0
 *
 * @param AutoSkipSwitchId
 * @text 既読自動スキップ通知スイッチID
 * @type switch
 * @desc 既読メッセージの時にこのスイッチがON、未読ならOFFに自動で切り替え。
 * @default 0
 *
 * @command ClearSeen
 * @text 既読データクリア
 * @desc 既読データを削除します（永続ファイルも削除）。
 *
 * @command SetPrefix
 * @text 既読用プレフィックス設定
 * @desc ランタイムで既読用プレフィックスを変更します。
 * @arg prefix
 * @type string
 * @default
 *
 * @help
 * NrTextFile.js の既読管理を拡張し、既読行に装飾やスキップ制御を追加します。
 *
 * バージョン：
 * v1.0.0 初回
 *
 * 利用規約：
 * - プラグイン作者に無断で使用、改変、再配布は不可です。
 */

(() => {
    const PLUGIN_NAME = "NrTextFile_EX";
    const params = PluginManager.parameters(PLUGIN_NAME);

    const UseCustomPrefix = params.UseCustomPrefix === "true";
    const NormalColorIndex = Number(params.NormalColorIndex || 7);
    let CustomPrefixText = String(params.CustomPrefixText || "\\NRC[#888888]");
    const BlockSwitchIds = JSON.parse(params.BlockSwitchIds || "[]").map(Number);
    const SkipControlSwitchId = Number(params.SkipControlSwitchId || 0);
    const AutoSkipSwitchId = Number(params.AutoSkipSwitchId || 0);
    const STORAGE_KEY = "nr_text_seen_global";

    let _seenMap = {};

    const loadSeenData = async () => {
        try {
            const data = await StorageManager.loadObject(STORAGE_KEY);
            _seenMap = data && typeof data === "object" ? data : {};
        } catch {
            _seenMap = {};
        }
    };

    const saveSeenData = async () => {
        try {
            await StorageManager.saveObject(STORAGE_KEY, _seenMap);
        } catch (e) {
            console.error("[NrTextFile_EX] Save failed:", e);
        }
    };

    const removeSeenDataFile = async () => {
        try {
            await StorageManager.remove(STORAGE_KEY);
        } catch {}
    };

    loadSeenData();

    const isBlocked = () => BlockSwitchIds.some(id => $gameSwitches.value(id));
    const makeSeenKey = (file, line, text) =>
        `${file || "unknown"}::${Number.isFinite(line) ? line : 0}::${text || ""}`;
    const hasSeen = (key) => !!_seenMap[key];
    const markSeen = (key) => {
        _seenMap[key] = true;
        saveSeenData();
    };

    const injectPrefixAfterColorEscapes = (text, prefix) => {
        if (!prefix) return text;
        const m = text.match(/^([^:]+(?::\d+)?::?)/);
        const head = m ? m[1] : "";
        const body = text.slice(head.length);
        const re = /^((?:\\(?:C|NRCG?|NRSH|NROWC?|NROWS?)\[[^\]]*\])*)/;
        const colorHead = (body.match(re) || [])[1] || "";
        const rest = body.slice(colorHead.length);
        return head + colorHead + prefix + rest;
    };

    if (window.NrTextFile && typeof window.NrTextFile.start === "function") {
        const _origStart = window.NrTextFile.start;
        window.NrTextFile.start = async function(fileName) {
            this._finished = false;
            const path = `data/text/${fileName.replace(/\.txt$/i, "")}.txt`;
            try {
                const text = await fetch(path).then(r => {
                    if (!r.ok) throw new Error("Failed to load " + path);
                    return r.text();
                });
                const lines = text.split(/\r?\n/).map(s => s.replace(/^\uFEFF/, ""));
                const proc = new SequentialTextProcessor(lines);
                proc._sourceFileName = fileName.replace(/\.txt$/i, "");
                await proc.start();
            } catch (e) {
                console.error("[NrTextFile_EX] Error loading:", e);
            }
            this._finished = true;
        };
    }

    if (typeof SequentialTextProcessor !== "undefined") {
        const _origShowMessage = SequentialTextProcessor.prototype.showMessage;
        SequentialTextProcessor.prototype.showMessage = async function(line) {
            const lineIndex = typeof this._index === "number" ? Math.max(0, this._index - 1) : 0;
            const fileName = this._sourceFileName || this._file || "unknown";

            let name = null, speakerIndex = null, text = "";
            let m = line.match(/^([^:]+):(\d+):(.*)$/);
            if (m) {
                name = m[1]; speakerIndex = Number(m[2]); text = m[3];
            } else if ((m = line.match(/^([^:]+)::(.*)$/))) {
                name = m[1]; text = m[2];
            } else if ((m = line.match(/^:(\d+):(.*)$/))) {
                speakerIndex = Number(m[1]); text = m[2];
            } else {
                text = line;
            }

            const key = makeSeenKey(fileName, lineIndex, text);
            const seen = hasSeen(key);

            if (AutoSkipSwitchId > 0) {
                $gameSwitches.setValue(AutoSkipSwitchId, seen);
            }

            if (seen && SkipControlSwitchId > 0 && $gameSwitches.value(SkipControlSwitchId)) {
                $gameMessage.setScrollMode(false);
                $gameMessage.setFastForward(true);
            }

            if (!isBlocked() && seen) {
                const prefix = UseCustomPrefix ? CustomPrefixText : `\\C[${NormalColorIndex}]`;
                text = injectPrefixAfterColorEscapes(text, prefix);

                if (name && typeof speakerIndex === "number") {
                    line = `${name}:${speakerIndex}:${text}`;
                } else if (name) {
                    line = `${name}::${text}`;
                } else if (typeof speakerIndex === "number") {
                    line = `:${speakerIndex}:${text}`;
                } else {
                    line = text;
                }
            }

            await _origShowMessage.call(this, line);
            markSeen(key);
        };
    }

    PluginManager.registerCommand(PLUGIN_NAME, "ClearSeen", async () => {
        _seenMap = {};
        await removeSeenDataFile();
    });

    PluginManager.registerCommand(PLUGIN_NAME, "SetPrefix", args => {
        CustomPrefixText = String(args.prefix || "");
    });

    window.NrTextReadMark = {
        clearSeen: async () => {
            _seenMap = {};
            await removeSeenDataFile();
        },
        setPrefix: (s) => { CustomPrefixText = String(s || ""); },
        saveNow: async () => { await saveSeenData(); },
        isSeenKey: (file, index, text) => hasSeen(makeSeenKey(file, index, text))
    };
})();
