//=============================================================================
// NrTextPresetEX.js
//=============================================================================
/*:
 * @target MZ
 * @plugindesc NrTextFile 専用拡張プリセット表示 v1.0.0
 * @base NrTextFile
 * @orderAfter NrTextFile
 *
 * @param PresetList
 * @type struct<Preset>[]
 * @default []
 *
 * @help
 * NrTextPresetEX.command("ファイル名") で読み込んだ場合のみ拡張表示が有効。
 * 通常の NrTextFile.command では影響を受けません。
 * 
 * バージョン：
 * v1.0.0 初回
 *
 * 利用規約：
 * - プラグイン作者に無断で使用、改変、再配布は不可です。
 */

/*~struct~Preset:
 * @param ID
 * @type number
 * @min 1
 * @desc プリセットのID（数値）。
 * @param 配置
 * @type select
 * @option 左揃え
 * @value left
 * @option 中央揃え
 * @value center
 * @option 右揃え
 * @value right
 * @default left
 * @desc テキストの配置（左揃え、中央揃え、右揃え）。
 * @param フォントサイズ
 * @type number
 * @default 28
 * @desc フォントサイズ。
 * @param 文字色
 * @type string
 * @default #ffffff
 * @desc テキストの色（16進数カラーコード）。
 * @param アウトライン色
 * @type string
 * @default #000000
 * @desc テキストのアウトラインの色（16進数カラーコード）。
 * @param アウトラインの太さ
 * @type number
 * @default 3
 * @desc アウトラインの太さ。
 * @param ウィンドウスキン
 * @type file
 * @dir img/system
 * @default Window
 * @desc 使用するウィンドウスキンのファイル名。
 * @param 不透明度
 * @type number
 * @default 255
 * @desc ウィンドウの不透明度（0〜255）。
 * @param テール画像
 * @type file
 * @dir img/system
 * @desc テール画像のファイル名。
 * @param テール位置
 * @type select
 * @option 右上
 * @value rt
 * @option 右下
 * @value rb
 * @option 中央上
 * @value ct
 * @option 中央下
 * @value cb
 * @option 左上
 * @value lt
 * @option 左下
 * @value lb
 * @default cb
 * @desc テールの位置（右上、右下、中央上、中央下、左上、左下）。
 * @param テールX座標
 * @type string
 * @default 0
 * @desc テールのX座標オフセット（負の値も入力可能）。
 * @param テールY座標
 * @type string
 * @default 0
 * @desc テールのY座標オフセット（負の値も入力可能）。
 * @param 入力待ち
 * @type boolean
 * @default true
 * @desc 入力待ちを有効にするかどうか。
 * @param 表示X座標
 * @type string
 * @default 0
 * @desc ウィンドウの表示位置のX座標（負の値も入力可能）。
 * @param 表示Y座標
 * @type string
 * @default 0
 * @desc ウィンドウの表示位置のY座標（負の値も入力可能）。
 * @param 追加テキスト
 * @type string
 * @default 
 * @desc プリセットごとにNrTextに渡す前に追加するテキスト。
 */

(() => {
    "use strict";
    const PN = "NrTextPresetEX";
    const params = PluginManager.parameters(PN);
    const PresetList = JSON.parse(params["PresetList"] || "[]").map(s => JSON.parse(s));
    const nrParams = PluginManager.parameters("NrTextFile") || {};
    const PrefixText = nrParams["PrefixText"] || "";
    const NameSuffixList = (nrParams["NameSuffixList"] ? JSON.parse(nrParams["NameSuffixList"]) : []).map(s => JSON.parse(s));
    const NameColorList = (nrParams["NameColorList"] ? JSON.parse(nrParams["NameColorList"]) : []).map(s => JSON.parse(s));

    function expandEscape(text) {
        if (!text) return "";
        return text.replace(/\\V\[(\d+)\]/gi, (_, v) => $gameVariables.value(Number(v)))
                   .replace(/\\N\[(\d+)\]/gi, (_, n) => {
                       const actor = $gameActors.actor(Number(n));
                       return actor ? actor.name() : "";
                   })
                   .replace(/\\\\/g, "\\")
    }

    function findPreset(id) {
        const preset = PresetList.find(p => Number(p.ID) === Number(id)) || null;
        return preset;
    }

    class Window_NrPreset extends Window_Base {
        constructor(rect) {
            super(rect);
            this.openness = 0;
            this._preset = null;
            this._name = null;
            this._text = "";
            this._tail = null;
        }

        setup(preset, name, text) {
            this._preset = preset;
            this._name = name;

            const additionalText = this._preset["追加テキスト"] || "";
            this._text = additionalText + text;

            this.x = parseInt(this._preset["表示X座標"] || 0, 10);
            this.y = parseInt(this._preset["表示Y座標"] || 0, 10);

            this.updateAppearance();
            this.updateSize();
            this.refresh();
        }

        refresh() {
            this.contents.clear();
            const fontSize = Number(this._preset["フォントサイズ"] || 28);
            this.contents.fontSize = fontSize;
            const outlineWidth = Number(this._preset["アウトラインの太さ"] ?? 0);
            const outlineColor = this._preset["アウトライン色"] ?? "#000000";
            if (outlineWidth === 0 || outlineColor === "transparent" || outlineColor === "none" || outlineColor === "off" || outlineColor === "rgba(0,0,0,0)") {
                this.contents.outlineWidth = 0;
                this.contents.outlineColor = "rgba(0,0,0,0)";
            } else {
                this.contents.outlineWidth = outlineWidth;
                this.contents.outlineColor = outlineColor;
            }
            let y = this.padding / 2;
            let maxLineWidth = 0;
            let totalLines = 0;

            if (this._name) {
                const nameWidth = this.textWidthEx(this._name);
                maxLineWidth = Math.max(maxLineWidth, nameWidth);
                totalLines += 1;
                this.drawTextEx(this._name, this.padding, y, this.contents.width - this.padding * 2);
                y += this.lineHeight() + 6;
            }

            const align = this._preset["配置"] || "left";
            const processedText = this.convertEscapeCharacters(this._text);
            processedText.split(/\n/).forEach(line => {
                this.contents.fontSize = fontSize;
                if (outlineWidth === 0 || outlineColor === "transparent" || outlineColor === "none" || outlineColor === "off" || outlineColor === "rgba(0,0,0,0)") {
                    this.contents.outlineWidth = 0;
                    this.contents.outlineColor = "rgba(0,0,0,0)";
                } else {
                    this.contents.outlineWidth = outlineWidth;
                    this.contents.outlineColor = outlineColor;
                }
                const lineWidth = this.textWidthEx(line);
                maxLineWidth = Math.max(maxLineWidth, lineWidth);
                totalLines += 1;

                if (align === "center") {
                    this.drawTextEx(line, Math.max((this.contents.width - lineWidth) / 2, 0), y, lineWidth);
                } else if (align === "right") {
                    this.drawTextEx(line, Math.max(this.contents.width - lineWidth - this.padding, 0), y, lineWidth);
                } else {
                    this.drawTextEx(line, Math.max(this.padding, 0), y, lineWidth);
                }
                y += this.lineHeight();
            });

            this.width = Math.ceil(maxLineWidth + this.padding * 2 + fontSize / 2);
            this.height = Math.ceil(totalLines * this.lineHeight() + this.padding + 16);
        }

        updateSize() {
            const padding = this.padding * 2;
            const fontSize = Number(this._preset["フォントサイズ"] || 28);
            let maxLineWidth = 0;
            let totalLines = 0;

            if (this._name) {
                const nameWidth = this.textWidthEx(this._name);
                maxLineWidth = Math.max(maxLineWidth, nameWidth);
                totalLines += 1;
            }

            const processedText = this.convertEscapeCharacters(this._text);
            processedText.split(/\n/).forEach(line => {
                const lineWidth = this.textWidthEx(line);
                maxLineWidth = Math.max(maxLineWidth, lineWidth);
                totalLines += 1;
            });

            this.width = Math.ceil(maxLineWidth + padding + fontSize);
            this.height = Math.ceil(totalLines * this.lineHeight() + padding);
            this.createContents();
        }

        textWidthEx(text) {
            if (!text) return 0;
            
            const tempRect = new Rectangle(0, 0, Graphics.width, this.lineHeight());
            const tempWindow = new Window_Base(tempRect);
            tempWindow.contents.fontSize = Number(this._preset["フォントサイズ"] || 28);
            tempWindow.contents.outlineWidth = Number(this._preset["アウトラインの太さ"] || 3);
            
            const textState = tempWindow.createTextState(text, 0, 0, Graphics.width);
            tempWindow.processAllText(textState);
            
            return textState.outputWidth || tempWindow.textWidth(tempWindow.convertEscapeCharacters(text));
        }

        updateAppearance() {
            const windowSkinKey = "ウィンドウスキン";
            if (this._preset[windowSkinKey]) {
                const skin = ImageManager.loadSystem(this._preset[windowSkinKey]);
                skin.addLoadListener(() => {
                    this.windowskin = skin;
                    this.refresh();
                });
                skin.onerror = () => {};
            }
                const frameOpacity = Number(this._preset["枠透明度"] ?? this._preset["不透明度"] ?? 255);
                const backOpacity = Number(this._preset["中身透明度"] ?? this._preset["不透明度"] ?? 255);
                this.opacity = frameOpacity;
                this.backOpacity = backOpacity;
                this.contentsOpacity = 255;
                if (this._backgroundSprite) {
                    this._backgroundSprite.opacity = backOpacity;
                }
                // フォント・アウトライン設定
                const fontSize = Number(this._preset["フォントサイズ"] || 28);
                this.contents.fontSize = fontSize;
                const outlineWidth = Number(this._preset["アウトラインの太さ"] ?? 0);
                const outlineColor = this._preset["アウトライン色"] ?? "#000000";
                if (outlineWidth === 0 || outlineColor === "transparent" || outlineColor === "none" || outlineColor === "off" || outlineColor === "rgba(0,0,0,0)") {
                    this.contents.outlineWidth = 0;
                    this.contents.outlineColor = "rgba(0,0,0,0)";
                } else {
                    this.contents.outlineWidth = outlineWidth;
                    this.contents.outlineColor = outlineColor;
                }
                this.refresh();
            const tailImageKey = "テール画像";
            if (this._preset[tailImageKey]) {
                const tailBitmap = ImageManager.loadSystem(this._preset[tailImageKey]);
                tailBitmap.addLoadListener(() => {
                    this._tail = new Sprite(tailBitmap);
                    this._tail.x = Number(this._preset["テールX座標"] || 0);
                    this._tail.y = Number(this._preset["テールY座標"] || 0);
                    this.addChild(this._tail);
                });
                tailBitmap.onerror = () => {};
            }
        }

        updateTail() {
            if (this._tail) {
                try {
                    this.removeChild(this._tail);
                } catch (e) {}
                this._tail = null;
            }
            if (!this._preset.TailImage) return;
            const bmp = ImageManager.loadSystem(this._preset.TailImage);
            bmp.addLoadListener(() => {
                const sp = new Sprite(bmp);
                const pos = this._preset.TailPos || "cb";
                const tx = Number(this._preset.TailX || 0);
                const ty = Number(this._preset.TailY || 0);
                let px = 0,
                    py = 0;
                switch (pos) {
                    case "rt":
                        px = this.x + this.width - sp.width + tx;
                        py = this.y + ty;
                        break;
                    case "rb":
                        px = this.x + this.width - sp.width + tx;
                        py = this.y + this.height - sp.height + ty;
                        break;
                    case "ct":
                        px = this.x + (this.width - sp.width) / 2 + tx;
                        py = this.y + ty;
                        break;
                    case "cb":
                        px = this.x + (this.width - sp.width) / 2 + tx;
                        py = this.y + this.height - sp.height + ty;
                        break;
                    case "lt":
                        px = this.x + tx;
                        py = this.y + ty;
                        break;
                    case "lb":
                        px = this.x + tx;
                        py = this.y + this.height - sp.height + ty;
                        break;
                }
                sp.x = px - this.x;
                sp.y = py - this.y;
                this._tail = sp;
                this.addChild(this._tail);
            });
            bmp.addErrorListener(() => {
                console.error(`[NrTextPresetEX] Failed to load tail image: ${this._preset.TailImage}`);
            });
        }

        drawTextEx(text, x, y) {
            const presetFontSize = Number(this._preset["フォントサイズ"] || 28);
            const textState = this.createTextState(text, x, y, this.contents.width);
            this.contents.fontSize = presetFontSize;

            this.processAllText(textState);
            return textState.outputWidth;
        }

        close() {
            if (this._tail) {
                try {
                    this.removeChild(this._tail);
                } catch (e) {}
                this._tail = null;
            }
            super.close();
        }
    }

    function ensurePresetWindow() {
        const scene = SceneManager._scene;
        if (!scene._nrPresetWindow) {
            const rect = new Rectangle(0, 0, Graphics.width, Graphics.height);
            scene._nrPresetWindow = new Window_NrPreset(rect);
            scene.addWindow(scene._nrPresetWindow);
        }
        return scene._nrPresetWindow;
    }

    function patch() {
        if (!SequentialTextProcessor || SequentialTextProcessor._nrPresetPatched) return;
        const _orig = SequentialTextProcessor.prototype.showMessage;
        SequentialTextProcessor.prototype.showMessage = async function(line) {
            if (!this._nrPresetMode) return _orig.call(this, line);
            let name = null, pid = null, text = line;
            let m = line.match(/^(.+?):(\d+):(.*)$/s);
            if (m) {
                name = m[1];
                pid = Number(m[2]);
                text = m[3];
            } else if ((m = line.match(/^:(\d+):(.*)$/s))) {
                pid = Number(m[1]);
                text = m[2];
            } else if ((m = line.match(/^(.+?)::(.*)$/s))) {
                name = m[1];
                text = m[2];
            }

            text = PrefixText + text;

            if (name) {
                const expandedName = expandEscape(name);
                const foundSuffix = NameSuffixList.find(cfg => expandEscape(cfg.Name || "") === expandedName);
                if (foundSuffix && foundSuffix.Suffix) text = foundSuffix.Suffix + text;
                const foundColor = NameColorList.find(cfg => expandEscape(cfg.Name || "") === expandedName);
                if (foundColor) {
                    const colorId = Number(foundColor.ColorId || 0);
                    name = "\\C[" + colorId + "]\\FB" + name + "\\C[0]";
                }
            }

            const preset = findPreset(pid);
            if (preset) {
                if (typeof this.sleep === 'function') {
                    await this.sleep(5);
                } else {
                    await new Promise(r => setTimeout(r, 5 * 1000 / 60));
                }
                const presetWindow = ensurePresetWindow();
                presetWindow.setup(preset, name, text);
                presetWindow.open();
                if (!text || text.trim() === "") {
                    console.warn("[NrTextPresetEX] テキストが空です。プリセットID:", pid, "name:", name);
                }
                const waitInput = preset["入力待ち"] !== "false" && preset["入力待ち"] !== false;
                if (waitInput) {
                    await new Promise(resolve => {
                        const waitHandler = () => {
                            if (Input.isTriggered('ok') || Input.isTriggered('cancel') || TouchInput.isTriggered()) {
                                resolve();
                                return;
                            }
                            requestAnimationFrame(waitHandler);
                        };
                        waitHandler();
                    });
                }
                presetWindow.close();
            }
        };
        SequentialTextProcessor._nrPresetPatched = true;
    }

    if (typeof SequentialTextProcessor !== "undefined") {
        patch();
    } else {
        const _Scene_Boot_start = Scene_Boot.prototype.start;
        Scene_Boot.prototype.start = function() { _Scene_Boot_start.call(this); patch(); };
    }

    window.NrTextPresetEX = {
        command: function(file) {
            const it = $gameMap._interpreter;
            if (it) {
                it._nrPresetMode = true;
                it.setWaitMode("nr_textpreset");
                return this.start(file).then(() => { it._nrPresetMode = false; it._nrTextFileFinished = true; });
            } else {
                return this.start(file);
            }
        },
        start: async function(file) {
            const path = `data/text/${file.replace(/\.txt$/i, "")}.txt`;
            const text = await fetch(path).then(r => r.text());
            const lines = text.split(/\r?\n/).map(s => s.replace(/^\uFEFF/, ""));
            const proc = new SequentialTextProcessor(lines);
            proc._nrPresetMode = true; await proc.start();
        }
    };

    const _uwm = Game_Interpreter.prototype.updateWaitMode;
    Game_Interpreter.prototype.updateWaitMode = function() {
        if (this._waitMode === "nr_textpreset") return !this._nrTextFileFinished;
        return _uwm.call(this);
    };
})();
